﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;
using System.Linq;
using LECore;
using LECore.Structures;
using LECore.Util;
using LECore.Structures.Core;

namespace LayoutEditor.Forms.ToolWindows.common
{
    class MultiSelectTreeNodeMgr
    {
        /// <summary>
        /// 規定のTreeViewの振る舞いを模倣するために、
        /// マウスダウンでTreeViewの選択アイテムを強制的に内部管理ノードと
        /// 一致させる必要があります。
        ///
        /// しかしマウスダウンでTreeViewの選択ノードを変更すると、マウスア
        /// ップイベントに反応してラベル編集がスタートしてしまう。
        /// そのため、マウスダウンで一時的に選択ノードをnullに設定後、
        /// マウスアップで一致動作を行う。
        ///
        /// </summary>

        #region イベント
        public delegate void OnChangeSelecetItemByMouseHandler( TreeNode[] nodeSet );
        public delegate bool OnCheckNodeContentSelectedHandler( TreeNode node );
        public delegate void OnUpdateNodeApperanceHandler( List<TreeNode> nodeSet );

        public event OnChangeSelecetItemByMouseHandler ChangeSelecetItemByMouse = null;
        public event OnCheckNodeContentSelectedHandler CheckNodeContentSelected = null;

        public event OnUpdateNodeApperanceHandler SetNodeApperanceSelected = null;
        public event OnUpdateNodeApperanceHandler ResetNodeApperanceSelected = null;

        public event EventHandler<KeyEventArgs> TreeViewKeyDown = null;

        #endregion イベント

        #region フィールド
        readonly TreeView _targetTreeView = null;
        readonly List<TreeNode> _oldSelectedSet = new List<TreeNode>();
        readonly List<TreeNode> _selectedNodeSet = new List<TreeNode>();
        bool _bUpdating = false;
        bool _bSelectRecursive = false;
        Keys _ModifierKeys = Keys.None;
        TreeNode _mouseDownNode = null;

        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// 最初の選択ノードを取得します。
        /// </summary>
        public TreeNode FirstSelectedNode
        {
            get
            {
                if( _selectedNodeSet.Count > 0 )
                {
                    return _selectedNodeSet[_selectedNodeSet.Count - 1];
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// 選択ノードセット
        /// </summary>
        public TreeNode[] SelectedNodeSet
        {
            get { return _selectedNodeSet.ToArray(); }
        }

        /// <summary>
        /// 更新中か
        /// </summary>
        public bool Updating { get { return _bUpdating; } }

        /// <summary>
        /// 選択セットが空か
        /// </summary>
        public bool Empty { get { return _selectedNodeSet.Count <= 0; } }

        /// <summary>
        /// 階層を再帰的に選択するか？
        /// </summary>
        public bool SelectRecursive
        {
            get { return _bSelectRecursive; }
            set { _bSelectRecursive = value; }
        }

        #endregion プロパティ

        // ペイン参照から、ノードを選択セットに追加するメソッド
        // OnSelectedSetChangedで呼び出して、選択セットを更新したい。

        /// <summary>
        /// リセット
        /// 更新中でなければ、選択セットをリセットします。
        /// </summary>
        public void Reset()
        {
            if( !Updating )
            {
                _selectedNodeSet.Clear();
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MultiSelectTreeNodeMgr( TreeView treeView )
        {
            _targetTreeView = treeView;

            // イベントの追加
            _targetTreeView.MouseDown += Event_TvwHierarchy_MouseDown;
            _targetTreeView.MouseUp += Event_TargetTreeView_MouseUp;

            _targetTreeView.BeforeExpand += Event_TargetTreeView_BeforeExpand;
            _targetTreeView.BeforeCollapse += Event_TargetTreeView_BeforeCollapse;
            _targetTreeView.BeforeSelect += Event_TargetTreeView_BeforeSelect;
            _targetTreeView.KeyDown += OnTreeViewKeyDown;
            _targetTreeView.BeforeLabelEdit += Event_TargetTreeView_BeforeLabelEdit;
        }



        /// <summary>
        /// 選択ノードを追加します。
        /// 表示状態の更新をおこないます。
        /// </summary>
        public void AddSelectedNode( List<TreeNode> nodeSet )
        {
            List<TreeNode> newSet = nodeSet.FindAll( delegate( TreeNode node )
            {
                return !_selectedNodeSet.Contains( node );
            } );

            _selectedNodeSet.AddRange( newSet );
        }

        /// <summary>
        ///
        /// </summary>
        void SetFirstSelectedNodeToTargetTreeView_()
        {
            // TreeViewメンバの選択ノードを更新します。
            if( _targetTreeView.SelectedNode != FirstSelectedNode )
            {
                if( FirstSelectedNode != null && FirstSelectedNode.IsVisible )
                {
                    _targetTreeView.SelectedNode = FirstSelectedNode;
                }
                else
                {
                    _targetTreeView.SelectedNode = null;
                }
            }
        }

        /// <summary>
        /// ノードを選択します。
        /// </summary>
        public void SetActiveNode(TreeNode[] nodeSet)
        {
            BeginSelectedSetUpdate_();

            _selectedNodeSet.Clear();
            _selectedNodeSet.AddRange(nodeSet);

            // ノードを展開します
            foreach (TreeNode node in nodeSet)
            {
                TreeNode parent = node.Parent;
                while(parent != null)
                {
                    parent.Expand();
                    parent = parent.Parent;
                }
            }

            EndSelectedSetUpdate_();
        }

        /// <summary>
        /// 選択アイテムを更新します。
        /// </summary>
        public void UpdateSelectedNodeByNodeContent()
        {
            // 正しくハンドラが設定されている必要がある。
            if( ResetNodeApperanceSelected == null ||
                SetNodeApperanceSelected == null )
            {
                return;
            }

            _targetTreeView.BeginUpdate();

            // 更新前の選択セットに対して、リセット処理を実施します。
            ResetNodeApperanceSelected( _selectedNodeSet );


            // リセット
            Reset();


            // 選択セットを更新します。
            if( CheckNodeContentSelected != null )
            {
                int countBeforeUpdate = _selectedNodeSet.Count;

                foreach( TreeNode treeNode in _targetTreeView.Nodes )
                {
                    UpdateSelectedNodeByNodeContent_( treeNode );
                }

                // 選択セット変更の通知は行いません。
            }

            // 更新後の選択セットに対して、セット処理を行います。
            SetNodeApperanceSelected( _selectedNodeSet );

            // TreeViewメンバの選択ノードを更新します。
            SetFirstSelectedNodeToTargetTreeView_();

            _targetTreeView.EndUpdate();
        }

        /// <summary>
        /// 選択アイテムを指定したペインセット順にソートします。
        /// </summary>
        public void SortSelectedSet(IPane[] paneSet)
        {
            if (paneSet == null || paneSet.Count() == 0)
            {
                return;
            }

            List<TreeNode> sortedTreeNode = new List<TreeNode>();
            var rootNode = _selectedNodeSet.Where(node => node.Parent == null).FirstOrDefault();
            if (rootNode != null)
            {
                sortedTreeNode.Add(rootNode);
            }

            foreach (IPane pane in paneSet)
            {
                foreach (TreeNode node in _selectedNodeSet)
                {
                    IPane selectedPane = node.Tag as IPane;
                    if (pane == selectedPane)
                    {
                        sortedTreeNode.Add(node);
                        break;
                    }
                }
            }

            _selectedNodeSet.Clear();
            _selectedNodeSet.AddRange(sortedTreeNode);
        }

        /// <summary>
        /// アトリビュートが紐付けられているノードを返します。
        /// </summary>
        public IEnumerable<TreeNode> GetRelativeNodes(IAnmAttribute[] attrSet)
        {
            List<TreeNode> relativeNodes = new List<TreeNode>();

            // ツリーノードを並列化します
            List<TreeNode> hierarchyNodes = new List<TreeNode>();
            foreach (TreeNode node in _targetTreeView.Nodes)
            {
                var nodes = GenericUtil.GetNodeRecursively(node, (x) => x.Nodes.OfType<TreeNode>());
                hierarchyNodes.AddRange(nodes);
            }

            foreach (IAnmAttribute attr in attrSet)
            {
                foreach (TreeNode node in hierarchyNodes)
                {
                    IAnmAttribute target = node?.Tag as IAnmAttribute;
                    if (target != null)
                    {
                        if (target == attr)
                        {
                            relativeNodes.Add(node);

                            // 子がいる場合はそれらも含めます
                            for (int index = 0; index < attr.NumSubAttribute; index++)
                            {
                                var childs = GetRelativeNodes(new IAnmAttribute[] { attr.FindSubAttributeByIdx(index) });
                                relativeNodes.AddRange(childs);
                            }
                        }
                    }
                }
            }

            return relativeNodes;
        }

        #region private メソッド


        /// <summary>
        /// 全てのノードについて、Content の状態を判定し、
        /// 選択リストに加えます。
        /// </summary>
        void UpdateSelectedNodeByNodeContent_( TreeNode treeNode )
        {
            Debug.Assert( CheckNodeContentSelected != null );

            // if( !_SelectedNodeSet.Contains( treeNode ) )
            {
                if( CheckNodeContentSelected( treeNode ) )
                {
                    // _SelectedNodeSet.Add( treeNode );
                    //AddSelectNode_( treeNode );
                    _selectedNodeSet.Add( treeNode );
                }
                else
                {
                    //RemoveSelectNode_( treeNode );
                    _selectedNodeSet.Remove( treeNode );
                }
            }

            foreach( TreeNode child in treeNode.Nodes )
            {
                UpdateSelectedNodeByNodeContent_( child );
            }
        }

        /// <summary>
        /// 選択アイテムの変更イベントを通知します。
        /// </summary>
        void NotifySelectedItemSetChanged_()
        {
            if( ChangeSelecetItemByMouse != null )
            {
                ChangeSelecetItemByMouse( SelectedNodeSet );
            }
        }

        /// <summary>
        /// ノードが見える位置に存在するか調査します。
        ///
        /// IsVisibleを利用すると、スクロールして、TreeViewの領域外のノード
        /// が不可視と判定されてしまうため、ここでの求める判定にならない。
        /// </summary>
        static bool IsNodeExposed_( TreeNode node )
        {
            node = node.Parent;
            while( node != null )
            {
                if( !node.IsExpanded )
                {
                    return false;
                }
                node = node.Parent;
            }

            // 一番親階層まで、開いていた。
            return true;
        }

        /// <summary>
        /// 範囲内のTreeNodeを選択します。
        /// </summary>
        void SelectTreeNodeRange_( TreeNode node, int topBound, int bottomBound )
        {
            // 選択する
            if( topBound <= node.Bounds.Y &&
                node.Bounds.Y <= bottomBound &&
                IsNodeExposed_( node ) )
            {
                if( !_selectedNodeSet.Contains( node ) )
                {
                    _selectedNodeSet.Add( node );
                }
            }
        }


        /// <summary>
        ///
        /// </summary>
        void AddSelectNode_( TreeNode node )
        {
            _selectedNodeSet.Add( node );

            if( _bSelectRecursive )
            {
                foreach( TreeNode child in node.Nodes )
                {
                    AddSelectNode_( child );
                }
            }
        }

        /// <summary>
        /// ノードの選択状態を反転します。
        /// Ctrkキー押下状態での振舞いにしようされます。
        /// </summary>
        void InvertNodeSelection_( TreeNode node )
        {
            // Ctrl
            if( _selectedNodeSet.Contains( node ) )
            {
                _selectedNodeSet.Remove( node );
            }
            else
            {
                _selectedNodeSet.Add( node );
            }

            if( _bSelectRecursive )
            {
                foreach( TreeNode child in node.Nodes )
                {
                    InvertNodeSelection_( child );
                }
            }
        }

        /// <summary>
        /// 選択セットが変更されたか?
        /// </summary>
        static bool IsSlectedSetChanged_( List<TreeNode> oldSet, List<TreeNode> newSet )
        {
            if( oldSet.Count != newSet.Count )
            {
                return true;
            }

            foreach( TreeNode node in oldSet )
            {
                if( !newSet.Contains( node ) )
                {
                    return true;
                }
            }
            return false;
        }

        #region イベントハンドラ
        /// <summary>
        /// キーダウン
        /// </summary>
        void OnTreeViewKeyDown(object sender, KeyEventArgs e)
        {
            // 選択ノードが無い場合、
            if (FirstSelectedNode == null)
            {
                // ルートノードを選択します。
                if (_targetTreeView.Nodes.Count > 0)
                {
                    _selectedNodeSet.Add(_targetTreeView.Nodes[0]);
                }
                return;
            }

            if (TreeViewKeyDown != null)
            {
                TreeViewKeyDown(sender, e);
                return;
            }

            switch ( e.KeyData )
            {
                case Keys.Left:
                {
                    LeftKeyAction();
                    break;
                }
                case Keys.Right:
                {
                    RightKeyAction();
                    break;
                }
                case Keys.Down:
                {
                    DownKeyAction();
                    break;
                }
                case Keys.Up:
                {
                    UpKeyAction();
                    break;
                }
            }
        }

        /// <summary>
        /// UPキー押下時の処理
        /// </summary>
        public void UpKeyAction()
        {
            TreeNode currenNode = FirstSelectedNode;
            TreeNode nextNode = FirstSelectedNode;

            if (currenNode.PrevNode != null)
            {
                // 前がいれば移動する
                nextNode = currenNode.PrevNode;
            }
            else if (currenNode.Parent != null)
            {
                // 親がいれば移動するってえ
                nextNode = currenNode.Parent;
            }


            BeginSelectedSetUpdate_();
            _selectedNodeSet.Clear();
            _selectedNodeSet.Add(nextNode);
            EndSelectedSetUpdate_();
        }

        /// <summary>
        /// DOWNキー押下時の処理
        /// </summary>
        public void DownKeyAction()
        {
            TreeNode currenNode = FirstSelectedNode;
            TreeNode nextNode = currenNode;

            // 子供があり開いていれば
            if (currenNode.Nodes.Count > 0 && currenNode.IsExpanded)
            {
                // 子供に移動する
                nextNode = currenNode.Nodes[0];
            }
            else if (currenNode.NextNode != null)
            {
                // 子供が無ければ次に移動する
                nextNode = currenNode.NextNode;
            }
            else
            {
                // いづれも無理ならば親をたどって、次の兄弟へと移動する
                TreeNode parent = currenNode.Parent;
                while (parent != null)
                {
                    if (parent.NextNode != null)
                    {
                        nextNode = parent.NextNode;
                        break;
                    }
                    parent = parent.Parent;
                }
            }

            BeginSelectedSetUpdate_();
            _selectedNodeSet.Clear();
            _selectedNodeSet.Add(nextNode);
            EndSelectedSetUpdate_();
        }

        /// <summary>
        /// LEFTキー押下時の処理
        /// </summary>
        public void LeftKeyAction()
        {
            TreeNode currenNode = FirstSelectedNode;

            // 親がいれば、移動する
            if (currenNode.Parent != null)
            {
                if (currenNode.IsExpanded)
                {
                    // 自身が展開されていれば、閉じます
                    currenNode.Collapse();
                }
                else
                {
                    BeginSelectedSetUpdate_();
                    _selectedNodeSet.Clear();
                    _selectedNodeSet.Add(currenNode.Parent);
                    EndSelectedSetUpdate_();
                }
            }
        }

        /// <summary>
        /// RIGHTキー押下時の処理
        /// </summary>
        public void RightKeyAction()
        {
            TreeNode currenNode = FirstSelectedNode;

            // 子供がいれば
            if (currenNode.Nodes.Count != 0)
            {
                if (!currenNode.IsExpanded)
                {
                    // 自身が閉じていれば展開する
                    currenNode.Expand();
                }
                else
                {
                    // 開いていれば子供に移動する
                    BeginSelectedSetUpdate_();
                    _selectedNodeSet.Clear();
                    _selectedNodeSet.Add(currenNode.Nodes[0]);
                    EndSelectedSetUpdate_();
                }
            }
        }

        /// <summary>
        /// 選択セット変更前処理
        /// </summary>
        void BeginSelectedSetUpdate_()
        {
            _bUpdating = true;
            _oldSelectedSet.Clear();
            _oldSelectedSet.AddRange( _selectedNodeSet );
        }

        /// <summary>
        /// 選択セット変更後処理
        /// </summary>
        bool EndSelectedSetUpdate_()
        {
            bool bUpdated = false;
            if( IsSlectedSetChanged_( _oldSelectedSet, _selectedNodeSet ) )
            {
                NotifySelectedItemSetChanged_();

                // 選択解除されたノードを求め、表示リセットします。
                foreach( TreeNode selectedNode in _selectedNodeSet )
                {
                    _oldSelectedSet.Remove( selectedNode );
                }

                // 表示の更新
                _targetTreeView.BeginUpdate();

                ResetNodeApperanceSelected( _oldSelectedSet );
                SetNodeApperanceSelected( _selectedNodeSet );

                _targetTreeView.EndUpdate();

                bUpdated = true;
            }

            _bUpdating = false;

            return bUpdated;
        }

        /// <summary>
        /// マウスダウン
        /// </summary>
        void Event_TvwHierarchy_MouseDown( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            _ModifierKeys = Keys.None;
            _mouseDownNode = _targetTreeView.GetNodeAt(new Point(e.X, e.Y));

            TreeNode node = _mouseDownNode;
            if( node == null )
            {
                return;
            }

            // 更新前のセットを保存しておく。
            BeginSelectedSetUpdate_();

            // 選択ノードに登録
            if( e.Button == MouseButtons.Left )
            {
                if( ( Control.ModifierKeys & Keys.Shift ) != 0 )
                {
                    _ModifierKeys = Keys.Shift;

                    if( _selectedNodeSet.Count > 0 )
                    {
                        // 検索開始、終了（上と下）ノードを決定します
                        TreeNode lastNode = _selectedNodeSet[0];

                        int topY = Math.Min( lastNode.Bounds.Y, node.Bounds.Y );
                        int bottomY = Math.Max( lastNode.Bounds.Y, node.Bounds.Y );

                        _selectedNodeSet.Clear();
                        // _selectedNodeSet.Add( lastNode );

                        // ツリーノードを並列化します
                        var hierarchyNodes = GenericUtil.GetNodeRecursively(_targetTreeView.Nodes[0], (x) => x.Nodes.OfType<TreeNode>());

                        // 範囲選択の上下を決定します
                        if ( lastNode.Bounds.Y > node.Bounds.Y )
                        {
                            hierarchyNodes = hierarchyNodes.Reverse();
                        }

                        // ノードを選択します
                        foreach ( TreeNode tn in hierarchyNodes )
                        {
                            SelectTreeNodeRange_( tn, topY, bottomY );
                        }
                    }
                }
                else if( ( Control.ModifierKeys & Keys.Control ) != 0 )
                {
                    // Ctrl
                    _ModifierKeys = Keys.Control;
                }
                else
                {
                    // 通常
                    if( e.Clicks != 2 && !_selectedNodeSet.Contains( node ) )
                    {
                        _selectedNodeSet.Clear();
                        AddSelectNode_( node );
                    }
                }
            }
            else
            {
                if( ( e.Clicks != 2 ) && !_selectedNodeSet.Contains( node ) )
                {
                    _selectedNodeSet.Clear();
                    AddSelectNode_( node );
                }
            }

            // 選択ノード上のマウスダウンでなければ、
            // 一旦選択を解除します。
            // 選択ノードはマウスアップで再度設定されます。
            //
            // マウス押し状態での規定の選択ノード描画効果を抑止するための処理です。
            if( _targetTreeView.SelectedNode != node )
            {
                _targetTreeView.SelectedNode = null;
            }

            // 終了処理
            EndSelectedSetUpdate_();
        }

        /// <summary>
        /// BeforeCollapse
        /// </summary>
        void Event_TargetTreeView_BeforeCollapse( object sender, TreeViewCancelEventArgs e )
        {
            // マウスによる操作は
            if( e.Action == TreeViewAction.ByMouse )
            {
                // Ctrl や Shift押し時はキャンセルする
                if( ( Control.ModifierKeys & Keys.Control ) != 0 ||
                    ( Control.ModifierKeys & Keys.Shift ) != 0 )
                {
                    e.Cancel = true;
                }
            }

        }

        /// <summary>
        /// BeforeExpand
        /// </summary>
        void Event_TargetTreeView_BeforeExpand( object sender, TreeViewCancelEventArgs e )
        {
            if( e.Action == TreeViewAction.ByMouse )
            {
                if( ( Control.ModifierKeys & Keys.Control ) != 0 ||
                    ( Control.ModifierKeys & Keys.Shift ) != 0 )
                {
                    e.Cancel = true;
                }
            }
        }

        /// <summary>
        /// マウスアップイベント
        /// </summary>
        void Event_TargetTreeView_MouseUp( object sender, MouseEventArgs e )
        {
            TreeNode node = _targetTreeView.GetNodeAt( new Point( e.X, e.Y ) );

            bool mouseUpAtDownNode = object.ReferenceEquals(_mouseDownNode, node);
            if (node != null && e.Button == MouseButtons.Left && mouseUpAtDownNode)
            {
                BeginSelectedSetUpdate_();
                if( _ModifierKeys == Keys.None )
                {
                    // 更新前のセットを保存しておく。
                    _selectedNodeSet.Clear();

                    AddSelectNode_( node );
                    // 終了処理
                }
                else if( _ModifierKeys == Keys.Control )
                {
                    InvertNodeSelection_( node );
                }

                EndSelectedSetUpdate_();
            }
            SetFirstSelectedNodeToTargetTreeView_();

            _mouseDownNode = null;
            _ModifierKeys = Keys.None;
        }

        /// <summary>
        /// 選択ノードを独自に管理するため、
        /// ユーザ操作による変更はキャンセルし、内部選択ノードを設定する。
        ///
        /// _targetTreeView.SelectedNode は内部で 選択変更イベントを生成し、
        /// BeforeSelect が呼ばれる点に注意が必要である。(e.Action で判定できる)
        ///
        ///
        /// </summary>
        void Event_TargetTreeView_BeforeSelect( object sender, TreeViewCancelEventArgs e )
        {
            // ユーザ操作による呼び出し
            if( e.Action != TreeViewAction.Unknown )
            {
                // 操作をキャンセルし、内部選択ノードに一致させる。
                e.Cancel = true;
                SetFirstSelectedNodeToTargetTreeView_();
            }
            else
            {
                // クラス内部からの呼び出し
                // 無限ループを避けるため、変更対象が、対応させるべき内部
                // ノードと異なる場合のみ設定します。
                if( e.Node != FirstSelectedNode )
                {
                    SetFirstSelectedNodeToTargetTreeView_();
                }
            }
        }

        /// <summary>
        /// ラベル編集前ハンドラ
        /// </summary>
        private void Event_TargetTreeView_BeforeLabelEdit( object sender, NodeLabelEditEventArgs e )
        {
            if( ( Control.ModifierKeys & Keys.Control ) != 0 ||
                ( Control.ModifierKeys & Keys.Shift ) != 0 )
            {
                e.CancelEdit = true;
            }
        }
        #endregion イベントハンドラ

        #endregion private メソッド
    }
}
