﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;


namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore;
    using LECore.Structures;
    using LECore.Structures.Core;
    using static LayoutWindow.PaneDragTargetAdapter;
    using AxisSnap = DragModifierHelper.AxisSnap;
    using MathUtil = LECore.Util.MathUtil;
    using OptionFlag = DragModifierHelper.OptionFlag;

    /// <summary>
    /// ドラッグ対象が実装すべきインタフェース
    /// </summary>
    public interface IDragTarget : IDrawable
    {
        float X { get; }
        float Y { get; }
        float Width { get; }
        float Height { get; }

        RectangleF BoundingRectangle { get; }
    }

    /// <summary>
    /// ヘルパクラス
    /// </summary>
    static class DragModifierHelper
    {
        //----------------------------------------------------------
        #region 定数・宣言
        /// <summary>
        /// 移動軸固定の種類
        /// インデックスとして、カーソル検索に利用します。
        /// </summary>
        public enum AxisSnap
        {
            None = 0,
            H = 1,
            V = 2,
            Detecting = 3,
            EightDirection = 4,
            SyncHAndV = 5,
            AxisSnap_Max = 6
        }

        /// <summary>
        /// 編集オプション
        /// </summary>
        [Flags]
        public enum OptionFlag
        {
            None = 0x00,
            GridSnapEnabled = 0x01,
            IgnoreShortDrag = 0x02,
            Option_Max
        }

        /// <summary>
        /// マウスカーソル衝突情報です。
        /// </summary>
        public class Contact
        {
            public PointF TransOrigin { get; set; }
            public AxisSnap AxisSnap { get; set; }
            public bool IsHit
            {
                get
                {
                    return
                        (float.IsInfinity(TransOrigin.X) && !float.IsNaN(TransOrigin.Y)) ||
                        (float.IsInfinity(TransOrigin.Y) && !float.IsNaN(TransOrigin.X));
                }
            }

            public bool IsControlAnchorClicked
            {
                get { return IsFloatValid_(TransOrigin.X) && IsFloatValid_(TransOrigin.Y); }
            }

            public Contact()
            {
                this.TransOrigin = new PointF(float.NaN, float.NaN);
            }

            public bool IsCornerClicked
            {
                get
                {
                    return Math.Sign(this.TransOrigin.X) != Math.Sign(this.TransOrigin.Y);
                }
            }
        }

        public static readonly PointF InvalidMousePos = new PointF( float.MaxValue, float.MaxValue );
        const float _BBSizeEpsilon = 0.001f;
        #endregion

        /// <summary>
        /// フロート値は有効か？
        /// </summary>
        static bool IsFloatValid_( float val )
        {
            return !float.IsNaN( val ) && !float.IsInfinity( val );
        }

        /// <summary>
        /// アンカーがクリックされたか？
        ///		TODO:CalcScaleOrigin内の処理を分離すること
        /// </summary>
        public static bool IsControlAnchorClicked(
            Matrix mtxLocal,
            RectangleF boundingRect,
            float magnify, PointF posMouse)
        {
            Contact modifierContact;
            return IsControlAnchorClicked(mtxLocal, boundingRect, magnify, posMouse, out modifierContact);
        }

        /// <summary>
        /// アンカーがクリックされたか？
        ///		TODO:CalcScaleOrigin内の処理を分離すること
        /// </summary>
        public static bool IsControlAnchorClicked(
            Matrix mtxLocal,
            RectangleF boundingRect,
            float magnify, PointF posMouse, out Contact contant)
        {
            PointF vScaling = new PointF(1.0f, 0.0f);
            if (mtxLocal.IsInvertible)
            {
                mtxLocal.Invert();
                posMouse = MathUtil.MtxTransformPoint(mtxLocal, posMouse);
                vScaling = MathUtil.MtxTransformVec(mtxLocal, vScaling);
            }
            float scale = MathUtil.GetVecLength(vScaling) / magnify;

            CalcTransformOrigin(posMouse, boundingRect, new PointF(scale, scale), out contant);

            return contant.IsControlAnchorClicked;
        }

        /// <summary>
        /// 変換原点を計算します。
        /// </summary>
        static public void CalcTransformOrigin(
            PointF posMouse,
            RectangleF boundRect,
            PointF dotSize,
            out Contact contant)
        {
            contant = new Contact();
            var transformOrigin = new PointF( float.NaN, float.NaN );

            float cX = ( boundRect.Right + boundRect.Left ) / 2.0f;
            float cY = ( boundRect.Top + boundRect.Bottom ) / 2.0f;

            float eX = 4.0f * Math.Abs( dotSize.X );
            float eY = 4.0f * Math.Abs( dotSize.Y );
            //DbgConsole.WriteLine("({0}, {1})", eX, eY);

            AxisSnap xSnap = AxisSnap.None;
            AxisSnap ySnap = AxisSnap.None;

            //------------------------------------------
            // X方向の変換原点を設定
            if( boundRect.Width <= eX )
            {
                // 大変小さい
                if( MathUtil.CheckValueNear( posMouse.X, boundRect.Left, eX ) )
                {
                    xSnap = AxisSnap.H;

                    if( boundRect.Left > posMouse.X )
                    {
                        transformOrigin.X = boundRect.Right;
                    }
                    else
                    {
                        transformOrigin.X = boundRect.Left;
                    }
                }
            }
            else
            {
                // 普通に大きい
                if( MathUtil.CheckValueNear( posMouse.X, boundRect.Left, eX ) )
                {
                    transformOrigin.X = boundRect.Right;
                }
                else if( MathUtil.CheckValueNear( posMouse.X, boundRect.Right, eX ) )
                {
                    transformOrigin.X = boundRect.Left;
                }
                else if( MathUtil.CheckValueNear( posMouse.X, cX, eX ) )
                {
                    transformOrigin.X = cX;
                    xSnap = AxisSnap.H;
                }
                else if ((boundRect.Left - eX <= posMouse.X) && (boundRect.Right + eX >= posMouse.X))
                {
                    transformOrigin.X = float.PositiveInfinity;
                }
            }

            //------------------------------------------
            // Y方向の変換原点を設定
            if( boundRect.Height <= eY )
            {
                if( MathUtil.CheckValueNear( posMouse.Y, boundRect.Top, eY ) )
                {
                    ySnap = AxisSnap.V;

                    if( boundRect.Bottom > posMouse.Y )
                    {
                        transformOrigin.Y = boundRect.Bottom;
                    }
                    else
                    {
                        transformOrigin.Y = boundRect.Top;
                    }
                }
            }
            else
            {
                if( MathUtil.CheckValueNear( posMouse.Y, boundRect.Top, eY ) )
                {
                    transformOrigin.Y = boundRect.Bottom;
                }
                else if( MathUtil.CheckValueNear( posMouse.Y, boundRect.Bottom, eY ) )
                {
                    transformOrigin.Y = boundRect.Top;
                }
                else if( MathUtil.CheckValueNear( posMouse.Y, cY, eY ) )
                {
                    transformOrigin.Y = cY;
                    ySnap = AxisSnap.V;
                }
                else if ((boundRect.Top -eY <= posMouse.Y) && (boundRect.Bottom + eY >= posMouse.Y))
                {
                    transformOrigin.Y = float.PositiveInfinity;
                }
            }

            // 中心点ならスナップしません。
            if( transformOrigin.X == cX && transformOrigin.Y == cY )
            {
                xSnap = AxisSnap.None;
                ySnap = AxisSnap.None;
            }

            // 両軸とも固定されるケースでは、
            if( xSnap != AxisSnap.None && ySnap != AxisSnap.None )
            {
                // スナップをキャンセルします。
                contant.AxisSnap = AxisSnap.None;
            }
            else
            {
                // 固定軸を固定する。
                contant.AxisSnap = xSnap != AxisSnap.None ? xSnap : ySnap;
            }

            contant.TransOrigin = transformOrigin;
        }

        /// <summary>
        ///
        /// </summary>
        public static RectangleF CalcBoundingRect( IDragTarget[] targetSet )
        {
            if( targetSet != null && targetSet.Length > 0 )
            {
                float l = targetSet[0].X,
                       r = targetSet[0].X + targetSet[0].Width,
                       t = targetSet[0].Y,
                       b = targetSet[0].Y + targetSet[0].Height;

                Array.ForEach( targetSet, delegate( IDragTarget key )
                {
                    l = Math.Min( l, key.X );
                    r = Math.Max( r, key.X + key.Width );
                    t = Math.Min( t, key.Y );
                    b = Math.Max( b, key.Y + key.Height );
                } );

                RectangleF result =
                    LECore.Util.MathUtil.MakePositiveSizeRectangle(
                    RectangleF.FromLTRB( l, t, r, b ) );

                result.Width = Math.Max( result.Width, _BBSizeEpsilon );
                result.Height = Math.Max( result.Height, _BBSizeEpsilon );

                return result;
            }

            return RectangleF.Empty;
        }
    }

    /// <summary>
    /// ドラッグ操作によって対象を編集するクラスです。
    /// レイアウトウインドウでのペイン編集などに利用されます。
    /// </summary>
    class DragModifier<TTargetType> : IDrawable where TTargetType : class, IDragTarget
    {
        //----------------------------------------------------------
        #region 定数・宣言


        /// <summary>
        /// 最小変更量
        /// </summary>
        const float _MinAbsoluteDifference = 4.0f;


        #endregion

        //----------------------------------------------------------
        #region フィールド
        /// <summary>
        /// 編集対象
        /// </summary>
        TTargetType[] _targetSet = null;
        /// <summary>
        /// 編集開始時のマウス点(通常はスクリーン座標系)
        /// </summary>
        PointF _dragStartPos = Point.Empty;
        /// <summary>
        /// 現在のマウス点(通常はスクリーン座標系)
        /// </summary>
        PointF _dragCurrentPos = Point.Empty;
        /// <summary>
        /// 最大変移量
        /// </summary>
        PointF _maxDifference = Point.Empty;
        /// <summary>
        /// 整列グリッドサイズ
        /// </summary>
        float _gridSize = 0.0f;
        /// <summary>
        /// 編集オプション
        /// </summary>
        OptionFlag _option = OptionFlag.None;
        /// <summary>
        /// 操作前：編集対象の領域矩形
        /// </summary>
        RectangleF _defaultBoundingRect = RectangleF.Empty;
        /// <summary>
        /// 軸固定タイプ
        /// </summary>
        AxisSnap _axisSnapKind = AxisSnap.None;

        #endregion フィールド

        //----------------------------------------------------------
        #region プロパティ
        /// <summary>
        /// ドラッグ開始点
        /// </summary>
        public PointF DragStartPointF
        {
            get{ return _dragStartPos;}
        }

        /// <summary>
        /// ドラッグ現在点
        /// </summary>
        public PointF DragCurrentPointF
        {
            get{ return _dragCurrentPos;}
        }

        /// <summary>
        /// 対象が存在するか
        /// </summary>
        public bool Empty
        {
            get { return _targetSet == null || _targetSet.Length <= 0; }
        }

        /// <summary>
        /// 操作対象セット
        /// </summary>
        public TTargetType[] TargetSet
        {
            get { return _targetSet; }
        }

        /// <summary>
        /// 軸固定状態
        /// </summary>
        public DragModifierHelper.AxisSnap AxisSnapKind
        {
            get { return _axisSnapKind; }
            set { _axisSnapKind = value; }
        }

        /// <summary>
        /// 変更前の状態の境界矩形
        /// </summary>
        public RectangleF DefaultBoundingRect
        {
            get { return _defaultBoundingRect; }
        }

        /// <summary>
        /// 差分を取得します。
        /// </summary>
        FVec2 _Difference
        {
            get
            {
                PointF temp;
                CalcDifference_( out temp );
                return new FVec2( temp );
            }
        }

        /// <summary>
        /// アクティブか
        /// </summary>
        bool _IsActive
        {
            get { return _targetSet != null; }
        }

        /// <summary>
        /// グリッド吸着が有効か判定します。
        /// </summary>
        protected bool _GridSnapEnable
        {
            get { return IsOptionEnable_( OptionFlag.GridSnapEnabled ) && _gridSize != 0.0f; }
        }

        /// <summary>
        /// 整列グリッドサイズ
        /// </summary>
        protected float _GridSize
        {
            get { return _gridSize; }
        }

        /// <summary>
        /// オプションを返します。
        /// </summary>
        public OptionFlag Option
        {
            get { return _option; }
        }

        /// <summary>
        /// グリッドサイズ
        /// </summary>
        public float GridSize
        {
            get { return _gridSize; }
        }

        #endregion プロパティ


        //----------------------------------------------------------
        #region private
        /// <summary>
        /// 指定フラグオプションが設定されているか調査します。
        /// </summary>
        bool IsOptionEnable_( OptionFlag flag )
        {
            return ( _option & flag ) != 0;
        }
        #endregion private

        //----------------------------------------------------------
        #region ヒットテスト関連
        /// <summary>
        /// フロート値が指定値に近いか判定します。
        /// </summary>
        static bool CheckValueNear_(
            float val,
            float threshold,
            float delta )
        {
            Debug.Assert( delta >= 0.0f );

            return Math.Abs( val - threshold ) < delta;
        }

        /// <summary>
        /// 変換原点は有効か
        /// </summary>
        public bool IsTransformOriginValid( PointF transformOrigin )
        {
            // 変換原点が有効なら...
            return ( IsFloatValid_( transformOrigin.X ) && IsFloatValid_( transformOrigin.Y ) );
        }

        /// <summary>
        /// ヒットテストを行います。
        /// </summary>
        public virtual bool CheckClicked( PointF posMouse, out TTargetType clicked )
        {
            if( _targetSet != null )
            {
                foreach( TTargetType target in _targetSet )
                {
                    if( target.BoundingRectangle.Contains( posMouse ) )
                    {
                        clicked = target;
                        return true;
                    }
                }
            }

            clicked = null;
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public virtual RectangleF CalcBoundingRect()
        {
            return DragModifierHelper.CalcBoundingRect( _targetSet as IDragTarget[] );
        }
        #endregion

        //---------------------------------------------------------
        #region　スケール計算関連
        /// <summary>
        /// フロート値は有効か？
        /// </summary>
        static bool IsFloatValid_( float val )
        {
            return !float.IsNaN( val ) && !float.IsInfinity( val );
        }

        /// <summary>
        /// スケールが有効か？
        /// </summary>
        static bool IsScaleValid_( double val )
        {
            return IsFloatValid_( (float)val ) && Math.Abs((float)val ) > float.Epsilon;
        }


        /// <summary>
        /// スケール値を計算します。
        /// </summary>
        protected bool CalcScale_( PointF difference, PointF transformOrigin, out PointD scale)
        {
            // 変換原点が無効なら...
            if( !IsTransformOriginValid( transformOrigin ) )
            {
                goto NG;
            }

            //-----------------------------------
            // クリック位置によって、ドラッグ量の解釈を変更します。
            Matrix mtxLoacl = CalcTargetLocalMatrix();
            if( mtxLoacl.IsInvertible )
            {
                mtxLoacl.Invert();

                PointF vDragStartLocal = MathUtil.MtxTransformPoint( mtxLoacl, DragStartPointF );
                PointF vDiff = MathUtil.SubVec( vDragStartLocal, transformOrigin );

                if( vDiff.X < 0.0f )
                {
                    difference.X *= -1.0f;
                }

                if( vDiff.Y < 0.0f )
                {
                    difference.Y *= -1.0f;
                }
            }

            //-----------------------------------
            // スケール値を計算
            RectangleF defaultBoundingRect = this.DefaultBoundingRect;
            double scaleX = ( difference.X + defaultBoundingRect.Width ) / (double)defaultBoundingRect.Width;
            double scaleY = ( difference.Y + defaultBoundingRect.Height ) / (double)defaultBoundingRect.Height;

            // 無効なら中断
            if( !IsScaleValid_( scaleX ) || !IsScaleValid_( scaleY ) )
            {
                goto NG;
            }

            scale = new PointD( scaleX, scaleY );
            return true;

            //-------------------------------
            // 失敗した場合
            NG:
            scale = new PointD( 1.0, 1.0 );
            return false;
        }
        #endregion



        //----------------------------------------------------------
        #region 変移量計算

        /// <summary>
        /// 変移量を計算します。
        /// </summary>
        protected void CalcDifference_( out PointF difference )
        {

            difference = new PointF(
                _dragCurrentPos.X - _dragStartPos.X,
                _dragCurrentPos.Y - _dragStartPos.Y );


            PointF vDiffInWorld = new PointF( difference.X, difference.Y );

            //------------------------------------------
            // 変化量をローカル座標系のベクトルに変換します。
            Matrix mtxLocal = CalcTargetLocalMatrix();
            if( mtxLocal.IsInvertible )
            {
                mtxLocal.Invert();
                difference = MathUtil.MtxTransformVec( mtxLocal, difference );
            }

            //------------------------------------------
            // 制約について調整をします。
            ConstraintDifference_( ref difference );

            // 移動値の絶対値の最大値を更新します。
            _maxDifference.X =
                Math.Max( Math.Abs( _maxDifference.X ), Math.Abs( vDiffInWorld.X ) );
            _maxDifference.Y =
                Math.Max( Math.Abs( _maxDifference.Y ), Math.Abs( vDiffInWorld.Y ) );

            // 移動値の絶対値の最大値が閾値以下であれば、移動をキャンセルします。
            if( IsOptionEnable_( OptionFlag.IgnoreShortDrag ) )
            {
                if( _maxDifference.X < _MinAbsoluteDifference &&
                    _maxDifference.Y < _MinAbsoluteDifference )
                {
                    difference = PointF.Empty;
                }
            }
        }

        /// <summary>
        /// 変更量を制約します。
        /// オーバーライド可能です。
        /// </summary>
        void ConstraintDifference_( ref PointF difference )
        {
            // 方向制限が有効ならば...
            if( _axisSnapKind == AxisSnap.EightDirection )
            {
                // difference ベクトルがX軸となす角度を計算して
                // 場合わけをします。
                double margin = Math.PI / 16.0;
                double rad = Math.Atan( difference.Y / difference.X );

                // X軸方向固定
                if( Math.Abs( rad ) < ( Math.PI / 4.0 - margin ) )
                {
                    difference.Y = 0.0f;
                }
                // Y軸方向固定
                else if( Math.Abs( rad ) > ( Math.PI / 4.0 + margin ) )
                {
                    difference.X = 0.0f;
                }
                // 斜め方向固定
                else
                {
                    float len = Convert.ToSingle( Math.Sqrt( ( difference.X * difference.X +
                                                              difference.Y * difference.Y ) * 0.5 ) );
                    difference.X = ( difference.X > 0 ) ? len : -len;
                    difference.Y = ( difference.Y > 0 ) ? len : -len;
                }

            }
            else if( _axisSnapKind == AxisSnap.Detecting )
            {
                // 軸固定指定が判定中なら....
                // 変化の大きかった方に、固定軸を設定します。
                if( Math.Abs( Math.Abs( difference.X ) - Math.Abs( difference.Y ) ) >
                    (int)_MinAbsoluteDifference )
                {
                    if( Math.Abs( difference.X ) > Math.Abs( difference.Y ) )
                    {
                        _axisSnapKind = AxisSnap.V;
                    }
                    else
                    {
                        _axisSnapKind = AxisSnap.H;
                    }
                    LECore.DbgConsole.WriteLine( "{0}", _axisSnapKind.ToString() );
                }
            }
            else if( _axisSnapKind == AxisSnap.H )
            {
                difference.X = 0.0f;
            }
            else if( _axisSnapKind == AxisSnap.V )
            {
                difference.Y = 0.0f;
            }
            else if( _axisSnapKind == AxisSnap.SyncHAndV )
            {
                float vMin = Math.Min( difference.X , difference.Y );
                difference.X = vMin;
                difference.Y = vMin;
            }
        }

        /// <summary>
        /// グリッドに整列するような、difference値を計算します。
        /// </summary>
        static public PointF CalcAlignedDifference(
            float gridSize,
            PointF basePos,
            PointF difference )
        {
            if( gridSize != 0.0f )
            {
                // 選択セットの先頭ペインを代表として、
                // 代表の移動先が、グリッド上になるように値を調整します。
                float resX = basePos.X + difference.X;
                float resY = basePos.Y + difference.Y;

                float dx = ( resX % gridSize );
                difference.X += ( dx < gridSize * 0.5f ) ? -dx : gridSize - dx;

                float dy = ( resY % gridSize );
                difference.Y += ( dy < gridSize * 0.5f ) ? -dy : gridSize - dy;
            }

            return difference;
        }

        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        public virtual Matrix CalcTargetLocalMatrix()
        {
            return new Matrix();
        }

        #endregion 変移量計算

        //----------------------------------------------------------
        #region public メソッド
        /// <summary>
        /// グリッド吸着を有効に設定します。
        /// </summary>
        public virtual void SetGridSnapEnable( float gridSize )
        {
            _gridSize = gridSize;
            _option |= OptionFlag.GridSnapEnabled;
        }

        /// <summary>
        /// ドラッグを開始します。
        /// </summary>
        public virtual void BeginDragging( TTargetType[] paneSet, PointF posInScreen, OptionFlag optionFlags )
        {
            _targetSet = paneSet;

            _dragStartPos = posInScreen;
            _dragCurrentPos = posInScreen;
            _maxDifference = PointF.Empty;

            _option = optionFlags;

            _defaultBoundingRect = CalcBoundingRect();
        }

        /// <summary>
        /// ドラッグ量を更新します。
        /// </summary>
        /// <param name="posInScene">現在の位置（シーン座標系）</param>
        public virtual void UpdateDragging( PointF posInScreen )
        {
            _dragCurrentPos = posInScreen;
        }

        /// <summary>
        /// 変移量によって更新します。
        /// </summary>
        public void UpdateDraggingByDifference( PointF differencePos )
        {
            _dragCurrentPos = MathUtil.AddVec( differencePos, _dragCurrentPos );
            UpdateDragging( _dragCurrentPos );
        }


        /// <summary>
        /// ドラッグ処理の開始点をリセットします。
        /// </summary>
        public virtual void ResetDragging( PointF posInScreen )
        {
            _dragStartPos = posInScreen;
            _dragCurrentPos = posInScreen;
        }

        /// <summary>
        /// ドラッグ終了
        /// </summary>
        /// <param name="difference">ドラックの結果生じた移動量（シーン座標系）</param>
        public virtual void EndDragging()
        {
            if( !Empty )
            {
                PointF difference;
                CalcDifference_( out difference );

                Clear();
            }
        }

        /// <summary>
        /// 内部状態のリセット。
        /// </summary>
        public virtual void Clear()
        {
            _targetSet = null;

            _gridSize = 0.0f;

            _dragStartPos = _dragCurrentPos;

            _option = OptionFlag.None;

            _axisSnapKind = AxisSnap.None;

            _defaultBoundingRect = RectangleF.Empty;
        }

        /// <summary>
        /// 内部状態を報告する文字列を取得します。
        /// </summary>
        public string ReportMsgString()
        {
            return string.Empty;
        }

        /// <summary>
        /// 移動に関する制約を処理します。
        /// </summary>
        public virtual void HandleConstraint( bool bConstraintEnabled )
        {
            // 既定の処理ではなにもしません。
        }

        #endregion public メソッド

        //----------------------------------------------------------
        #region IDrawable メンバ
        /// <summary>
        /// 有効であれば、対象ペインを描画します。
        /// 既定の処理では、平行移動操作に対応した描画をおこなっています。
        /// </summary>
        /// <param name="renderer"></param>
        public virtual void Draw( IRenderer renderer, DrawableOption option )
        {
            //if( _IsActive )
            //{
            //    // 変移量を算出します。
            //    PointF difference;
            //    CaclDifference_( out difference );


            //    renderer.PushMtx();
            //    renderer.Trans( difference.X, difference.Y );


            //    // 色の変更
            //    Color oldColor = renderer.Color;
            //    renderer.Color = Color.Gray;



            //    // 簡易描画フラグの指定
            //    uint flag = renderer.OptionFlag;
            //    renderer.OptionFlag = flag | IRenderOptionFlag.DrawOutline;

            //    // すべての対象を描画します。
            //    foreach( TTargetType pane in _targetSet )
            //    {
            //        pane.Draw( renderer, option );
            //    }

            //    // 描画フラグの復元
            //    renderer.OptionFlag = flag;

            //    // 色の復元
            //    renderer.Color = oldColor;

            //    renderer.PopMtx();

            //}
        }

        #endregion

        /// <summary>
        /// 編集アンカー付き矩形を描画します。
        /// </summary>
        static public void DrawAnchorRectangle( IRenderer renderer, RectangleF rectangle, float anchorRectSize )
        {
            float lineW = renderer.LineWidth;

            renderer.LineWidth = 1.0f;

            //---------------------------------------------
            // 矩形
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                rectangle );

            //---------------------------------------------
            // 操作点
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Left, rectangle.Top ), anchorRectSize );

            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Right, rectangle.Top ), anchorRectSize );
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Left, rectangle.Bottom ), anchorRectSize );
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Right, rectangle.Bottom ), anchorRectSize );

            float cX = ( rectangle.Left + rectangle.Right ) / 2.0f;
            float cY = ( rectangle.Top + rectangle.Bottom ) / 2.0f;

            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( cX, rectangle.Top ), anchorRectSize );
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( cX, rectangle.Bottom ), anchorRectSize );
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Left, cY ), anchorRectSize );
            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( rectangle.Right, cY ), anchorRectSize );

            RendererHelper.DrawRectangleWithoutTransform(
                renderer,
                new PointF( cX, cY ), anchorRectSize );

            renderer.LineWidth = lineW;
        }
    }
}
