﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;
using LayoutEditor.Forms.Dialogs;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
using LECore;
using LECore.Manipulator;
using LECore.Structures;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    public partial class PartsPanePage : PropertyPage
    {
        //----------------------------------------------------------
        private Initializeparamaters initializeparamaters = null;
        private PaneGuiAdapterGroup _oldTarget = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPanePage()
        {
            InitializeComponent();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPanePage(Initializeparamaters param)
        {
            this.initializeparamaters = param;
            this.IsOverridable = param.ParentPartsLayout != null;

            InitializeComponent();

            // サイズ制約は、最親階層だけ有効
            this._grpSizeConstraint.Enabled = param.ParentPartsLayout == null;
        }

        public class Initializeparamaters
        {
            public readonly IPartsLayout TargetPartsLayout;
            public readonly IPartsLayout ParentPartsLayout;

            public Initializeparamaters(IPartsLayout target, IPartsLayout patent)
            {
                TargetPartsLayout = target;
                ParentPartsLayout = patent;
            }
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static PropertyPage CreateInstance(object arg)
        {
            var newInstance = new PartsPanePage(arg as Initializeparamaters);
            return newInstance;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 部品レイアウトが上書きできるか？
        /// </summary>
        public bool IsOverridable
        {
            get;
            set;
        }

        /// <summary>
        /// 先頭の操作対象ペインを取得します。
        /// </summary>
        private ISubScene _SubScene
        {
            get
            {
                return LECore.LayoutEditorCore.Scene.CurrentISubScene;
            }
        }

        /// <summary>
        /// 操作対象ペインを取得します。
        /// </summary>
        private IEnumerable<PartsLayoutManipulator> _TargetPartsLayouts
        {
            get
            {
                foreach (PaneGUIAdapter targetObject in Target.Objects)
                {
                    var partsMnp = targetObject.FindTargetPartsLayoutManipulator(this.initializeparamaters.TargetPartsLayout);
                    if(partsMnp != null)
                    {
                        yield return partsMnp;
                    }
                }
            }
        }

        /// <summary>
        /// 先頭の操作対象ペインを取得します。
        /// </summary>
        private PartsLayoutManipulator _FirstTargetPartsLayout
        {
            get
            {
                return _TargetPartsLayouts.FirstOrDefault((parts) => parts != null);
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// UI状態を初期化します。
        /// </summary>
        protected override void InitializeProperty()
        {
        }

        /// <summary>
        /// UI状態を更新します。
        /// </summary>
        protected override void UpdateProperty()
        {
            _lblPartsLayoutName.Text = this.initializeparamaters.TargetPartsLayout.PartsLayoutName;
            _lblPartsLayoutName.BackColor = GetMultiEditColor();

            _nudMagnify.FVec2Value = this.initializeparamaters.TargetPartsLayout.Magnify.FVec2;
            _nudSizeConstraint.FVec2Value = this.initializeparamaters.TargetPartsLayout.SizeConstraint.FVec2;

            // 連動ボタンの更新
            if (_oldTarget != Target)
            {
                _oldTarget = Target;
                UpdateLinkButton(this.initializeparamaters.TargetPartsLayout);
            }

            this.Target.Active.Refresh();
        }

        /// <summary>
        /// 連動ボタンを更新します
        /// </summary>
        private void UpdateLinkButton(IPartsLayout partsLayout)
        {
            if (partsLayout == null) return;

            _nudMagnify.UpdateLinkButton(partsLayout.Magnify.X == partsLayout.Magnify.Y);
        }

        /// <summary>
        /// ターゲット変更時にすべてのページについて呼び出される関数
        /// </summary>
        public override void OnTargetChangeHandler()
        {
            using (new GUIUpdating(this))
            {
                UpdateProperty();
            }
        }

        //----------------------------------------------------------
        /// <summary>
        /// 部品レイアウト変更ボタン
        /// </summary>
        private void Event_BtnSetPartsLayout_Click(object sender, EventArgs e)
        {
            if (_FirstTargetPartsLayout == null)
            {
                Debug.Assert(false);
                return;
            }

            // 部品レイアウトを選ばせる
            string newPartsLayoutName = string.Empty;
            {
                string currentPartsLayoutName = _FirstTargetPartsLayout.IPartsLayout.PartsLayoutName;
                SelectPartsLayoutFiles selectPartsLayoutFiles = new SelectPartsLayoutFiles();
                selectPartsLayoutFiles.Initialize(currentPartsLayoutName);
                if (selectPartsLayoutFiles.ShowDialog() != DialogResult.OK ||
                    selectPartsLayoutFiles.SelectedPartsLayoutFileName == currentPartsLayoutName)
                {
                    // 変更指定がなかったので終了。
                    return;
                }

                newPartsLayoutName = selectPartsLayoutFiles.SelectedPartsLayoutFileName;
            }

            _SubScene.BeginMassiveModify();
            {
                if (this.initializeparamaters.ParentPartsLayout != null)
                {
                    // 複数編集には対応していない。
                    // 理由：各 PartsLayout から ParentPartsLayout を取得できず、
                    // 複数の ParentPartsLayout をうまく渡す方法がないため。
                    // 必要性が高ければ修正する。
                    var ownerPane = (_FirstTargetPartsLayout.IPartsLayout as IPaneExParamater).OwnerPane;
                    Ensure.Operation.ObjectNotNull(ownerPane.Parent);
                    if (this.initializeparamaters.ParentPartsLayout != null)
                    {
                        PartsLayoutManipulator partsLayoutMnp = new PartsLayoutManipulator();
                        partsLayoutMnp.BindTarget(this.initializeparamaters.ParentPartsLayout);
                        LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(newPartsLayoutName);

                        if (IsNewPartsFileValid_(newPartsLayoutName))
                        {
                            partsLayoutMnp.OverrideChildePartsPane(ownerPane.PaneName, newPartsLayoutName);
                        }
                    }
                }
                else
                {
                    LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(newPartsLayoutName);

                    if (IsNewPartsFileValid_(newPartsLayoutName))
                    {
                        PartsLayoutManipulator mnp = new PartsLayoutManipulator();
                        foreach (PaneGUIAdapter targetObject in Target.Objects)
                        {
                            mnp.BindTarget(targetObject.Target.IPartsLayout);
                            mnp.ChangePartsPaneLayout(newPartsLayoutName);
                        }
                    }
                }
            }
            _SubScene.EndMassiveModify();
        }

        /// <summary>
        /// 新しい部品ファイルが妥当なものか判断します。
        /// </summary>
        private bool IsNewPartsFileValid_(string newPartsLayoutName)
        {
            IPartsSubScene newPartsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(newPartsLayoutName);
            if (newPartsSubScene == null)
            {
                return false;
            }

            if (this.Owner == null || this.Owner.Owner == null)
            {
                return true;
            }

            // 自分自身に部品ペインを配置しようとしていないかチェックします。
            string layoutFilePath = this.Owner.Owner.GetLayoutFileName();
            if (!string.IsNullOrEmpty(layoutFilePath))
            {
                if (!PartsSubSceneHelper.IsPartsSubSceneValidForBodyLayout(newPartsSubScene, layoutFilePath))
                {
                    // 警告：処理をキャンセルしました：無限ループを生む不正な部品ペインの追加
                    _ttpCancelWarn.Show(StringResMgr.Get("PARTS_PANE_WANING_ADD_INVALID"), _lblPartsLayoutName, 0, 0, 3000);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 部品サイズ倍率変更。
        /// </summary>
        private void Event_NudMagnify_OnFVec2Edit(
            object sender, LayoutEditor.Controls.FVec2EditEventArgs args)
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            subScene.BeginMassiveModify();

            _FirstTargetPartsLayout.SizeConstraint = FVec3.Empty;
            _FirstTargetPartsLayout.Magnify = new FVec3(args.Value);

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// サイズ制約変更。
        /// </summary>
        private void Event_NudSizeConstraint_OnFVec2Edit(
            object sender, LayoutEditor.Controls.FVec2EditEventArgs args)
        {
            if (_FirstTargetPartsLayout.IPartsLayout.SizeConstraint.FVec2 == args.Value)
            {
                return;
            }

            _SubScene.BeginMassiveModify();

            _FirstTargetPartsLayout.Magnify = FVec3.One;
            _FirstTargetPartsLayout.SizeConstraint = FVec3.Empty;
            _FirstTargetPartsLayout.SizeConstraint = new FVec3(args.Value);

            _SubScene.EndMassiveModify();
        }

        /// <summary>
        /// 部品サイズリセット。
        /// </summary>
        private void Event_BtnSetMagnifyDisabled_Click(object sender, EventArgs e)
        {
            if (_FirstTargetPartsLayout.IPartsLayout.Magnify == FVec3.One)
            {
                return;
            }

            _FirstTargetPartsLayout.Magnify = FVec3.One;
        }

        /// <summary>
        /// サイズ制約リセット。
        /// </summary>
        private void Event_BtnSetSizeConstraintDisabled_Click(object sender, EventArgs e)
        {
            if (_FirstTargetPartsLayout.IPartsLayout.SizeConstraint == FVec3.Empty)
            {
                return;
            }

            _FirstTargetPartsLayout.SizeConstraint = FVec3.Empty;
        }

        /// <summary>
        /// サイズ制約、部品サイズの設定。
        /// </summary>
        private void Event_BtnSetSizeConstraintDefault_Click(object sender, EventArgs e)
        {
            if (_FirstTargetPartsLayout.IPartsLayout.SizeConstraint == _FirstTargetPartsLayout.IPartsLayout.InitialSize)
            {
                return;
            }

            _SubScene.BeginMassiveModify();
            _FirstTargetPartsLayout.Magnify = FVec3.One;
            _FirstTargetPartsLayout.SizeConstraint = FVec3.Empty;
            _FirstTargetPartsLayout.SizeConstraint = _FirstTargetPartsLayout.IPartsLayout.InitialSize;
            _SubScene.EndMassiveModify();
        }
    }
}
