﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using System.Collections.Generic;
    using Adapters;
    using LayoutEditor.Controls;
    using LayoutEditor.Forms.ToolWindows.common;
    using LayoutEditor.Utility;
    using LECore;
    using LECore.Structures;
    using LayoutEditor.Plugin;
    using System;

    /// <summary>
    /// PropWndPaneBasicPanel の概要の説明です。
    /// TODO : 詳細マテリアルページの開閉
    /// </summary>
    public class PaneBasicPanel : PropertyPanel
    {
        private readonly PropertyPanelType _type;
        private bool _lowLevelMaterialModeEnabled = false;
        private bool _combinerUserShaderSettingsEnabled = false;

        //----------------------------------------------------------
        // コンストラクタ
        //----------------------------------------------------------

        /// <summary>
        /// フォームデザイナ用デフォルトコンストラクタ
        /// </summary>
        public PaneBasicPanel()
            : this(PropertyPanelType.Null, null, null)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="type">ペインの種類</param>
        public PaneBasicPanel(PropertyPanelType type, PaneGuiAdapterGroup target, Action<ViewManagerMessage> sendMessageToViewManagerAction)
        {
            InitializeComponent();
            _type = type;
            _sendMessageToViewManagerAction = sendMessageToViewManagerAction;

            bool isMaskEnabled;
            bool isDropShadowEnabled;
            List<TreeNode> newNodePane = RegisterCategoryViewNodePaneSpecific_(out isMaskEnabled, out isDropShadowEnabled, this._type, target);
            SetCategoryViewNodes(newNodePane, isMaskEnabled, isDropShadowEnabled);

            // 初期選択ノートはペイン基本情報
            this.CategoryView.SelectedNode = this.CategoryView.Nodes[0];
        }

        /// <summary>
        /// GUI初期化
        /// </summary>
        private void InitializeComponent()
        {
            this.pnlLeft.SuspendLayout();
            this.SuspendLayout();
            //
            // tvwCategory
            //
            this.tvwCategory.LineColor = System.Drawing.Color.Black;
            //
            // PaneBasicPanel
            //
            this.Name = "PaneBasicPanel";
            this.pnlLeft.ResumeLayout(false);
            this.ResumeLayout(false);

        }

        /// <summary>
        /// 操作対象のペイン
        /// </summary>
        private IPane GetActivePane_(PaneGuiAdapterGroup target)
        {
            if (target != null &&
                target.Active is IBaseGuiAdapter)
            {
                return (target.Active as IBaseGuiAdapter).Target;
            }

            return null;
        }

        //----------------------------------------------------------
        // カテゴリノードの初期化
        //----------------------------------------------------------

        /// <summary>
        /// カテゴリ登録。
        /// </summary>
        private PropWndTreeNode CreateCategoryNode_(
            string textStrID,
            CreatePropertyPageFunction func)
        {
            PropWndTreeNode node = new PropWndTreeNode(
                StringResMgr.Get("PROP_PANE_CATEGORY_" + textStrID), 0, 0, func, null);

            return node;
        }

        /// <summary>
        /// 初期化パラメータ
        /// </summary>
        private class PartsPageNodeInitParams
        {
            public int NumPicture { get; set; }
            public int NumText { get; set; }
            public int NumWindow { get; set; }
            public int NumCapture { get; set; }
        }

        //----------------------------------------------------------

        private TreeNode AddMaterialNodes_(PaneGuiAdapterGroup target, PropWndTreeNode parentNode)
        {
            SetupMaterialCategoryNode(GetActivePane_(target), PartsPropertyUsageOptions.None, parentNode);

            parentNode.Expand();
            return parentNode;
        }

        /// <summary>
        /// ペイン種類ごとの固有カテゴリノードの登録を行います。
        /// </summary>
        /// <param name="type"></param>
        private List<TreeNode> RegisterCategoryViewNodePaneSpecific_(
            out bool isMaskEnabled,
            out bool isDropShadowEnabled,
            PropertyPanelType type, PaneGuiAdapterGroup target)
        {
            List<TreeNode> nodes = new List<TreeNode>();
            isMaskEnabled = false;
            isDropShadowEnabled = false;
            switch (type)
            {
                case PropertyPanelType.Null:
                case PropertyPanelType.NullPane:
                    break;
                case PropertyPanelType.PicturePane:
                    PropWndTreeNode pictNode = CreateCategoryNode_("PICTURE", PicturePanePage.CreateInstance);
                    nodes.Add(pictNode);
                    AddMaterialNodes_(target, pictNode);

                    isMaskEnabled = true;
                    isDropShadowEnabled = true;
                    break;
                case PropertyPanelType.TextBoxPane:
                    PropWndTreeNode textNode = CreateCategoryNode_("TEXTBOX", TextBoxPanePage.CreateInstance);
                    nodes.Add(textNode);
                    AddMaterialNodes_(target, textNode);
                    isMaskEnabled = true;
                    isDropShadowEnabled = true;
                    break;
                case PropertyPanelType.WindowPane:
                    PropWndTreeNode wndNode = CreateCategoryNode_("WINDOW", WindowPanePage.CreateInstance);
                    nodes.Add(wndNode);
                    AddMaterialNodes_(target, wndNode);
                    isMaskEnabled = true;
                    isDropShadowEnabled = true;
                    break;
                case PropertyPanelType.RegionPane:
                    nodes.Add(CreateCategoryNode_("BOUNDINGREGION", BoundingRegionPanePage.CreateInstance));
                    break;
                case PropertyPanelType.CapturePane:
                    nodes.Add(CreateCategoryNode_("CAPTURE", CapturePanePage.CreateInstance));
                    break;
                case PropertyPanelType.Alignment:
                    nodes.Add(CreateCategoryNode_("ALIGNMENT", AlignmentPanePage.CreateInstance));
                    break;
                case PropertyPanelType.Scissor:
                    nodes.Add(CreateCategoryNode_("SCISSOR", ScissorPanePage.CreateInstance));
                    break;
                case PropertyPanelType.PartsPane:
                    PartsPageNodeInitParams param = new PartsPageNodeInitParams();
                    PropWndTreeNode paramNode = SetupPartsLayoutNode(GetActivePane_(target).IPartsLayout, null, param);
                    nodes.Add(paramNode);
                    break;
                default:
                    // 意図しないケース
                    Debug.Assert(false);
                    break;
            }

            return nodes;
        }

        /// <summary>
        /// マテリアルカテゴリーノードを初期化します。
        /// </summary>
        private void SetupMaterialCategoryNode(
            IPane owner,
            PartsPropertyUsageOptions option,
            PropWndTreeNode parentNode,
            bool isOverwrite = false)
        {
            // テクスチャ
            TreeNode ctgTexture = RegisterCategory("PROP_MAT_CATEGORY_TEXTURE", 1, MaterialPage.CreateMaterialPage<MaterialTexturePage>, owner, isOverwrite);
            if (owner.PaneKind != PaneKind.Textbox)
            {
                parentNode.Nodes.Add(ctgTexture);
            }

            // カラー補間
            TreeNode ctgColorBlend = RegisterCategory("PROP_MAT_CATEGORY_COLORBLEND", 1, MaterialPage.CreateMaterialPage<MaterialColorBlendPage>, owner, isOverwrite);
            parentNode.Nodes.Add(ctgColorBlend);

            // 詳細なマテリアル項目
            // テクスチャコンバイナ
            if (owner.PaneKind != PaneKind.Textbox)
            {
                IRevHWMaterial rawMat = null;
                if (!isOverwrite)
                {
                    rawMat = this.GetTargetCoreMaterial();
                }
                else
                {
                    // 上書きの場合 owner は部品レイアウト内のペインが渡される
                    rawMat = this.GetTargetCoreMaterialFromPane(owner);
                }
                _lowLevelMaterialModeEnabled = rawMat != null && rawMat.LowLevelCombinerSettingsEnabled;
                _combinerUserShaderSettingsEnabled = rawMat != null && rawMat.CombinerUserShaderSettingsEnabled;

                if (_combinerUserShaderSettingsEnabled)
                {
                    TreeNode ctgTexComb = RegisterCategory("PROP_MAT_CATEGORY_TEXCOMB", 1, MaterialPage.CreateMaterialPage<MaterialCombinerUserShaderPage>, owner, isOverwrite);
                    parentNode.Nodes.Add(ctgTexComb);
                }
                else if (_lowLevelMaterialModeEnabled)
                {
                    TreeNode ctgTexComb = RegisterCategory("PROP_MAT_CATEGORY_TEXCOMB", 1, MaterialPage.CreateMaterialPage<MaterialTexCombStagePageDetailCtr>, owner, isOverwrite);
                    parentNode.Nodes.Add(ctgTexComb);
                }
                else
                {
                    TreeNode ctgTexComb = RegisterCategory("PROP_MAT_CATEGORY_TEXCOMB", 1, MaterialPage.CreateMaterialPage<MaterialTexCombStagePage>, owner, isOverwrite);
                    parentNode.Nodes.Add(ctgTexComb);
                }
            }

            // 角丸はピクチャペインのみ対応している
            // 角丸は上書きに対応していないため上書き設定時はプロパティを表示しないようにする
            if (owner.PaneKind == PaneKind.Picture &&
                !isOverwrite)
            {
                // 角丸
                TreeNode nodeProceduralShape = RegisterCategory("PROP_PANE_CATEGORY_PROCEDURAL_SHAPE", 0, PropWndPaneProceduralShapePage.CreateInstance, null);
                parentNode.Nodes.Add(nodeProceduralShape);
            }

            // フラグメント処理
            TreeNode ctgFOp = RegisterCategory("PROP_MAT_CATEGORY_FRAGMENTOPERATION", 1,
                MaterialPage.CreateMaterialPage<MaterialFragmentOperationPage>, owner, isOverwrite);
            parentNode.Nodes.Add(ctgFOp);

            // ペイン種類固有情報の、部分的な上書きが指定されていれば対応します。
            if (PartsPropertyUsageOptionsHelper.IsPaneExDataPartlyOverwrote(option))
            {
                // ペイン情報ノード
                {
                    parentNode.ForceDisabled = true;

                    // テキスト
                    if (option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText))
                    {
                        parentNode.ForceDisabled = false;
                    }
                }

                // マテリアルノード
                {
                    // 一旦マテリアルを全部無効にする。
                    foreach (PropWndTreeNode materialNode in parentNode.Nodes)
                    {
                        SetPropWndTreeNodeForceDisabledFlag_(materialNode, true);
                    }

                    // カラー補間を有効にする。
                    if (option.HasFlag(PartsPropertyUsageOptions.UsePaneExParamaterColorBlend))
                    {
                        int index = parentNode.Nodes.IndexOf(ctgColorBlend);
                        Debug.Assert(index != -1);
                        (parentNode.Nodes[index] as PropWndTreeNode).ForceDisabled = false;
                    }

                    // テクスチャを有効にする。
                    if (option.HasFlag(PartsPropertyUsageOptions.UsePaneExParameterTexture))
                    {
                        int index = parentNode.Nodes.IndexOf(ctgTexture);
                        Debug.Assert(index != -1);
                        (parentNode.Nodes[index] as PropWndTreeNode).ForceDisabled = false;
                        (parentNode.Nodes[index] as PropWndTreeNode).IsTextureOverwritingEnabled = true;
                    }
                }
            }
        }

        /// <summary>
        /// 部品ペイン用ノードを設定します。
        /// </summary>
        private PropWndTreeNode SetupPartsLayoutNode(
            IPartsLayout partsLayout,
            IPartsLayout parentPartsLayout,
            PartsPageNodeInitParams param)
        {
            // 部品ペインノード
            PropWndTreeNode parentNode = null;
            {
                var createFunc = (parentPartsLayout != null) ?
                    new CreatePropertyPageFunction(PartsPropertyPage.CreatePartsPageInstance) :
                    new CreatePropertyPageFunction(PartsPanePage.CreateInstance);
                var nodeText = StringResMgr.Get("PROP_PANE_CATEGORY_PARTS") + " [" + partsLayout.PartsLayoutName + "]";
                var initParams = new PartsPanePage.Initializeparamaters(partsLayout, parentPartsLayout);
                parentNode = new PropWndTreeNode(nodeText, 1, 1, createFunc, initParams);
                parentNode.ExpandAll();
            }

            // 部品プロパティ
            foreach (var subPartsPropaerty in partsLayout.PartsPropaerties)
            {
                // 上書きペインのルート
                CreatePropertyPageFunction pageCreateFunc = (CreatePropertyPageFunction)DefaultPartsPanePage.CreateInstance;
                var nodeText = StringResMgr.Get("PROP_PANE_CATEGORY_PARTSPROPERTY") + " [" + subPartsPropaerty.TargetName + "]";
                int imageIndex = CategoryImageList.Images.IndexOfKey(PaneItemImageHelper.GetPaneImageKindAsString(subPartsPropaerty.PaneKind, subPartsPropaerty.Option != PartsPropertyUsageOptions.None));
                PropWndTreeNode overwritePaneRoot = NewNode_(partsLayout, parentPartsLayout, nodeText, subPartsPropaerty, 0, pageCreateFunc, imageIndex);
                parentNode.Nodes.Add(overwritePaneRoot);

                // コメントを ToolTipに設定
                IPartsPropaertySettings setting = partsLayout.GetPartsPropaertySettings(subPartsPropaerty.TargetName);
                overwritePaneRoot.ToolTipText = (setting != null) ? PaneTreeViewHelper.GetUICommentText(setting.Description) : string.Empty;

                // ペイン基本情報用ノードを初期化
                if ((subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePanePosition) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneSize) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneScale) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneRotate) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneTransparency) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneVisibility) != 0 ||
                    (subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneBasicUserData) != 0)
                {
                    var node = RegisterCategory("PROP_PANE_CATEGORY_GENERAL", 1, PaneBasicPropertyPage.CreateInstance, subPartsPropaerty);
                    overwritePaneRoot.Nodes.Add(node);
                }

                // 各種ペインの上書き
                if ((subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneExParamater) != 0)
                {
                    if (subPartsPropaerty.PaneKind == PaneKind.Picture)
                    {
                        PropWndTreeNode nodeEx = NewNode_(partsLayout, parentPartsLayout, StringResMgr.Get("PROP_PANE_CATEGORY_" + "PICTURE"), subPartsPropaerty, param.NumPicture, PicturePropertyPage.CreatePartsPageInstance, true);
                        SetupMaterialCategoryNode(subPartsPropaerty.Paramater.OwnerPane, subPartsPropaerty.Option, nodeEx, true);

                        overwritePaneRoot.Nodes.Add(nodeEx);
                        param.NumPicture++;
                    }
                    else if (subPartsPropaerty.PaneKind == PaneKind.Textbox)
                    {
                        PropWndTreeNode nodeEx = NewNode_(partsLayout, parentPartsLayout, StringResMgr.Get("PROP_PANE_CATEGORY_" + "TEXTBOX"), subPartsPropaerty, param.NumText, TextBoxPropertyPage.CreatePartsPageInstance, true);
                        SetupMaterialCategoryNode(subPartsPropaerty.Paramater.OwnerPane, subPartsPropaerty.Option, nodeEx, true);

                        overwritePaneRoot.Nodes.Add(nodeEx);
                        param.NumText++;
                    }
                    else if (subPartsPropaerty.PaneKind == PaneKind.Window)
                    {
                        PropWndTreeNode nodeEx = NewNode_(partsLayout, parentPartsLayout, StringResMgr.Get("PROP_PANE_CATEGORY_" + "WINDOW"), subPartsPropaerty, param.NumWindow, WindowPropertyPage.CreatePartsPageInstance, true);
                        SetupMaterialCategoryNode(subPartsPropaerty.Paramater.OwnerPane, subPartsPropaerty.Option, nodeEx, true);

                        overwritePaneRoot.Nodes.Add(nodeEx);
                        param.NumWindow++;
                    }
                    else if (subPartsPropaerty.PaneKind == PaneKind.Capture)
                    {
                        PropWndTreeNode nodeEx = NewNode_(partsLayout, parentPartsLayout, StringResMgr.Get("PROP_PANE_CATEGORY_" + "CAPTURE"), subPartsPropaerty, param.NumCapture, CapturePropertyPage.CreatePartsPageInstance, true);

                        overwritePaneRoot.Nodes.Add(nodeEx);
                        param.NumCapture++;
                    }
                    else if (subPartsPropaerty.PaneKind == PaneKind.Parts)
                    {
                        // 再帰的に処理を行います。
                        var subPartsNode = SetupPartsLayoutNode(
                            subPartsPropaerty.Paramater as IPartsLayout, partsLayout, param);
                        overwritePaneRoot.Nodes.Add(subPartsNode);
                    }
                }

                // ユーザデータ用ノードを初期化
                if ((subPartsPropaerty.Option & PartsPropertyUsageOptions.UsePaneExUserData) != 0)
                {
                    var node = RegisterCategory("PROP_PANE_CATEGORY_USERDATA", 1, UserDataPage.CreateInstance, subPartsPropaerty.Paramater.OwnerPane);
                    overwritePaneRoot.Nodes.Add(node);

                    // ページプラグインノードを初期化
                    foreach (var pagePluginInfo in PluginManager.Instance.PagePlugins)
                    {
                        node = RegisterPluginCategory(pagePluginInfo.NodeName, 1, pagePluginInfo.CreatePropertyFunction, subPartsPropaerty.Paramater.OwnerPane);
                        overwritePaneRoot.Nodes.Add(node);
                    }
                }

                overwritePaneRoot.ExpandAll();
            }

            return parentNode;
        }

        private static PropWndTreeNode NewNode_(IPartsLayout partsLayout, IPartsLayout parentPartsLayout, string nodeText, IPartsPropaerty subPartsPropaerty, int targetIndex, CreatePropertyPageFunction pageCreateFunc, int imageIndex, bool isOverwrite = false)
        {
            var initParam = new PropertyPageInitializeParamaters(partsLayout, parentPartsLayout, subPartsPropaerty.TargetName, subPartsPropaerty.PaneKind, targetIndex, subPartsPropaerty.Option);
            PropWndTreeNode pageNode = new PropWndTreeNode(nodeText, imageIndex, imageIndex, pageCreateFunc, initParam, isOverwrite);
            return pageNode;
        }

        private static PropWndTreeNode NewNode_(IPartsLayout partsLayout, IPartsLayout parentPartsLayout, string nodeText, IPartsPropaerty subPartsPropaerty, int targetIndex, CreatePropertyPageFunction pageCreateFunc, bool isOverwrite = false)
        {
            return NewNode_(partsLayout, parentPartsLayout, nodeText, subPartsPropaerty, targetIndex, pageCreateFunc, 1, isOverwrite);
        }

        /// <summary>
        /// 有効状態を設定します。
        /// </summary>
        private static void SetPropWndTreeNodeForceDisabledFlag_(PropWndTreeNode node, bool forceDisabled)
        {
            node.ForceDisabled = forceDisabled;
            foreach (var childNode in node.Nodes)
            {
                SetPropWndTreeNodeForceDisabledFlag_(childNode as PropWndTreeNode, forceDisabled);
            }
        }

        /// <summary>
        /// カテゴリノードを設定します。
        /// </summary>
        /// <param name="newNodePane"></param>
        private void SetCategoryViewNodes(List<TreeNode> newNodePane, bool isMaskEnabled, bool isDropShadowEnabled)
        {
            TreeNode nodeBasic = RegisterCategory("PROP_PANE_CATEGORY_GENERAL", 0, PropWndPaneBasicPage.CreateInstance, null);
            this.CategoryView.Nodes.Add(nodeBasic);

            // -------------- ペイン基本の子ノード
            if (isMaskEnabled)
            {
                TreeNode nodeMask = RegisterCategory("PROP_PANE_CATEGORY_MASK", 0, PropWndPaneMaskPage.CreateInstance, null);
                nodeBasic.Nodes.Add(nodeMask);
                nodeBasic.ExpandAll();
            }
            if (isDropShadowEnabled)
            {
                TreeNode nodeDropShadow = RegisterCategory("PROP_PANE_CATEGORY_DROP_SHADOW", 0, PropWndPaneDropShadowPage.CreateInstance, null);
                nodeBasic.Nodes.Add(nodeDropShadow);
                nodeBasic.ExpandAll();
            }

            // -------------- ペイン種類ごとの設定
            this.CategoryView.Nodes.AddRange(newNodePane.ToArray());

            // -------------- ユーザ拡張情報
            TreeNode nodeUserData = RegisterCategory("PROP_PANE_CATEGORY_USERDATA", 0, UserDataPage.CreateInstance, null);
            this.CategoryView.Nodes.Add(nodeUserData);

            // -------------- ページプラグイン
            foreach (var pagePluginInfo in PluginManager.Instance.PagePlugins)
            {
                var node = RegisterPluginCategory(pagePluginInfo.NodeName, 0, pagePluginInfo.CreatePropertyFunction, null);
                this.CategoryView.Nodes.Add(node);
            }

            // オーバーレイ初期化
            foreach (TreeNode node in this.CategoryView.Nodes)
            {
                UpdateOverrayIcons_(node);
            }
        }

        /// <summary>
        /// オーバーレイアイコンの更新
        /// </summary>
        void UpdateOverrayIcons_(TreeNode node)
        {
            if(node.ToolTipText != string.Empty)
            {
                OverlayIconHelper.TreeViewOverlay(node, (uint)_CommentOverrayIconIdx);
            }

            foreach(TreeNode childNode in node.Nodes)
            {
                UpdateOverrayIcons_(childNode);
            }
        }

        /// <summary>
        /// ターゲット変更時の更新処理
        /// </summary>
        public override void OnTargetChangeCategoryUpdate()
        {
            DoCategoryUpdate_();
        }

        /// <summary>
        /// 選択された上書きペインノードを取得します。
        /// </summary>
        void GetSelectedPropertyNodeText_(out string propNodeText, out string selectedNodeText)
        {
            propNodeText = string.Empty;
            selectedNodeText = string.Empty;

            if (this.CategoryView.SelectedNode == null)
            {
                return;
            }

            selectedNodeText = this.CategoryView.SelectedNode.Text;
            propNodeText = selectedNodeText;

            TreeNode node = this.CategoryView.SelectedNode;
            while(node.Parent != null &&
                !node.Text.Contains(StringResMgr.Get("PROP_PANE_CATEGORY_PARTSPROPERTY")))
            {
                node = node.Parent;
            }

            propNodeText = node.Text;
        }

        /// <summary>
        /// 編集対象の、マテリアル(IRevHWMaterial)を取得します。
        /// </summary>
        protected virtual IRevHWMaterial GetTargetCoreMaterial()
        {
            if (this.Owner == null)
            {
                return null;
            }

            IPane owner = GetActivePane_(this.Owner.Target);
            if (owner != null)
            {
                return owner.EnumrateDetailMaterial().FirstOrDefault();
            }

            return null;
        }

        /// <summary>
        /// 編集対象の、マテリアル(IRevHWMaterial)を取得します。
        /// </summary>
        protected virtual IRevHWMaterial GetTargetCoreMaterialFromPane(IPane pane)
        {
            if (pane == null)
            {
                return null;
            }

            return pane.EnumrateDetailMaterial().FirstOrDefault();
        }

        /// <summary>
        /// カテゴリ項目を更新する必要があるかどうか
        /// </summary>
        protected virtual bool NeedToUpdateCategory_()
        {
            var rawMat = this.GetTargetCoreMaterial();
            if (rawMat != null)
            {
                if (_lowLevelMaterialModeEnabled != rawMat.LowLevelCombinerSettingsEnabled ||
                    _combinerUserShaderSettingsEnabled != rawMat.CombinerUserShaderSettingsEnabled)
                {
                    return true;
                }
            }

            return _type == PropertyPanelType.PartsPane;
        }

        /// <summary>
        /// カテゴリーノードの更新
        /// </summary>
        private void DoCategoryUpdate_()
        {
            if (NeedToUpdateCategory_())
            {
                bool isMaskEnabled;
                bool isDropShadowEnabled;
                List<TreeNode> newNodePanes = RegisterCategoryViewNodePaneSpecific_(out isMaskEnabled, out isDropShadowEnabled, this._type, this.Owner.Target);

                // 変更前の状態を保存しておきます。
                string propNodeName, selectedNodeName;
                GetSelectedPropertyNodeText_(out propNodeName, out selectedNodeName);
                int lastScrollPosV = LECore.Win32.User32.GetScrollPos(this.CategoryView.Handle, LECore.Win32.SB.SB_VERT);

                this.CategoryView.BeginUpdate();
                List<PropertyPage> oldPages = this.ResetAllNodesAndGatherOldPages();

                SetCategoryViewNodes(newNodePanes, isMaskEnabled, isDropShadowEnabled);

                PropWndTreeNode newActiveNode = null;

                // まったく同じ名前で選択できないか試みる。
                newActiveNode = PaneTreeViewHelper.GetNodeByNodeText(this.CategoryView, propNodeName) as PropWndTreeNode;
                if (newActiveNode == null)
                {
                    // ダメなら、ペイン固有ノードの終端もしくは先頭ノード
                    if (newNodePanes.Count > 0)
                    {
                        newActiveNode = newNodePanes[newNodePanes.Count - 1] as PropWndTreeNode;
                    }
                    else
                    {
                        newActiveNode = this.CategoryView.Nodes[0] as PropWndTreeNode;
                    }

                    lastScrollPosV = 0;
                }

                // 選択とスクロール位置の復元
                {
                    // 上書きペインノードの子ノードを選択していた場合はさらに選択を試みる。
                    if (propNodeName != selectedNodeName)
                    {
                        var newsubNode = PaneTreeViewHelper.GetNodeByNodeText(newActiveNode, selectedNodeName) as PropWndTreeNode;
                        if (newsubNode != null)
                        {
                            newActiveNode = newsubNode;
                        }
                    }

                    this.CategoryView.SelectedNode = newActiveNode;
                    this.CategoryView.EndUpdate();

                    // スクロール位置の復元。
                    // 復元してノードが見えていない場合には見えるように調整。
                    LECore.Win32.User32.SetScrollPos(this.CategoryView.Handle, LECore.Win32.SB.SB_VERT, lastScrollPosV, true);
                    if (!this.CategoryView.SelectedNode.IsVisible)
                    {
                        LECore.Win32.User32.SetScrollPos(this.CategoryView.Handle, LECore.Win32.SB.SB_VERT, this.CategoryView.SelectedNode.Bounds.Y, true);
                    }
                }

                SetActiveCategory(newActiveNode);

                // すべての初期化が完了したら、古いページを破棄します。
                oldPages.ForEach((page) => page.Dispose());
            }
            else
            {
                Ensure.Operation.True(_type != PropertyPanelType.PartsPane);
                for (int i = 1; i < this.CategoryView.Nodes.Count - 1; i++)
                {
                    (CategoryView.Nodes[i] as PropWndTreeNode).UpdateTarget(this.Target.Active.Target);
                }
            }
        }
    }
}
