﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Controls;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
using System.Diagnostics;
using LECore.Structures.Nsrif.Attributes;
using LECore;
using LECore.Structures;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using src.Controls;
    using ToolWindows.LayoutWindow;

    public partial class MaterialTexCombStagePageDetailCtr : MaterialPage
    {
        private int _stageNo = 0;
        private UIRadioButton[] _stageButtons = new UIRadioButton[6];

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        /// ターゲットステージ。
        /// </summary>
        private TevStage TargetStage
        {
            get
            {
                Debug.Assert(_stageNo != -1);
                return this.ActiveTarget.TevData.Stages[_stageNo];
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override PropertyPageID PageID
        {
            get { return PropertyPageID.MaterialTevStageLowLevel; }
        }

        /// <summary>
        /// コンバイナユーザーシェーダを利用することが出来るか。
        /// </summary>
        /// <remarks>
        /// プロジェクト設定ファイルで有効になっており、上書きでの利用では無い場合のみ有効
        /// </remarks>
        private bool IsCombinerUserShaderEnabled
        {
            get { return AppSetting != null && AppSetting.ProjectSettings.IsCombinerUserShaderEnabled && !IsOverwrite; }
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        public MaterialTexCombStagePageDetailCtr()
        {
            InitializeComponent();

            // コンテキストメニューを追加する
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblBlackColor, _ctlBlackColor.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblWhiteColor, _ctlWhiteColor.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblConstColor0, _ctlConstColor0.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblConstColor1, _ctlConstColor1.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblConstColor2, _ctlConstColor2.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblConstColor3, _ctlConstColor3.AnimationMarkArray);
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblConstColor4, _ctlConstColor4.AnimationMarkArray);
        }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        protected override void InitializeProperty()
        {
            InitializeKonstantRegisterComboBox_(_cmbConstantRegRGB);
            InitializeKonstantRegisterComboBox_(_cmbConstantRegA);

            _tbxNumStage.MaxValue = LECore.LayoutEditorCore.PlatformDetail.MaxTevStageCount;

            _stageButtons[0] = _radStage1;
            _stageButtons[1] = _radStage2;
            _stageButtons[2] = _radStage3;
            _stageButtons[3] = _radStage4;
            _stageButtons[4] = _radStage5;
            _stageButtons[5] = _radStage6;

            for (int i = 0; i < _stageButtons.Length; i++)
            {
                _stageButtons[i].Tag = i;
            }

            _ctlStageRGB.InitializeProperty();
            _ctlStageA.InitializeProperty();

            _ctlConstColor0.Tag = 0;
            _ctlConstColor1.Tag = 1;
            _ctlConstColor2.Tag = 2;
            _ctlConstColor3.Tag = 3;
            _ctlConstColor4.Tag = 4;

            // ViewManagerへのメッセージハンドラを設定します
            _ctlBlackColor.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlWhiteColor.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlConstColor0.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlConstColor1.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlConstColor2.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlConstColor3.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
            _ctlConstColor4.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        protected override void UpdateProperty()
        {
            TevStages stages = this.ActiveTarget.TevData.Stages;

            // ステージ番号調整
            if (_stageNo >= stages.NumStages)
            {
                _stageNo = 0;
            }

            // ステージ数
            _tbxNumStage.SetValue(stages.NumStages);

            // ステージ番号
            for (int i = 0; i < _stageButtons.Length; i++)
            {
                UIRadioButton button = _stageButtons[i];

                button.SetCheckedByTag(_stageNo);
                button.Enabled = i < stages.NumStages;

                if (i < stages.NumStages)
                {
                    button.Font = LayoutEditor.AppMain.GuiFontBold;
                    if (i == _stageNo) { button.ForeColor = Color.Red; }
                    else { button.ForeColor = SystemColors.ControlText; }
                }
                else
                {
                    button.Font = this.Font;
                    button.ForeColor = SystemColors.ControlText;
                }
            }

            // ステージ設定
            TevStage stage = stages[_stageNo];
            _ctlStageRGB.SetData(_stageNo, stage.ColorStage);
            _ctlStageA.SetData(_stageNo, stage.AlphaStage);

            //
            _cmbConstantRegRGB.SetSelectedItemData(stage.ColorStage.Konst);
            _cmbConstantRegA.SetSelectedItemData(stage.AlphaStage.Konst);

            // アニメーションマークとコンテキストメニューの有効無効
            bool bUse = CanUseAnimationMark(this._OwnerPane.OwnerSubScene);
            IEnumerable<IAnimationMarkUsable> animCtrlList = GetAnimationMarkUsableControl();
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UseAnimationMark = bUse;
            }

            // TEV カラー
            {
                _ctlBlackColor.Color = this.ActiveTarget.IMaterial.BlackColor;
                _ctlWhiteColor.Color = this.ActiveTarget.IMaterial.WhiteColor;

                _ctlConstColor0.Color = this.ActiveTarget.TevData.GetColor(0);
                _ctlConstColor1.Color = this.ActiveTarget.TevData.GetColor(1);
                _ctlConstColor2.Color = this.ActiveTarget.TevData.GetColor(2);
                _ctlConstColor3.Color = this.ActiveTarget.TevData.GetColor(3);
                _ctlConstColor4.Color = this.ActiveTarget.TevData.GetColor(4);
            }

            _grpStage.Enabled = stages.NumStages > 0;

            //// 左上のマテリアル共有の設定によるパネルの有効/無効の設定
            //this.Enabled = !EnabledUseLTMaterial;

            btnMoveLeft.Enabled = _stageNo > 0;
            btnMoveRight.Enabled = _stageNo + 1 < stages.NumStages;

            // コンバイナユーザーシェーダー が利用可能な場合は次のプロパティに変更する
            if(IsCombinerUserShaderEnabled)
            {
                _btnChangeMaterialMode.Text = StringResMgr.Get("PROP_MAT_CUS_CHANGE_TEXT");
            }

            // アニメーションマークで利用するアトリビュートを関連付け
            _ctlBlackColor.SetTargetAttribute(this.ActiveTarget.IMaterial.BlackColorIAnmAttr);
            _ctlWhiteColor.SetTargetAttribute(this.ActiveTarget.IMaterial.WhiteColorIAnmAttr);
            var tevAttribute = this.ActiveTarget.CoreMaterial.GetAnimationTargetAttributeSet(ParamaterKind.Animation_MaterialColors);
            _ctlConstColor0.SetTargetAttribute(tevAttribute[0].OwnerNode);
            _ctlConstColor1.SetTargetAttribute(tevAttribute[4].OwnerNode);
            _ctlConstColor2.SetTargetAttribute(tevAttribute[8].OwnerNode);
            _ctlConstColor3.SetTargetAttribute(tevAttribute[12].OwnerNode);
            _ctlConstColor4.SetTargetAttribute(tevAttribute[16].OwnerNode);

            // アニメーションマークの更新
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UpdateAnimationMark();
            }
        }

        /// <summary>
        /// アニメーションマークが利用可能なコントロールの一覧を取得する。
        /// </summary>
        protected override IEnumerable<IAnimationMarkUsable> GetAnimationMarkUsableControl()
        {
            List<IAnimationMarkUsable> list = new List<IAnimationMarkUsable>();

            list.Add(_lblBlackColor);
            list.Add(_lblWhiteColor);
            list.Add(_lblConstColor0);
            list.Add(_lblConstColor1);
            list.Add(_lblConstColor2);
            list.Add(_lblConstColor3);
            list.Add(_lblConstColor4);
            list.Add(_ctlBlackColor);
            list.Add(_ctlWhiteColor);
            list.Add(_ctlConstColor0);
            list.Add(_ctlConstColor1);
            list.Add(_ctlConstColor2);
            list.Add(_ctlConstColor3);
            list.Add(_ctlConstColor4);

            return list;
        }

        /// <summary>
        /// コンスタントレジスタ用コンボボックスの初期化
        /// </summary>
        private void InitializeKonstantRegisterComboBox_(UIComboBox cmb)
        {

            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST_BLACKCOLOR", AttrTevKonst.BlackColor, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST_WHITECOLOR", AttrTevKonst.WhiteColor, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST0", AttrTevKonst.K0, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST1", AttrTevKonst.K1, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST2", AttrTevKonst.K2, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST3", AttrTevKonst.K3, true);
            cmb.AddItem("PROP_MAT_TEV_STAGE_KONST4", AttrTevKonst.K4, true);
        }

        //----------------------------------------------------------
        // イベントハンドラ
        //----------------------------------------------------------

        private void Event_tbxNumStage_TextChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            int preNumStages = this.ActiveTarget.TevData.Stages.NumStages;
            int numStages = _tbxNumStage.Value;

            this.ActiveTarget.TevData.Stages.NumStages = numStages;
            if (_stageNo >= numStages)
            {
                _stageNo = 0;
            }

            using (new GUIUpdating(this))
            {
                UpdateProperty();
            }
            NotifyPropertyChanged();
        }

        private void Event_radStage1_CheckedChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            UIRadioButton button = (UIRadioButton)sender;

            if (button.Checked)
            {
                button.Font = LayoutEditor.AppMain.GuiFontBold;
                button.ForeColor = Color.Red;

                // ステージ変更
                _stageNo = (int)button.Tag;
                using (new GUIUpdating(this))
                {
                    UpdateProperty();
                }
            }
            else
            {
                button.Font = this.Font;
                button.ForeColor = SystemColors.ControlText;
            }
        }

        private void _cmbConstantRegRGB_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            _ctlStageRGB.Data.Konst = (AttrTevKonst)(sender as UIComboBox).SelectedItemData;
            Event_ColorPanel_DataChanged(sender, new EventArgs());
        }

        private void _cmbConstantRegA_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            _ctlStageA.Data.Konst = (AttrTevKonst)(sender as UIComboBox).SelectedItemData;
            Event_AlphaPanel_DataChanged(sender, new EventArgs());
        }

        private void Event_ColorPanel_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            LECore.DbgConsole.WriteLine("ColorStage_DataChanged");
            LECore.DbgConsole.WriteLine(_ctlStageRGB.Data.ToString());

            this.TargetStage.ColorStage.Set(_ctlStageRGB.Data);
            NotifyPropertyChanged();
        }

        private void Event_AlphaPanel_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            LECore.DbgConsole.WriteLine("AlphaStage_DataChanged");
            LECore.DbgConsole.WriteLine(_ctlStageA.Data.ToString());

            this.TargetStage.AlphaStage.Set(_ctlStageA.Data);
            NotifyPropertyChanged();
        }

        /// <summary>
        /// カラー編集
        /// </summary>
        private void Event_ctlConstColor0_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            ColorEditPanel control = (ColorEditPanel)sender;
            int index = (int)control.Tag;

            EditColor(Target, e,
                (adapter) => adapter.TevData.GetColor(index),
                (adapter, color) => adapter.TevData.SetColor(color, index));

            NotifyPropertyChanged();
        }

        private void Event_btnChangeMaterialMode_Click(object sender, EventArgs e)
        {
            // コンバイナユーザーシェーダーが利用可能な場合は、次に開くプロパティを分岐する
            if (IsCombinerUserShaderEnabled)
            {
                this.ActiveTarget.CombinerUserShaderSettingsEnabled = true;
            }
            this.ActiveTarget.LowLevelCombinerSettingsEnabled = false;
            NotifyPropertyChanged();
        }

        private void Event_ctlBlackColor_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            EditColor(Target, e,
                (adapter) => adapter.BlackColor,
                (adapter, color) => adapter.BlackColor = color);

            NotifyPropertyChanged();
        }

        private void Event_ctlWhiteColor_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            EditColor(Target, e,
                (adapter) => adapter.WhiteColor,
                (adapter, color) => adapter.WhiteColor = color);

            NotifyPropertyChanged();
        }

        private bool ValidateStage0(TevDirectStage stage)
        {
            if (stage.CopyReg)
            {
                return false;
            }

            int count = MaterialPages.TexCombStagePanelDetailCtr.GetEnabledSourceCount(stage.CombineMode);
            for (int i = 0; i < count; i++)
            {
                switch (stage.TevArgs[i].Source)
                {
                    case AttrTevSource.Previous:
                    case AttrTevSource.Register:
                        return false;
                }
            }

            return true;
        }

        private void btnMoveLeft_Click(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (_stageNo - 1 == 0)
            {
                if (!ValidateStage0(this.ActiveTarget.TevData.Stages[1].ColorStage) ||
                    !ValidateStage0(this.ActiveTarget.TevData.Stages[1].AlphaStage))
                {
                    MessageBox.Show(StringResMgr.Get("ERROR_STAGE_SWAP"), AppConstants.AppName);
                    return;
                }
            }

            // 差し替える
            var tmp = new TevStage(_stageNo - 1);
            tmp.Set(this.ActiveTarget.TevData.Stages[_stageNo - 1]);
            this.ActiveTarget.TevData.Stages[_stageNo - 1].Set(this.ActiveTarget.TevData.Stages[_stageNo]);
            this.ActiveTarget.TevData.Stages[_stageNo].Set(tmp);

            // 選択ステージ番号を変える
            _stageNo--;

            // ステージの入れ替えによって値が変わらなくてもコマンドの実行を強制する
            ForcePushCommand = true;
            NotifyPropertyChanged();
        }

        private void btnMoveRight_Click(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (_stageNo == 0)
            {
                if (!ValidateStage0(this.ActiveTarget.TevData.Stages[1].ColorStage) ||
                    !ValidateStage0(this.ActiveTarget.TevData.Stages[1].AlphaStage))
                {
                    MessageBox.Show(StringResMgr.Get("ERROR_STAGE_SWAP"), AppConstants.AppName);
                    return;
                }
            }

            // 差し替える
            var tmp = new TevStage(_stageNo + 1);
            tmp.Set(this.ActiveTarget.TevData.Stages[_stageNo + 1]);
            this.ActiveTarget.TevData.Stages[_stageNo + 1].Set(this.ActiveTarget.TevData.Stages[_stageNo]);
            this.ActiveTarget.TevData.Stages[_stageNo].Set(tmp);

            // 選択ステージ番号を変える
            _stageNo++;

            // ステージの入れ替えによって値が変わらなくてもコマンドの実行を強制する
            ForcePushCommand = true;
            NotifyPropertyChanged();
        }
    }
}
