﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using LayoutEditor.Controls;

using LayoutEditor.Forms.ToolWindows.PropertyEditWindow;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.ColorControls;

using LECore;
using LECore.Structures.Nsrif.Attributes;


namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages
{

    using Utility;
   using Adapters;

    /// <summary>
    /// マテリアルカラーページクラス。
    /// </summary>
    public sealed class MaterialColorPage : MaterialPage
    {
        #region デザイナ変数

        private System.Windows.Forms.GroupBox grpColor;
        private LayoutEditor.Controls.UICheckBox chkLightingAlpha;
        private LayoutEditor.Controls.UICheckBox chkLightingSpecular;
        private LayoutEditor.Controls.ColorButton btnSpecular;
        private LayoutEditor.Controls.ColorButton btnAmbient;
        private LayoutEditor.Controls.ColorButton btnDiffuse;
        private System.Windows.Forms.GroupBox grpASrc;
        private LayoutEditor.Controls.UIRadioButton radASrcVtx;
        private LayoutEditor.Controls.IntEditSlider ctlAlpha;
        private LayoutEditor.Controls.IntTrackBar trbDiffuse;
        private LayoutEditor.Controls.IntTrackBar trbAmbient;
        private LayoutEditor.Controls.IntTrackBar trbSpecular;
        private System.Windows.Forms.GroupBox grpDSrc;
        private LayoutEditor.Controls.UIRadioButton radDSrcMat;
        private LayoutEditor.Controls.UIRadioButton radDSrcVtx;
        private LayoutEditor.Controls.UIRadioButton radASrcMat;
        private System.Windows.Forms.GroupBox grpLighting;
        private LayoutEditor.Controls.UICheckBox chkLightingDiffAmb;
        private System.Windows.Forms.Panel pnlDiffuse;
        private System.Windows.Forms.Panel pnlAmbient;
        private System.Windows.Forms.Panel pnlSpecular;
        private System.Windows.Forms.Panel pnlAlpha;
        private LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.ColorControls.ColorShadingBox ctlColorShading;
        private LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.ColorControls.AlphaShadingBox ctlAlphaShading;
        private System.Windows.Forms.Label lblColorShading;
        private System.Windows.Forms.Label lblAlphaShading;
        private LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.MaterialTevAutoMark ctlTevAuto;
        private System.Windows.Forms.Label lblAssign;
        private System.Windows.Forms.Label lblAlpha;
        private System.Windows.Forms.Label lblAlphaAssign;
        private System.Windows.Forms.Label lblDiffuse;
        private System.Windows.Forms.Label lblDiffuseAssign;
        private System.Windows.Forms.Label lblAmbient;
        private System.Windows.Forms.Label lblAmbientAssign;
        private System.Windows.Forms.Label lblSpecular;
        private System.Windows.Forms.Label lblSpecularAssign;
        private System.Windows.Forms.ToolTip ttpHint;
        private System.ComponentModel.IContainer components;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager( typeof( MaterialColorPage ) );
            this.grpDSrc = new System.Windows.Forms.GroupBox();
            this.radDSrcMat = new LayoutEditor.Controls.UIRadioButton();
            this.radDSrcVtx = new LayoutEditor.Controls.UIRadioButton();
            this.grpASrc = new System.Windows.Forms.GroupBox();
            this.radASrcMat = new LayoutEditor.Controls.UIRadioButton();
            this.radASrcVtx = new LayoutEditor.Controls.UIRadioButton();
            this.grpLighting = new System.Windows.Forms.GroupBox();
            this.chkLightingAlpha = new LayoutEditor.Controls.UICheckBox();
            this.chkLightingSpecular = new LayoutEditor.Controls.UICheckBox();
            this.chkLightingDiffAmb = new LayoutEditor.Controls.UICheckBox();
            this.ctlTevAuto = new LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.MaterialTevAutoMark();
            this.grpColor = new System.Windows.Forms.GroupBox();
            this.lblAssign = new System.Windows.Forms.Label();
            this.pnlAlpha = new System.Windows.Forms.Panel();
            this.ctlAlpha = new LayoutEditor.Controls.IntEditSlider();
            this.lblAlpha = new System.Windows.Forms.Label();
            this.lblAlphaAssign = new System.Windows.Forms.Label();
            this.pnlDiffuse = new System.Windows.Forms.Panel();
            this.btnDiffuse = new LayoutEditor.Controls.ColorButton();
            this.trbDiffuse = new LayoutEditor.Controls.IntTrackBar();
            this.lblDiffuse = new System.Windows.Forms.Label();
            this.lblDiffuseAssign = new System.Windows.Forms.Label();
            this.pnlAmbient = new System.Windows.Forms.Panel();
            this.btnAmbient = new LayoutEditor.Controls.ColorButton();
            this.trbAmbient = new LayoutEditor.Controls.IntTrackBar();
            this.lblAmbient = new System.Windows.Forms.Label();
            this.lblAmbientAssign = new System.Windows.Forms.Label();
            this.pnlSpecular = new System.Windows.Forms.Panel();
            this.btnSpecular = new LayoutEditor.Controls.ColorButton();
            this.trbSpecular = new LayoutEditor.Controls.IntTrackBar();
            this.lblSpecular = new System.Windows.Forms.Label();
            this.lblSpecularAssign = new System.Windows.Forms.Label();
            this.ctlColorShading = new LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.ColorControls.ColorShadingBox();
            this.ctlAlphaShading = new LayoutEditor.Forms.ToolWindows.PropertyEditWindow.MaterialPages.ColorControls.AlphaShadingBox();
            this.lblColorShading = new System.Windows.Forms.Label();
            this.lblAlphaShading = new System.Windows.Forms.Label();
            this.ttpHint = new System.Windows.Forms.ToolTip( this.components );
            this.grpDSrc.SuspendLayout();
            this.grpASrc.SuspendLayout();
            this.grpLighting.SuspendLayout();
            this.grpColor.SuspendLayout();
            this.pnlAlpha.SuspendLayout();
            this.pnlDiffuse.SuspendLayout();
            ( (System.ComponentModel.ISupportInitialize)( this.trbDiffuse ) ).BeginInit();
            this.pnlAmbient.SuspendLayout();
            ( (System.ComponentModel.ISupportInitialize)( this.trbAmbient ) ).BeginInit();
            this.pnlSpecular.SuspendLayout();
            ( (System.ComponentModel.ISupportInitialize)( this.trbSpecular ) ).BeginInit();
            this.SuspendLayout();
            //
            // grpDSrc
            //
            this.grpDSrc.Controls.Add( this.radDSrcMat );
            this.grpDSrc.Controls.Add( this.radDSrcVtx );
            this.grpDSrc.FlatStyle = System.Windows.Forms.FlatStyle.System;
            resources.ApplyResources( this.grpDSrc, "grpDSrc" );
            this.grpDSrc.Name = "grpDSrc";
            this.grpDSrc.TabStop = false;
            //
            // radDSrcMat
            //
            resources.ApplyResources( this.radDSrcMat, "radDSrcMat" );
            this.radDSrcMat.Name = "radDSrcMat";
            //
            // radDSrcVtx
            //
            resources.ApplyResources( this.radDSrcVtx, "radDSrcVtx" );
            this.radDSrcVtx.Name = "radDSrcVtx";
            //
            // grpASrc
            //
            this.grpASrc.Controls.Add( this.radASrcMat );
            this.grpASrc.Controls.Add( this.radASrcVtx );
            this.grpASrc.FlatStyle = System.Windows.Forms.FlatStyle.System;
            resources.ApplyResources( this.grpASrc, "grpASrc" );
            this.grpASrc.Name = "grpASrc";
            this.grpASrc.TabStop = false;
            //
            // radASrcMat
            //
            resources.ApplyResources( this.radASrcMat, "radASrcMat" );
            this.radASrcMat.Name = "radASrcMat";
            //
            // radASrcVtx
            //
            resources.ApplyResources( this.radASrcVtx, "radASrcVtx" );
            this.radASrcVtx.Name = "radASrcVtx";
            //
            // grpLighting
            //
            this.grpLighting.Controls.Add( this.chkLightingAlpha );
            this.grpLighting.Controls.Add( this.chkLightingSpecular );
            this.grpLighting.Controls.Add( this.chkLightingDiffAmb );
            this.grpLighting.Controls.Add( this.ctlTevAuto );
            this.grpLighting.FlatStyle = System.Windows.Forms.FlatStyle.System;
            resources.ApplyResources( this.grpLighting, "grpLighting" );
            this.grpLighting.Name = "grpLighting";
            this.grpLighting.TabStop = false;
            //
            // chkLightingAlpha
            //
            resources.ApplyResources( this.chkLightingAlpha, "chkLightingAlpha" );
            this.chkLightingAlpha.Name = "chkLightingAlpha";
            //
            // chkLightingSpecular
            //
            resources.ApplyResources( this.chkLightingSpecular, "chkLightingSpecular" );
            this.chkLightingSpecular.Name = "chkLightingSpecular";
            //
            // chkLightingDiffAmb
            //
            resources.ApplyResources( this.chkLightingDiffAmb, "chkLightingDiffAmb" );
            this.chkLightingDiffAmb.Name = "chkLightingDiffAmb";
            //
            // ctlTevAuto
            //
            resources.ApplyResources( this.ctlTevAuto, "ctlTevAuto" );
            this.ctlTevAuto.Name = "ctlTevAuto";
            this.ttpHint.SetToolTip( this.ctlTevAuto, resources.GetString( "ctlTevAuto.ToolTip" ) );
            //
            // grpColor
            //
            this.grpColor.Controls.Add( this.lblAssign );
            this.grpColor.Controls.Add( this.pnlAlpha );
            this.grpColor.Controls.Add( this.pnlDiffuse );
            this.grpColor.Controls.Add( this.pnlAmbient );
            this.grpColor.Controls.Add( this.pnlSpecular );
            this.grpColor.FlatStyle = System.Windows.Forms.FlatStyle.System;
            resources.ApplyResources( this.grpColor, "grpColor" );
            this.grpColor.Name = "grpColor";
            this.grpColor.TabStop = false;
            //
            // lblAssign
            //
            resources.ApplyResources( this.lblAssign, "lblAssign" );
            this.lblAssign.Name = "lblAssign";
            //
            // pnlAlpha
            //
            this.pnlAlpha.Controls.Add( this.ctlAlpha );
            this.pnlAlpha.Controls.Add( this.lblAlpha );
            this.pnlAlpha.Controls.Add( this.lblAlphaAssign );
            resources.ApplyResources( this.pnlAlpha, "pnlAlpha" );
            this.pnlAlpha.Name = "pnlAlpha";
            //
            // ctlAlpha
            //
            resources.ApplyResources( this.ctlAlpha, "ctlAlpha" );
            this.ctlAlpha.MaxValue = 255;
            this.ctlAlpha.MinValue = 0;
            this.ctlAlpha.Name = "ctlAlpha";
            this.ctlAlpha.ShowRangeLabel = false;
            this.ctlAlpha.TextBoxSize = LayoutEditor.Controls.EditSliderTextBoxSize.Size72;
            //
            // lblAlpha
            //
            resources.ApplyResources( this.lblAlpha, "lblAlpha" );
            this.lblAlpha.Name = "lblAlpha";
            //
            // lblAlphaAssign
            //
            this.lblAlphaAssign.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            resources.ApplyResources( this.lblAlphaAssign, "lblAlphaAssign" );
            this.lblAlphaAssign.Name = "lblAlphaAssign";
            //
            // pnlDiffuse
            //
            this.pnlDiffuse.Controls.Add( this.btnDiffuse );
            this.pnlDiffuse.Controls.Add( this.trbDiffuse );
            this.pnlDiffuse.Controls.Add( this.lblDiffuse );
            this.pnlDiffuse.Controls.Add( this.lblDiffuseAssign );
            resources.ApplyResources( this.pnlDiffuse, "pnlDiffuse" );
            this.pnlDiffuse.Name = "pnlDiffuse";
            //
            // btnDiffuse
            //
            this.btnDiffuse.Color = System.Drawing.Color.Empty;
            resources.ApplyResources( this.btnDiffuse, "btnDiffuse" );
            this.btnDiffuse.Name = "btnDiffuse";
            //
            // trbDiffuse
            //
            resources.ApplyResources( this.trbDiffuse, "trbDiffuse" );
            this.trbDiffuse.Maximum = 255;
            this.trbDiffuse.Name = "trbDiffuse";
            //
            // lblDiffuse
            //
            resources.ApplyResources( this.lblDiffuse, "lblDiffuse" );
            this.lblDiffuse.Name = "lblDiffuse";
            //
            // lblDiffuseAssign
            //
            this.lblDiffuseAssign.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            resources.ApplyResources( this.lblDiffuseAssign, "lblDiffuseAssign" );
            this.lblDiffuseAssign.Name = "lblDiffuseAssign";
            //
            // pnlAmbient
            //
            this.pnlAmbient.Controls.Add( this.btnAmbient );
            this.pnlAmbient.Controls.Add( this.trbAmbient );
            this.pnlAmbient.Controls.Add( this.lblAmbient );
            this.pnlAmbient.Controls.Add( this.lblAmbientAssign );
            resources.ApplyResources( this.pnlAmbient, "pnlAmbient" );
            this.pnlAmbient.Name = "pnlAmbient";
            //
            // btnAmbient
            //
            this.btnAmbient.Color = System.Drawing.Color.Empty;
            resources.ApplyResources( this.btnAmbient, "btnAmbient" );
            this.btnAmbient.Name = "btnAmbient";
            //
            // trbAmbient
            //
            resources.ApplyResources( this.trbAmbient, "trbAmbient" );
            this.trbAmbient.Maximum = 255;
            this.trbAmbient.Name = "trbAmbient";
            //
            // lblAmbient
            //
            resources.ApplyResources( this.lblAmbient, "lblAmbient" );
            this.lblAmbient.Name = "lblAmbient";
            //
            // lblAmbientAssign
            //
            this.lblAmbientAssign.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            resources.ApplyResources( this.lblAmbientAssign, "lblAmbientAssign" );
            this.lblAmbientAssign.Name = "lblAmbientAssign";
            //
            // pnlSpecular
            //
            this.pnlSpecular.Controls.Add( this.btnSpecular );
            this.pnlSpecular.Controls.Add( this.trbSpecular );
            this.pnlSpecular.Controls.Add( this.lblSpecular );
            this.pnlSpecular.Controls.Add( this.lblSpecularAssign );
            resources.ApplyResources( this.pnlSpecular, "pnlSpecular" );
            this.pnlSpecular.Name = "pnlSpecular";
            //
            // btnSpecular
            //
            this.btnSpecular.Color = System.Drawing.Color.Empty;
            resources.ApplyResources( this.btnSpecular, "btnSpecular" );
            this.btnSpecular.Name = "btnSpecular";
            //
            // trbSpecular
            //
            resources.ApplyResources( this.trbSpecular, "trbSpecular" );
            this.trbSpecular.Maximum = 255;
            this.trbSpecular.Name = "trbSpecular";
            //
            // lblSpecular
            //
            resources.ApplyResources( this.lblSpecular, "lblSpecular" );
            this.lblSpecular.Name = "lblSpecular";
            //
            // lblSpecularAssign
            //
            this.lblSpecularAssign.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            resources.ApplyResources( this.lblSpecularAssign, "lblSpecularAssign" );
            this.lblSpecularAssign.Name = "lblSpecularAssign";
            //
            // ctlColorShading
            //
            resources.ApplyResources( this.ctlColorShading, "ctlColorShading" );
            this.ctlColorShading.Name = "ctlColorShading";
            this.ctlColorShading.ShadingInfo = null;
            this.ctlColorShading.TabStop = false;
            //
            // ctlAlphaShading
            //
            resources.ApplyResources( this.ctlAlphaShading, "ctlAlphaShading" );
            this.ctlAlphaShading.Name = "ctlAlphaShading";
            this.ctlAlphaShading.ShadingInfo = null;
            this.ctlAlphaShading.TabStop = false;
            //
            // lblColorShading
            //
            resources.ApplyResources( this.lblColorShading, "lblColorShading" );
            this.lblColorShading.Name = "lblColorShading";
            //
            // lblAlphaShading
            //
            resources.ApplyResources( this.lblAlphaShading, "lblAlphaShading" );
            this.lblAlphaShading.Name = "lblAlphaShading";
            //
            // MaterialColorPage
            //
            this.Controls.Add( this.grpLighting );
            this.Controls.Add( this.grpColor );
            this.Controls.Add( this.grpASrc );
            this.Controls.Add( this.grpDSrc );
            this.Controls.Add( this.ctlColorShading );
            this.Controls.Add( this.ctlAlphaShading );
            this.Controls.Add( this.lblColorShading );
            this.Controls.Add( this.lblAlphaShading );
            resources.ApplyResources( this, "$this" );
            this.Name = "MaterialColorPage";
            this.grpDSrc.ResumeLayout( false );
            this.grpASrc.ResumeLayout( false );
            this.grpLighting.ResumeLayout( false );
            this.grpColor.ResumeLayout( false );
            this.pnlAlpha.ResumeLayout( false );
            this.pnlDiffuse.ResumeLayout( false );
            ( (System.ComponentModel.ISupportInitialize)( this.trbDiffuse ) ).EndInit();
            this.pnlAmbient.ResumeLayout( false );
            ( (System.ComponentModel.ISupportInitialize)( this.trbAmbient ) ).EndInit();
            this.pnlSpecular.ResumeLayout( false );
            ( (System.ComponentModel.ISupportInitialize)( this.trbSpecular ) ).EndInit();
            this.ResumeLayout( false );

        }
        #endregion

        // カラーピッカーアダプタ
        private readonly ColorPickerAdapter _adapterDiffuse  = null;
        private readonly ColorPickerAdapter _adapterAmbient  = null;
        private readonly ColorPickerAdapter _adapterSpecular = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialColorPage()
        {
            InitializeComponent();

            // カラーピッカーアダプタ作成
            _adapterDiffuse = new ColorPickerAdapter(btnDiffuse);
            _adapterDiffuse.AddControl(trbDiffuse);
            _adapterDiffuse.ColorEdit += new ColorEditEventHandler(Event_ColorPicker_ColorEdit);

            _adapterAmbient = new ColorPickerAdapter(btnAmbient);
            _adapterAmbient.AddControl(trbAmbient);
            _adapterAmbient.ColorEdit += new ColorEditEventHandler(Event_ColorPicker_ColorEdit);

            _adapterSpecular = new ColorPickerAdapter(btnSpecular);
            _adapterSpecular.AddControl(trbSpecular);
            _adapterSpecular.ColorEdit += new ColorEditEventHandler(Event_ColorPicker_ColorEdit);
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static ObjPropertyPage CreateInstance(object arg)
        {
             ObjPropertyPage page = new MaterialColorPage();
            return page;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override PropertyPageID PageID
        {
            get { return PropertyPageID.MaterialColor; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void InitializeProperty()
        {
            // ディフューズソース
            radDSrcMat.Tag = AttrColorSource.Reg;
            radDSrcVtx.Tag = AttrColorSource.Vtx;
            radDSrcMat.RadioChecked += new EventHandler(Event_RadDSrc_RadioChecked);
            radDSrcVtx.RadioChecked += new EventHandler(Event_RadDSrc_RadioChecked);

            // アルファソース
            radASrcMat.Tag = AttrColorSource.Reg;
            radASrcVtx.Tag = AttrColorSource.Vtx;
            radASrcMat.RadioChecked += new EventHandler(Event_RadASrc_RadioChecked);
            radASrcVtx.RadioChecked += new EventHandler(Event_RadASrc_RadioChecked);

            // ディフューズ
            trbDiffuse.ScrollChanged += new ScrollChangedEventHandler(Event_TrbColor_ScrollChanged);

            // アンビエント
            trbAmbient.ScrollChanged += new ScrollChangedEventHandler(Event_TrbColor_ScrollChanged);

            // スペキュラ
            trbSpecular.ScrollChanged += new ScrollChangedEventHandler(Event_TrbColor_ScrollChanged);

            // アルファ
            ctlAlpha.ValueEdit += new ValueEditEventHandler(Event_CtlAlpha_ValueEdit);

            // ライト計算
            chkLightingDiffAmb.CheckedChanged  += new EventHandler(Event_ChkLightingDiffAmb_CheckedChanged);
            chkLightingSpecular.CheckedChanged += new EventHandler(Event_ChkLightingSpecular_CheckedChanged);
            chkLightingAlpha.CheckedChanged    += new EventHandler(Event_ChkLightingAlpha_CheckedChanged);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateProperty()
        {
            MaterialGUIAdapter target = this.ActiveTarget;
            ChanCtrl chanD  = this.DiffuseCtrl;
            ChanCtrl chanA  = this.AlphaCtrl;
            ChanCtrl chanS  = this.SpecularCtrl;

            Rgba diffuse  = this.DiffuseRegister.Color;
            Rgba ambient  = this.AmbientRegister.Color;
            Rgba specular = this.SpecularRegister.Color;
            Rgba alpha    = this.AlphaRegister.Color;

            // シェーディングボックス
            ctlColorShading.ShadingInfo = new MaterialColorShadingInfo(
                diffuse.ToColorNoAlpha(),
                ambient.ToColorNoAlpha(),
                specular.ToColorNoAlpha(),
                chanD.Lighting,
                chanS.Lighting
            );
            ctlAlphaShading.ShadingInfo = new MaterialAlphaShadingInfo(
                alpha.A,
                chanA.Lighting
            );

            ctlColorShading.Invalidate();
            ctlAlphaShading.Invalidate();

            // ディフューズソース
            radDSrcMat.SetCheckedByTag(chanD.MatSrc);
            radDSrcVtx.SetCheckedByTag(chanD.MatSrc);
            radDSrcVtx.Enabled = target.HasVtxColors;

            // アルファソース
            radASrcMat.SetCheckedByTag(chanA.MatSrc);
            radASrcVtx.SetCheckedByTag(chanA.MatSrc);
            radASrcVtx.Enabled = target.HasVtxColors;

            // ディフューズ
            // TODO: 有効条件
            // マテリアルレジスタ割り当てならソース選択で有効状態を切り替える
            // 現状はマテリアルレジスタ固定割り当てなのでこのままで良い
            btnDiffuse.Color = diffuse.ToColorNoAlpha();
            trbDiffuse.SetValue(diffuse.ToHsv().V);
            lblDiffuseAssign.Text = target.ColorAssignData.Diffuse.TargetText;
            pnlDiffuse.Enabled = (chanD.MatSrc == AttrColorSource.Reg);

            // アンビエント
            btnAmbient.Color = ambient.ToColorNoAlpha();
            trbAmbient.SetValue(ambient.ToHsv().V);
            lblAmbientAssign.Text = target.ColorAssignData.Ambient.TargetText;
            pnlAmbient.Enabled = chanD.Lighting;

            // スペキュラ
            btnSpecular.Color = specular.ToColorNoAlpha();
            trbSpecular.SetValue(specular.ToHsv().V);
            lblSpecularAssign.Text = target.ColorAssignData.Specular.TargetText;
            pnlSpecular.Enabled = chanS.Lighting;

            // アルファ
            // TODO: 有効条件
            // マテリアルレジスタ割り当てならソース選択で有効状態を切り替える
            // 現状はマテリアルレジスタ固定割り当てなのでこのままで良い
            ctlAlpha.SetValue(alpha.A);
            lblAlphaAssign.Text = target.ColorAssignData.Alpha.TargetText;
            pnlAlpha.Enabled = (chanA.MatSrc == AttrColorSource.Reg);

            // ディフューズ＆アンビエントライト計算
            chkLightingDiffAmb.SetChecked(chanD.Lighting);
            chkLightingDiffAmb.Enabled = target.HasVtxNormals;

            // スペキュラライト計算
            chkLightingSpecular.SetChecked(chanS.Lighting);
            chkLightingSpecular.Enabled = target.HasVtxNormals;

            // アルファライト計算
            chkLightingAlpha.SetChecked(chanA.Lighting);
            chkLightingAlpha.Enabled = target.HasVtxNormals;

            // ＴＥＶ自動計算マーク
            ctlTevAuto.Visible = target.TevAutoCalculation && chkLightingSpecular.Enabled;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy() { return true; }

        /// <summary>
        /// コピー。
        /// </summary>
        public override Object Copy()
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static Object Copy(MaterialGUIAdapter target)
        {
            ChanCtrl chanD  = target.ChannelData.ChanCtrls[0];
            ChanCtrl chanA  = target.ChannelData.ChanCtrls[1];
            ChanCtrl chanS  = target.ChannelData.ChanCtrls[2];
            ColorAssignData colorAssign = target.ColorAssignData;

            ArrayList data = new ArrayList();
            data.Add(chanD.MatSrc);
            data.Add(chanA.MatSrc);
            data.Add(new Rgba(colorAssign.Diffuse.GetTargetRegister(target).Color));
            data.Add(new Rgba(colorAssign.Ambient.GetTargetRegister(target).Color));
            data.Add(new Rgba(colorAssign.Specular.GetTargetRegister(target).Color));
            data.Add(new Rgba(colorAssign.Alpha.GetTargetRegister(target).Color));
            data.Add(chanD.Lighting);
            data.Add(chanA.Lighting);
            data.Add(chanS.Lighting);
            return data;
        }

        /// <summary>
        /// ペーストが可能か。
        /// </summary>
        public override string CanPaste(Object pasteObject)
        {
            return CanPaste(ActiveTarget, pasteObject);
        }

        /// <summary>
        /// ペーストが可能か。
        /// </summary>
        public static string CanPaste(MaterialGUIAdapter destination, Object pasteObject)
        {
            Debug.Assert(pasteObject is ArrayList);
            ArrayList source = (ArrayList)pasteObject;
            Debug.Assert(source.Count == 9);
            string result = "";

            // 頂点カラーが無いのに頂点カラーを利用しようとしていないか
            bool noColorError = false;
            AttrColorSource colorSource;
            colorSource = (AttrColorSource)source[0];
            if((colorSource == AttrColorSource.Vtx) && (!destination.HasVtxColors))
            {
                noColorError = true;
            }
            colorSource = (AttrColorSource)source[1];
            if((colorSource == AttrColorSource.Vtx) && (!destination.HasVtxColors))
            {
                noColorError = true;
            }
            if(noColorError)
            {
                result += StringResMgr.Get("MSG_MATERIAL_COLOR_PASTE_FAILED_NO_VTXCOLOR") + "\n";
            }

            // 法線が無いのにライティングを行おうとしていないか
            bool noNormalError = false;
            bool lighting;
            lighting = (bool)source[6];
            if(lighting && (!destination.HasVtxNormals))
            {
                noNormalError = true;
            }
            lighting = (bool)source[7];
            if(lighting && (!destination.HasVtxNormals))
            {
                noNormalError = true;
            }
            lighting = (bool)source[8];
            if(lighting && (!destination.HasVtxNormals))
            {
                noNormalError = true;
            }
            if(noNormalError)
            {
                result += StringResMgr.Get("MSG_MATERIAL_COLOR_PASTE_FAILED_NO_VTXNORMAL") + "\n";
            }
            return result;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(Object pasteObject)
        {
            Paste(ActiveTarget, pasteObject);
            UpdateProperty();
            NotifyPropertyChanged();
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static void Paste(MaterialGUIAdapter destination, Object pasteObject)
        {
            ChanCtrl chanD  = destination.ChannelData.ChanCtrls[0];
            ChanCtrl chanA  = destination.ChannelData.ChanCtrls[1];
            ChanCtrl chanS  = destination.ChannelData.ChanCtrls[2];
            ColorAssignData colorAssign = destination.ColorAssignData;
            Debug.Assert(pasteObject is ArrayList);
            ArrayList source = (ArrayList)pasteObject;
            Debug.Assert(source.Count == 9);

            chanD.MatSrc = (AttrColorSource)source[0];
            chanA.MatSrc = (AttrColorSource)source[1];
            colorAssign.Diffuse.GetTargetRegister(destination).Color.Set(
                (Rgba)source[2]);
            colorAssign.Ambient.GetTargetRegister(destination).Color.Set(
                (Rgba)source[3]);
            colorAssign.Specular.GetTargetRegister(destination).Color.Set(
                (Rgba)source[4]);
            colorAssign.Alpha.GetTargetRegister(destination).Color.Set(
                (Rgba)source[5]);
            chanD.Lighting = (bool)source[6];
            chanA.Lighting = (bool)source[7];
            chanS.Lighting = (bool)source[8];
        }
        #endregion

        #region ターゲット取得ヘルパ


        /// <summary>
        /// ディフューズコントロール。
        /// </summary>
        private ChanCtrl DiffuseCtrl
        {
            get { return this.ActiveTarget.ChannelData.ChanCtrls[0]; }
        }

        /// <summary>
        /// アルファコントロール。
        /// </summary>
        private ChanCtrl AlphaCtrl
        {
            get { return this.ActiveTarget.ChannelData.ChanCtrls[1]; }
        }

        /// <summary>
        /// スペキュラコントロール。
        /// </summary>
        private ChanCtrl SpecularCtrl
        {
            get { return this.ActiveTarget.ChannelData.ChanCtrls[2]; }
        }

        /// <summary>
        /// ディフューズ割り当てレジスタ。
        /// </summary>
        private ColorRegister DiffuseRegister
        {
            get
            {
                MaterialGUIAdapter m = this.ActiveTarget;
                return m.ColorAssignData.Diffuse.GetTargetRegister(m);
            }
        }

        /// <summary>
        /// アンビエント割り当てレジスタ。
        /// </summary>
        private ColorRegister AmbientRegister
        {
            get
            {
                MaterialGUIAdapter m = this.ActiveTarget;
                return m.ColorAssignData.Ambient.GetTargetRegister(m);
            }
        }

        /// <summary>
        /// スペキュラ割り当てレジスタ。
        /// </summary>
        private ColorRegister SpecularRegister
        {
            get
            {
                MaterialGUIAdapter m = this.ActiveTarget;
                return m.ColorAssignData.Specular.GetTargetRegister(m);
            }
        }

        /// <summary>
        /// アルファ割り当てレジスタ。
        /// </summary>
        private ColorRegister AlphaRegister
        {
            get
            {
                MaterialGUIAdapter m = this.ActiveTarget;
                return m.ColorAssignData.Alpha.GetTargetRegister(m);
            }
        }
        #endregion

        #region イベントハンドラ
        // ディフューズソース
        private void Event_RadDSrc_RadioChecked(object sender, EventArgs e)
        {
            this.DiffuseCtrl.MatSrc = (AttrColorSource)((UIRadioButton)sender).Tag;
            UpdateProperty();
            NotifyPropertyChanged();
        }
        // アルファソース
        private void Event_RadASrc_RadioChecked(object sender, EventArgs e)
        {
            this.AlphaCtrl.MatSrc = (AttrColorSource)((UIRadioButton)sender).Tag;
            UpdateProperty();
            NotifyPropertyChanged();
        }

        // カラーピッカー
        private void Event_ColorPicker_ColorEdit(object sender, ColorEditEventArgs e)
        {
            // ディフューズ
            if (sender == _adapterDiffuse)
            {
                OnEventColorPickerColorEdit(this.DiffuseRegister, e);
            }
            // アンビエント
            else if (sender == _adapterAmbient)
            {
                OnEventColorPickerColorEdit(this.AmbientRegister, e);
            }
            // スペキュラ
            else if (sender == _adapterSpecular)
            {
                OnEventColorPickerColorEdit(this.SpecularRegister, e);
            }
        }
        private void OnEventColorPickerColorEdit(ColorRegister register, ColorEditEventArgs e)
        {
            Rgba  color  = register.Color;
            Color result = e.Color;

            // 編集確定
            if (e.EditFixed)
            {
                // ＲＧＢのみ変更
                color.R = result.R;
                color.G = result.G;
                color.B = result.B;

                UpdateProperty();
                NotifyPropertyChanged();
            }
        }

        // カラートラックバー
        private void Event_TrbColor_ScrollChanged(object sender, ScrollChangedEventArgs e)
        {
            // ディフューズ
            if (sender == trbDiffuse)
            {
                OnEventTrbColorScrollChanged(trbDiffuse, btnDiffuse, _adapterDiffuse, this.DiffuseRegister, e);
            }
            // アンビエント
            else if (sender == trbAmbient)
            {
                OnEventTrbColorScrollChanged(trbAmbient, btnAmbient, _adapterAmbient, this.AmbientRegister, e);
            }
            // スペキュラ
            else if (sender == trbSpecular)
            {
                OnEventTrbColorScrollChanged(trbSpecular, btnSpecular, _adapterSpecular, this.SpecularRegister, e);
            }
        }
        private void OnEventTrbColorScrollChanged(IntTrackBar trb, ColorButton btn, ColorPickerAdapter adapter, ColorRegister register, ScrollChangedEventArgs e)
        {
            Rgba color  = register.Color;
            Hsv  hsv    = color.ToHsv();
            Rgba result = new Hsv(hsv.H, hsv.S, trb.Value).ToRgba();

            // 編集確定
            if (e.EndScroll)
            {
                // ボタン更新
                btn.Color = result.ToColorNoAlpha();

                // アダプタに通知
                adapter.NotifyUpdate();

                // ＲＧＢのみ変更
                color.R = result.R;
                color.G = result.G;
                color.B = result.B;

                UpdateProperty();
                NotifyPropertyChanged();
            }
            // 編集中
            else
            {
                // ボタン更新
                btn.Color = result.ToColorNoAlpha();

                // シェーディング更新
                if      (trb == trbDiffuse)  { ctlColorShading.ShadingInfo.Diffuse  = result.ToColorNoAlpha(); }
                else if (trb == trbAmbient)  { ctlColorShading.ShadingInfo.Ambient  = result.ToColorNoAlpha(); }
                else if (trb == trbSpecular) { ctlColorShading.ShadingInfo.Specular = result.ToColorNoAlpha(); }
                ctlColorShading.Invalidate();
            }
        }

        // アルファ
        private void Event_CtlAlpha_ValueEdit(object sender, ValueEditEventArgs e)
        {
            // 編集確定
            if (e.EditFixed)
            {
                this.AlphaRegister.Color.A = ctlAlpha.Value;
                UpdateProperty();
                NotifyPropertyChanged();
            }
            // 編集中
            else
            {
                ctlAlphaShading.ShadingInfo.Alpha = ctlAlpha.Value;
                ctlAlphaShading.Invalidate();
            }
        }

        // ライト計算
        private void Event_ChkLightingDiffAmb_CheckedChanged(object sender, EventArgs e)
        {
            this.DiffuseCtrl.Lighting = chkLightingDiffAmb.Checked;

            UpdateProperty();
            NotifyPropertyChanged();
        }
        private void Event_ChkLightingSpecular_CheckedChanged(object sender, EventArgs e)
        {
            this.SpecularCtrl.Lighting = chkLightingSpecular.Checked;
            this.ActiveTarget.CalculateTev(); // ＴＥＶ自動計算

            UpdateProperty();
            NotifyPropertyChanged();
        }
        private void Event_ChkLightingAlpha_CheckedChanged(object sender, EventArgs e)
        {
            this.AlphaCtrl.Lighting = chkLightingAlpha.Checked;

            UpdateProperty();
            NotifyPropertyChanged();
        }
        #endregion
    }
}
