﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters
{
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures.Nsrif.Elements;
    using LECore.Structures.LECoreInterface;
    using LayoutEditor.Utility;
    using LECore.Structures;

    //
    // Model クラスは削除されました
    // 以前は、マテリアルのオーナーとして、Model クラスが存在しており、
    // マテリアルのGUIがModel クラスに依存していました。
    //

    // Polygon クラスは消去されました。




    /// <summary>
    /// ユーザデータ(現状では、Mockオブジェクトです。)
    /// </summary>
    public class UserDataHolder
    {
        /// <summary>
        /// コンストラクタ。
        /// 以前は、引数にElem_user_data_arrayをとって、
        /// 内部データを構築していました。
        /// </summary>
        public UserDataHolder()
        {
        }

        /// <summary>
        /// 未設定かどうか。
        /// Mockメソッドです。
        /// 従来は get { return _datas.Count == 0; } となっていました。
        /// </summary>
        public bool IsEmpty
        {
            get { return true; }
        }

        #region ドキュメント(消去されました)
        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_material element)
        {
            UpdateDocElement(element.user_data_array);
        }

        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_node element)
        {
            UpdateDocElement(element.user_data_array);
        }

        /// <summary>
        /// ドキュメント要素更新。// 消去されました。
        /// </summary>
        private void UpdateDocElement(Elem_user_data_array element)
        {
        }
        #endregion
    }


    /// <summary>
    /// サムネイルインタフェース。
    /// </summary>
    public interface IThumbnail
    {
        /// <summary>
        /// 名前。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// カラーイメージ。
        /// </summary>
        Bitmap ColorImage { get; }

        /// <summary>
        /// アルファイメージ。
        /// </summary>
        Bitmap AlphaImage { get; }

        /// <summary>
        /// カラーイメージ描画。
        /// </summary>
        void DrawColorImage(Graphics g, Rectangle rect, bool enable);

        /// <summary>
        /// アルファイメージ描画。
        /// </summary>
        void DrawAlphaImage(Graphics g, Rectangle rect, bool enable);

        /// <summary>
        /// GDIイメージ描画。
        /// </summary>
        void DrawImageOnCheckerBoard(Graphics g, Rectangle rect, bool enable);
    }

    /// <summary>
    /// サムネイル描画クラス。
    /// </summary>
    public sealed class ThumbnailDraw
    {
        /// <summary>
        /// イメージ描画。
        /// </summary>
        public static void DrawImage(Graphics g, Image image, Rectangle rect, bool enable)
        {
            // 補間モード変更
            InterpolationMode im = g.InterpolationMode;
            g.InterpolationMode = InterpolationMode.NearestNeighbor;

            // サンプリングポイントがずれるのを考慮して半ピクセルずらして補間
            if (enable)
            {
                g.DrawImage(image, rect, -0.5f, -0.5f, image.Width, image.Height, GraphicsUnit.Pixel);
            }
            else
            {
                // グレースケール化
                ColorMatrix cm = new ColorMatrix();

                cm.Matrix00 = 0.3f;
                cm.Matrix10 = 0.3f;
                cm.Matrix20 = 0.3f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.3f;
                cm.Matrix11 = 0.3f;
                cm.Matrix21 = 0.3f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.3f;
                cm.Matrix12 = 0.3f;
                cm.Matrix22 = 0.3f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;

                using (ImageAttributes ia = new ImageAttributes())
                {
                    ia.SetColorMatrix(cm);
                    g.DrawImage(image, rect, -0.5f, -0.5f, image.Width, image.Height, GraphicsUnit.Pixel, ia);
                }
            }

            // 補間モード復帰
            g.InterpolationMode = im;
        }

#if true
        /// <summary>
        /// イメージ描画。
        /// </summary>
        public static void DrawImage(Graphics g, Image image, Rectangle destRect, int srcW, int srcH, bool enable)
        {
            // 補間モード変更
            InterpolationMode im = g.InterpolationMode;
            g.InterpolationMode = InterpolationMode.NearestNeighbor;

            // サンプリングポイントがずれるのを考慮して半ピクセルずらして補間
            if (enable)
            {
                //g.DrawImage(image, destRect, -0.5f, -0.5f, image.Width, image.Height, GraphicsUnit.Pixel);
                g.DrawImage(image, destRect, -0.5f, -0.5f, srcW, srcH, GraphicsUnit.Pixel);
            }
            else
            {
                // グレースケール化
                ColorMatrix cm = new ColorMatrix();

                cm.Matrix00 = 0.3f;
                cm.Matrix10 = 0.3f;
                cm.Matrix20 = 0.3f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.3f;
                cm.Matrix11 = 0.3f;
                cm.Matrix21 = 0.3f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.3f;
                cm.Matrix12 = 0.3f;
                cm.Matrix22 = 0.3f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.0f;

                using (ImageAttributes ia = new ImageAttributes())
                {
                    ia.SetColorMatrix(cm);
                    //g.DrawImage(image, destRect, -0.5f, -0.5f, image.Width, image.Height, GraphicsUnit.Pixel, ia);
                    g.DrawImage(image, destRect, -0.5f, -0.5f, srcW, srcH, GraphicsUnit.Pixel, ia);
                }
            }

            // 補間モード復帰
            g.InterpolationMode = im;
        }
#endif

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private ThumbnailDraw() {}
    }

    /// <summary>
    /// テクスチャイメージクラス。
    /// DocumentObjectを基底クラスとはしない。
    /// テクスチャ = ファイル（ドキュメント）の図式は成り立たない。
    /// </summary>
    public sealed class TexImage : IThumbnail
    {
        // テクスチャ＆パレット
        private readonly Texture _texture;

        // インスタンスリスト
        private static ArrayList _instances = new ArrayList();

        /// <summary>
        /// テクスチャ。
        /// </summary>
        public Texture Texture
        {
            get { return _texture; }
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static TexImage Create(string texName, string palName)
        {
            return Create(TextureManager.Search(texName));
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static TexImage Create(Texture texture)
        {
            if( texture == null)
            {
                return null;
            }

            // 同じ組み合わせのインスタンスを検索
            TexImage instance = Search(texture);

            // なければ作成してリスト登録
            if (instance == null)
            {
                instance = new TexImage(texture);
                _instances.Add(instance);
            }

            return instance;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private TexImage(Texture texture)
        {
            Debug.Assert(texture != null);
            _texture = texture;
        }

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static TexImage()
        {
            // インスタンスリストに存在しないデータがあると困るので
            // 閉じられた時にリストから外すようにする
            // 勿論それらはどのマテリアルからも参照されていないはず
        }

        /// <summary>
        /// 検索。
        /// </summary>
        private static TexImage Search(Texture texture)
        {
            foreach (TexImage instance in _instances)
            {
                if (instance.Texture == texture)
                {
                    return instance;
                }
            }
            return null;
        }

        #region IThumbnail実装
        /// <summary>
        /// 名前です。
        /// </summary>
        public string Name
        {
            get
            {
                string name = _texture.Name;
                return name;
            }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public Bitmap ColorImage
        {
            get { return _texture.ColorImage; }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public Bitmap AlphaImage
        {
            get { return _texture.AlphaImage; }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawColorImage(Graphics g, Rectangle rect, bool enable)
        {
            _texture.DrawColorImage(g, rect, enable);
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawAlphaImage(Graphics g, Rectangle rect, bool enable)
        {
            _texture.DrawAlphaImage(g, rect, enable);
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawImageOnCheckerBoard(Graphics g, Rectangle rect, bool enable)
        {
            _texture.DrawImageOnCheckerBoard(g, rect, enable);
        }
        #endregion
    }



    /// <summary>
    /// テクスチャクラス。
    /// </summary>
    public class Texture : IThumbnail
    {
        // プロパティデータ
        private DataSize _dataSize = null;
        int              _mipmapLevel = 0;

        readonly ITextureImage _texImage = null;

        /// <summary>
        /// コンストラクタ
        /// 従来は、NsrifDocument document を引数にとって、初期化を行うコンストラクタが
        /// 提供されていました。
        /// </summary>
        public Texture( ITextureImage texImage, bool isOwnedByCurrentSubScene = false )
        {
            _texImage = texImage;
            // TODO:データサイズを計算する
            _dataSize = new DataSize( 0 );

            this.IsOwnedByCurrentSubScene = isOwnedByCurrentSubScene;
        }

        #region アトリビュート

        /// <summary>
        /// フォーマット。
        /// </summary>
        public TexImagePixelFmt PixelFormat
        {
            get { return _texImage.PixelFmt; }
        }

        /// <summary>
        /// 幅。
        /// </summary>
        public int Width
        {
            get { return _texImage.Size.X; }
        }

        /// <summary>
        /// 高さ。
        /// </summary>
        public int Height
        {
            get { return _texImage.Size.Y; }
        }

        /// <summary>
        /// サイズ。
        /// </summary>
        public Size Size
        {
            get { return new Size(this.Width, this.Height); }
        }

        /// <summary>
        /// 実際のサイズ。
        /// </summary>
        public Size ActualSize
        {
            get { return new Size(_texImage.ActualSize.X, _texImage.ActualSize.Y); }
        }

        /// <summary>
        /// ミップマップレベル。
        /// </summary>
        public int MipmapLevel
        {
            get { return _mipmapLevel; }
        }

        /// <summary>
        /// データサイズ。
        /// </summary>
        public DataSize DataSize
        {
            get { return _dataSize; }
        }

        public string Name
        {
            get { return _texImage.Name; }
        }

        public string FilePath
        {
            get { return _texImage.FilePath; }
        }

        /// <summary>
        /// ミップマップ可能かどうか。
        /// </summary>
        public bool IsMipmapAvailable
        {
            get { return this.MipmapLevel >= 2; }
        }

        /// <summary>
        /// 現在の編集対象サブシーンが保持しているかどうか？
        /// UI に選択肢として表示できるのは、編集対象サブシーンの持っているテクスチャーだけ。
        /// </summary>
        public bool IsOwnedByCurrentSubScene
        {
            get; private set;
        }

        public bool IsHidingFromList
        {
            get { return _texImage.IsHidingFromList; }
        }

        #endregion

        #region ＵＩテキスト
        /// <summary>
        /// フォーマットテキスト。
        /// </summary>
        public string FormatText
        {
            get { return _texImage.PixelFmt.ToString(); }
        }

        /// <summary>
        /// サイズテキスト。
        /// </summary>
        public string SizeText
        {
            get { return string.Format("{0} x {1}", this.Width, this.Height); }
        }

        /// <summary>
        /// ミップマップレベルテキスト。
        /// </summary>
        public string MipmapLevelText
        {
            get
            {
                if (this.MipmapLevel > 1) { return StringResMgr.Get("TEX_MMLV_ENABLE", this.MipmapLevel); }
                else                      { return StringResMgr.Get("TEX_MMLV_DISABLE"); }
            }
        }

        /// <summary>
        /// テクスチャソースの形式を取得
        /// </summary>
        public TextureSourceType SourceType
        {
            get
            {
                return _texImage.SourceType;
            }
        }
        #endregion

        #region サムネイル

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public Bitmap RenderingImage
        {
            get { return _texImage.GDIBitmap; }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public Bitmap ColorImage
        {
            get { return _texImage.ColorGDIBitmap; }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public Bitmap AlphaImage
        {
            get { return _texImage.AlphaGDIBitmap; }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawColorImage(Graphics g, Rectangle rect, bool enable)
        {
            DrawImage(g, ColorImage, rect, enable);
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawAlphaImage(Graphics g, Rectangle rect, bool enable)
        {
            if (AlphaImage != null)
            {
                DrawImage(g, AlphaImage, rect, enable);
            }
            else
            {
                // サイズ計算
                Rectangle rcDraw = CalcDrawRectangle(_texImage, rect);

                // 白で描画
                g.FillRectangle(new SolidBrush(Color.White), rcDraw);
            }
        }

        /// <summary>
        /// インタフェース実装。
        /// </summary>
        public void DrawImageOnCheckerBoard(Graphics g, Rectangle rect, bool enable)
        {
            TextureThumbnailHelper.DrawThumbnail(
                g,
                RenderingImage.Size.Width,
                RenderingImage.Size.Height,
                RenderingImage,
                rect.Size.Width,
                rect.Size.Height,
                rect.X,
                rect.Y,
                false);
        }

        /// <summary>
        /// イメージ描画。
        /// </summary>
        public void DrawImage(Graphics g, Image image, Rectangle rect, bool enable)
        {
            if (!TextureThumbnailHelper.CheckImageValid(image))
            {
                return;
            }

            // サイズ計算
            Rectangle rcDraw = CalcDrawRectangle(_texImage, rect);

            ThumbnailDraw.DrawImage( g,
                                     image,
                                     rcDraw,
                                     _texImage.Size.X,
                                     _texImage.Size.Y,
                                     enable);
        }

        /// <summary>
        /// 描画領域を計算します。
        /// </summary>
        private Rectangle CalcDrawRectangle(ITextureImage texImage, Rectangle rect)
        {
            float scaleW = 1.0F;
            float scaleH = 1.0F;
            int w = texImage.Size.X;
            int h = texImage.Size.Y;

            if (w < h)
            {
                scaleW = (float)w / (float)h;
                scaleH = 1.0F;
            }
            else
            {
                if (w > h)
                {
                    scaleW = 1.0F;
                    scaleH = (float)h / (float)w;
                }
            }

            return new Rectangle(rect.X, rect.Y,
                        (int)((float)rect.Width * scaleW),
                        (int)((float)rect.Height * scaleH));
        }
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public PropertyPanelType ObjectID
        {
            get { return PropertyPanelType.Texture; }
        }

        /// <summary>
        /// オーバーライド。
        /// 規定クラスを削除するために、消去されました。
        /// get { return RtexObject.ElementName; }となっていました。
        /// </summary>
        public string FileExt
        {
            get { return "TODO:このような情報は、別クラスが提供するべき"; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public string FileFilter
        {
            get { return StringResMgr.Get("FILEFILTER_RTEX"); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public void UpdateDocument()
        {
            // 特に更新する必要なし
        }
        #endregion

        /// <summary>
        ///
        /// </summary>
        public bool IsWidthPow2
        {
            get { return IsPow2( this.Width); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsHeightPow2
        {
            get { return IsPow2( this.Height); }
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsPow2( int value)
        {
            return value == 0 ? false : ( value & ( value - 1 )) == 0;
        }
    }
}
