﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using LECore.Structures;
using System.Collections.Generic;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters
{
    #region DocComponentChangedEvent

    /// <summary>
    /// ドキュメント構成変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocComponentChangedEventHandler( object sender, DocComponentChangedEventArgs e);

    /// <summary>
    /// ドキュメント構成変更イベントデータクラス。
    /// </summary>
    public sealed class DocComponentChangedEventArgs : EventArgs
    {
    }

    #endregion // DocComponentChangedEvent

    #region DocPropertyChangedEvent
    /// <summary>
    /// プロパティ編集イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocPropertyChangedEventHandler(object sender, DocPropertyChangedEventArgs e);

    /// <summary>
    /// プロパティ編集イベントデータクラス。
    /// </summary>
    public sealed class DocPropertyChangedEventArgs : EventArgs
    {
    }
    #endregion // DocPropertyChangedEvent

    /// <summary>
    /// ペインを対象とする、IGuiAdapterです。
    /// <remarks>
    /// ページ種類固有の処理(OnModified(PropertyPage sender))を、ページ側に移植しました。
    /// </remarks>
    /// </summary>
    public interface IBaseGuiAdapter
    {
        /// <summary>
        /// 名前。
        /// </summary>
        string Name { get; }

        // アダプタオブジェクトがターゲット対象としているかどうかを調査します。
        bool HasSameTargets(object anotherAdapter);

        //// 内部状態を更新します。
        void Refresh();

        IPane Target { get; }
    }

    /// <summary>
    /// ＧＵＩオブジェクトＩＤ。
    /// PropertyPanelType
    /// </summary>
    public enum PropertyPanelType
    {
        Null,
        NullPane,
        PicturePane,
        TextBoxPane,
        WindowPane,
        RegionPane,
        PartsPane,
        Texture,
        Palette,
        CapturePane,
        Alignment,
        Scissor,
    }

    /// <summary>
    /// ＧＵＩオブジェクトグループクラス。
    /// </summary>
    public sealed class PaneGuiAdapterGroup
    {
        // オブジェクトリスト
        private readonly List<IBaseGuiAdapter> _objects = new List<IBaseGuiAdapter>();
        // アクティブオブジェクト
        private PaneGUIAdapter _active = null;

        /// <summary>内容変更イベント。</summary>
        public event EventHandler Changed = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PaneGuiAdapterGroup()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PaneGuiAdapterGroup(PaneGuiAdapterGroup src)
        {
            _objects.AddRange(src.Objects);
            _active   = src._active;
        }

        /// <summary>
        /// オブジェクトリスト。
        /// </summary>
        public IEnumerable<IBaseGuiAdapter> Objects
        {
            get { return _objects; }
        }

        /// <summary>
        /// アクティブオブジェクト。
        /// </summary>
        public PaneGUIAdapter Active
        {
            get { return _active;  }
            set
            {
                if (_active != value)
                {
                    _active = value;
                    InvokeChanged();
                }
            }
        }

        /// <summary>
        /// 未設定かどうか。
        /// </summary>
        public bool IsEmpty
        {
            get { return _objects.Count == 0; }
        }

        /// <summary>
        /// 複数設定かどうか。
        /// </summary>
        public bool IsMulti
        {
            get { return _objects.Count >= 2; }
        }

        /// <summary>
        /// サイズモードを取得します。
        /// </summary>
        public PaneSizeMode PaneSizeMode
        {
            get
            {
                PaneSizeMode sizeMode = PaneSizeMode.All;
                foreach (IBaseGuiAdapter pane in Objects)
                {
                    if (pane.Target != null)
                    {
                        sizeMode &= pane.Target.PaneSizeMode;
                    }
                }

                return sizeMode;
            }
        }

        /// <summary>
        /// クリア。
        /// </summary>
        public void Clear()
        {
            if (_objects.Count > 0)
            {
                _objects.Clear();
                _active = null;
                InvokeChanged();
            }
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PaneGUIAdapter obj)
        {
            Debug.Assert(obj != null);

            if (!(_objects.Count == 1 && _active == obj))
            {
                _objects.Clear();
                _objects.Add(obj);
                _active = obj;
                InvokeChanged();
            }
        }

        /// <summary>
        /// 追加。
        /// </summary>
        public void Add(PaneGUIAdapter obj)
        {
            Add(obj, true);
        }

        /// <summary>
        /// 追加。
        /// </summary>
        public void Add(PaneGUIAdapter obj, bool activate)
        {
            Debug.Assert(obj != null);
            Debug.Assert(!_objects.Contains(obj));

            _objects.Add(obj);

            if (_active == null || activate)
            {
                _active = obj;
            }
            InvokeChanged();
        }

        /// <summary>
        /// 削除。
        /// </summary>
        public void Remove(PaneGUIAdapter obj)
        {
            Debug.Assert(obj != null);
            Debug.Assert(_objects.Contains(obj));

            _objects.Remove(obj);

            if (_objects.Count > 0 && _active == obj)
            {
                _active = _objects[0] as PaneGUIAdapter;
            }
            else if (_objects.Count == 0)
            {
                _active = null;
            }
            InvokeChanged();
        }

        #region 比較

        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PaneGuiAdapterGroup src)
        {
            if (src == null) return false;
            if (src == this) return false;

            // 比較処理
            if (_active == src._active)
            {
                if (_objects.Count == src._objects.Count)
                {
                    foreach (IBaseGuiAdapter obj in _objects)
                    {
                        if (!src._objects.Contains(obj))
                        {
                            return false;
                        }
                    }
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PaneGuiAdapterGroup);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            // TODO: 文字列化
            return base.ToString();
        }

        /// <summary>
        /// 内容変更イベント発行。
        /// </summary>
        private void InvokeChanged()
        {
            if (Changed != null)
            {
                Changed(this, EventArgs.Empty);
            }
        }
    }
}
