﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Drawing;
using System.Diagnostics;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows
{
    using LECore.Structures;
    using LECore.Manipulator;
    using LayoutEditor.Structures.SerializableObject;
    using Controls;
    using common;
    using Utility;

    /// <summary>
    /// 検索、置換ダイアログ
    /// </summary>
    public partial class PaneSearchWindow : LEToolWindow
    {
        /// <summary>
        /// 検索対象文字列の種類。
        /// </summary>
        enum TargetStringKind
        {
            None = 0,
            PaneName = 1,
            Commemnt = 2,
            UserData = 3
        }

       #region MatchSubString

        /// <summary>
        /// 検索一致、部分文字列
        /// </summary>
        struct MatchSubString
        {
            IPane       _pane;
            string      _targetString;
            bool        _bForward;

            int _matchStartIndex;
            int _matchLength;

            public IPane Pane { get { return _pane; } }
            public string PaneName
            {
                get
                {
                    if( _pane != null )
                    {
                        return _pane.PaneName;
                    }
                    else
                    {
                        return string.Empty;
                    }
                }
            }
            public string TargetString {get { return _targetString; }}
            public int MatchStartIndex { get { return _matchStartIndex; } }
            public int MatchLength { get { return _matchLength; } }
            public bool IsMatch {get { return _matchLength != 0; }}

            /// <summary>
            /// 検索開始インデックス
            /// </summary>
            public int SearchStartIndex
            {
                get
                {
                    if( _bForward )
                    {
                        return _matchStartIndex + MatchLength;
                    }
                    else
                    {
                        return _matchStartIndex;
                    }
                }
            }

            /// <summary>
            /// 同一の内容か判定します。
            /// </summary>
            public bool IsSame( IPane pane, string baseString, bool bForward )
            {
                return ( pane == _pane && _targetString == baseString && _bForward == bForward );
            }

            /// <summary>
            /// 初期化します。
            /// </summary>
            public void Initialize( IPane pane, string baseString, bool bForward )
            {
                Debug.Assert( pane != null );
                Debug.Assert( baseString != null );

                _pane = pane;
                _targetString = baseString;
                _bForward = bForward;

                if( _bForward )
                {
                    _matchStartIndex = 0;
                }
                else
                {
                    _matchStartIndex = TargetString.Length;
                }
                _matchLength = 0;
            }

            /// <summary>
            /// 検索一致部分を設定します。
            /// </summary>
            public void SetMatchSubString( int startIndex, int length )
            {
                _matchStartIndex = startIndex;
                _matchLength = length;
            }
        }

       #endregion MatchSubString

       #region SearchOptions
        struct SearchOptions
        {
            /// <summary>
            /// 検索対象文字列
            /// </summary>
            TargetStringKind _targetStringKind;
            /// <summary>
            /// 正規表現を使用するか
            /// </summary>
            bool _bUseRegx;
            /// <summary>
            /// 順方向検索か？
            /// </summary>
            bool _searchForward;

          #region プロパティ
            public bool SearchForward
            {
                set { _searchForward = value; }
                get { return _searchForward; }
            }

            public bool UseRegex
            {
                set { _bUseRegx = value; }
                get { return _bUseRegx; }
            }

            public TargetStringKind TargetStringKind
            {
                set { _targetStringKind = value; }
                get { return _targetStringKind; }
            }
           #endregion プロパティ
        }

       #endregion

       #region
        /// <summary>
        /// ハッシュ関数のみをオーバーライドした、文字列キュー
        /// </summary>
        class HistoryQueue : Queue<string>
        {
            public override int GetHashCode()
            {
                int hashCode = 1;
                string[] itemSet = base.ToArray();
                for( int i = 0 ; i < itemSet.Length ; i++ )
                {
                    hashCode = hashCode * 31 + itemSet[i].GetHashCode();
                }
                return hashCode;
            }
        }
       #endregion

       #region フィールド
        /// <summary>
        /// 検索文字列
        /// </summary>
        string _searchedString = string.Empty;
        /// <summary>
        /// 最後に検索したシーン
        /// </summary>
        private ISubScene _finallyTargetSubScene = null;
        /// <summary>
        /// 最後に検索で見つかったペイン
        /// </summary>
        private IPane _finallyTargetPane = null;
        /// <summary>
        /// 検索結果
        /// </summary>
        MatchSubString _matchString = new MatchSubString();
        /// <summary>
        /// 検索オプション
        /// </summary>
        SearchOptions _options = new SearchOptions();
        /// <summary>
        /// 検索文字列：履歴キュー
        /// </summary>
        HistoryQueue _searchedStringHistory = new HistoryQueue();
        /// <summary>
        /// 置換文字列：履歴キュー
        /// </summary>
        HistoryQueue _replacedStringHistory = new HistoryQueue();

        const int MaxHistoryNum = 10;

        ShortcutHandler _shortCutHandler = null;
       #endregion フィールド

       #region プロパティ


        /// <summary>
        /// ペインセット
        /// </summary>
        IEnumerable<IPane> _CurrentPaneSet
        {
            get
            {
                if( _ISubScene != null )
                {
                    return _ISubScene.IPaneArray;
                }
                else
                {
                    return new IPane[0];
                }
            }
        }

        /// <summary>
        /// サブシーン
        /// </summary>
        ISubScene _ISubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// 置換結果の文字列が正しいか？
        /// </summary>
        bool _IsReplacedStringValid
        {
            get
            {
                string str = GetReplacedString_();
                switch( _options.TargetStringKind )
                {
                    case TargetStringKind.PaneName:
                        ISubScene   subScene = _ISubScene;
                        if( subScene == null ||
                            SubSceneHelper.FindPaneByName( subScene, str ) != null )
                        {
                            return false;
                        }
                        return PaneHelper.CheckPaneNameValid( str );

                    case TargetStringKind.Commemnt: return true;
                    case TargetStringKind.UserData: return PaneHelper.CheckUserDataValid( str );
                    default: return false;
                }
            }
        }
        #endregion プロパティ

        #region オーバーライド

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (_shortCutHandler.ProcessCmdKey(keyData))
            {
                return true;
            }

            switch (keyData)
            {
                case Keys.Control | Keys.Z:
                case Keys.Control | Keys.Shift | Keys.Z:
                    // テキストボックスにフォーカスがあるかにかかわらずメッセージを伝播します。
                    if (NotifyCmdKeyMessageToOwner(ref msg, keyData))
                    {
                        return true;
                    }
                    break;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        #region 設定保存、読み込み関連
        /// <summary>
        /// 履歴キュー用データ名取得
        /// </summary>
        static string GetHistoryDataName_( string dataName, int i )
        {
            return dataName + "-" + i.ToString();
        }

        /// <summary>
        /// 履歴キュー保存
        /// </summary>
        void SaveHistoryQueue_( HistoryQueue history, string dataName, LEToolFormSetting setting )
        {
            string[] strSet = history.ToArray();
            for( int i = 0;i < strSet.Length; i++ )
            {
                LEControlUserDataChunk data =
                    new LEControlUserDataChunk( GetHistoryDataName_( dataName, i ), strSet[i] );
                setting.UserDataSet.Add( data );
            }
        }

        /// <summary>
        /// 履歴キュー読み込み
        /// </summary>
        void LoadHistoryQueue_( LEToolFormSetting setting, string dataName, HistoryQueue history )
        {
            LEControlUserDataChunk data = null;
            for( int i = 0 ; i < MaxHistoryNum ; i++ )
            {
                data = setting.FindUserDataByName( GetHistoryDataName_( dataName, i ) );
                if( data != null )
                {
                    history.Enqueue( data.Value );
                }
            }
        }

        /// <summary>
        /// 状態を保存します。
        /// 最小化状態のウインドウの状態は保存しません。
        /// </summary>
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            if (option.AlsoSaveOtherThanWorkspace)
            {
                SaveHistoryQueue_(_searchedStringHistory, "searchHistory", setting);
                SaveHistoryQueue_(_replacedStringHistory, "replaceHistory", setting);
            }

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 状態を読み込みます。
        /// </summary>
        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            if (option.AlsoLoadOtherThanWorkspace)
            {
                LoadHistoryQueue_(setting, "searchHistory", _searchedStringHistory);
                LoadHistoryQueue_(setting, "replaceHistory", _replacedStringHistory);

                UpdateHistories_();
            }

            base.LoadSetting(setting, option);
        }
        #endregion 設定保存、読み込み関連
        public override bool AllowToChangeVisible { get { return true; } }
        public override Keys CustomShortcut { get { return Keys.Control | Keys.F; } }
       #endregion


        /// <summary>
        /// GUI初期化
        /// </summary>
        void InitializeProperties_()
        {
            // TODO:リソース化
            _tcbTargetString.Items.Add( StringResMgr.Get( "PANESEARCH_TARGETSTR_PANENAME" ) );
            _tcbTargetString.Items.Add(StringResMgr.Get("TAG_COMMENT"));
            _tcbTargetString.Items.Add( StringResMgr.Get( "PANESEARCH_TARGETSTR_USERDATA" ) );
            _tcbTargetString.SelectedIndex = 0;

            _options.UseRegex = false;
            _options.SearchForward = true;
            _options.TargetStringKind = TargetStringKind.PaneName;

            _cmbFindString.Tag = 0;
            _cmbReplacedString.Tag = 0;

            UpdateProperties_();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneSearchWindow()
        {
            InitializeComponent();
            InitializeProperties_();

            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_mspMain);

            // ドッキングウインドウ用のショートカットの自前処理の設定
            _shortCutHandler = new ShortcutHandler(_mspMain);
        }

        /// <summary>
        /// GUI更新
        /// </summary>
        void UpdateProperties_()
        {
            UpdateSearchResult_();
            UpdateButtonState_();
            UpdateHistories_();

            _cmbReplacedString.Enabled = _chkReplace.Checked;
        }

        /// <summary>
        /// 置換された文字列を取得します。
        /// </summary>
        string GetReplacedString_()
        {
            string newStr = _matchString.TargetString;
            newStr = newStr.Remove( _matchString.MatchStartIndex, _matchString.MatchLength );
            newStr = newStr.Insert( _matchString.MatchStartIndex, _cmbReplacedString.Text );

            return newStr;
        }

        /// <summary>
        /// 検索結果の更新
        /// </summary>
        void UpdateSearchResult_()
        {
            if( _matchString.IsMatch )
            {
                _tbxPaneName.Text = _matchString.PaneName;

                _tbxBeforeReplace.ResetText();
                _tbxBeforeReplace.Text = _matchString.TargetString;
                _tbxBeforeReplace.SelectionStart = _matchString.MatchStartIndex;
                _tbxBeforeReplace.SelectionLength = _matchString.MatchLength;
                _tbxBeforeReplace.SelectionColor = Color.Red;

                if( _chkReplace.Checked )
                {
                    _tbxAfterReplace.ResetText();
                    _tbxAfterReplace.Text = GetReplacedString_();

                    _tbxAfterReplace.SelectionStart = _matchString.MatchStartIndex;
                    _tbxAfterReplace.SelectionLength = _cmbReplacedString.Text.Length;
                    _tbxAfterReplace.SelectionColor = Color.Red;
                }
                else
                {
                    _tbxAfterReplace.ResetText();
                }

                _pnlReplace.Enabled = _chkReplace.Checked;
            }
            else
            {
                _tbxPaneName.Text = string.Empty;
                _tbxBeforeReplace.ResetText();
                _tbxAfterReplace.ResetText();

                _pnlReplace.Enabled = false;
            }
            UpdateButtonState_();
        }

        /// <summary>
        /// ボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            _btnFindNext.Enabled = _searchedString != string.Empty;
            _btnReplace.Enabled =
                _matchString.IsMatch &&
                _chkReplace.Checked &&
                _IsReplacedStringValid;

        }

        /// <summary>
        /// 履歴更新
        /// </summary>
        void UpdateHistory_( Queue<string> historyQueue, ComboBox cmbHistory )
        {
            int hashComboboxHistory = (int)cmbHistory.Tag;
            int hashCurrentHistory = historyQueue.GetHashCode();
            if( hashComboboxHistory != hashCurrentHistory )
            {
                string[] historyStr = historyQueue.ToArray();
                Array.Reverse( historyStr );

                cmbHistory.Items.Clear();
                cmbHistory.Items.AddRange( historyStr );
                cmbHistory.Tag = hashCurrentHistory;
            }
        }

        /// <summary>
        /// 履歴更新
        /// </summary>
        void UpdateHistories_()
        {
            UpdateHistory_( _searchedStringHistory, _cmbFindString );
            UpdateHistory_( _replacedStringHistory, _cmbReplacedString );
        }


       #region 検索関連

        /// <summary>
        /// 次のペインの番号を取得する。
        /// </summary>
        int GetNextPaneIdx_( int idxCurrentPane )
        {
            if( _options.SearchForward )
            {
                if( idxCurrentPane + 1 < _CurrentPaneSet.Count() )
                {
                    return idxCurrentPane + 1;
                }
                else
                {
                    return 0;
                }
            }
            else
            {
                if( idxCurrentPane > 0 )
                {
                    return idxCurrentPane - 1;
                }
                else
                {
                    return _CurrentPaneSet.Count() - 1;
                }
            }
        }

        /// <summary>
        /// 検索対象文字列を取得します。
        /// </summary>
        string GetTargetString_( IPane pane )
        {
            switch( _options.TargetStringKind )
            {
                case TargetStringKind.PaneName: return pane.PaneName;
                case TargetStringKind.Commemnt: return pane.UserCommentString;
                case TargetStringKind.UserData: return pane.UserData;
                default: Debug.Assert( false ); return pane.PaneName;
            }
        }

        /// <summary>
        /// 文字列比較を利用した、一致の検査
        /// </summary>
        bool CheckMatchingByString_( ref MatchSubString matchString )
        {
            string targetString = matchString.TargetString;
            int startIdx = matchString.SearchStartIndex;

            // 不正な状況では、失敗とします。
            if( _searchedString.Length <= 0 ||
                targetString.Length <= 0 ||
                startIdx < 0 )
            {
                return false;
            }


            if( !_options.UseRegex )
            {
                // 正規表現を利用しない場合
                int idx = -1;
                if( _options.SearchForward )
                {
                    // 順方向検索
                    idx = targetString.IndexOf( _searchedString, startIdx );
                }
                else
                {
                    // 逆方向検索
                    if( startIdx - 1 >= 0 )
                    {
                        idx = targetString.LastIndexOf( _searchedString, startIdx - 1 );
                    }
                }

                // 発見された場合…
                if( idx != -1 )
                {
                    matchString.SetMatchSubString( idx, _searchedString.Length );
                    return true;
                }
            }
            else
            {
                // 正規表現を利用する場合
                Match m = null;
                if( _options.SearchForward )
                {
                    Regex forewordRegex =
                        new Regex( _searchedString, RegexOptions.None );
                    // 順方向検索
                    m = forewordRegex.Match( targetString, startIdx );
                }
                else
                {
                    // 逆方向検索
                    Regex backwordRegex =
                        new Regex( _searchedString, RegexOptions.RightToLeft );
                    m = backwordRegex.Match( targetString, startIdx );
                }

                // 発見された場合…
                if( m.Success )
                {
                    matchString.SetMatchSubString( m.Index, m.Value.Length );
                    return true;
                }
            }

            // 調査したが、発見されなかった。
            return false;
        }

        /// <summary>
        /// 検索します
        /// </summary>
        void SearchTargetPane_()
        {
            var currentPaneSet = _CurrentPaneSet;
            int currentIndex = 0;

            // 検索開始
            // 不正な場合は、検索を行いません。
            if( currentPaneSet.Count() <= 0 )
            {
                _finallyTargetSubScene = null;
                _finallyTargetPane = null;
                UpdateSearchResult_();
                return;
            }

            // 前回の検索とシーンが違う場合には検索開始の検索情報を初期化する
            if( _ISubScene != _finallyTargetSubScene )
            {
                _finallyTargetSubScene = null;
                _finallyTargetPane = null;
            }

            // 前回の検索で見つかったペインを使い検索開始インデックスを決定する
            if( _finallyTargetPane != null )
            {
                for( int index = 0; index < currentPaneSet.Count(); index++ )
                {
                    if( _finallyTargetPane == currentPaneSet.ElementAt( index ) )
                    {
                        currentIndex = index;
                        break;
                    }
                }
            }

            _finallyTargetPane = null;

            //
            int startIndex = currentIndex;
            do
            {
                IPane currentPane = _CurrentPaneSet.ElementAt(currentIndex);
                bool bSame = _matchString.IsSame(
                    currentPane,
                    GetTargetString_( currentPane ),
                    _options.SearchForward );

                if( !bSame )
                {
                    _matchString.Initialize(
                        currentPane,
                        GetTargetString_( currentPane ),
                        _options.SearchForward );
                }

                if( CheckMatchingByString_( ref _matchString ) )
                {
                    // 発見
                    // 必要なら、選択を行う。
                    if( _chkAutoSelect.Checked )
                    {
                        SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                        subSceneMnp.BindTarget( _ISubScene );
                        subSceneMnp.BeginSelectSetChange();
                        subSceneMnp.ResetSelectedSet();
                        subSceneMnp.SelectPanesByPaneRef( _matchString.Pane );
                        subSceneMnp.EndSelectSetChange();
                    }
                    break;
                }
                currentIndex = GetNextPaneIdx_( currentIndex );
            }
            while( startIndex != currentIndex );

            // 履歴に登録
            StoreToHistoryQueue_( _searchedStringHistory, _searchedString );

            UpdateSearchResult_();

            // 検索したシーン、ペインを記録しておく
            _finallyTargetSubScene = _ISubScene;
            _finallyTargetPane = _matchString.Pane;
        }

        /// <summary>
        /// 履歴キューに記録する。
        ///
        /// </summary>
        void StoreToHistoryQueue_( Queue<string> historyQueue, string newString )
        {
            // 重複していなければ、登録する
            if( !historyQueue.Contains( newString ) )
            {
                // 登録
                historyQueue.Enqueue( newString );
            }
            else
            {
                string[] strSet = historyQueue.ToArray();

                // newString と同一の要素を除いたキューを作る。
                historyQueue.Clear();
                foreach( string str in strSet )
                {
                    if( newString != str )
                    {
                        historyQueue.Enqueue( str );
                    }
                }
                // 改めて、newStringを最新の位置に登録する。
                historyQueue.Enqueue( newString );
            }

            // 要素数を調整する。
            if( historyQueue.Count > MaxHistoryNum )
            {
                historyQueue.Dequeue();
            }
            // GUI更新
            UpdateHistories_();
        }

       #endregion 検索関連

       #region イベントハンドラ
        /// <summary>
        /// メニュードロップダウン
        /// </summary>
        private void Event_TmiOperation_DropDownOpening( object sender, EventArgs e )
        {

            _tmiOperationSelect.Enabled = _matchString.IsMatch;
            _tmiOperationReplace.Enabled = _matchString.IsMatch;
        }

        /// <summary>
        /// 選択
        /// </summary>
        private void Event_TmiOperationSelect_Click( object sender, EventArgs e )
        {
            if( _matchString.IsMatch )
            {
                // 選択
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget( _ISubScene );
                subSceneMnp.BeginSelectSetChange();
                // 現在の選択セットをリセットしません。
                subSceneMnp.SelectPanesByPaneRef( _matchString.Pane );
                subSceneMnp.EndSelectSetChange();
            }
        }

        /// <summary>
        /// 全て選択
        /// </summary>
        private void Event_TmiOperationSelectAll_Click( object sender, EventArgs e )
        {
            MatchSubString matchString = new MatchSubString();

            List<IPane> selectedSet = new List<IPane>();
            // 全ての検索マッチペインを選択します。
            foreach( IPane pane in _CurrentPaneSet )
            {
                matchString.Initialize( pane, GetTargetString_( pane ), true );
                if( CheckMatchingByString_( ref matchString ) )
                {
                    selectedSet.Add( pane );
                }
            }

            // 一致が存在すれば選択します。
            if( selectedSet.Count > 0 )
            {
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget( _ISubScene );
                subSceneMnp.BeginSelectSetChange();
                subSceneMnp.ResetSelectedSet();
                foreach( IPane pane in selectedSet )
                {
                    subSceneMnp.SelectPanesByPaneRef( pane );
                }
                subSceneMnp.EndSelectSetChange();
                // 履歴に登録
                StoreToHistoryQueue_( _searchedStringHistory, _searchedString );
            }
        }

        /// <summary>
        /// 対象文字列、選択番号変更
        /// </summary>
        private void Event_TcbTargetString_SelectedIndexChanged( object sender, EventArgs e )
        {
            _options.TargetStringKind = (TargetStringKind)(_tcbTargetString.SelectedIndex + 1);
        }

        /// <summary>
        /// 正規表現チェックボックス
        /// </summary>
        private void Event_ChkRegularExpression_CheckedChanged( object sender, EventArgs e )
        {
            CheckBox cb = sender as CheckBox;
            _options.UseRegex = cb.Checked;
        }

        /// <summary>
        /// 置換チェックボックス
        /// </summary>
        private void Event_ChkReplace_CheckedChanged( object sender, EventArgs e )
        {
            UpdateProperties_();
        }

        /// <summary>
        /// 次を検索ボタンハンドラ
        /// </summary>
        private void Event_BtnFindNext_Click( object sender, EventArgs e )
        {
            _options.SearchForward = true;
            SearchTargetPane_();
        }

        /// <summary>
        /// 次を検索
        /// </summary>
        private void Event_BtnFindPrev_Click( object sender, EventArgs e )
        {
            _options.SearchForward = false;
            SearchTargetPane_();
        }

        /// <summary>
        /// 検索文字列変更
        /// </summary>
        private void Event_CmbFindString_TextChanged( object sender, EventArgs e )
        {
            ComboBox cb = sender as ComboBox;
            if( cb.Text != string.Empty )
            {
                if( cb.Text != _searchedString )
                {
                    _searchedString = cb.Text;
                }
            }
            else
            {
                _searchedString = string.Empty;
            }

            Debug.Assert( _searchedString != null );
            UpdateButtonState_();
        }

        /// <summary>
        /// 置換文字列変更
        /// </summary>
        private void Event_CmbReplacedString_TextChanged(object sender, EventArgs e)
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// 置換
        /// </summary>
        private void Event_BtnReplace_Click( object sender, EventArgs e )
        {
            Debug.Assert( _matchString.IsMatch );
            PaneManipulator paneMnp = new PaneManipulator();
            paneMnp.BindTarget( _matchString.Pane );

            string str = GetReplacedString_();

            switch( _options.TargetStringKind )
            {
                case TargetStringKind.PaneName: paneMnp.PaneName = str; break;
                case TargetStringKind.Commemnt: paneMnp.UserCommentString = str; break;
                case TargetStringKind.UserData: paneMnp.UserData = str; break;
                default: Debug.Assert( false ); break;
            }

            // 履歴に登録
            StoreToHistoryQueue_( _replacedStringHistory, _cmbReplacedString.Text );

            // 次を検索
            SearchTargetPane_();
        }

       #endregion イベントハンドラ
    }
}
