﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{


    using LECore;
    using LECore.Structures;
    using LECore.Structures.Core;

    using LECore.Util;
    using LECore.Manipulator;
    using LayoutEditor.Forms.ToolWindows.common;

    #region TweakedKey
    /// <summary>
    /// 選択されたキーをあらわすクラス。
    /// キーを保持するアニメーションカーブとペアで初期化されます。
    ///
    /// キーに対する、パラメータ操作は ModifyTemporary() メソッドで行います。
    /// FixModify()を実行すると、変更がアンドゥ可能な状態で適用されます。
    /// </summary>
    class TweakedKey : IDragTarget
    {
        //---------------------------------------
        // フィールド
        //---------------------------------------
        #region フィールド
        readonly IAnmKeyFrame _keyFrame = null;
        readonly AnmKeyFrameManipulator _keyFrameMnp = new AnmKeyFrameManipulator();

        /// <summary>
        /// 一時的な変更を適用するまえの値。
        /// </summary>
        float _defaultTime = 0;
        float _defaultValue = 0.0f;
        #endregion フィールド

        //---------------------------------------
        // プロパティ
        //---------------------------------------
        #region プロパティ
        public float DefaultTime { get { return _defaultTime; } }
        public float DefaultValue { get { return _defaultValue; } }
        #endregion プロパティ

        /// <summary>
        /// IAnmKeyFrame[] から、TweakedKey[]を生成します。
        /// </summary>
        public static TweakedKey[] FromKeySet( IAnmKeyFrame[] srcKeySet )
        {
            if( srcKeySet == null )
            {
                return new TweakedKey[0];
            }

            TweakedKey[] dstKeySet = new TweakedKey[srcKeySet.Length];
            for( int i = 0 ; i < dstKeySet.Length ; i++ )
            {
                dstKeySet[i] = new TweakedKey( srcKeySet[i] );
            }
            return dstKeySet;
        }

        /// <summary>
        /// 一時的な変更前の前の値を更新します。
        /// </summary>
        void UpdateDefaultValue_()
        {
            _defaultTime = _keyFrame.Time;
            _defaultValue = _keyFrame.ValueAsFloat;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="anmCurve"></param>
        /// <param name="keyFrame"></param>
        public TweakedKey( IAnmKeyFrame keyFrame )
        {
            _keyFrame = keyFrame;
            _keyFrameMnp.BindTarget( keyFrame );

            UpdateDefaultValue_();
        }

        /// <summary>
        /// グラフ上の座標として取得
        /// </summary>
        public PointF AsPointF
        {
            get { return new PointF( _keyFrame.Time, _keyFrame.ValueAsFloat ); }
        }

        /// <summary>
        /// グラフ上の座標として取得
        /// </summary>
        public PointF DefaultAsPointF
        {
            get { return new PointF( DefaultTime, DefaultValue ); }
        }

        /// <summary>
        /// IAnmKeyFrame に変換
        /// </summary>
        /// <returns></returns>
        public IAnmKeyFrame IAnmKeyFrame
        {
            get { return this._keyFrame; }
        }

        /// <summary>
        /// 一時的な変更。Undoには記録されません。
        /// </summary>
        /// <param name="time"></param>
        /// <param name="val"></param>
        public void ModifyTemporary( float time, float val )
        {
            _keyFrameMnp.TimeWithoutCommand = time;
            _keyFrameMnp.ValueAsFloatWithoutCommand = val;
        }

        /// <summary>
        /// 引数で指定されたキーフレームが同一のものか、判定します。
        /// </summary>
        /// <param name="anmCurve"></param>
        /// <param name="keyFrame"></param>
        /// <returns></returns>
        public bool IsSame( IAnmKeyFrame keyFrame )
        {
            return ( this._keyFrame == keyFrame );
        }

        public bool IsSame( TweakedKey tweakedKey )
        {
            return ( this._keyFrame == tweakedKey.IAnmKeyFrame );
        }

        #region IDragTarget メンバ

        public float X
        {
            get { return AsPointF.X; }
        }

        public float Y
        {
            get { return AsPointF.Y; }
        }

        public float Width
        {
            get { return 0.0f; }
        }

        public float Height
        {
            get { return 0.0f; }
        }

        public RectangleF BoundingRectangle
        {
            get { return new RectangleF( X, Y, Width, Height ); }
        }

        #endregion

        #region IDrawable メンバ

        public void Draw( IRenderer renderer, DrawableOption option )
        {
            // 何もしません。
        }

        #endregion
    }
    #endregion TweakedKey
}

