﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LayoutEditor.Forms.ToolWindows.common;
    using LECore.Structures;
    using LECore.Structures.Core;
    using DbgConsole = LECore.DbgConsole;
    using System.Collections;
    using System.Collections.Generic;
    using System.Linq;
    using LECore.Manipulator;

    public partial class GraphView : System.Windows.Forms.Control
    {
        #region MouseState クラスと、その派生クラス

        /// <summary>
        /// マウス編集種類。
        /// </summary>
        public enum GraphViewMouseTweakMode
        {
            /// <summary>
            /// 通常
            /// </summary>
            Normal,
            /// <summary>
            /// 最も近いキーを編集
            /// </summary>
            TweakNear,
            /// <summary>
            /// 選択カーブにキーを追加
            /// </summary>
            AddKeyToSelectedCurve,
            /// <summary>
            /// 選択カーブにキーを挿入
            /// </summary>
            InsertKeyToSelectedCurve,
        }

        /// <summary>
        /// マウス状態クラスです。
        /// </summary>
        public class GraphViewMouseState
        {
            //----------------------------------------------------
            #region 型定義
            /// <summary>
            /// 矢印キー移動量
            /// </summary>
            public struct ArrowKeySpeed
            {
                public readonly float _up;
                public readonly float _down;
                public readonly int _left;
                public readonly int _right;
                public readonly bool _bEnabled;

                /// <summary>
                /// コンストラクタ
                /// </summary>
                public ArrowKeySpeed( bool bEnabled, float u, float d, int l, int r )
                {
                    Debug.Assert( CheckParamatersValid( u, d, l, r ) );

                    _up = u;
                    _down = d;
                    _left = l;
                    _right = r;
                    _bEnabled = bEnabled;
                }

                /// <summary>
                /// 有効な状態かどうか取得します。
                /// </summary>
                public bool Enabled
                {
                    get { return CheckParamatersValid( _up, _down, _left, _right ) && _bEnabled; }
                }

                /// <summary>
                /// パラメータが有効かチェックします。
                /// </summary>
                static public bool CheckParamatersValid( float u, float d, int l, int r )
                {
                    return ( u > 0.0f && d > 0.0f && l > 0 && r > 0 );
                }

                /// <summary>
                /// 無効な値。
                /// </summary>
                public static readonly ArrowKeySpeed Disabled = new ArrowKeySpeed();
                public static readonly ArrowKeySpeed Default = new ArrowKeySpeed( true, 1.0f, 1.0f, 1, 1 );
            }
            #endregion

            //----------------------------------------------------
            #region フィールド
            public event EventHandler OnMouseModeChange = null;

            // マウスモード
            MouseState _curerntMoouseState = new MouseState_Idle();
            readonly GraphView _graphView = null;
            PointF _mouseTewakStartPos = PointF.Empty; // MouseStateに移す？

            // 矢印キーの移動量。
            ArrowKeySpeed _arrowKeySpeed = ArrowKeySpeed.Default;

            /// <summary>
            /// マウスモード
            /// </summary>
            MouseState _CurerntMoouseState
            {
                set
                {
                    _curerntMoouseState.LeaveState( this );

                    // デバックメッセージ表示
                    LECore.DbgConsole.WriteLine( "GraphView : MouseMode Change --- [ {0} => {1} ]",
                    _curerntMoouseState.StateName, value.StateName );
                    _curerntMoouseState = value;

                    value.GraphView = this._graphView;
                    value.EnterState( this );


                    // 変更イベントハンドラを呼びます
                    if( OnMouseModeChange != null )
                    {
                        OnMouseModeChange( _curerntMoouseState, null );
                    }
                }
                get { return _curerntMoouseState; }
            }

            /// <summary>
            /// 矢印キーの移動量を設定、取得します。
            /// </summary>
            public ArrowKeySpeed ArrowKeySpeedSettings
            {
                get { return _arrowKeySpeed; }
                set { _arrowKeySpeed = value; }
            }

            /// <summary>
            /// 値のスナップモード
            /// </summary>
            public GraphValueSnap GraphValueSnap
            {
                get;
                set;
            }

            #endregion フィールド

            /// <summary>
            /// マウス編集モードです。
            /// </summary>
            public GraphViewMouseTweakMode MouseTweakMode
            {
                get;
                set;
            }

            /// <summary>
            /// マウスカーソルリストの初期化
            /// </summary>
            void InitializeMouseCursor_()
            {
            }


            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="graphView"></param>
            public GraphViewMouseState( GraphView graphView )
            {
                Debug.Assert( graphView != null );
                _graphView = graphView;

                // マウスカーソルリストの初期化
                InitializeMouseCursor_();

                _CurerntMoouseState = MouseState._Idling;
            }

            /// <summary>
            /// 規定の状態にリセットします。
            /// </summary>
            public void SetDefault()
            {
                _CurerntMoouseState = MouseState._Idling;
            }

            #region --------- マウスイベント、キーイベントハンドラ ---------

            /// <summary>
            /// 軸固定設定を取得します。
            /// 仮実装：FIXME!
            /// </summary>
            /// <returns></returns>
            DragModifierHelper.AxisSnap GetAxisSnapType_()
            {
                if( ( Control.ModifierKeys & Keys.Control ) != 0 )
                {
                    return DragModifierHelper.AxisSnap.H;
                }
                else if( ( Control.ModifierKeys & Keys.Shift ) != 0 )
                {
                    return DragModifierHelper.AxisSnap.V;
                }
                else
                {
                    return DragModifierHelper.AxisSnap.None;
                }
            }

            /// <summary>
            /// ステートパターンによる実装。
            /// 各ステートクラスに処理が委譲されます。
            /// </summary>
            /// <param name="e"></param>
            public void OnMouseDown( MouseEventArgs e )
            {
                // フォーカスを移します。
                _graphView.Focus();

                _mouseTewakStartPos = new PointF( e.X, e.Y );

                _curerntMoouseState.OnMouseDown( e, this );
            }

            /// <summary>
            /// マウス移動
            /// </summary>
            public void OnMouseMove( MouseEventArgs e )
            {
                _curerntMoouseState.OnMouseMove( e, this );
            }

            /// <summary>
            /// マウスアップ
            /// </summary>
            public void OnMouseUp( MouseEventArgs e )
            {
                _curerntMoouseState.OnMouseUp( e, this );

                _mouseTewakStartPos = PointF.Empty;
            }

            /// <summary>
            /// ホイール
            /// </summary>
            public void OnWheelMove(MouseEventArgs e)
            {
                MouseState current = _CurerntMoouseState;

                _CurerntMoouseState = MouseState._View;
                _curerntMoouseState.OnWheelMove(e, this);

                _CurerntMoouseState = current;
            }

            /// <summary>
            /// キーダウン
            /// </summary>
            public void OnKeyDown( KeyEventArgs e )
            {
                if( e.KeyData == Keys.Space &&
                    _CurerntMoouseState != MouseState._View )
                {
                    _CurerntMoouseState = MouseState._View;
                }

                _curerntMoouseState.OnKeyDown( e, this );
            }


            /// <summary>
            /// キーアップ
            /// </summary>
            public void OnKeyUp( KeyEventArgs e )
            {
                if( ( e.KeyData & Keys.Space ) != 0 &&
                    _CurerntMoouseState == MouseState._View )
                {
                    _CurerntMoouseState = MouseState._Idling;
                    _graphView.Cursor = Cursors.Default;
                }

                _curerntMoouseState.OnKeyUp( e, this );
            }

            /// <summary>
            /// マウスモード固有描画
            /// </summary>
            public void Draw( IRenderer renderer )
            {
                _curerntMoouseState.Draw( renderer );
            }

            /// <summary>
            /// 調整ステートクラスを取得します。
            /// </summary>
            /// <returns></returns>
            private MouseState GetTweakState_()
            {
                switch (this.MouseTweakMode)
                {
                    case GraphViewMouseTweakMode.TweakNear: return MouseState._KeyTweakNear;
                    case GraphViewMouseTweakMode.InsertKeyToSelectedCurve: return MouseState._InsertKey;
                    case GraphViewMouseTweakMode.AddKeyToSelectedCurve: return MouseState._AddKey;
                    default: return MouseState._KeyTweak;
                }
            }

            #endregion --------- マウスイベント、キーイベントハンドラ ---------

            #region マウス状態クラス(基底クラス)
            /// <summary>
            /// マウス状態クラス。
            /// 仮想基底クラスです。
            /// </summary>
            abstract class MouseState
            {
                // ステートクラスインスタンス。
                public static readonly MouseState _Idling = new MouseState_Idle();
                public static readonly MouseState _View = new MouseState_View();
                public static readonly MouseState _KeySelect = new MouseState_Select();
                public static readonly MouseState _KeyTweak = new MouseState_Tweak();
                public static readonly MouseState _KeyTweakNear = new MouseState_KeyTweakNear();
                public static readonly MouseState _InsertKey = new MouseState_InsertKey();
                public static readonly MouseState _AddKey = new MouseState_AddKey();


                public static readonly MouseState _KeyScale = new MouseState_Scale();
                GraphView _graphView = null;

                /// <summary>
                /// コンストラクタ
                /// </summary>
                public MouseState()
                {
                }

                /// <summary>
                /// ステートの名称を取得します。
                /// </summary>
                public abstract string StateName
                {
                    get;
                }

                /// <summary>
                /// グラフビューです。
                /// </summary>
                public GraphView GraphView
                {
                    set { _graphView = value; }
                    get { return _graphView; }
                }

                protected ISubScene _CoreScene
                {
                    get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
                }

                /// <summary>
                /// MouseDownハンドラ
                /// </summary>
                protected void OnMouseDown_( MouseEventArgs e, GraphViewMouseState mouseState ){}

                /// <summary MouseDownハンドラ </summary>
                public virtual void OnMouseDown( MouseEventArgs e, GraphViewMouseState mouseState ) { }

                /// <summary MouseMoveハンドラ </summary>
                public virtual void OnMouseMove( MouseEventArgs e, GraphViewMouseState mouseState ) { }

                /// <summary MouseUpハンドラ </summary>
                public virtual void OnMouseUp( MouseEventArgs e, GraphViewMouseState mouseState )
                {
                    // マウスモードをリセットします
                    mouseState._CurerntMoouseState = MouseState._Idling;
                    GraphView.Cursor = Cursors.Default;
                }

                public virtual void OnWheelMove(MouseEventArgs e, GraphViewMouseState mouseState) { }

                /// <summary> KeyDownハンドラ </summary>
                public virtual void OnKeyDown( KeyEventArgs e, GraphViewMouseState mouseState ) { }

                /// <summary> KeyUpハンドラ </summary>
                public virtual void OnKeyUp( KeyEventArgs e, GraphViewMouseState mouseState ) { }

                /// <summary>
                /// ステート開始時の処理を記述します。
                /// </summary>
                public virtual void EnterState( GraphViewMouseState mouseState )
                {
                    GraphView.Invalidate();
                }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public virtual void LeaveState( GraphViewMouseState mouseState ) { }

                /// <summary>
                /// マウス状態固有描画
                /// </summary>
                public virtual void Draw( IRenderer renderer )
                {
                    // 何もしません。
                }
            }
            #endregion マウス状態クラス

            #region マウス状態クラス(待機)
            /// <summary>
            /// マウス状態クラス(待機)
            /// </summary>
            class MouseState_Idle : MouseState
            {
                public override string StateName
                {
                    get { return "Idle"; }
                }

                /// <summary>
                /// マウスダウン
                /// </summary>
                public override void OnMouseDown(
                    MouseEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    // 左クリック
                    if( e.Button == MouseButtons.Left )
                    {
                        // 選択モード
                        // 選択開始(Mayaの挙動を真似している。)
                        mouseState._CurerntMoouseState = MouseState._KeySelect;
                    }
                    else if (e.Button != MouseButtons.Right || e.Button != MouseButtons.Middle)
                    {
                        mouseState._CurerntMoouseState = mouseState.GetTweakState_();
                    }
                    base.OnMouseDown( e, mouseState );
                }

                /// <summary>
                /// キーアップ
                /// </summary>
                public override void OnKeyDown(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    if( e.KeyData == Keys.T )
                    {
                        mouseState._CurerntMoouseState = MouseState._KeyScale;
                        return;
                    }
                    if (
                        e.KeyCode  == Keys.Up ||
                        e.KeyCode  == Keys.Down ||
                        e.KeyCode  == Keys.Left ||
                        e.KeyCode  == Keys.Right)
                    {
                        mouseState._CurerntMoouseState = mouseState.GetTweakState_();
                        return;
                    }

                    base.OnKeyDown( e, mouseState );
                }
            }
            #endregion マウス状態クラス(待機)

            #region マウス状態クラス(選択)
            /// <summary>
            /// マウス状態クラス(選択)
            /// </summary>
            class MouseState_Select : MouseState
            {
                public override string StateName
                {
                    get { return "Select"; }
                }

                /// <summary MouseMoveハンドラ </summary>
                public override void OnMouseMove( MouseEventArgs e, GraphViewMouseState mouseState )
                {
                    GraphView._selectRect.UpDataSize( GraphView.ConvertPosFromScrnToGraphSpace_( new PointF( e.X, e.Y ) ) );
                    GraphView.Invalidate();
                    base.OnMouseMove( e, mouseState );
                }

                /// <summary MouseUpハンドラ </summary>
                public override void OnMouseUp( MouseEventArgs e, GraphViewMouseState mouseState )
                {
                    // キー選択
                    RectangleF selRect = GraphView._selectRect.EndSelecting();
                    GraphView.SelectKeyFrame_( selRect, Control.ModifierKeys == Keys.Shift );
                    GraphView.Invalidate();
                    base.OnMouseUp( e, mouseState );
                }

                /// <summary>
                ///
                /// </summary>
                public override void EnterState( GraphViewMouseState mouseState )
                {
                    GraphView._selectRect.BeginSelecting( GraphView.ConvertPosFromScrnToGraphSpace_( mouseState._mouseTewakStartPos ) );

                    base.EnterState( mouseState );
                }

                /// <summary>
                ///
                /// </summary>
                public override void LeaveState( GraphViewMouseState mouseState )
                {
                    GraphView._selectRect.EndSelecting();
                    GraphView.Invalidate();

                    base.LeaveState( mouseState );
                }

            }
            #endregion マウス状態クラス(選択)

            #region マウス状態クラス(キー調整)

            /// <summary>
            ///
            /// </summary>
            partial class MouseState_Tweak : MouseState
            {
                #region 型定義
                /// <summary>
                /// 調整している、UI機器
                /// </summary>
                protected enum TweakDevice
                {
                    None,
                    Mouse,
                    KeyBoard
                }

                /// <summary>
                /// 調整対象
                /// </summary>
                protected enum TweakMode
                {
                    None,
                    KeysTrans,
                    KeySlope
                }

                #endregion 型定義

                //------------------------------------
                // フィールド
                //------------------------------------
                #region フィールド

                static readonly Cursor[] _curesorSet = new Cursor[(int)DragModifierHelper.AxisSnap.AxisSnap_Max];
                static readonly Cursor _cursorSlopeTweak = ImageResMgr.GetManifestResourceCursor("curve_slope_tweak.cur");

                /// <summary>
                /// ドラッグ編集対象
                /// </summary>
                protected DragKeyModifier _draggedKeySet = new DragKeyModifier();
                DragAnchorModifier _draggedAnchorSet = new DragAnchorModifier();

                protected TweakMode _tweakMode = TweakMode.None;
                protected TweakDevice _tweakDevice = TweakDevice.None;

                #endregion フィールド


                #region プロパティ
                /// <summary>
                /// 現在、調整中か取得します。
                /// </summary>
                bool _IsTweaking
                {
                    get { return !_draggedKeySet.Empty || !_draggedAnchorSet.Empty; }
                }

                #endregion

                /// <summary>
                /// コンストラクタ
                /// </summary>
                public MouseState_Tweak()
                {
                    _curesorSet[(int)DragModifierHelper.AxisSnap.None] = ImageResMgr.GetManifestResourceCursor("curve_key_tweak_HV.cur");
                    _curesorSet[(int)DragModifierHelper.AxisSnap.H] = ImageResMgr.GetManifestResourceCursor("curve_key_tweak_H.cur");
                    _curesorSet[(int)DragModifierHelper.AxisSnap.V] = ImageResMgr.GetManifestResourceCursor("curve_key_tweak_V.cur");
                    _curesorSet[(int)DragModifierHelper.AxisSnap.Detecting] = ImageResMgr.GetManifestResourceCursor("curve_key_tweak_HV.cur");
                }

                /// <summary>
                ///
                /// </summary>
                PointF ConvertDragDiffFromScrnToGraphSpace_(PointF pos)
                {
                    SizeF viewScale = GraphView._viewScale;
                    return new PointF(pos.X * viewScale.Width, -pos.Y * viewScale.Height);
                }

                /// <summary>
                ///
                /// </summary>
                public override string StateName
                {
                    get { return "Tweak"; }
                }

                /// <summary>
                ///
                /// </summary>
                protected void SetupTweakingKeysMode_(IAnmKeyFrame[] targetKeys, PointF mouseTewakStartPos)
                {
                    //-------------------------------------------
                    // 操作クラスに登録
                    // 操作開始
                    TweakedKey[] keySet = TweakedKey.FromKeySet(targetKeys);

                    _draggedKeySet.BeginDragging(keySet, mouseTewakStartPos, DragModifierHelper.OptionFlag.None);

                    SizeF viewScale = GraphView._viewScale;
                    _draggedKeySet.ScreenScale = new PointF(viewScale.Width, -viewScale.Height);

                    // 軸固定設定を行います
                    if ((Control.ModifierKeys & Keys.Shift) != 0)
                    {
                        _draggedKeySet.AxisSnapKind = DragModifierHelper.AxisSnap.Detecting;
                    }
                    　
                    // カーソル変更
                    GraphView.Cursor = _curesorSet[(int)_draggedKeySet.AxisSnapKind];
                }

                /// <summary>
                ///
                /// </summary>
                protected void SetupTweakingAnchorMode_(KeyAanchor[] anchorSet, PointF mouseTewakStartPos)
                {
                    if (anchorSet.Length <= 0)
                    {
                        // 空なら何もしない
                        return;
                    }

                    //-------------------------------------------
                    // 操作基点となる点を、すべてのキーの平均から計算します。
                    {
                        PointF posKey = PointF.Empty;
                        foreach (KeyAanchor keyAnchor in anchorSet)
                        {
                            posKey.X += keyAnchor.IAnmKeyFrame.Time;
                            posKey.Y += keyAnchor.IAnmKeyFrame.ValueAsFloat;
                        }
                        posKey.X /= anchorSet.Length;
                        posKey.Y /= anchorSet.Length;

                        _draggedAnchorSet.AnchorTweakPivotPoint = GraphView.ConvertPosFromGraphToScrnSpace_(posKey);
                    }

                    //-------------------------------------------
                    // 操作クラスに登録
                    // 操作開始
                    _draggedAnchorSet.BeginDragging(
                        AnchorTweakedKey.FromKeyAanchor(anchorSet),
                        mouseTewakStartPos,
                        DragModifierHelper.OptionFlag.IgnoreShortDrag);
                    _draggedAnchorSet.AspectRatio = this.GraphView.AspectRatio;

                    //-------------------------------------------
                    // カーソル変更
                    GraphView.Cursor = _cursorSlopeTweak;
                }

                /// <summary>
                /// 編集を開始します。
                /// </summary>
                protected virtual void BeginTweking_(GraphViewMouseState mouseState)
                {
                    _draggedKeySet.GraphView = this.GraphView;
                    _draggedKeySet.SetGraphValueSnap(mouseState.GraphValueSnap);

                    // 通常の調整
                    // 操作モードを決定します。
                    if (GraphView._targetAnmAttributeSet._SelectedAnchorSet.Length != 0)
                    {
                        _tweakMode = TweakMode.KeySlope;
                        SetupTweakingAnchorMode_(GraphView._targetAnmAttributeSet._SelectedAnchorSet, mouseState._mouseTewakStartPos);
                    }
                    else if (GraphView.SelectedKeyFrameSet_.Length != 0)
                    {
                        _tweakMode = TweakMode.KeysTrans;
                        SetupTweakingKeysMode_(GraphView.SelectedKeyFrameSet_, mouseState._mouseTewakStartPos);
                    }
                    else
                    {
                        _tweakMode = TweakMode.None;
                        mouseState._CurerntMoouseState = MouseState._Idling;
                    }
                }

                /// <summary>
                /// 更新量を計算します。
                /// </summary>
                float CalcModifyAmount_(GraphViewMouseState mouseState)
                {

                    float scaleY = mouseState._graphView.ViewScale.Height * 10.0f;

                    // 3段階に切り替えます。
                    scaleY = Math.Abs(scaleY);
                    if (scaleY > 1.0f)
                    {
                        scaleY = 1.0f;
                    }
                    else if (scaleY > 0.1f)
                    {
                        scaleY = 0.11f;
                    }
                    else if (scaleY <= 0.1f)
                    {
                        scaleY = 0.01f;
                    }

                    return scaleY;
                }

                /// <summary>
                ///
                /// </summary>
                static bool CheckAllTargetKeysAreIntType_(TweakedKey[] targetSet)
                {
                    if (targetSet != null)
                    {
                        foreach (TweakedKey twkf in targetSet)
                        {

                            if (twkf.IAnmKeyFrame.OwnerIAnmCurve.TargetAttrType == AttributeType.Float)
                            {
                                return false;
                            }
                        }
                    }

                    // すべてが整数キー
                    return true;
                }

                /// <summary>
                /// 矢印キーの移動量を決定します。
                /// </summary>
                PointF CalcArrowKeySpeed_(Keys inputKey, GraphViewMouseState mouseState)
                {
                    // 矢印キーの移動量を決定します。
                    PointF difference;

                    float accerate = ModifierKeys == Keys.Shift ? 10.0f : 1.0f;
                    float invScaleX = accerate / mouseState._graphView.ViewScale.Width;
                    float invScaleY = accerate / mouseState._graphView.ViewScale.Height;

                    if (mouseState.ArrowKeySpeedSettings.Enabled)
                    {
                        ArrowKeySpeed aks = mouseState.ArrowKeySpeedSettings;
                        // 指定があれば、利用します。
                        switch (inputKey)
                        {
                            case Keys.Up: difference = new PointF(0.0f, -aks._up * invScaleY); break;
                            case Keys.Down: difference = new PointF(0.0f, aks._down * invScaleY); break;
                            case Keys.Left: difference = new PointF(-aks._left * invScaleX, 0.0f); break;
                            case Keys.Right: difference = new PointF(aks._right * invScaleX, 0.0f); break;
                            default: difference = new PointF(0.0f, 0.0f); break;
                        }
                    }
                    else
                    {
                        // なければ、適切な値を計算します。
                        float dx = 1.0f;
                        float dy = CheckAllTargetKeysAreIntType_(_draggedKeySet.TargetSet) ?
                            1.0f : CalcModifyAmount_(mouseState);

                        dx *= invScaleX;
                        dy *= invScaleY;

                        switch (inputKey)
                        {
                            case Keys.Up: difference = new PointF(0.0f, -dy); break;
                            case Keys.Down: difference = new PointF(0.0f, dy); break;
                            case Keys.Left: difference = new PointF(-dx, 0.0f); break;
                            case Keys.Right: difference = new PointF(dx, 0.0f); break;
                            default: difference = new PointF(0.0f, 0.0f); break;
                        }
                    }
                    return difference;
                }

                /// <summary>
                /// 矢印キーによるキーフレームの調整。
                /// </summary>
                void TweakByArrowKey_(Keys inputKey, GraphViewMouseState mouseState)
                {
                    if (inputKey != Keys.Up &&
                        inputKey != Keys.Down &&
                        inputKey != Keys.Left &&
                        inputKey != Keys.Right)
                    {
                        return;
                    }

                    bool bFirstTime = !_IsTweaking;
                    if (bFirstTime)
                    {
                        _tweakDevice = TweakDevice.KeyBoard;
                        BeginTweking_(mouseState);
                    }

                    // 矢印キーの移動量を決定します。
                    PointF difference = CalcArrowKeySpeed_(inputKey, mouseState);

                    if (bFirstTime)
                    {
                        // 変移量で、切捨て動作を行います。
                        _draggedKeySet.UpdateDraggingByDifferenceWithRoundDown(difference);
                    }
                    else
                    {
                        _draggedKeySet.UpdateDraggingByDifference(difference);
                    }

                    GraphView.Invalidate();
                }

                /// <summary>
                ///　
                /// </summary>
                public override void EnterState(GraphViewMouseState mouseState)
                {
                    base.EnterState(mouseState);
                }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public override void LeaveState(GraphViewMouseState mouseState)
                {
                    _tweakMode = TweakMode.None;
                    _draggedKeySet.Clear();
                    _draggedAnchorSet.Clear();
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, GraphViewMouseState mouseState)
                {
                    if (e.Button != MouseButtons.Right && e.Button != MouseButtons.Middle)
                    {
                        return;
                    }

                    if (!_IsTweaking)
                    {
                        _tweakDevice = TweakDevice.Mouse;
                        BeginTweking_(mouseState);
                    }

                    // -------------------- カーソルの更新
                    Cursor newCursol =
                        _curesorSet[(int)_draggedKeySet.AxisSnapKind];
                    if (newCursol != GraphView.Cursor)
                    {
                        GraphView.Cursor = newCursol;
                    }

                    // --------------------
                    switch (_tweakMode)
                    {
                        case TweakMode.KeysTrans:
                            {
                                _draggedKeySet.UpdateDragging(new PointF(e.X, e.Y));
                                GraphView.Invalidate();
                                break;
                            }
                        case TweakMode.KeySlope:
                            {
                                _draggedAnchorSet.UpdateDragging(new PointF(e.X, e.Y));
                                GraphView.Invalidate();
                                break;
                            }
                        default:
                            {
                                // 何もしません。
                                break;
                            }
                    }
                    base.OnMouseMove(e, mouseState);
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, GraphViewMouseState mouseState)
                {
                    if (!_IsTweaking)
                    {
                        _tweakDevice = TweakDevice.Mouse;
                        BeginTweking_(mouseState);
                    }

                    switch (_tweakMode)
                    {
                        case TweakMode.KeysTrans:
                            {
                                _draggedKeySet.EndDragging();
                                GraphView.Invalidate();
                                break;
                            }
                        case TweakMode.KeySlope:
                            {
                                _draggedAnchorSet.EndDragging();
                                GraphView.Invalidate();
                                break;
                            }
                        default:
                            {
                                break;
                            }
                    }

                    _tweakDevice = TweakDevice.None;
                    base.OnMouseUp(e, mouseState);
                }


                /// <summary>
                /// キーダウン
                /// </summary>
                public override void OnKeyDown(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState)
                {
                    // マウスでの編集が開始されていたらキーでの編集を開始できないようにする
                    if (_tweakDevice == TweakDevice.Mouse)
                    {
                        return;
                    }

                    TweakByArrowKey_(e.KeyCode, mouseState);
                }

                /// <summary>
                /// キーアップ
                /// </summary>
                public override void OnKeyUp(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState)
                {
                    // マウスでの編集が開始されていたらキーでの編集を開始できないようにする
                    if (_tweakDevice == TweakDevice.Mouse)
                    {
                        return;
                    }

                    TweakByArrowKey_(e.KeyCode, mouseState);


                    _draggedKeySet.EndDragging();

                    GraphView.Invalidate();
                    mouseState._CurerntMoouseState = MouseState._Idling;



                    base.OnKeyUp(e, mouseState);
                }

                /// <summary>
                /// マウス状態固有描画
                /// </summary>
                public override void Draw(IRenderer renderer)
                {
                    // デバック描画
                    // _draggedAnchorSet.DebugDraw( renderer );
                }
            }

            /// <summary>
            /// マウスカーソル近くの調整
            /// </summary>
            class MouseState_KeyTweakNear : MouseState_Tweak
            {
                const float _MinSelectDistance = 10.0f;

                /// <summary>
                ///
                /// </summary>
                public override string StateName
                {
                    get { return "KeyTweakNear"; }
                }

                /// <summary>
                /// 編集を開始します。
                /// </summary>
                protected override void BeginTweking_(GraphViewMouseState mouseState)
                {
                    _draggedKeySet.GraphView = this.GraphView;
                    _draggedKeySet.SetGraphValueSnap(mouseState.GraphValueSnap);

                    // マウスカーソル近くの調整
                    // キーボードでの操作は別で扱います。
                    if (_tweakDevice == TweakDevice.KeyBoard)
                    {
                        _tweakMode = TweakMode.KeysTrans;
                        SetupTweakingKeysMode_(GraphView.SelectedKeyFrameSet_, mouseState._mouseTewakStartPos);
                        return;
                    }

                    // まず、マウスカーソル近傍のアンカーを探す
                    KeyAanchor nearAnchor = FindNearAnchor_(mouseState);
                    if (nearAnchor.IsValid)
                    {
                        _tweakMode = TweakMode.KeySlope;
                        SetupTweakingAnchorMode_(new KeyAanchor[] { nearAnchor }, mouseState._mouseTewakStartPos);
                    }
                    else
                    {
                        // アンカーが近くなければ、キーを探す
                        IAnmKeyFrame nearKey = FindNearKey_(mouseState);
                        if (nearKey != null)
                        {
                            _tweakMode = TweakMode.KeysTrans;
                            SetupTweakingKeysMode_(new IAnmKeyFrame[] { nearKey }, mouseState._mouseTewakStartPos);
                        }
                    }
                }

                /// <summary>
                /// マウスカーソル近傍のキーを取得します。
                /// </summary>
                private static IAnmKeyFrame FindNearKey_(GraphViewMouseState mouseState)
                {
                    var downPosInGraph = new FVec2(mouseState._graphView.ConvertPosFromScrnToGraphSpace_(mouseState._mouseTewakStartPos));

                    IAnmKeyFrame nearKey = null;
                    float minDist = float.MaxValue;
                    foreach (var key in GetNeraKetCandidate_(mouseState))
                    {
                        var keyPos = new FVec2(mouseState._graphView.GetKeyAsPointF_(key));
                        var dist = (downPosInGraph - keyPos).Length;

                        if (minDist > dist)
                        {
                            minDist = dist;
                            nearKey = key;
                        }
                    }

                    if (minDist >= _MinSelectDistance)
                    {
                        nearKey = null;
                    }

                    return nearKey;
                }

                /// <summary>
                /// マウスカーソル近傍のキー候補を取得します。
                /// </summary>
                private static IEnumerable<IAnmKeyFrame> GetNeraKetCandidate_(GraphViewMouseState mouseState)
                {
                    if (mouseState._graphView.SelectedKeyFrameSet_.Length > 0)
                    {
                        foreach (var key in mouseState._graphView.SelectedKeyFrameSet_)
                        {
                            yield return key;
                        }
                    }
                    else
                    {
                        foreach (var curve in mouseState._graphView.TargetCurveSet)
                        {
                            foreach (var key in curve.IKeyFrameSet)
                            {
                                yield return key;
                            }
                        }
                    }
                }

                /// <summary>
                /// マウスカーソル近傍のアンカーを取得します。
                /// </summary>
                private static KeyAanchor FindNearAnchor_(GraphViewMouseState mouseState)
                {
                    KeyAanchor nearAnchor = new KeyAanchor();
                    if (mouseState._graphView.SelectedKeyFrameSet_.Length > 0)
                    {
                        float minDist = float.MaxValue;
                        var downPosInScrn = new FVec2(mouseState._mouseTewakStartPos);
                        foreach (IAnmKeyFrame keyInfo in mouseState._graphView.SelectedKeyFrameSet_)
                        {
                            // 編集が行えない場合は、接線選択セットには登録 しない。
                            if (!keyInfo.TangentModifyEnabled)
                            {
                                continue;
                            }

                            // キーが矩形に含まれているなら
                            PointF pntKey = mouseState._graphView.ConvertPosFromGraphToScrnSpace_(mouseState._graphView.GetKeyAsPointF_(keyInfo));
                            var inTanPnt = new FVec2(mouseState._graphView.GetTangetAnchorPoint_(pntKey, new PointF(-32.0f, 0.0f), keyInfo.InTangent));
                            var outTanPnt = new FVec2(mouseState._graphView.GetTangetAnchorPoint_(pntKey, new PointF(32.0f, 0.0f), keyInfo.OutTangent));

                            // In
                            float distIn = (downPosInScrn - inTanPnt).Length;
                            if (minDist > distIn)
                            {
                                minDist = distIn;
                                nearAnchor = KeyAanchor.MakeLeftSlope(keyInfo);
                            }

                            // Out
                            float distOut = (downPosInScrn - outTanPnt).Length;
                            if (minDist > distOut)
                            {
                                minDist = distOut;
                                nearAnchor = KeyAanchor.MakeRightSlope(keyInfo);
                            }
                        }

                        if (minDist >= _MinSelectDistance)
                        {
                            nearAnchor = new KeyAanchor();
                        }
                    }
                    return nearAnchor;
                }

            }

            #endregion マウス状態クラス(キー調整)

            #region マウス状態クラス(ビュー変更)
            /// <summary>
            ///
            /// </summary>
            class MouseState_View : MouseState
            {
                static readonly Cursor _CursorScaleHV = ImageResMgr.GetManifestResourceCursor( "curve_scale_HV.cur" );
                static readonly Cursor _CursorScaleH = ImageResMgr.GetManifestResourceCursor( "curve_scale_H.cur" );
                static readonly Cursor _CursorScaleV = ImageResMgr.GetManifestResourceCursor( "curve_scale_V.cur" );

                static readonly Cursor _CursorHandClose = ImageResMgr.GetManifestResourceCursor( "hand_close.cur" );
                static readonly Cursor _CursorHandOpen = ImageResMgr.GetManifestResourceCursor( "hand_open.cur" );

                static readonly Cursor _CursorZoom = ImageResMgr.GetManifestResourceCursor("MagnifyMinus.cur");
                static readonly Cursor _CursorPan = ImageResMgr.GetManifestResourceCursor("MagnifyPlus.cur");

                PointF _tempViewPos = new PointF( 0, -20 );
                SizeF _tempViewScale = new SizeF( 0.5f, 0.5f );

                bool _bMouseButtonDown = false;

                MouseTewakMode _tweakMode = MouseTewakMode.None;

                /// <summary>
                /// ステート開始時の処理を記述します。
                /// </summary>
                public override void EnterState( GraphViewMouseState mouseState )
                {
                    GraphView.Cursor = _CursorHandOpen;
                    _bMouseButtonDown = false;
                    base.EnterState( mouseState );
                }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public override void LeaveState( GraphViewMouseState mouseState )
                {
                    GraphView.Cursor = Cursors.Default;
                    _bMouseButtonDown = false;
                    base.LeaveState( mouseState );
                }

                /// <summary>
                ///
                /// </summary>
                void AddViewPosInGraphSpace_( float px, float py )
                {
                    // コンパイラ エラー CS1690  対策
                    // System.MarshalByRefObject から派生するクラスの値型のメンバに対する
                    // アクセサ呼び出しは禁止らしい。対策として、ローカルコピーを用意する。
                    SizeF viewScale = GraphView._viewScale;
                    PointF newPos = new PointF(
                        _tempViewPos.X + px * viewScale.Width,
                        _tempViewPos.Y - py * viewScale.Height );

                    GraphView._viewPosInGraphSpace = newPos;
                    GraphView.Invalidate();
                }

                /// <summary>
                /// ビュースケールを設定します。
                /// </summary>
                void AddViewScale_( float sx, float sy )
                {
                    sx = _tempViewScale.Width +
                        sx * _tempViewScale.Width * 0.002f +
                        sx * GraphView.ViewScale.Width * 0.001f;

                    sy = _tempViewScale.Height +
                        sy * _tempViewScale.Height * 0.002f +
                        sy * GraphView.ViewScale.Height * 0.001f;

                    GraphView.ViewScale = new SizeF( sx, sy );
                }

                /// <summary>
                ///
                /// </summary>
                public override string StateName
                {
                    get { return "View"; }
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseDown(
                    MouseEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    switch( e.Button )
                    {
                        case MouseButtons.Left:
                        case MouseButtons.Middle:
                        {
                            if ((Control.ModifierKeys & Keys.Control) != 0)
                            {
                                GraphView.Cursor = _CursorPan;
                            }
                            else if ((Control.ModifierKeys & Keys.Alt) != 0)
                            {
                                GraphView.Cursor = _CursorZoom;
                            }
                            else
                            {
                                // ビュー平行移動
                                // 左クリック
                                _tweakMode = MouseTewakMode.ViewTrans;
                                _tempViewPos = GraphView._viewPosInGraphSpace;
                                GraphView.Cursor = _CursorHandClose;
                            }
                            break;
                        }
                        case MouseButtons.Right:
                        {
                            // ビュースケール変更
                            if( ( Control.ModifierKeys & Keys.Shift ) != 0 )
                            {
                                _tweakMode = MouseTewakMode.ViewScale_H_OR_V;
                                _tempViewScale = GraphView._viewScale;
                            }
                            else
                            {
                                _tweakMode = MouseTewakMode.ViewScale_HV;
                                _tempViewScale = GraphView._viewScale;

                            }
                            GraphView.Cursor = _CursorScaleHV;
                            break;
                        }
                    }

                    _bMouseButtonDown = true;
                    base.OnMouseDown( e, mouseState );
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseMove(
                    MouseEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    if (_bMouseButtonDown)
                    {
                        float dx = e.X - mouseState._mouseTewakStartPos.X;
                        float dy = e.Y - mouseState._mouseTewakStartPos.Y;

                        switch (_tweakMode)
                        {
                            case MouseTewakMode.ViewTrans:
                                AddViewPosInGraphSpace_(-dx, -dy);
                                break;
                            case MouseTewakMode.ViewScale_HV:
                                AddViewScale_(dx, dx);
                                GraphView.Cursor = _CursorScaleHV;
                                break;
                            case MouseTewakMode.ViewScale_H:
                                dy = 0.0f;
                                AddViewScale_(dx, dy);
                                GraphView.Cursor = _CursorScaleV;
                                break;
                            case MouseTewakMode.ViewScale_V:
                                dx = 0.0f;
                                AddViewScale_(dx, dy);
                                GraphView.Cursor = _CursorScaleH;
                                break;
                            case MouseTewakMode.ViewScale_H_OR_V:
                                // 水平、垂直どちらのスケールを操作すべきか、判定します。
                                float sq_dx = dx * dx;
                                float sq_dy = dy * dy;
                                if (sq_dx + sq_dy > 100)
                                {
                                    if (sq_dx > sq_dy)
                                    {
                                        _tweakMode = MouseTewakMode.ViewScale_H;
                                    }
                                    else
                                    {
                                        _tweakMode = MouseTewakMode.ViewScale_V;
                                    }
                                }
                                break;
                            default:
                                // 処理されなかっった
                                // return false;
                                break;
                        }
                    }

                    base.OnMouseMove( e, mouseState );
                    // return true;
                }

                /// <summary>
                /// マウスアップ
                /// </summary>
                public override void OnMouseUp( MouseEventArgs e, GraphViewMouseState mouseState )
                {
                    // ズーム処理
                    if(e.Button == MouseButtons.Left)
                    {
                        if ((Control.ModifierKeys & Keys.Control) != 0)
                        {
                            mouseState._graphView._viewPosInGraphSpace = mouseState._graphView.ConvertPosFromScrnToGraphSpace_(new PointF(e.X, e.Y));
                            mouseState._graphView.ZoomViewScale();
                        }
                        else if ((Control.ModifierKeys & Keys.Alt) != 0)
                        {
                            mouseState._graphView.PanViewScale();
                        }
                    }

                    // モード切替をしたくないので、base を呼ばない。
                    GraphView.Cursor = _CursorHandOpen;
                    _tweakMode = MouseTewakMode.None;
                    _bMouseButtonDown = false;
                }

                /// <summary>
                /// ホイール
                /// </summary>
                public override void OnWheelMove(MouseEventArgs e, GraphViewMouseState mouseState)
                {
                    if (e.Delta > 0)
                    {
                        mouseState._graphView.ZoomViewScale();
                    }
                    else if (e.Delta < 0)
                    {
                        mouseState._graphView.PanViewScale();
                    }

                    base.OnWheelMove(e, mouseState);
                }

                /// <summary>
                /// キーアップ
                /// </summary>
                public override void OnKeyUp(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    _tempViewScale = GraphView._viewScale;
                    _tweakMode = MouseTewakMode.None;

                    base.OnKeyUp( e, mouseState );
                }


            }
            #endregion マウス状態クラス(ビュー変更)

            #region マウス状態クラス(キースケール変更)
            /// <summary>
            /// マウス状態クラス(キースケール変更)
            /// </summary>
            class MouseState_Scale : MouseState
            {
                //----------------------------------
                // フィールド
                //----------------------------------
                #region フィールド
                /// <summary>
                /// カーソル
                /// </summary>
                static readonly Cursor _CursorKeyScaling =
                    ImageResMgr.GetManifestResourceCursor( "curve_scale_HV.cur" );

                /// <summary>
                /// 調整キーセット
                /// </summary>
                DragKeyScaleModifier _scaledKeySet = new DragKeyScaleModifier();


                /// <summary>
                /// マウスボタンフラグ
                /// </summary>
                bool _bMouseButtonPushed = false;
                #endregion フィールド

                /// <summary>
                /// ステート名
                /// </summary>
                public override string StateName
                {
                    get { return "Key-Scaling"; }
                }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                public MouseState_Scale()
                {
                }

                /// <summary>
                /// 操作クラスに登録します。
                /// </summary>
                void RegisterDragedSet_()
                {
                    IAnmKeyFrame[] keySet = GraphView.SelectedKeyFrameSet_;
                    _scaledKeySet.BeginDragging( TweakedKey.FromKeySet( keySet ), _scaledKeySet.DragStartPointF, DragModifierHelper.OptionFlag.None );
                }

                /// <summary>
                /// ステート開始時の処理を記述します。
                /// </summary>
                public override void EnterState( GraphViewMouseState mouseState )
                {
                    IAnmKeyFrame[] keySet = GraphView.SelectedKeyFrameSet_;

                    //-------------------------------------------
                    // 選択キーがない場合は、キャンセルします。
                    if( keySet.Length < 2 )
                    {
                        DbgConsole.WriteLine( "Entering key-transfomr-mode is canceled. Please select more than two keys." );
                        mouseState._CurerntMoouseState = MouseState._Idling;
                        return;
                    }
                    else
                    {
                        GraphView.Cursor = _CursorKeyScaling;

                        // 操作クラスに登録
                        RegisterDragedSet_();

                        GraphView.Invalidate();
                        base.EnterState( mouseState );
                    }
                }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public override void LeaveState( GraphViewMouseState mouseState )
                {
                    GraphView.Cursor = Cursors.Default;
                    _scaledKeySet.Clear();
                    _bMouseButtonPushed = false;


                    GraphView.Invalidate();
                    base.LeaveState( mouseState );
                }

                /// <summary>
                /// マウスダウン
                /// </summary>
                public override void OnMouseDown(
                    MouseEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    // 中・右ボタンクリック以外ならモードを抜けます。
                    if (e.Button != MouseButtons.Right && e.Button != MouseButtons.Middle)
                    {
                        return;
                    }

                    // 編集開始
                    PointF posMouse = GraphView.ConvertPosFromScrnToGraphSpace_( new PointF( e.X, e.Y ) );
                    bool bAxisSnap = ( Control.ModifierKeys & Keys.Shift ) != 0;

                    Debug.Assert( !_scaledKeySet.Empty );
                    _scaledKeySet.ResetDragging( posMouse );

                    _scaledKeySet.SetTransformOrigin(
                        posMouse,
                        new PointF( GraphView.ViewScale.Width, GraphView.ViewScale.Height ) );

                    _bMouseButtonPushed = true;

                    GraphView.Invalidate();

                    base.OnMouseDown( e, mouseState );
                }

                /// <summary>
                /// OnMouseMove
                /// </summary>
                public override void OnMouseMove(
                    MouseEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    if( _bMouseButtonPushed )
                    {
                        _scaledKeySet.UpdateDragging(
                            GraphView.ConvertPosFromScrnToGraphSpace_( new PointF( e.X, e.Y ) ) );
                        GraphView.Invalidate();
                    }
                    base.OnMouseMove( e, mouseState );
                }

                /// <summary>
                /// マウスアップ
                /// </summary>
                public override void OnMouseUp( MouseEventArgs e, GraphViewMouseState mouseState )
                {
                    _scaledKeySet.EndDragging();
                    _bMouseButtonPushed = false;

                    RegisterDragedSet_();
                    // 規定の処理(base.OnMouseUp)はIdleモードに遷移するので、呼ばない。
                }

                /// <summary>
                /// キーダウン
                /// </summary>
                public override void OnKeyDown(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    // 現状は何もしません。
                    base.OnKeyDown( e, mouseState );
                }

                /// <summary>
                /// KeyUpハンドラ
                /// </summary>
                public override void OnKeyUp(
                    KeyEventArgs e,
                    GraphViewMouseState mouseState )
                {
                    // T(transform) キーでモードを抜ける
                    if( e.KeyData == Keys.T )
                    {
                        mouseState._CurerntMoouseState = _Idling;
                        return;
                    }
                    base.OnKeyUp( e, mouseState );
                }

                /// <summary>
                /// マウス状態固有描画
                /// </summary>
                public override void Draw( IRenderer renderer )
                {
                    _scaledKeySet.Draw( renderer, DrawableOption.Empty );
                }
            }
            #endregion

            #region マウス状態クラス(キー追加)
            /// <summary>
            /// キーの挿入
            /// </summary>
            partial class MouseState_InsertKey : MouseState
            {
                private FVec2 _currentMousePos = new FVec2();
                private FVec2 _targetCurveTop = new FVec2();
                private FVec2 _targetCurveBottom = new FVec2();

                /// <summary>
                /// 現在の時間
                /// </summary>
                protected float _CurrentTimeFrame
                {
                    get { return (float)Math.Ceiling(_currentMousePos.X); }
                }

                /// <summary>
                /// 現在のマウス位置
                /// </summary>
                protected float _CurrentMouseY
                {
                    get { return _currentMousePos.Y; }
                }

                /// <summary>
                ///
                /// </summary>
                public override string StateName
                {
                    get { return "InsertKey"; }
                }

                /// <summary>
                ///　
                /// </summary>
                public override void EnterState(GraphViewMouseState mouseState)
                {
                    UpdateCursorState_(mouseState._mouseTewakStartPos, mouseState);
                    base.EnterState(mouseState);
                }

                /// <summary>
                /// ステート終了時の処理を記述します。
                /// </summary>
                public override void LeaveState(GraphViewMouseState mouseState)
                {
                    base.LeaveState(mouseState);
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseMove(MouseEventArgs e, GraphViewMouseState mouseState)
                {
                    UpdateCursorState_(new PointF(e.X, e.Y), mouseState);
                    base.OnMouseMove(e, mouseState);
                }

                /// <summary>
                /// カーソル状態を更新
                /// </summary>
                private void UpdateCursorState_(PointF mousePos, GraphViewMouseState mouseState)
                {
                    _currentMousePos = new FVec2(GraphView.ConvertPosFromScrnToGraphSpace_(mousePos));

                    _targetCurveTop = new FVec2(_CurrentTimeFrame, GraphView._ViewTopInGraphSpace);
                    _targetCurveBottom = new FVec2(_CurrentTimeFrame, GraphView._ViewBottomInGraphSpace);

                    GraphView.Invalidate();
                }

                /// <summary>
                ///
                /// </summary>
                public override void OnMouseUp(MouseEventArgs e, GraphViewMouseState mouseState)
                {
                    // シーンが読み込まれていない場合は処理しない
                    if (GraphView._CurrentSubScene == null)
                    {
                        return;
                    }

                    // キーを追加する
                    // 選択あれば、選択に、選択なければすべてに
                    try
                    {
                        GraphView._CurrentSubScene.BeginMassiveModify();
                        foreach (var targetCurve in GetTargetCurve_())
                        {
                            InterporationType interpType = targetCurve.IKeyFrameSet.Length > 0 ? targetCurve.IKeyFrameSet[0].InterporationType : InterporationType.Spline;

                            AnmCurveManipulator AnmCurveMnp = new AnmCurveManipulator();
                            AnmCurveMnp.BindTarget(targetCurve);
                            AnmCurveMnp.MakeKeyFrame((int)_CurrentTimeFrame, GetNewKeyValue_(targetCurve));
                        }
                    }
                    finally
                    {
                        GraphView._CurrentSubScene.EndMassiveModify();
                    }

                    base.OnMouseUp(e, mouseState);
                }

                /// <summary>
                /// 新しいキーの値を取得します。
                /// </summary>
                protected virtual object GetNewKeyValue_(IAnmCurve targetCurve)
                {
                    return (object)targetCurve.Evaluate(_CurrentTimeFrame);
                }

                /// <summary>
                /// 対象カーブを取得します。
                /// </summary>
                private List<IAnmCurve> GetTargetCurve_()
                {
                    List<IAnmCurve> targetCurves = new List<IAnmCurve>();
                    if (GraphView._targetAnmAttributeSet._SelectedKeyFrameSet.Length > 0)
                    {
                        // キーが選択されていれば、その持ち主カーブすべて
                        foreach (var key in GraphView._targetAnmAttributeSet._SelectedKeyFrameSet)
                        {
                            if (!targetCurves.Contains(key.OwnerIAnmCurve))
                            {
                                targetCurves.Add(key.OwnerIAnmCurve);
                            }
                        }
                    }
                    else
                    {
                        // キーが選択されていなければ、表示対象カーブすべて
                        targetCurves.AddRange(GraphView._targetAnmAttributeSet._TargetCurveSet);
                    }

                    return targetCurves;
                }

                /// <summary>
                /// マウス状態固有描画
                /// </summary>
                public override void Draw(IRenderer renderer)
                {
                    renderer.Color = Color.Red;
                    renderer.DrawLine(_targetCurveTop.AsPointF, _targetCurveBottom.AsPointF);

                    DrawNewKeyCandidate_(renderer);
                }

                /// <summary>
                /// 新しいキー候補の描画
                /// </summary>
                protected virtual void DrawNewKeyCandidate_(IRenderer renderer)
                {
                    foreach (var targetCurve in GetTargetCurve_())
                    {
                        renderer.Color = Color.Black;
                        renderer.LineWidth = 4;
                        renderer.DrawPoint(new PointF(_CurrentTimeFrame, targetCurve.Evaluate(_CurrentTimeFrame)));
                    }
                }
            }

            /// <summary>
            /// キーの挿入
            /// </summary>
            partial class MouseState_AddKey : MouseState_InsertKey
            {
                /// <summary>
                /// 新しいキーの値を取得します。
                /// </summary>
                protected override object GetNewKeyValue_(IAnmCurve targetCurve)
                {
                    return (object)_CurrentMouseY;
                }

                /// <summary>
                /// マウス状態固有描画
                /// </summary>
                protected override void DrawNewKeyCandidate_(IRenderer renderer)
                {
                    renderer.Color = Color.Black;
                    renderer.LineWidth = 4;
                    renderer.DrawPoint(new PointF(_CurrentTimeFrame, _CurrentMouseY));
                }
            }
            #endregion マウス状態クラス(キー追加)
        }
        #endregion マウスステートクラス

    }
}
