﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LayoutEditor.Structures.SerializableObject;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Util;




    /// <summary>
    /// グラフ描画を行うクラスです。
    ///
    /// 本クラスは、内部データを、
    /// X軸(右が正方向)を時間、
    /// Y軸(上が正方向！注意)を 値とする GraphSpace 座標系で扱います。
    ///
    /// </summary>
    public partial class GraphView : System.Windows.Forms.Control
    {
        #region 定数、クラス定義
        /// <summary>
        /// マウス操作モード
        /// </summary>
        enum MouseTewakMode
        {
            None,
            ViewTrans,       // Alt + センターホイールボタン
            ViewScale_HV,    // Alt + R
            ViewScale_H_OR_V,// Alt + SHIFT + R
            ViewScale_H,     // Alt + SHIFT + R
            ViewScale_V,     // Alt + SHIFT + R
            SelectingKeys,
            // TweakingKeys,
            // TweakingAnchor,
        }

        /// <summary>
        /// 更新イベント
        /// </summary>
        public enum ChangeEvent
        {
            KeySelectionChange,
            TargetCurveChange,
        }

        // Viewスケール増加分係数。
        const float _ViewScaleMinimum = 0.00001f;
        const float _ViewScaleStep = 0.3f;
        const float _GridMinH = 0.1f;
        const float _GridMinV = 0.01f;

        const float _KeyDotSize = 3.0f;

        #endregion 定数

        #region イベント
        public delegate void OnChangeHandler( ChangeEvent kind );
        public event OnChangeHandler OnChange;
        /// <summary>
        /// マウスモード変更イベント
        /// </summary>
        public event EventHandler OnMouseModeChange
        {
            add { _mouseState.OnMouseModeChange += value; }
            remove { _mouseState.OnMouseModeChange -= value; }
        }

        #endregion イベント

        #region ----------------- フィールド -----------------

        // グラフ内で使用される、フォント
        Font _font = new Font( "Tahoma", 8.25F );
        Font _fontBold = new Font("Tahoma", 8.25F, FontStyle.Bold);
        Font _keyInfoFont = new Font("Tahoma", 8.25F);

        // ビューの表示倍率( グラフ空間 / スクリーン空間 )
        SizeF _viewScale = new SizeF( 0.5f, 0.5f );
        // グラフ空間でのビューのサイズ
        SizeF _viewSizeInGraphSpace = new SizeF( 100, 100 );
        // グラフ空間でのビューの位置(ウインドウの中心の位置)
        PointF _viewPosInGraphSpace = new PointF( 0, 60 );


        // マウス操作関連のパラメータ
        GraphViewMouseState _mouseState = null;


        // 選択矩形
        SelectRect _selectRect = new SelectRect();

        // プロット対象となっている、アニメーションカーブのセット(IAnmCurve)
        TargetAnmAttributeSet _targetAnmAttributeSet = null;


        IRenderer _renderer = null;

        // キーフレーム操作クラス
        AnmKeyFrameManipulator _anmKeyFrameMnp = new AnmKeyFrameManipulator();
        AnmCurveManipulator _anmCurveMnp = new AnmCurveManipulator();

        // グリッドカラー
        Color _gridColor = Color.Gray;
        // 現在の時間カラー
        Color _currentTimeColor = Color.DimGray;
        // キーフレーム外領域を描画するか
        bool _drawInifinity = true;

        // キーとキーの間の描画分割数
        int _numSubCurveDivision = 8;

        // Renderer 生成に使用したウインドウハンドル
        private IntPtr rendererHandle { get; set; } = IntPtr.Zero;


        #endregion //  ----------------- フィールド -----------------

        #region ----------------- プロパティ -----------------

        /// <summary>
        /// ビュー縦横比
        /// </summary>
        public float AspectRatio
        {
            get { return _viewScale.Height / _viewScale.Width; }
        }

        /// <summary>
        /// ビュースケール
        /// </summary>
        public SizeF ViewScale
        {
            set
            {
                bool isChanged = false;

                if (value.Width < (float)byte.MaxValue && value.Width > GraphView._ViewScaleMinimum &&
                    value.Width != _viewScale.Width )
                {
                    _viewScale.Width = value.Width;
                    _viewSizeInGraphSpace.Width = ClientRectangle.Width * _viewScale.Width;
                    isChanged = true;
                }

                if (value.Height < (float)byte.MaxValue && value.Height > GraphView._ViewScaleMinimum &&
                    value.Height != _viewScale.Height )
                {
                    _viewScale.Height = value.Height;
                    _viewSizeInGraphSpace.Height = ClientRectangle.Height * _viewScale.Height;
                    isChanged = true;
                }

                if (isChanged)
                {
                    this.Invalidate();
                }
            }

            get
            {
                return _viewScale;
            }
        }

        /// <summary>
        /// ターゲットアトリビュートセット
        /// </summary>
        public TargetAnmAttributeSet TargetAnmAttributeSet
        {
            get { return _targetAnmAttributeSet; }
            set
            {
                if( value != null )
                {
                    _targetAnmAttributeSet = value;
                    _targetAnmAttributeSet.OnSelectedKeySetChange += OnSelectedKeySetChanged_;
                }
            }
        }

        /// <summary>
        /// 操作対象アニメーションカーブセットを取得します。
        /// </summary>
        public IAnmCurve[] TargetCurveSet
        {
            get
            {
                if( _targetAnmAttributeSet == null )
                {
                    return new IAnmCurve[0];
                }
                return _targetAnmAttributeSet._TargetCurveSet;

            }
        }

        /// <summary>
        /// 前後区間キーの表示・非表示を設定、取得します。
        /// </summary>
        public bool DispBeforeAfterKeyInfo
        {
            get; set;
        }

        /// <summary>
        /// 矢印キーの移動量を設定、取得します。
        /// </summary>
        public GraphViewMouseState.ArrowKeySpeed ArrowKeySpeedSettings
        {
            get { return _mouseState.ArrowKeySpeedSettings; }
            set { _mouseState.ArrowKeySpeedSettings = value; }
        }

        /// <summary>
        /// クランプ設定を設定、取得します。
        /// </summary>
        public GraphValueSnap GraphValueSnap
        {
            get { return _mouseState.GraphValueSnap; }
            set { _mouseState.GraphValueSnap = value; }
        }

        //
        /// <summary>
        /// マウス編集モードです。
        /// </summary>
        public GraphViewMouseTweakMode MouseTweakMode
        {
            get { return _mouseState.MouseTweakMode; }
            set { _mouseState.MouseTweakMode = value; }
        }

        /// <summary>
        /// グリットカラー
        /// </summary>
        public Color GridColor
        {
            get { return _gridColor; }
            set { _gridColor = value; }
        }

        /// <summary>
        /// 現在の時間カラー
        /// </summary>
        public Color CurrentTimeColor
        {
            get { return _currentTimeColor; }
            set { _currentTimeColor = value; }
        }

        /// <summary>
        /// キーフレーム外領域を描画するか設定、取得します。
        /// </summary>
        public bool DrawInifinity
        {
            get { return _drawInifinity; }
            set
            {
                if( _drawInifinity != value )
                {
                    _drawInifinity = value;
                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// 選択キーの配列を取得します。
        /// </summary>
        IAnmKeyFrame[] SelectedKeyFrameSet_
        {
            get
            {
                if( _targetAnmAttributeSet == null )
                {
                    return new IAnmKeyFrame[0];
                }
                return _targetAnmAttributeSet._SelectedKeyFrameSet;
            }
        }

        /// <summary>
        /// 選択アンカーを取得します。
        /// </summary>
        KeyAanchor[] _SelectedAnchorSet
        {
            get
            {
                if( _targetAnmAttributeSet == null )
                {
                    return new KeyAanchor[0];
                }
                return _targetAnmAttributeSet._SelectedAnchorSet;
            }
        }

        float _ViewLeftInGraphSpace
        {
            get { return _viewPosInGraphSpace.X - _viewSizeInGraphSpace.Width / 2.0f; }
        }

        float _ViewRightInGraphSpace
        {
            get { return _viewPosInGraphSpace.X + _viewSizeInGraphSpace.Width / 2.0f; }
        }

        float _ViewTopInGraphSpace
        {
            get { return _viewPosInGraphSpace.Y - _viewSizeInGraphSpace.Height / 2.0f; }
        }

        float _ViewBottomInGraphSpace
        {
            get { return _viewPosInGraphSpace.Y + _viewSizeInGraphSpace.Height / 2.0f; }
        }

        /// <summary>
        /// 表示時間範囲を取得します。
        /// </summary>
        public float ViewRangeInGraphSpace
        {
            get { return _viewSizeInGraphSpace.Width; }
        }

        ISubScene _CurrentSubScene
        {
            get
            {
                return LECore.LayoutEditorCore.Scene.CurrentISubScene;
            }
        }

        /// <summary>
        /// サブカーブ描画分割数(1-32)
        /// </summary>
        int NumSubCurveDivision
        {
            get { return _numSubCurveDivision; }
            set
            {
                if( value > 0 && value <= 32 )
                {
                    _numSubCurveDivision = value;
                }
            }
        }

        /// <summary>
        // 親ウインドウ
        /// </summary>
        public LEToolWindow ParentWindow { get; set; }


        #endregion // ----------------- プロパティ -----------------

        #region デザイナ生成コード
        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            //
            // GraphView
            //
            this.SizeChanged += new System.EventHandler( this.Event_GraphView_SizeChanged );
            this.MouseUp += new System.Windows.Forms.MouseEventHandler( this.Event_GraphView_MouseUp );
            this.KeyUp += new System.Windows.Forms.KeyEventHandler( this.GraphView_KeyUp );
            this.KeyDown += new System.Windows.Forms.KeyEventHandler( this.GraphView_KeyDown );
            this.MouseMove += new System.Windows.Forms.MouseEventHandler( this.Event_GraphView_MouseMove );
            this.MouseDown += new System.Windows.Forms.MouseEventHandler( this.Event_GraphView_MouseDown );

        }
        #endregion
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;
        #endregion デザイナ生成コード

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GraphView()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();
            _mouseState = new GraphViewMouseState( this );

            this.MouseWheel += Event_GraphView_MouseWheel;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                    components.Dispose();
            }

            if (_renderer != null)
            {
                _renderer.Dispose();
            }

            base.Dispose( disposing );
        }

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            // レンダラの初期化
            InitializeRenderer();
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if (_renderer == null)
            {
                InitializeRenderer();
            }

            base.OnSizeChanged(e);
        }

        private void InitializeRenderer()
        {
            if (Size.Width > 0 && Size.Height > 0 && IsHandleCreated)
            {
                if (_renderer != null)
                {
                    if (Handle != rendererHandle)
                    {
                        // ドッキングウインドウを使用しているときにハンドルが変わることがある
                        // ハンドルが切り替わっているので再度初期化
                        _renderer.Dispose();
                        _renderer = null;
                    }
                    else
                    {
                        // 既に初期化されているので何もしない
                        return;
                    }
                }

                _renderer = RendererFactory.Create(AppSetting.RendererType.D3D);
                SetStyle(RendererFactory.GetControlState(AppSetting.RendererType.D3D), true);

                bool bInitSuccess = _renderer.Initialize(this);
                Trace.Assert(bInitSuccess);

                rendererHandle = Handle;
            }
        }

        #region ------------------- 外部非公開メソッド -------------------

        /// <summary>
        /// キー選択の変更を通知する。
        /// </summary>
        void NotifyKeySelectionChange_()
        {
            if( OnChange != null )
            {
                OnChange( ChangeEvent.KeySelectionChange );
            }
        }

        /// <summary>
        /// 対象カーブ変更を通知する。
        /// </summary>
        void NotifyTargetCurveChange_()
        {
            if( OnChange != null )
            {
                OnChange( ChangeEvent.TargetCurveChange );
            }
        }

        /// <summary>
        /// 選択キー変更ハンドラ
        /// </summary>
        void OnSelectedKeySetChanged_( object sender, EventArgs args )
        {
            _mouseState.SetDefault();
            Invalidate();
        }

        #region 座標変換
        /// <summary>
        /// 座標変換
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        PointF ConvertPosFromScrnToGraphSpace_( PointF pos )
        {
            // ウインドウ中心からの座標に変換
            PointF posCtr = MathUtil.ScaleVec( new Point( ClientRectangle.Size ), 0.5f );
            pos = MathUtil.SubVec( pos, posCtr );

            pos.X = _viewPosInGraphSpace.X + pos.X * _viewScale.Width;
            pos.Y = _viewPosInGraphSpace.Y - pos.Y * _viewScale.Height;
            return pos;
        }

        /// <summary>
        /// 座標変換
        /// </summary>
        /// <param name="pos"></param>
        /// <returns></returns>
        PointF ConvertPosFromGraphToScrnSpace_( PointF pos )
        {
            // スクリーンの中心に対応する点からの相対位置に変換します。
            pos = MathUtil.SubVec( pos, _viewPosInGraphSpace );

            // 相対位置をスクリーン上でのピクセル値に変換します。
            pos.X /= _viewScale.Width;
            pos.Y /= -_viewScale.Height;

            // 画面の中心位置座標を足します。
            PointF posCtr = MathUtil.ScaleVec( new Point( ClientRectangle.Size ), 0.5f );
            pos = MathUtil.AddVec( pos, posCtr );

            return pos;
        }

        /// <summary>
        /// 矩形：座標変換
        /// </summary>
        RectangleF ConvertRectangleFromGraphToScrnSpace_( RectangleF rectInGraph )
        {
            PointF pntRectTL = ConvertPosFromGraphToScrnSpace_( new PointF( rectInGraph.Left, rectInGraph.Bottom ) );
            PointF pntRectRB = ConvertPosFromGraphToScrnSpace_( new PointF( rectInGraph.Right, rectInGraph.Top ) );
            RectangleF rectScrn = RectangleF.FromLTRB( pntRectTL.X, pntRectTL.Y, pntRectRB.X, pntRectRB.Y );

            return MathUtil.MakePositiveSizeRectangle( rectScrn );
        }
        #endregion 座標変換

        #region 表示関連パラメータの変更


        /// <summary>
        /// 指定範囲を表示するように、ビューパラメータを設定します。
        /// </summary>
        public void SetViewParamaters( float minT, float maxT, float minVal, float maxVal, float fitRatio )
        {
            SetViewParamatersH_(minT, maxT, fitRatio);
            SetViewParamatersV_(minVal, maxVal, fitRatio);

            Invalidate();
        }

        /// <summary>
        /// 指定範囲を表示するように、ビューパラメータを設定します。
        /// </summary>
        public void SetViewParamatersH(float minT, float maxT, float fitRatio)
        {
            SetViewParamatersH_(minT, maxT, fitRatio);
            Invalidate();
        }

        /// <summary>
        /// 指定範囲を表示するように、ビューパラメータを設定します。
        /// </summary>
        void SetViewParamatersH_(float minT, float maxT, float fitRatio)
        {
            _viewPosInGraphSpace.X = (minT + maxT) * 0.5f;
            _viewSizeInGraphSpace.Width = (maxT - minT) / fitRatio;

            if (_viewSizeInGraphSpace.Width == 0.0f)
            {
                _viewSizeInGraphSpace.Width = this.ClientRectangle.Width;
            }

            _viewScale.Width = _viewSizeInGraphSpace.Width / this.ClientRectangle.Width;
        }

        /// <summary>
        /// 指定範囲を表示するように、ビューパラメータを設定します。
        /// </summary>
        void SetViewParamatersV_(float minVal, float maxVal, float fitRatio)
        {
            // 文字が重ならないように fitRatio を調整する
            var margin = _fontBold.Height * 4; // 時間軸, 現在時刻, 選択キーの情報(2行)
            fitRatio = Math.Max(0.1f, Math.Min(fitRatio, (ClientRectangle.Height - margin * 2) / (float)ClientRectangle.Height));

            _viewPosInGraphSpace.Y = (minVal + maxVal) * 0.5f;

            _viewSizeInGraphSpace.Height = (maxVal - minVal) / fitRatio;

            if (_viewSizeInGraphSpace.Height == 0.0f)
            {
                _viewSizeInGraphSpace.Height = this.ClientRectangle.Height;
            }

            _viewScale.Height = _viewSizeInGraphSpace.Height / this.ClientRectangle.Height;
        }

        /// <summary>
        /// スケールを加算します。
        /// </summary>
        /// <param name="sx"></param>
        /// <param name="sy"></param>
        public void PanViewScale()
        {
            var sx = _viewScale.Width + _ViewScaleStep * _viewScale.Width;
            var sy = _viewScale.Height + _ViewScaleStep * _viewScale.Height;

            ViewScale = new SizeF( sx, sy );
        }

        /// <summary>
        /// スケールを減算します。
        /// </summary>
        /// <param name="sx"></param>
        /// <param name="sy"></param>
        public void ZoomViewScale()
        {
            var sx = _viewScale.Width - _ViewScaleStep * _viewScale.Width;
            var sy = _viewScale.Height - _ViewScaleStep * _viewScale.Height;

            ViewScale = new SizeF(sx, sy);
        }

        #endregion 表示関連パラメータの変更

        #region キー選択関連
        /// <summary>
        /// 矩形内に存在するキーフレームを返します。
        /// </summary>
        /// <param name="selRect"></param>
        /// <returns></returns>
        List<IAnmKeyFrame> GetKeyFramesInsideRect_( RectangleF selRect )
        {
            List<IAnmKeyFrame> keySet = new List<IAnmKeyFrame>();
            // 矩形内のキーを選択セットに登録します。
            foreach( IAnmCurve anmCurve in TargetCurveSet )
            {
                int numKey = anmCurve.NumKeyFrame;
                foreach( IAnmKeyFrame key in anmCurve.IKeyFrameSet )
                {
                    if( selRect.Contains( GetKeyAsPointF_( key ) ) )
                    {
                        if( !keySet.Contains( key ) )
                        {
                            keySet.Add( key );
                        }
                    }
                }
            }
            return keySet;
        }


        /// <summary>
        /// アンカーをあらわすスクリーン上の点を計算します。
        /// </summary>
        /// <param name="keyInfo"></param>
        /// <param name="posAnchor"></param>
        /// <param name="tangent"></param>
        /// <returns></returns>
        PointF GetTangetAnchorPoint_( PointF pntKey, PointF posAnchor, float tangent )
        {
            float rotAngle = -MathUtil.RadToDeg( (float)Math.Atan( tangent / AspectRatio ) );

            Matrix rotMtx = new Matrix();
            rotMtx.RotateAt( rotAngle, pntKey );
            return MathUtil.MtxTransformPoint( rotMtx, new PointF( pntKey.X + posAnchor.X, pntKey.Y + posAnchor.Y ) );
        }

        /// <summary>
        /// 矩形内に存在するキーフレームを選択します。
        /// </summary>
        /// <param name="selRect"></param>
        void SelectKeyFrame_( RectangleF selRect, bool bAdditional )
        {
            // アニメーションカーブ中のキーフレーム選択を試みる
            List<IAnmKeyFrame> keysInRect = GetKeyFramesInsideRect_( selRect );
            IAnmKeyFrame[] selKeyArray = SelectedKeyFrameSet_;

            TargetAnmAttributeSet.BeginKeySelectChange();

            // 現在の設定をリセットして再設定
            if( !bAdditional )
            {
                TargetAnmAttributeSet.ResetSelectKeyAll();
                TargetAnmAttributeSet.ResetSelectedTangentAnchor();
            }

            // 矩形内にキーがあれば、
            if( keysInRect.Count != 0 )
            {
                TargetAnmAttributeSet.SelectKey( keysInRect.ToArray(), bAdditional );
            }
            else
            {
                // 選択がなければ、
                // 選択キーフレームセット内から、アンカーポイントの選択を行う
                bool isAnchorSelected = false;
                if (selKeyArray != null)
                {
                    // 接線選択セットに登録
                    List<KeyAanchor> anchorSet = GetSelectedAnchorFormKeys_(selKeyArray, selRect);
                    if (anchorSet.Count > 0)
                    {
                        TargetAnmAttributeSet.SelectKeyAnchor(anchorSet.ToArray(), bAdditional);
                        isAnchorSelected = true;
                    }
                }

                // 選択がなかった場合は、カーブ曲線からの選択を試みます。
                if (!isAnchorSelected)
                {
                    foreach (IAnmCurve anmCurve in TargetCurveSet)
                    {
                        if (CheckRectHitCurve_(selRect, anmCurve))
                        {
                            TargetAnmAttributeSet.SelectKey(anmCurve.IKeyFrameSet, bAdditional);
                        }
                    }
                }
            }

            TargetAnmAttributeSet.EndKeySelectChange();

            NotifyKeySelectionChange_();
            // NOTICE:マウスモードの変更は、行わない。
        }

        /// <summary>
        /// キーセットから選択接線アンカーセットを取得します。
        /// </summary>
        List<KeyAanchor> GetSelectedAnchorFormKeys_(IEnumerable<IAnmKeyFrame> keys, RectangleF selRectInGraph)
        {
            List<KeyAanchor> anchorSet = new List<KeyAanchor>();

            // すべての選択キーについて...
            RectangleF selRectInScrn = ConvertRectangleFromGraphToScrnSpace_(selRectInGraph);
            foreach (IAnmKeyFrame key in keys)
            {
                // 編集が行えない場合は、接線選択セットには登録 しない。
                if (!key.TangentModifyEnabled)
                {
                    continue;
                }

                // キーが矩形に含まれているなら
                PointF pntKey = ConvertPosFromGraphToScrnSpace_(GetKeyAsPointF_(key));
                PointF inTanPnt = GetTangetAnchorPoint_(pntKey, new PointF(-32.0f, 0.0f), key.InTangent);
                PointF outTanPnt = GetTangetAnchorPoint_(pntKey, new PointF(32.0f, 0.0f), key.OutTangent);

                if (selRectInScrn.Contains(inTanPnt))
                {
                    anchorSet.Add(KeyAanchor.MakeLeftSlope(key));
                }
                else if (selRectInScrn.Contains(outTanPnt))
                {
                    anchorSet.Add(KeyAanchor.MakeRightSlope(key));
                }
            }

            return anchorSet;
        }

        /// <summary>
        ///
        /// </summary>
        bool CheckRectHitCurve_(RectangleF selRect, IAnmCurve curve)
        {
            PointF pR0 = new PointF(selRect.Left, selRect.Bottom);
            PointF pR1 = new PointF(selRect.Right, selRect.Top);

            // すべてのキーフレーム間において...
            for (int i = 0; i < curve.IKeyFrameSet.Length - 1; i++)
            {
                float t0 = curve.IKeyFrameSet[i].Time;
                float t1 = curve.IKeyFrameSet[i + 1].Time;

                // 明らかに範囲外なので処理をパス
                if (pR1.X < t0 || pR0.X > t1)
                {
                    continue;
                }

                // キー間を分割して...
                const int numStep = 64;
                float tStep = (t1 - t0) / (float)numStep;
                for (int ii = 0; ii < numStep; ii++)
                {
                    float t00 = t0 +  ii      * tStep;
                    float t01 = t0 + (ii + 1) * tStep;

                    // 明らかに範囲外なので処理をパス
                    if (pR1.X < t00 || pR0.X > t01)
                    {
                        continue;
                    }

                    PointF p00 = new PointF(t00, curve.Evaluate(t00));
                    PointF p01 = new PointF(t01, curve.Evaluate(t01));

                    // 直線 p00 - p01 と pR0 - pR1 との交差判定を行う
                    if (AreLinesCrossed_(p00, p01, pR0, pR1))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// 線分の交差判定
        /// </summary>
        bool AreLinesCrossed_(PointF pA0, PointF pA1, PointF pB0, PointF pB1)
        {
            var vA0 = GetLineEquationValue_(pA0, pA1, pB0);
            var vA1 = GetLineEquationValue_(pA0, pA1, pB1);

            var vB0 = GetLineEquationValue_(pB0, pB1, pA0);
            var vB1 = GetLineEquationValue_(pB0, pB1, pA1);
            return vA0 * vA1 <= 0.0f && vB0 * vB1 <= 0.0f;
        }

        /// <summary>
        /// 点における、直線の方程式の値を求めます。
        /// </summary>
        float GetLineEquationValue_(PointF pA0, PointF pA1, PointF p)
        {
            return (pA1.X - pA0.X) * (p.Y - pA0.Y) - (pA1.Y - pA0.Y) * (p.X - pA0.X);
        }

        /// <summary>
        /// キーフレームをポイントに変換します。
        /// プロット用です。
        /// </summary>
        /// <param name="anmCurve"></param>
        /// <param name="kf"></param>
        /// <returns></returns>
        PointF GetKeyAsPointF_( IAnmKeyFrame kf )
        {
            return new PointF( (float)kf.Time, kf.ValueAsFloat );
        }

        /// <summary>
        /// キーフレームをポイントに変換します。(スクリーン座標系)
        /// </summary>
        PointF GetKeyAsScreenPointF_( IAnmKeyFrame kf )
        {
            return ConvertPosFromGraphToScrnSpace_( GetKeyAsPointF_( kf ) );
        }
        #endregion キー選択関連



        #endregion // ------------------- 外部非公開メソッド -------------------

        #region ------------- イベントハンドラ -------------

        #region ------------------- 描画関連 -------------------

        /// <summary>
        /// 1ドット表示されるような、線の太さを取得します。
        /// </summary>
        float Get1DotLineWidth_()
        {
            return Math.Min( _viewScale.Width, _viewScale.Height );
        }

        #region キーフレームの描画
        /// <summary>
        /// キーフレームの接線を描画します。
        /// </summary>
        void DrawTangent_( IRenderer renderer, PointF pO, PointF vAnchor, float tangent )
        {
            PointF[] temp = new PointF[] { pO, vAnchor };
            renderer.TransformPointsToView(temp);
            pO = temp[0];

            // 正規アスペクト比における回転角をもとめるため、tangent / AspectRatio とします。
            float rotAngle = MathUtil.RadToDeg( (float)Math.Atan( tangent / AspectRatio ) );
            Matrix m = new Matrix();
            m.Rotate( rotAngle );
            vAnchor = MathUtil.MtxTransformPoint( m, vAnchor );
            vAnchor = MathUtil.AddVec( pO, vAnchor );


            // 単位行列で描画を行い、その後行列を復元します。
            renderer.PushMtx();
            renderer.IdentityMtx();

            renderer.LineWidth = _KeyDotSize;
            renderer.DrawPoint(pO);
            renderer.DrawPoint(vAnchor);

            renderer.LineWidth = 1.0f;
            renderer.DrawLine(pO, vAnchor);

            renderer.PopMtx();
        }

        /// <summary>
        /// キーフレームを描画します。
        /// </summary>
        void DrawBeforeAfterKeyFrame_(IRenderer renderer, PointF pos)
        {
            renderer.Color = Color.FromArgb(127, Color.DeepPink);
            renderer.LineWidth = _KeyDotSize;
            renderer.DrawPoint(pos);
        }

        /// <summary>
        /// キーフレームを描画します。
        /// </summary>
        void DrawKeyFrame_( IRenderer renderer, IAnmKeyFrame kf )
        {
            PointF pos = GetKeyAsPointF_( kf );

            var isZero = kf.ValueAsFloat == 0.0f;
            renderer.Color = kf.IsSelected ? (isZero ? Color.Cyan: Color.Yellow) : (isZero ? Color.Blue : Color.Black);
            renderer.LineWidth = _KeyDotSize;
            renderer.DrawPoint(pos);

            // 接線を描画します。
            if( kf.IsSelected && kf.TangentModifyEnabled )
            {
                renderer.Color = Color.Green;
                DrawTangent_( renderer, pos, new PointF( 32.0f, 0 ), kf.OutTangent );
                DrawTangent_( renderer, pos, new PointF( -32.0f, 0 ), kf.InTangent );
            }
        }

        /// <summary>
        /// キーフレームを描画します。
        /// </summary>
        void DrawKeyFrames_(
            IRenderer renderer,
            IAnmCurve anmCurve )
        {
            renderer.Color = Color.Black;

            // 最後のキーフレーム以外のすべてのキーフレームについて...
            foreach( IAnmKeyFrame kf in anmCurve.IKeyFrameSet )
            {
                // 両端を調べ、キーが完全に画面外なら、処理しません。
                if( kf.Time < _ViewLeftInGraphSpace ||
                    kf.Time > _ViewRightInGraphSpace )
                {
                    continue;
                }

                // キーフレームの描画
                DrawKeyFrame_( renderer, kf );
            }
        }
        #endregion キーフレームの描画

        #region カーブの描画
        /// <summary>
        /// 時間が、表示範囲にふくまれるか判定します。
        /// </summary>
        bool IsTimeInsideView_( float time )
        {
            return ( time < _ViewLeftInGraphSpace || time > _ViewRightInGraphSpace );
        }

        /// <summary>
        /// キーフレーム区間の長さを計算します。
        /// </summary>
        float CalcKeyedAreaLength_( IAnmCurve anmCurve )
        {
            IAnmKeyFrame[] keyFrameSet = anmCurve.IKeyFrameSet;

            float firstKeyTime = anmCurve.FrameOfHead;
            float lastKeyTime = anmCurve.FrameOfTail;

            // lastKeyTime - firstKeyTime + 1 ではありません
            // t == lastKeyTime はキー領域ですが、
            // t > lastKeyTime 領域は post infinity 領域である点に注意してください。
            return ( lastKeyTime - firstKeyTime );
        }

        /// <summary>
        /// キーフレーム前区間の描画
        /// </summary>
        void DrawPreInfinity_(
            IRenderer renderer,
            IAnmCurve anmCurve,
            Color lineCol)
        {
            IAnmKeyFrame[] keyFrameSet = anmCurve.IKeyFrameSet;
            float keyAreaLength = CalcKeyedAreaLength_( anmCurve );

            if( anmCurve.PreInfinityType == AnmCurveInfinityType.Cycle &&
                keyAreaLength != 0 )
            {
                // AnmCurveInfinityType.Cycle の場合
                float firstKeyTime = anmCurve.FrameOfHead;
                float preAreaLength = firstKeyTime - (int)_ViewLeftInGraphSpace + 1;
                Debug.Assert( keyAreaLength != 0 );

                int numDraw = (int)( preAreaLength / keyAreaLength ) + 1;

                float drawStartTime = firstKeyTime - ( numDraw * keyAreaLength );

                for( int i = 0 ; i < numDraw ; i++ )
                {
                    DrawCurve_( renderer, lineCol, lineCol, anmCurve, drawStartTime );

                    drawStartTime += keyAreaLength;

                    PointF posEnd = new PointF( (float)( drawStartTime ), anmCurve.ValueOfTail );
                    PointF posNextStart = new PointF( (float)( drawStartTime ), anmCurve.ValueOfHead );
                    renderer.DrawLine( posEnd, posNextStart );
                }
            }
            else
            {
                // AnmCurveInfinityType.Constant の場合
                // 画面左端から、先頭キーへ直線を描画します。
                PointF pFirstKey = new PointF(anmCurve.FrameOfHead, anmCurve.ValueOfHead);
                PointF pL = new PointF( _ViewLeftInGraphSpace, pFirstKey.Y );
                renderer.Color = lineCol;
                renderer.DrawLine( pL, pFirstKey );
            }
        }

        /// <summary>
        /// キーフレーム後区間の描画
        /// </summary>
        void DrawPostInfinity_(
            IRenderer renderer,
            IAnmCurve anmCurve,
            Color lineCol)
        {
            IAnmKeyFrame[] keyFrameSet = anmCurve.IKeyFrameSet;
            float keyAreaLength = CalcKeyedAreaLength_( anmCurve );

            if( anmCurve.PostInfinityType == AnmCurveInfinityType.Cycle &&
                keyAreaLength != 0 )
            {
                float lastKeyTime = anmCurve.FrameOfTail;
                float postAreaLength = (int)_ViewRightInGraphSpace - lastKeyTime + 1;
                Debug.Assert( keyAreaLength != 0 );

                int numDraw = (int)( postAreaLength / keyAreaLength ) + 1;
                float drawStartTime = lastKeyTime;

                // キーフレーム区間と preInfinity 区間をつなぐ直線を描画
                PointF p0 = new PointF( (float)( lastKeyTime ), anmCurve.ValueOfTail );
                PointF p1 = new PointF( (float)( lastKeyTime ), anmCurve.ValueOfHead );
                renderer.DrawLine( p0, p1 );


                for( int i = 0 ; i < numDraw ; i++ )
                {
                    Color col = Color.FromArgb( 64, Color.Red );
                    DrawCurve_( renderer, lineCol, lineCol, anmCurve, drawStartTime );

                    drawStartTime += keyAreaLength;

                    PointF posEnd = new PointF( drawStartTime, anmCurve.ValueOfTail );
                    PointF posNextStart = new PointF( drawStartTime, anmCurve.ValueOfHead );
                    renderer.DrawLine( posEnd, posNextStart );
                }
            }
            else
            {
                // AnmCurveInfinityType.Constant の場合
                // 最終 キーから、画面右端へ直線を描画します。
                PointF pLastKey = new PointF(anmCurve.FrameOfTail, anmCurve.ValueOfTail);
                PointF pR = new PointF( _ViewRightInGraphSpace, pLastKey.Y );

                renderer.Color = lineCol;
                renderer.DrawLine( pLastKey, pR );
            }
        }

        /// <summary>
        /// キーフレームとキーフレームの間の部分カーブを描画します。
        /// </summary>
        void DrawSubCurve_(
            IRenderer renderer,
            IAnmCurve anmCurve,
            bool isStep,
            float kfsTime,
            float kfsValue,
            float kfeTime,
            float kfeValue,
            float timeOffset )
        {

            PointF p0 = PointF.Empty;
            PointF p1 = PointF.Empty;

            // 最初の１点目の設定
            p0.X = kfsTime + timeOffset;
            p0.Y = kfsValue;

            // ステップキーの場合は直線描画となります。
            if(isStep)
            {
                p1 = new PointF( kfeTime + timeOffset, kfeValue );
                PointF p0_1 = new PointF( p1.X, p0.Y );

                renderer.DrawLine( p0, p0_1 );
                renderer.DrawLine( p0_1, p1 );
                return;
            }

            // 分割数、（時間軸スケールによって、増加させます。）
            int numDivide = (int)MathUtil.Clamp( NumSubCurveDivision / _viewScale.Width, NumSubCurveDivision, 32.0f );

            float timeStep = (float)( kfeTime - kfsTime ) / (float)numDivide;

            // timeStep ずつ 時間をずらしながら...
            for( int i = 0 ; i < numDivide ; i++ )
            {
                float t = kfsTime + timeStep * i;

                // 1点目が画面外なら描画を終了します。
                if( p0.X > _ViewRightInGraphSpace )
                {
                    return;
                }

                // 2点目の設定
                // カーブの評価に使用する時間は time ではなく t である
                float time = t + timeOffset;
                p1.X = time;
                p1.Y = anmCurve.Evaluate( t );


                // 2点目の時間が最小時間以下の場合スキップ
                if( time >= _ViewLeftInGraphSpace )
                {
                    // 直線の描画
                    renderer.DrawLine( p0, p1 );
                }

                // 2点目を次の1点目とします。
                p0 = p1;
            }

            // 最終フレームに確実に線がつながるように描画します。
            p1.X = kfeTime + timeOffset;
            p1.Y = kfeValue;
            renderer.DrawLine( p0, p1 );
        }

        /// <summary>
        /// カーブ部分の描画を行います。
        /// </summary>
        void DrawCurve_(
            IRenderer renderer,
            Color baseColor,
            Color hilightColor,
            IAnmCurve anmCurve,
            float timeOffset )
        {
            float firstKeyTime = anmCurve.FrameOfHead;
            float lastKeyTime = anmCurve.FrameOfTail;
            float drawOffsetTime = timeOffset - firstKeyTime;

            if (anmCurve.IsParameterizedAnim)
            {
                int divisionNum = 10; // 曲線の分割数
                float delta = (lastKeyTime - firstKeyTime) / (float)(divisionNum - 1);
                for (int i = 0; i < divisionNum; i++)
                {
                    float kf0Time = firstKeyTime + i * delta;
                    float kf1Time = kf0Time + delta;

                    // 両端を調べ、曲線が完全に画面外なら、処理しません。
                    if (kf1Time + drawOffsetTime < _ViewLeftInGraphSpace ||
                        kf0Time + drawOffsetTime > _ViewRightInGraphSpace)
                    {
                        continue;
                    }

                    float kf0Value = anmCurve.Evaluate(kf0Time);
                    float kf1Value = anmCurve.Evaluate(kf1Time);
                    DrawSubCurve_(renderer, anmCurve, false, kf0Time, kf0Value, kf1Time, kf1Value, drawOffsetTime);
                }
            }
            else
            {
                // 描画開始キー、終了キーを捜します
                IAnmKeyFrame[] keyFrameSet = anmCurve.IKeyFrameSet;

                int numKey = keyFrameSet.Length;

                IAnmKeyFrame kf0 = null;
                IAnmKeyFrame kf1 = null;

                // 最後のキーフレーム以外のすべてのキーフレームについて...
                for (int i = 0; i < numKey - 1; i++)
                {
                    kf0 = keyFrameSet[i];
                    kf1 = keyFrameSet[i + 1];

                    // 両端を調べ、曲線が完全に画面外なら、処理しません。
                    if (kf1.Time + drawOffsetTime < _ViewLeftInGraphSpace ||
                        kf0.Time + drawOffsetTime > _ViewRightInGraphSpace)
                    {
                        continue;
                    }

                    // 選択されているキー以降は色を変えて描画します。
                    if (kf0.IsSelected || kf1.IsSelected)
                    {
                        renderer.Color = hilightColor;
                    }
                    else
                    {
                        renderer.Color = baseColor;
                    }

                    // カーブの描画
                    // カーブの描画ステップは、表示されれるピクセルサイズに応じて可変させている。
                    bool isStep = kf0.InterporationType == InterporationType.Step;
                    DrawSubCurve_(renderer, anmCurve, isStep, kf0.Time, kf0.ValueAsFloat, kf1.Time, kf1.ValueAsFloat, drawOffsetTime);
                }
            }
        }

        /// <summary>
        /// アニメーションカーブを描画します。
        /// </summary>
        /// <param name="g"></param>
        void DrawAnmCurves_( IRenderer renderer )
        {
            if (_targetAnmAttributeSet == null)
            {
                return;
            }

            renderer.LineWidth = Get1DotLineWidth_();

            // 選択対象であるすべてのカーブについて...
            foreach (TargetAnmAttributeSet.DrawAnmCurve drwaAnmCurve in _targetAnmAttributeSet._TargetDrawCurveSet)
            {
                var anmCurve = drwaAnmCurve.AnmAttr.ICurrentAnimationCurve;
                int numKey = anmCurve.NumKeyFrame;

                // 前後区間の描画
                // キーが存在しない場合でも前後区間の描画を行います
                DrawBeforeAfterCurve_(renderer, drwaAnmCurve);

                // キーが存在しないカーブは処理をスキップします。
                if( numKey <= 0 && !anmCurve.IsParameterizedAnim )
                {
                    continue;
                }

                // キーフレーム設定領域外の描画
                if( _drawInifinity )
                {
                    DrawPreInfinity_(renderer, anmCurve, Color.FromArgb(64, drwaAnmCurve.DrawColor));
                    DrawPostInfinity_(renderer, anmCurve, Color.FromArgb(64, drwaAnmCurve.DrawColor));
                }

                // キーフレーム領域の描画
                IAnmKeyFrame[] keyFrameSet = anmCurve.IKeyFrameSet;
                DrawCurve_(renderer, drwaAnmCurve.DrawColor, Color.White, anmCurve, anmCurve.FrameOfHead);
                DrawKeyFrames_( renderer, anmCurve );
            }
        }

        /// <summary>
        /// 前後区間のカーブを描画します。
        /// </summary>
        void DrawBeforeAfterCurve_(IRenderer renderer, TargetAnmAttributeSet.DrawAnmCurve drawAnmCurve)
        {
            if (!this.DispBeforeAfterKeyInfo || !_CurrentSubScene.IsAnimEditSeparateMode())
            {
                return;
            }

            var anmCurve = drawAnmCurve.AnmAttr.ICurrentAnimationCurve;

            // 現在区間の前後の区間を取得
            IAnimFrameSection targetSection = _CurrentSubScene.IAnimFrameSectionSet.TargetIAnimFrameSection;
            IAnimFrameSection beforeSection;
            IAnimFrameSection afterSection;
            _CurrentSubScene.GetBeforeAfterAnimSection(out beforeSection, out afterSection);

            // 前後区間のカーブを取得
            IAnmCurve beforeCurve = drawAnmCurve.AnmAttr.GetAnmCurveRelationTag(beforeSection?.Name);
            IAnmCurve afterCurve = drawAnmCurve.AnmAttr.GetAnmCurveRelationTag(afterSection?.Name);

            // 前区間の描画
            if (beforeCurve != null && beforeCurve.NumKeyFrame > 0)
            {
                IAnmKeyFrame key = anmCurve?.GetFirstKey();

                float x = targetSection != null ? targetSection.StartFrame : 0;
                x = (key != null && key.Time < x) ? key.Time : x;

                float y = beforeCurve.Evaluate(beforeSection.EndFrame);

                PointF pos = new PointF(x - 1, y);

                // カーブの描画
                if (key != null)
                {
                    renderer.Color = Color.FromArgb(127, Color.HotPink);
                    renderer.DrawLine(GetKeyAsPointF_(key), pos);
                }

                // キーフレームの描画
                if (pos.X >= _ViewLeftInGraphSpace && pos.X <= _ViewRightInGraphSpace)
                {
                    DrawBeforeAfterKeyFrame_(renderer, pos);
                    DrawBeforeAfterKeyInfoSrting_(renderer, pos.X, pos.Y);
                }
            }


            // 後区間の描画
            if (afterCurve != null && afterCurve.NumKeyFrame > 0)
            {
                IAnmKeyFrame key = anmCurve?.GetLastKey();

                float x = targetSection != null ? targetSection.EndFrame : 0;
                x = (key != null && key.Time > x) ? key.Time : x;

                float y = afterCurve.Evaluate(afterSection.StartFrame);

                PointF pos = new PointF(x + 1, y);

                // カーブの描画
                if (key != null)
                {
                    renderer.Color = Color.FromArgb(127, Color.HotPink);
                    renderer.DrawLine(GetKeyAsPointF_(key), pos);
                }

                // キーフレームの描画
                if (pos.X >= _ViewLeftInGraphSpace && pos.X <= _ViewRightInGraphSpace)
                {
                    DrawBeforeAfterKeyFrame_(renderer, pos);
                    DrawBeforeAfterKeyInfoSrting_(renderer, pos.X, pos.Y);
                }
            }
        }
        #endregion カーブの描画

        /// <summary>
        /// 文字列描画
        /// </summary>
        void DrawString_( IRenderer renderer, Color color, string str, float px, float py )
        {
            DrawString_( renderer, _font, color, str, px, py );
        }

        /// <summary>
        /// 文字列を描画します。
        ///
        /// 座標変換処理が内部で行われるため、パフォーマンスに不安があります。
        /// とりあえずはこのままにしておき、将来リファクタリングすることとします。
        /// DirectX描画が導入されれば、その必要もないかもしれません。
        ///
        /// メモ：座標系の差異を吸収するために、Y座標スケールに -1.0f を乗じている。
        /// そのために、通常の文字列描画を行うと、文字がさかさまに描画されてしまう。
        ///
        /// </summary>
        void DrawString_( IRenderer renderer, Font font, Color color, string str, float px, float py )
        {
            RendererHelper.DrawStringWithoutTransform( renderer, font, color, str, px, py );
        }


        /// <summary>
        /// グリッドなどの背景の描画
        /// </summary>
        void DrawBackGround_( IRenderer renderer )
        {

            float left = _ViewLeftInGraphSpace;
            float right = _ViewRightInGraphSpace;
            float top = _viewPosInGraphSpace.Y + _viewSizeInGraphSpace.Height / 2.0f;
            float bottom = _viewPosInGraphSpace.Y - _viewSizeInGraphSpace.Height / 2.0f;


            // -------------- グリッド幅の算出
            float blankH = Math.Max(_GridMinH, NumberLineDrawer.GetNumberLineBlank(ClientRectangle.Width, left, right, 96));
            float blankV = Math.Max(_GridMinV, NumberLineDrawer.GetNumberLineBlank(ClientRectangle.Height, bottom, top, 96));

            // 描画開始位置はグリッド幅で整列します
            float lefts = left - ( left % blankH );
            float bottoms = bottom - ( bottom % blankV );
            float fontHeight = _font.Height * _viewScale.Height;

            renderer.Color = Color.DarkGray;
            renderer.FillRectangle(left, top, 0, right - left, bottom - top);

            // 対象アニメーション区間タグの描画
            if (_targetAnmAttributeSet != null && _targetAnmAttributeSet.TargetAnimFrameSection != null)
            {
                renderer.Color = Color.FromArgb(96, Color.LightGray);

                renderer.FillRectangle(_targetAnmAttributeSet.TargetAnimFrameSection.StartFrame, top, 0, _targetAnmAttributeSet.TargetAnimFrameSection.EndFrame - _targetAnmAttributeSet.TargetAnimFrameSection.StartFrame, bottom - top);

                DrawString_(renderer, Color.DimGray, string.Format("TAG:[{0}]",_targetAnmAttributeSet.TargetAnimFrameSection.Name), _targetAnmAttributeSet.TargetAnimFrameSection.StartFrame, top);

                DrawString_(renderer, Color.DimGray, _targetAnmAttributeSet.TargetAnimFrameSection.StartFrame.ToString(), _targetAnmAttributeSet.TargetAnimFrameSection.StartFrame, bottom + (fontHeight * 2));
                DrawString_(renderer, Color.DimGray, _targetAnmAttributeSet.TargetAnimFrameSection.EndFrame.ToString(), _targetAnmAttributeSet.TargetAnimFrameSection.EndFrame, bottom + (fontHeight * 2));
            }

            // 線の太さを設定
            renderer.LineWidth = Get1DotLineWidth_();

            // -------------- グリッドの描画
            renderer.Color = _gridColor;
            for( float i = lefts ; i < right ; i += blankH )
            {
                renderer.DrawLine( new PointF( i, top ), new PointF( i, bottom ) );
            }

            for( float i = bottoms ; i < top ; i += blankV )
            {
                renderer.DrawLine( new PointF( left, i ), new PointF( right, i ) );
            }
            // ゼロ軸
            renderer.Color = Color.Black;
            renderer.DrawLine( new PointF( left, 0 ), new PointF( right, 0 ) );
            renderer.DrawLine( new PointF( 0, top ), new PointF( 0, bottom ) );


            // -------------- 文字の描画
            // Y軸数値
            for( float i = lefts ; i < right ; i += blankH )
            {
                float ii = (float)Math.Round(i, 3, MidpointRounding.AwayFromZero);
                DrawString_(renderer, Color.Black, ii.ToString("g"), i, bottom + fontHeight);
            }

            // X軸数値
            for( float i = bottoms ; i < top ; i += blankV )
            {
                float ii = (float)Math.Round(i, 3, MidpointRounding.AwayFromZero);
                DrawString_(renderer, Color.Black, ii.ToString("g"), left, i);
            }
        }

        /// <summary>
        /// グリッドなどの背景の描画
        /// </summary>
        void DrawTime_(IRenderer renderer, int time, Color color)
        {
            float top = _viewPosInGraphSpace.Y + _viewSizeInGraphSpace.Height / 2.0f;
            float bottom = _viewPosInGraphSpace.Y - _viewSizeInGraphSpace.Height / 2.0f;
            float fontHeight = _fontBold.Height * _viewScale.Height;

            renderer.Color = color;
            renderer.DrawLine(new PointF(time, top), new PointF(time, bottom));

            DrawString_(renderer, _fontBold, color, time.ToString(), time, bottom + (fontHeight * 2));
        }


        /// <summary>
        /// Tangent編集用コントローラを描画します。
        /// </summary>
        void DrawKeyTangentContorollers_( IRenderer renderer )
        {
            if( _SelectedAnchorSet.Length > 0 )
            {
                PointF vAnchor;
                float tangent;
                PointF vAnchorAnother;
                float tangentAnother;

                foreach( KeyAanchor keyAnchor in TargetAnmAttributeSet._SelectedAnchorSet )
                {
                    PointF keyPosScrn = GetKeyAsPointF_( keyAnchor.IAnmKeyFrame );

                    if( keyAnchor.bLeft )
                    {
                        vAnchor = new PointF( -32.0f, 0 );
                        tangent = keyAnchor.IAnmKeyFrame.InTangent;
                        vAnchorAnother = new PointF( 32.0f, 0 );
                        tangentAnother = keyAnchor.IAnmKeyFrame.OutTangent;
                    }
                    else
                    {
                        vAnchor = new PointF( 32.0f, 0 );
                        tangent = keyAnchor.IAnmKeyFrame.OutTangent;
                        vAnchorAnother = new PointF( -32.0f, 0 );
                        tangentAnother = keyAnchor.IAnmKeyFrame.InTangent;
                    }

                    renderer.Color = Color.YellowGreen;
                    DrawTangent_( renderer, keyPosScrn, vAnchor, tangent );

                    renderer.Color = Color.Green;
                    DrawTangent_( renderer, keyPosScrn, vAnchorAnother, tangentAnother );
                }
            }
        }

        /// <summary>
        /// 選択キー文字列情報の描画
        /// </summary>
        void DrawSelectedKeyInfoSrting_( IRenderer renderer )
        {
            if( SelectedKeyFrameSet_.Length != 0 )
            {
                IAnmKeyFrame key = SelectedKeyFrameSet_[0];

                string str = string.Format( "T = {0}\nV = {1}",
                    key.Time.ToString().PadLeft( 7 ),
                    key.ValueAsFloat.ToString( "f2" ).PadLeft( 7 ) );

                DrawString_( _renderer, _keyInfoFont, Color.Green, str, (float)key.Time, key.ValueAsFloat );
            }
        }

        /// <summary>
        /// 前後区間キー文字列情報の描画
        /// </summary>
        void DrawBeforeAfterKeyInfoSrting_(IRenderer renderer, float px, float value)
        {
            string str = string.Format("{0}", value.ToString("f2"));
            DrawString_(_renderer, _keyInfoFont, Color.FromArgb(127, Color.DeepPink), str, px, value);
        }

        /// <summary>
        /// グリッドなどの背景の描画
        /// </summary>
        void DrawDisableMask_(IRenderer renderer)
        {
            float left = _ViewLeftInGraphSpace;
            float right = _ViewRightInGraphSpace;
            float top = _viewPosInGraphSpace.Y + _viewSizeInGraphSpace.Height / 2.0f;
            float bottom = _viewPosInGraphSpace.Y - _viewSizeInGraphSpace.Height / 2.0f;

            // 無効エリアの描画
            renderer.Color = Color.FromArgb(200, Color.LightGray);
            renderer.FillRectangle(left, top, 0, right - left, bottom - top);
        }

        #endregion // ------------------- 描画関連 -------------------

        /// <summary>
        /// 再描画
        /// </summary>
        protected override void OnPaint( PaintEventArgs pe )
        {
            if (_renderer == null)
            {
                return;
            }

            Graphics g = pe.Graphics;

            _renderer.BeginRendering( g );

            // ビュー行列を設定する
            double halfFovy = (double)(_renderer.FOV) * 0.5;
            float cameraDist = _renderer.PerseNear - (this.Height * 0.5f / (float)Math.Tan(halfFovy));

            _renderer.SetViewTransform(
                1.0f / _viewScale.Width,
                1.0f / _viewScale.Height,
                _viewPosInGraphSpace.X,
                _viewPosInGraphSpace.Y,
                cameraDist,
                0.0f);

            // グリッドなどの背景の描画
            DrawBackGround_( _renderer );

            // キーフレーム、カーブの描画
            DrawAnmCurves_( _renderer );

            // 接線コントローラの描画
            DrawKeyTangentContorollers_( _renderer );


            // 選択対象であるすべてのカーブについて...
            if (_targetAnmAttributeSet != null)
            {
                foreach (var drwaAnmCurve in _targetAnmAttributeSet._TargetDrawCurveSet)
                {
                    // カスタム時間ソースの時間を描画
                    if (drwaAnmCurve.AnmAttr.ICurrentAnimationCurve.NumKeyFrame > 1 && drwaAnmCurve.AnmAttr.ICurrentAnimationCurve.HasCustomTimeSource)
                    {
                        int customTimeFrame = drwaAnmCurve.AnmAttr.ICurrentAnimationCurve.CustomTimeFrame;

                        PointF pos = new PointF(customTimeFrame, drwaAnmCurve.AnmAttr.ICurrentAnimationCurve.Evaluate(customTimeFrame));
                        _renderer.Color = Color.DarkBlue;
                        _renderer.LineWidth = _KeyDotSize;
                        _renderer.DrawPoint(pos);

                        DrawTime_(_renderer, customTimeFrame, Color.DarkBlue);
                    }
                }
            }

            // -------------- カレント時間の描画
            DrawTime_(_renderer, GlobalTime.Inst.Time, Color.Crimson);

            // 選択矩形の描画
            DrawableOption option = DrawableOption.Empty;
            option.InitializeSystemColors(false, false);

            _selectRect.Draw(_renderer, option);

            // 選択キー文字列情報の描画
            DrawSelectedKeyInfoSrting_( _renderer );

            // マウスモード固有描画
            _mouseState.Draw( _renderer );

            // 無効時のマスク描画
            if (!this.Enabled)
            {
                DrawDisableMask_(_renderer);
            }

            // カレント行列を戻す
            _renderer.CurrentMtx = new Matrix();



            _renderer.EndRendering();

            // OnPaint で基本クラスを呼び出し中
            base.OnPaint( pe );
        }

        /// <summary>
        /// サイズ変更
        /// </summary>
        private void Event_GraphView_SizeChanged( object sender, System.EventArgs e )
        {

            _viewSizeInGraphSpace.Width = ClientRectangle.Width * _viewScale.Width;
            _viewSizeInGraphSpace.Height = ClientRectangle.Height * _viewScale.Height;

            Invalidate();
        }

        #region ------------- マウスイベント、キーイベント関連 -------------
        private void Event_GraphView_MouseDown( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            _mouseState.OnMouseDown( e );
        }

        private void Event_GraphView_MouseMove( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            _mouseState.OnMouseMove( e );
        }

        private void Event_GraphView_MouseUp( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            _mouseState.OnMouseUp( e );
        }

        private void Event_GraphView_MouseWheel(object sender, MouseEventArgs e)
        {

            Point clientPos = this.PointToClient(new Point(MousePosition.X, MousePosition.Y));
            if (Rectangle.FromLTRB(0, 0, this.Width, this.Height).Contains(clientPos))
            {
                _mouseState.OnWheelMove(e);
            }
        }

        private void GraphView_KeyDown( object sender, System.Windows.Forms.KeyEventArgs e )
        {
            _mouseState.OnKeyDown( e );
        }

        private void GraphView_KeyUp( object sender, System.Windows.Forms.KeyEventArgs e )
        {
            _mouseState.OnKeyUp( e );
        }
        #endregion // ------------- マウスイベント、キーイベント -------------




        #endregion // ------------- イベントハンドラ -------------
    }
}
