﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.IO;
using System.Drawing;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Forms.Dialogs
{
    using LayoutEditor.Controls;
    using LECore;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Core;
    using Structures.SerializableObject;

    public partial class TextureFileSetImporterDlg : LEBaseDlg
    {
        //------------------------------------------------
        #region 型宣言
        /// <summary>
        /// テクスチャグループ
        /// </summary>
        class TextureGroup
        {
            readonly List<string> _texturePathSet = new List<string>();
            readonly string _name;

            public List<string> TexturePathSet
            {
                get { return _texturePathSet; }
            }

            public string Name { get { return _name; } }

            public TextureGroup( string name )
            {
                _name = name;
            }
        }

        #endregion

        //------------------------------------------------
        #region フィールド
        System.Windows.Forms.FolderBrowserDialog _folderBrowserDialog;

        readonly AppSetting _appSetting = null;

        #endregion フィールド

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureFileSetImporterDlg(AppSetting appSetting)
        {
            InitializeComponent();

            _appSetting = appSetting;

            _folderBrowserDialog = new FolderBrowserDialog();
        }

        /// <summary>
        /// フォルダ読みこみ
        /// </summary>
        private void Event_BtnFolder_Click( object sender, EventArgs e )
        {
            if( Directory.Exists( _tbxFolder.Text ) )
            {
                _folderBrowserDialog.SelectedPath = _tbxFolder.Text;
            }

            if( _folderBrowserDialog.ShowDialog() == DialogResult.OK )
            {
                _tbxFolder.Text = _folderBrowserDialog.SelectedPath;
            }
        }

        /// <summary>
        /// フォルダ名、テキスト変更
        /// </summary>
        private void Event_TbxFolder_TextChanged( object sender, EventArgs e )
        {
            // 有効化判定する
            _btnImport.Enabled = Directory.Exists( _tbxFolder.Text );
        }

        /// <summary>
        /// 繰り返し数変更
        /// </summary>
        private void Event_NudRepeatNum_ValueChanged( object sender, EventArgs e )
        {
            UINumericUpDown nud = sender as UINumericUpDown;
            _pnlRepeatType.Enabled = ( nud.Value > 0 );
        }



        /// <summary>
        /// 入力ボタンクリック
        /// </summary>
        private void Event_BtnImport_Click( object sender, EventArgs e )
        {
            string inputPath = _tbxFolder.Text;
            if( !Directory.Exists( inputPath ) )
            {
                return;
            }

            //---------------------------------------------------------
            // フォルダ内のテクスチャを列挙
            string[] tgaFilePathSet = Directory.GetFiles( inputPath, "*.tga" ).OrderBy(x => x).ToArray();
            List<TextureGroup> textureGroupSet = new List<TextureGroup>();
            // 接頭子を抽出
            foreach( string filePath in tgaFilePathSet )
            {
                string fileName = Path.GetFileName( filePath );

                // 正規表現で確認
                Regex regexFileName = new Regex( @"(\.[0-9]+\.tga)$" );
                if( regexFileName.IsMatch( fileName ) )
                {

                    // 頭の部分だけ取り出す
                    // string prefix = match.Groups["name"].Value;
                    string prefix = regexFileName.Replace( fileName, "" );

                    if( prefix == null || prefix == string.Empty )
                    {
                        continue;
                    }

                    TextureGroup textureGroup =
                        textureGroupSet.Find( delegate( TextureGroup group )
                        {
                            return group.Name == prefix;
                        } );

                    // 接頭子によってグループ分けして、パスを格納
                    if( textureGroup == null )
                    {
                        textureGroup = new TextureGroup( prefix );
                        textureGroupSet.Add( textureGroup );
                    }

                    if( textureGroup.TexturePathSet.IndexOf( filePath ) == -1 )
                    {
                        textureGroup.TexturePathSet.Add( filePath );
                    }
                }
            }

            //---------------------------------------------------------
            // 不正なデータを削除します。
            TextureGroup[] textureGroupSetTemp = textureGroupSet.ToArray();
            foreach( TextureGroup textureGroup in textureGroupSetTemp )
            {
                // テクスチャをもつ。
                if( textureGroup.TexturePathSet.Count <= 0 )
                {
                    textureGroupSet.Remove( textureGroup );
                    break;
                }

                // すべてのテクスチャが有効か確認する
                foreach( string filePath in textureGroup.TexturePathSet )
                {
                    if( !File.Exists( filePath ) )
                    {
                        textureGroupSet.Remove( textureGroup );
                        break;
                    }
                }
            }

            //---------------------------------------------------------
            // シーンに読み込みます。
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            subScene.BeginMassiveModify();
            // グループごとに
            foreach( TextureGroup textureGroup in textureGroupSet )
            {
                // すべてのテクスチャを読み込む
                TextureMgrManipulator
                    textureMgrManipulator = new TextureMgrManipulator();

                textureMgrManipulator.BindTarget( subScene.ITextureMgr );
                textureMgrManipulator.RegisterITextureImageSetFromFile( textureGroup.TexturePathSet.ToArray() );


                // ペインを作成する
                SubSceneManipulator subSceneManipulator = new SubSceneManipulator();
                subSceneManipulator.BindTarget( subScene );

                var picturePane = subSceneManipulator.AddPicturePane
                    (_appSetting.ProjectSettings.GetNewPicturePaneName(),
                    Path.GetFileNameWithoutExtension(textureGroup.TexturePathSet[0]),
                    _appSetting.ProjectSettings.DefaultPaneNames.UseTextureNameAsPicturePaneName);

                // パターンリストを作る
                string[]		textureNameSet = new string[textureGroup.TexturePathSet.Count];

                for( int i = 0; i < textureGroup.TexturePathSet.Count; i++ )
                {
                    textureNameSet[i] = Path.GetFileNameWithoutExtension(
                        textureGroup.TexturePathSet[i] );
                }

                IMaterial material = picturePane.IMaterial[0];
                IMaterialTexMap materialTexMap = material.IMaterialTexMapSet[0];

                TextureChangerManipulator textureChangerManipulator = new TextureChangerManipulator();
                textureChangerManipulator.BindTarget( materialTexMap.ITextureChanger );
                textureChangerManipulator.SetAll( textureNameSet );


                // パターンアニメーションキーフレームを作成する。
                IAnmCurve curve =
                    materialTexMap.ITextureChanger.TextureIndexIAnmAttr.ICurrentAnimationCurve;
                AnmCurveManipulator anmCurveManipulator = new AnmCurveManipulator();
                anmCurveManipulator.BindTarget( curve );

                int timeBase = (int)_nudPatternStartFrame.Value;
                int timeStep = (int)_nudPatternIntervalFrame.Value;
                int numRepeat = (int)_nudRepeatNum.Value;
                bool bRepeatMirror = _radRepeatMirror.Checked;

                // 指定された繰り返し回数だけキーフレームを追加していく...
                for( int repeatCnt = 0 ; repeatCnt < numRepeat; repeatCnt++ )
                {
                    for( int i = 0 ; i < textureNameSet.Length ; i++ )
                    {
                        int textureIndex = i;
                        if( bRepeatMirror )
                        {
                            textureIndex = ((repeatCnt % 2) == 0 ) ?
                                textureIndex : ( textureNameSet.Length - 1 ) - textureIndex;
                        }

                        anmCurveManipulator.AddNewKeyFrame(
                            timeBase + i * timeStep,
                            (float)textureIndex,
                            true,
                            0.0f, 0.0f,
                            LECore.Structures.Core.InterporationType.Step );
                    }

                    timeBase += textureNameSet.Length * timeStep;
                }

            }
            subScene.EndMassiveModify();

            //---------------------------------------------------------
            // 入力作業の結果をメッセージボックスで報告します
            string messageString;
            if( textureGroupSet.Count > 0 )
            {
                StringBuilder strb = new StringBuilder( StringResMgr.Get( "TEXTURESET_IMPORT_MSG_LOADFILES" ) );
                strb.AppendLine( "" );
                strb.AppendLine( "" );
                foreach( TextureGroup textureGroup in textureGroupSet )
                {
                    string textureStr = string.Format(
                        " name = {0} \t pattern={1}",
                        textureGroup.Name, textureGroup.TexturePathSet.Count );

                    strb.AppendLine( textureStr );
                }
                messageString = strb.ToString();
            }
            else
            {
                messageString = StringResMgr.Get( "TEXTURESET_IMPORT_MSG_LOADNOFILES" );
            }

            MessageBox.Show( this, messageString, StringResMgr.Get( "TEXTURESET_IMPORT_MSG_TITLE" ) );

            DialogResult = DialogResult.OK;
        }
    }
}
