﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Forms.Dialogs;
using LECore.Structures;
using LECore;
using LayoutEditor.Utility;
using LayoutEditor.src.Forms.Dialogs;
using System.IO;

namespace LayoutEditor.Forms.Dialogs
{
    /// <summary>
    /// 部品設定ダイアログ
    /// </summary>
    public partial class SetPartsKindDialog : LEBaseDlg
    {
        private const int NormalLayoutIndex = 0;
        Func<string, string> _getUniqueDescriptionNameFunk = null;

        private string _initialControlName;
        private string _initialDescriptionName;
        private bool _IsControlMode = false;

        /// <summary>
        /// コンボボックス用アイテムです。
        /// </summary>
        private class PartsControlSettingItem
        {
            private readonly IPartsControlSetting _controlSetting;
            public string CustomLable {get;set;}

            private readonly Controls.UIComboBox _Owner;

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public PartsControlSettingItem(IPartsControlSetting controlSetting, Controls.UIComboBox owner)
            {
                _controlSetting = controlSetting;
                _Owner = owner;
            }

            static private string AdjustWidth_(string str, float maxWidth)
            {
                SizeF size;

                do
                {
                    str += " ";
                    size = TextRenderer.MeasureText(str, SetPartsKindDialog.DefaultFont);
                } while (size.Width < maxWidth);

                return str;
            }

            /// <summary>
            /// 文字列に変換します。
            /// </summary>
            public override string ToString()
            {
                if (!string.IsNullOrEmpty(this.CustomLable))
                {
                    return this.CustomLable;
                }

                return string.Format("{0}{1}{2}",
                    AdjustWidth_("[" + Path.GetFileNameWithoutExtension(this._controlSetting.FilePath) + "]", _Owner.DropDownWidth / 3),
                    AdjustWidth_(this._controlSetting.UIName, _Owner.DropDownWidth / 3),
                    this._controlSetting.Name);
            }

            /// <summary>
            /// 部品コントロール定義です。
            /// </summary>
            public IPartsControlSetting PartsControlSetting
            {
                get { return this._controlSetting; }
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SetPartsKindDialog()
        {
            InitializeComponent();
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// ユーザによって選択されたコントロール設定ひな形です。
        /// </summary>
        public IPartsControlSetting PartsControlSetting
        {
            get
            {
                if (_cmbKind.SelectedIndex == NormalLayoutIndex)
                {
                    return null;
                }

                var item = _cmbKind.SelectedItem as PartsControlSettingItem;
                return item != null ? item.PartsControlSetting : null;
            }
        }

        /// <summary>
        /// ひとつ前に選択されていたコントロール設定設定ひな形です。
        /// </summary>
        IPartsControlSetting _LastSelectedPartsControlSetting { get; set; }

        /// <summary>
        /// 説明名称です。
        /// </summary>
        public string PartsDescriptionName
        {
            get { return _tbxPartsDescriptionName.Text; }
        }

        /// <summary>
        /// ダイアログ初期状態から変更されているか？
        /// </summary>
        public bool IsChanged
        {
            get { return
                _initialDescriptionName != _tbxPartsDescriptionName.Text ||
                _initialControlName != this._PartsControlName;}
        }

        /// <summary>
        /// コントロール名
        /// </summary>
        private string _PartsControlName
        {
            get { return this.PartsControlSetting != null ? this.PartsControlSetting.Name : string.Empty; }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 最初のコントロールのラベル文字列を初期化します。
        /// </summary>
        public void InitFirstControlItemLabel(string firstItemLabel)
        {
            var firstItem = this._cmbKind.Items[0] as PartsControlSettingItem;
            firstItem.CustomLable = firstItemLabel;

            this._cmbKind.Items[0] = firstItem;
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        public void Initialize(string controlName, string descriptionName)
        {
            Initialize_(controlName, descriptionName, null);
        }

              /// <summary>
        /// 初期化します。
        /// </summary>
        public void InitializeAsControlSelectMode(string controlName, string descriptionName, Func<string, string> getUniqueDescriptionNameFunk)
        {
            _IsControlMode = true;
            Initialize_(controlName, descriptionName, getUniqueDescriptionNameFunk);
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        void Initialize_(string controlName, string descriptionName, Func<string, string> getUniqueDescriptionNameFunk)
        {
            _initialDescriptionName = descriptionName;
            _initialControlName = controlName;

            // _tbxPartsDescriptionName
            if (_IsControlMode)
            {
                _getUniqueDescriptionNameFunk = getUniqueDescriptionNameFunk;

                _tbxPartsDescriptionName.InitializeTextValidator((text) => PaneHelper.CheckPaneNameValid(text));
                _tbxPartsDescriptionName.GuideText = StringResMgr.Get("PARTS_CTRLNAME_GUIDE");
            }

            // 現在の状態
            {
                var controlSetting = LayoutEditorCore.Scene.PartsControlSettings.FirstOrDefault(
                    (partsTemplate) => partsTemplate.Name == controlName);

                _tbxPartsDescriptionName.Text = descriptionName;
            }

            // 部品コンボボックス
            {
                foreach (var partsTemplate in LayoutEditorCore.Scene.PartsControlSettings)
                {
                    this._cmbKind.Items.Add(new PartsControlSettingItem(partsTemplate, this._cmbKind));
                }
                this._cmbKind.ExpandDropDownWidth();

                this._cmbKind.SelectByCondition(
                    (item) => (item as PartsControlSettingItem).PartsControlSetting.Name == controlName);

                if (this._cmbKind.SelectedIndex < 0)
                {
                    this._cmbKind.SelectedIndex = 0;
                }

                _LastSelectedPartsControlSetting = PartsControlSetting;
            }

            // UI の更新。
            this.UpdateUIState();
        }

        //----------------------------------------------------------

        /// <summary>
        /// UIを更新します。
        /// </summary>
        private void UpdateUIState()
        {
            var item = _cmbKind.SelectedItem as PartsControlSettingItem;
            if (item != null)
            {
                // TODO:説明文章を更新します。
                _lblTempleteDescriptionContent.Text = item.PartsControlSetting.Description;

                String labelText = string.Format("{0} : {1}", LECoreStringResMgr.Get("LECORE_DEFINITION_FILE"), string.IsNullOrEmpty(item.PartsControlSetting.FilePath) ? "" : Path.GetFileName(item.PartsControlSetting.FilePath));
                _lblFilePath.Text = labelText;
            }

            // OK ボタンの状態を更新します。
            _btnOK.Enabled = _IsControlMode ? IsControlModeSettingValid_() : IsSettingValid_();
        }

        /// <summary>
        /// 設定が有効か判定します
        /// </summary>
        bool IsSettingValid_()
        {
            // コントロール未設定ならいかなるときも有効
            if (this._cmbKind.SelectedIndex == NormalLayoutIndex)
            {
                return true;
            }

            // 空でなければ有効
            return !string.IsNullOrEmpty(this._tbxPartsDescriptionName.Text);
        }

        /// <summary>
        /// 設定が有効か判定します
        /// </summary>
        bool IsControlModeSettingValid_()
        {
            // 新規追加であれば、何かコントロールが選択されていること
            if (string.IsNullOrEmpty(_initialControlName))
            {
                if (this.PartsControlSetting == null)
                {
                    return false;
                }
            }

            // コントロール名前が設定されていない。
            if (string.IsNullOrEmpty(this._tbxPartsDescriptionName.Text))
            {
                return false;
            }

            // 初期状態から変更されていないなら有効
            if(_initialDescriptionName == _tbxPartsDescriptionName.Text)
            {
                return true;
            }

            // ユニークな名前であるなら有効
            return _tbxPartsDescriptionName.Text == _getUniqueDescriptionNameFunk(_tbxPartsDescriptionName.Text);
        }

        //----------------------------------------------------------
        // イベントハンドラです。
        //----------------------------------------------------------

        /// <summary>
        /// 部品名変更。
        /// </summary>
        private void Event_tbxPartsDescriptionName_ValueChanged(object sender, EventArgs e)
        {
            bool error = false;
            if (_IsControlMode)
            {
                if (_initialDescriptionName != _tbxPartsDescriptionName.Text)
                {
                    string newText = _getUniqueDescriptionNameFunk(_tbxPartsDescriptionName.Text);
                    if (_tbxPartsDescriptionName.Text != newText)
                    {
                        // 重複しない名前に改名されました。と通知する。
                        error = true;
                        _errMain.SetError(_tbxPartsDescriptionName, StringResMgr.Get("RENAME_MSG", _tbxPartsDescriptionName.Text, newText));
                        _tbxPartsDescriptionName.Text = newText;
                    }
                }
            }

            if (!error)
            {
                _errMain.SetError(_tbxPartsDescriptionName, string.Empty);
            }

            this.UpdateUIState();
        }

        /// <summary>
        /// 部品種類コンボボックス変更。
        /// </summary>
        private void Event_CmbKind_SelectedIndexChanged(object sender, EventArgs e)
        {
            // コントロール選択に伴って、名前を調整する
            if (_IsControlMode)
            {
                if(PartsControlSetting != null)
                {
                    if (string.IsNullOrEmpty(_tbxPartsDescriptionName.Text))
                    {
                        // 初期値はコントロール名を設定する。
                        _tbxPartsDescriptionName.Text = PartsControlSetting.Name;
                    }
                    else if (_LastSelectedPartsControlSetting != null)
                    {
                        // 前に選択されていたコントロール名の部分を新しいコントロール名で置換する。
                        _tbxPartsDescriptionName.Text = _tbxPartsDescriptionName.Text.Replace(_LastSelectedPartsControlSetting.Name, PartsControlSetting.Name);
                    }
                }
            }

            _LastSelectedPartsControlSetting = PartsControlSetting;

            this.UpdateUIState();
        }

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// テキスト更新
        /// </summary>
        private void Event_tbxPartsDescriptionName_TextChanged(object sender, EventArgs e)
        {
            this.UpdateUIState();
        }
    }
}
