﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace LayoutEditor
{
    /// <summary>
    /// フォルダ参照ユーティリティ(エディットボックスが表示されるバージョン)
    /// </summary>
    public class FolderBrowserUility
    {
        /// <summary>
        /// 定数
        /// </summary>
        public const int WM_DESTROY = 0x0002;
        public const int WM_GETMINMAXINFO = 0x0024;
        public const int WM_USER = 0x0400;
        public const int BFFM_INITIALIZED = 1;
        public const int BFFM_SELCHANGED = 2;
        public const int BFFM_VALIDATEFAILEDA = 3;
        public const int BFFM_VALIDATEFAILEDW = 4;
        public const int BFFM_IUNKNOWN = 5;
        public const int BFFM_SETSTATUSTEXTA = WM_USER + 100;
        public const int BFFM_ENABLEOK = WM_USER + 101;
        public const int BFFM_SETSELECTIONA = WM_USER + 102;
        public const int BFFM_SETSELECTIONW = WM_USER + 103;
        public const int BFFM_SETSTATUSTEXTW = WM_USER + 104;
        public const int BFFM_SETOKTEXT = WM_USER + 105;
        public const int BFFM_SETEXPANDED = WM_USER + 106;

        public const int GWL_WNDPROC = -4;

        //private uint BIF_RETURNONLYFSDIRS = 0x0001;
        //private uint BIF_DONTGOBELOWDOMAIN = 0x0002;
        //private uint BIF_STATUSTEXT = 0x0004;

        //private uint BIF_RETURNFSANCESTORS = 0x0008;
        private uint BIF_EDITBOX = 0x0010;
        private uint BIF_VALIDATE = 0x0020;

        private uint BIF_NEWDIALOGSTYLE = 0x0040;

        //private uint BIF_USENEWUI = 0x0040 + 0x0010;

        //private uint BIF_BROWSEINCLUDEURLS = 0x0080;
        //private uint BIF_UAHINT = 0x0100;
        //private uint BIF_NONEWFOLDERBUTTON = 0x0200;
        //private uint BIF_NOTRANSLATETARGETS = 0x0400;

        //private uint BIF_BROWSEFORCOMPUTER = 0x1000;
        //private uint BIF_BROWSEFORPRINTER = 0x2000;
        //private uint BIF_BROWSEINCLUDEFILES = 0x4000;
        private uint BIF_SHAREABLE = 0x8000;

        /// <summary>
        /// PInvoke
        /// </summary>
        [DllImport("shell32.dll")]
        static extern IntPtr SHBrowseForFolder(ref BROWSEINFO lpbi);

        [DllImport("shell32.dll", CharSet = CharSet.Unicode)]
        static extern bool SHGetPathFromIDList(IntPtr pidl, IntPtr pszPath);

        [DllImport("user32.dll", PreserveSig = true)]
        public static extern IntPtr SendMessage(HandleRef hWnd, uint Msg, int wParam, IntPtr lParam);

        [DllImport("user32.dll", CharSet = CharSet.Auto)]
        public static extern IntPtr SendMessage(HandleRef hWnd, int msg, int wParam, string lParam);

        [DllImport("user32.dll", SetLastError = true)]
        public static extern IntPtr FindWindowEx(HandleRef hwndParent, IntPtr hwndChildAfter, string lpszClass, string lpszWindow);

        [DllImport("user32.dll", SetLastError = true)]
        public static extern Boolean SetWindowText(IntPtr hWnd, String text);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr SetWindowLongPtr(IntPtr hWnd, int nIndex, IntPtr dwNewLong);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr SetWindowLongPtr(IntPtr hWnd, int nIndex, WndProc dwNewLong);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr CallWindowProc(IntPtr lpPrevWndFunc, IntPtr hwnd, uint msg, IntPtr wParam, IntPtr lParam);

        private delegate int BrowseCallBackProc(IntPtr hwnd, int msg, IntPtr lp, IntPtr wp);

        private delegate IntPtr WndProc(IntPtr hWnd, uint msg, IntPtr wParam, IntPtr lParam);

        /// <summary>
        /// Win32API の BROWSEINFO 構造体
        /// </summary>
        struct BROWSEINFO
        {
            public IntPtr hwndOwner;
            public IntPtr pidlRoot;
            public string pszDisplayName;
            public string lpszTitle;
            public uint ulFlags;
            public BrowseCallBackProc lpfn;
            public IntPtr lParam;
            public int iImage;
        }

        /// <summary>
        /// Win32API の POINT 構造体
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        struct POINT
        {
            public int x;
            public int y;
        }

        /// <summary>
        /// Win32API の MINMAXINFO 構造体
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        struct MINMAXINFO
        {
            public POINT ptReserved;
            public POINT ptMaxSize;
            public POINT ptMaxPosition;
            public POINT ptMinTrackSize;
            public POINT ptMaxTrackSize;
        }

        /// <summary>
        /// 初期選択ディレクトリ
        /// </summary>
        private string _initialPath;

        /// <summary>
        /// フック処理を行うウインドウプロシージャ
        /// ガベージコレクトされないようにメンバ変数として保持します
        /// </summary>
        private WndProc _hookWndProc;

        /// <summary>
        /// デフォルトのウインドウプロシージャ
        /// </summary>
        private IntPtr _oldWndProc;

        /// <summary>
        /// 定型イベント処理
        /// </summary>
        public int OnBrowseEvent(IntPtr hWnd, int msg, IntPtr lp, IntPtr lpData)
        {
            switch (msg)
            {
                // 初期化
                case BFFM_INITIALIZED:
                    {
                        // フック用のウインドウプロシージャを設定
                        _hookWndProc = new WndProc(HookWndProc);
                        _oldWndProc = SetWindowLongPtr(hWnd, GWL_WNDPROC, _hookWndProc);

                        // initialPathを設定
                        if (this._initialPath.Length != 0)
                        {
                            IntPtr hwndEdit = FindWindowEx(new HandleRef(null, hWnd), IntPtr.Zero, "Edit", null);
                            SetWindowText(hwndEdit, this._initialPath);

                            SendMessage(new HandleRef(null, hWnd), BFFM_SETSELECTIONW, 1, this._initialPath);
                        }

                        break;
                    }
                // 選択変更
                case BFFM_SELCHANGED:
                    {

                        IntPtr pathPtr = IntPtr.Zero;
                        try
                        {
                            pathPtr = Marshal.AllocHGlobal((int)(260 * Marshal.SystemDefaultCharSize));
                            if (SHGetPathFromIDList(lp, pathPtr))
                            {
                                SendMessage(new HandleRef(null, hWnd), BFFM_SETSTATUSTEXTW, 0, pathPtr);

                                IntPtr hwndEdit = FindWindowEx(new HandleRef(null, hWnd), IntPtr.Zero, "Edit", null);
                                SetWindowText(hwndEdit, Marshal.PtrToStringAuto(pathPtr));
                            }
                        }
                        finally
                        {
                            Marshal.FreeHGlobal(pathPtr);
                        }

                        break;
                    }
                // エディットボックスに不正なフォルダ名を入力した
                case BFFM_VALIDATEFAILEDA:
                case BFFM_VALIDATEFAILEDW:
                    {
                        var windowHandle = Control.FromHandle(hWnd);
                        string path = Marshal.PtrToStringAnsi(lp);

                        string text = StringResMgr.Get("FOLDERBROWSER_VALIDATEFAILED_TEXT", path);
                        string caption = StringResMgr.Get("FOLDERBROWSER_VALIDATEFAILED_CAPTION");
                        MessageBox.Show(windowHandle, text, caption, MessageBoxButtons.OK, MessageBoxIcon.Error);

                        return 1;
                    }
            }

            return 0;
        }

        /// <summary>
        /// フォルダ選択ダイアログのウインドウプロシージャのフック処理
        /// </summary>
        public IntPtr HookWndProc(IntPtr hWnd, uint msg, IntPtr wParam, IntPtr lParam)
        {
            switch (msg)
            {
                // ウインドウを破棄する直前にウインドウプロシージャを元に戻す
                case WM_DESTROY:
                    {
                        SetWindowLongPtr(hWnd, GWL_WNDPROC, _oldWndProc);

                        return CallWindowProc(_oldWndProc, hWnd, msg, wParam, lParam);
                    }
                // フォルダの新規作成ボタンが重ならないように最小ウインドウサイズを調整する
                case WM_GETMINMAXINFO:
                    {
                        const int minWidth = 427;
                        IntPtr result = CallWindowProc(_oldWndProc, hWnd, msg, wParam, lParam);
                        MINMAXINFO minmaxinfo = (MINMAXINFO)Marshal.PtrToStructure(lParam, typeof(MINMAXINFO));
                        minmaxinfo.ptMinTrackSize.x = Math.Max(minmaxinfo.ptMinTrackSize.x, minWidth);
                        Marshal.StructureToPtr(minmaxinfo, lParam, false);

                        return result;
                    }
            }

            return CallWindowProc(_oldWndProc, hWnd, msg, wParam, lParam);
        }

        /// <summary>
        /// フォルダを選択します。
        /// </summary>
        public string SelectFolder(string caption, string initialPath, IntPtr parentHandle)
        {
            if (Directory.Exists(initialPath))
            {
                DirectoryInfo di = new DirectoryInfo(initialPath);
                _initialPath = di.FullName;
            }
            else
            {
                _initialPath = initialPath;
            }

            BROWSEINFO bi = new BROWSEINFO();
            bi.hwndOwner = parentHandle;
            bi.pidlRoot = IntPtr.Zero;
            bi.lpszTitle = caption;
            bi.ulFlags = BIF_NEWDIALOGSTYLE | BIF_SHAREABLE | BIF_EDITBOX | BIF_VALIDATE;
            bi.lpfn = new BrowseCallBackProc(OnBrowseEvent);
            bi.lParam = IntPtr.Zero;
            bi.iImage = 0;

            IntPtr bufferAddress = Marshal.AllocHGlobal(256); ;
            IntPtr pidl = IntPtr.Zero;
            try
            {
                pidl = SHBrowseForFolder(ref bi);
                if (true != SHGetPathFromIDList(pidl, bufferAddress))
                {
                    return null;
                }

                string resultPath = Marshal.PtrToStringAuto(bufferAddress);

                return resultPath;
            }
            finally
            {
                Marshal.FreeCoTaskMem(pidl);
                Marshal.FreeCoTaskMem(bufferAddress);
            }
        }
    }

}
