﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;

namespace LayoutEditor.Controls
{
    using LECore;
    using LECore.Util;
    using TextBoxState = System.Windows.Forms.VisualStyles.TextBoxState;

    /// <summary>
    /// 値テキストボックスクラス。
    /// </summary>
    public abstract class ValueTextBox : TextBox, FixEditingState
    {
        /// <summary>値変更イベント</summary>
        public event EventHandler ValueChanged = null;

        /// <summary>イベント発行制御</summary>
        protected static bool _invokeEvent = true;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ValueTextBox()
        {
            this.Enter += Event_ValueTextBox_Enter_;
        }

        /// <summary>
        /// フォーカスを取得した場合、テキストを全選択します。
        /// </summary>
        void Event_ValueTextBox_Enter_( object sender, EventArgs e )
        {
            this.SelectAll();
        }

        /// <summary>
        /// 未設定状態にする。
        /// </summary>
        public void SetEmpty()
        {
            this.Text = string.Empty;
        }

        /// <summary>
        /// テキスト更新。
        /// </summary>
        protected abstract void UpdateText();

        /// <summary>
        /// 値更新。
        /// </summary>
        protected abstract bool UpdateValue();

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnValueChanged(EventArgs e)
        {
            if (_invokeEvent)
            {
                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, e);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool IsInputKey(Keys keyData)
        {
            if ((keyData & Keys.Alt) != Keys.Alt)
            {
                Keys keys = keyData & Keys.KeyCode;
                if (keys == Keys.Return)
                {
                    return true;
                }
            }
            return base.IsInputKey(keyData);
        }

        /// <summary>
        /// 編集中の値を確定します。
        /// </summary>
        public void FixValue()
        {
            // 値更新
            bool changed = UpdateValue();

            // テキスト更新
            UpdateText();

            // イベント発行
            if (changed)
            {
                OnValueChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            FixValue();

            base.OnLostFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// OnLostFocus と異なり、値変更状態にかかわらず
        /// ValueChanged イベントを発生します。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
            case Keys.Return:
            {
                if (!this.Multiline)
                {
                    // 値更新
                    UpdateValue();

                    // テキスト更新
                    UpdateText();

                    // 全選択
                    SelectAll();

                    // イベント発行
                    // 常にValueChanged イベントを発生します。
                    OnValueChanged(EventArgs.Empty);
                }

                break;
            }
            default:
                base.OnKeyDown(e);
                break;
            }
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            this.ResumeLayout( false );

        }
    }

    /// <summary>
    /// 数値テキストボックスクラス。
    /// </summary>
    public abstract class NumberTextBox : ValueTextBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected NumberTextBox() {}

        /// <summary>
        /// 入力可能文字かどうか。
        /// </summary>
        protected bool IsEnableInputChar(char ch)
        {
            // マイナス
            if (ch == '-' && IsEnableInputMinus())
            {
                return true;
            }
            // 数字か編集キー（VK_BACK=0x08）
            else if (('0' <= ch && ch <= '9')
                || ch == 0x08 // backspace
                || ch == 0x01 // ctrl+a
                || ch == 0x03 // ctrl+c
                || ch == 0x16 // ctrl+v
                || ch == 0x18 // ctrl+x
                )
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// マイナスが入力可能かどうか。
        /// </summary>
        protected bool IsEnableInputMinus()
        {
            string text = this.Text;

            // 未入力
            if (text == string.Empty)
            {
                return true;
            }

            // キャレットが先頭
            if (this.SelectionStart == 0)
            {
                // 非選択状態
                if (this.SelectionLength == 0)
                {
                    if (text.IndexOf('-') == -1)
                    {
                        return true;
                    }
                }
                // 選択状態
                else
                {
                    if (text.IndexOf('-', this.SelectionLength) == -1)
                    {
                        return true;
                    }
                }
            }
            return false;
        }
    }

    /// <summary>
    /// 整数値テキストボックスクラス。
    /// </summary>
    [DefaultProperty("Value")]
    [ToolboxBitmap(typeof(TextBox))]
    public class IntTextBox : NumberTextBox
    {
        internal const int _defValue     = 0;
        internal const int _defMinValue  = -99999;
        internal const int _defMaxValue  = 99999;
        internal const int _defMaxLength = 6;

        private int _value    = _defValue;
        private int _minValue = _defMinValue;
        private int _maxValue = _defMaxValue;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntTextBox()
        {
            this.MaxLength = _defMaxLength;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(_defValue)]
        [Description("値。")]
        public int Value
        {
            get { return _value; }
            set
            {
                if (_value != value)
                {
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(_defMinValue)]
        [Description("最小値。")]
        public int MinValue
        {
            get { return _minValue;  }
            set { _minValue = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(_defMaxValue)]
        [Description("最大値。")]
        public int MaxValue
        {
            get { return _maxValue;  }
            set { _maxValue = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(_defMaxLength)]
        public new int MaxLength
        {
            get { return base.MaxLength;  }
            set { base.MaxLength = value; }
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// </summary>
        public void SetValue(int value)
        {
            _invokeEvent = false;
            {
                this.Value = value;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            this.Text = _value.ToString();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            int prev = _value;
            try
            {
                _value = int.Parse(this.Text);
                if      (_value < _minValue) { _value = _minValue; }
                else if (_value > _maxValue) { _value = _maxValue; }
            }
            catch (SystemException)
            {
                _value = _minValue;
            }
            return _value != prev;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            if (m.Msg == LECore.Win32.WM.WM_CHAR)
            {
                char ch = (char)m.WParam;

                // 数値制御文字
                if (IsEnableInputChar(ch))
                {
                    base.WndProc(ref m);
                }
                return;
            }
            base.WndProc(ref m);
        }
    }

    /// <summary>
    /// 実数値テキストボックスクラス。
    /// </summary>
    [DefaultProperty("Value")]
    [ToolboxBitmap(typeof(TextBox))]
    public class FloatTextBox : NumberTextBox
    {
        internal const float  _defValue     = 0;
        internal const float  _defMinValue  = -99999;
        internal const float  _defMaxValue  = 99999;
        internal const string _defFormat    = "f6";
        internal const int    _defMaxLength = 13;

        private float  _value    = _defValue;
        private float  _minValue = _defMinValue;
        private float  _maxValue = _defMaxValue;
        private string _format   = _defFormat;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatTextBox()
        {
            this.MaxLength = _defMaxLength;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(_defValue)]
        [Description("値。")]
        public float Value
        {
            get { return _value; }
            set
            {
                if (_value != value)
                {
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(_defMinValue)]
        [Description("最小値。")]
        public float MinValue
        {
            get { return _minValue;  }
            set { _minValue = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(_defMaxValue)]
        [Description("最大値。")]
        public float MaxValue
        {
            get { return _maxValue;  }
            set { _maxValue = value; }
        }

        /// <summary>
        /// 書式指定フォーマット。
        /// </summary>
        [DefaultValue(_defFormat)]
        [Description("書式指定フォーマット。")]
        public string Format
        {
            get { return _format;  }
            set
            {
                _format = value;
                UpdateText();
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(_defMaxLength)]
        public new int MaxLength
        {
            get { return base.MaxLength;  }
            set { base.MaxLength = value; }
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// </summary>
        public void SetValue(float value)
        {
            _invokeEvent = false;
            {
                this.Value = value;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            this.Text = _value.ToString(_format);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            float prev = _value;
            try
            {
                _value = float.Parse(this.Text);
                if      (_value < _minValue) { _value = _minValue; }
                else if (_value > _maxValue) { _value = _maxValue; }
            }
            catch (SystemException)
            {
                _value = _minValue;
            }
            return _value != prev;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            if (m.Msg == LECore.Win32.WM.WM_CHAR)
            {
                char ch = (char)m.WParam;

                // 数値制御文字
                if (IsEnableInputChar(ch))
                {
                    base.WndProc(ref m);
                }
                // 小数点
                else if (ch == '.' && IsEnableInputPeriod())
                {
                    base.WndProc(ref m);
                }
                return;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// 小数点が入力可能かどうか。
        /// </summary>
        private bool IsEnableInputPeriod()
        {
            // 入力済みじゃない
            return (this.Text.IndexOf('.') == -1);
        }
    }

    /// <summary>
    /// 文字列テキストボックスクラス。
    /// </summary>
    [DefaultProperty("Value")]
    [ToolboxBitmap(typeof(TextBox))]
    public class StringTextBox : ValueTextBox
    {
        private const string _defValue = "";

        private string _value = _defValue;

        /// <summary>
        /// ユーザが設定する、妥当性を判断するハンドラ
        /// </summary>
        private Predicate<string> _validTextPredicate = null;

        /// <summary>
        /// 最後に有効だった文字列です。
        /// </summary>
        private string ValidTextCache { get; set; }

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public StringTextBox()
        {
        }

        //----------------------------------------------------------

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(_defValue)]
        [Description("値。")]
        public string Value
        {
            get { return _value; }
            set
            {
                if (_value != value)
                {
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 文字列の妥当性チェックを初期化します。
        /// </summary>
        public void InitializeTextValidator(Predicate<string> validTextPredicate)
        {
            // 初期化時に、一度だけ呼ばれる想定です。
            Debug.Assert(_validTextPredicate == null);
            Debug.Assert(validTextPredicate != null);

            _validTextPredicate = validTextPredicate;

            this.TextChanged += OnTextChangedValidation_;
        }

        /// <summary>
        /// Valueプロパティ設定。
        /// </summary>
        public void SetValue(string value)
        {
            _invokeEvent = false;
            {
                this.Value = value;
            }
            _invokeEvent = true;
        }

        //----------------------------------------------------------

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            this.Text = _value;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            string prev = _value;
            _value = this.Text;
            return _value != prev;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 文字更新時バリデーションと復元処理
        /// </summary>
        void OnTextChangedValidation_(object sender, EventArgs e)
        {
            if (_validTextPredicate == null)
            {
                return;
            }

            // 文字列がない場合は処理しません。
            StringTextBox tbx = sender as StringTextBox;
            if (string.IsNullOrEmpty(tbx.Text))
            {
                return;
            }

            // 不正な文字列だったら、以前の状態に戻す。
            if (!_validTextPredicate(tbx.Text))
            {
                int cursorPos = tbx.SelectionStart;
                tbx.Text = tbx.ValidTextCache;
                tbx.Select(cursorPos, 0);
            }

            // 正しい文字列をキャッシュする
            tbx.ValidTextCache = tbx.Text;
        }
    }

    /// <summary>
    /// Textが空のときに、説明文章を表示するTextBox
    /// </summary>
    class UIGuideStringTextBox : StringTextBox
    {
        Color _guideStringColor = Color.Gray;
        string _guideText = "Guide-Text";
        Brush _backBrush = null;

        /// <summary>
        /// 説明文章の色
        /// </summary>
        public Color GuideStringColor
        {
            get { return _guideStringColor; }
            set { _guideStringColor = value; }
        }

        /// <summary>
        /// 説明文章
        /// </summary>
        [Localizable(true)]
        public string GuideText
        {
            get { return _guideText; }
            set { _guideText = value; }
        }

        /// <summary>
        /// 案内文字列を描画します。
        ///
        /// TextBox WM_PAINT でgoogle検索して発見したページ
        /// を参考にしました。
        ///
        /// WM_PAINTを使ったTextBoxのカスタム描画
        /// http://blogs.wankuma.com/umebayashi/archive/2007/02/15/62360.aspx
        /// </summary>
        void DrawGuideString_()
        {
            TextFormatFlags tff = TextFormatFlags.Top;
            switch (this.TextAlign)
            {
                case HorizontalAlignment.Center:
                {
                    tff |= TextFormatFlags.HorizontalCenter;
                    break;
                }
                case HorizontalAlignment.Left:
                {
                    tff |= TextFormatFlags.Left;
                    break;
                }
                case HorizontalAlignment.Right:
                {
                    tff |= TextFormatFlags.Right;
                    break;
                }
            }

            if (this.Multiline)
            {
                tff |= TextFormatFlags.WordBreak;
            }

            using (Graphics g = CreateGraphics())
            {
                if( _backBrush == null )
                {
                    _backBrush = new SolidBrush( this.BackColor );
                }

                //BackColorで背景を塗りつぶす。
                g.FillRectangle( _backBrush, this.ClientRectangle );

                TextRenderer.DrawText( g, GuideText, this.Font, this.ClientRectangle, GuideStringColor, tff );
            }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void WndProc( ref Message m )
        {
            base.WndProc( ref m );
            if( m.Msg == LECore.Win32.WM.WM_PAINT )
            {
                if( Text == string.Empty )
                {
                    if( !Focused )
                    {
                        DrawGuideString_();
                    }
                }
            }
        }
    }
}

#region ToolStripTextBox

/// <summary>
/// 値テキストボックスクラス。
/// </summary>
public abstract class ToolStripValueTextBox : ToolStripTextBox
{
    /// <summary>値変更イベント</summary>
    public event EventHandler ValueChanged = null;

    /// <summary>イベント発行制御</summary>
    protected static bool _invokeEvent = true;

    /// <summary>
    /// コンストラクタ。
    /// </summary>
    protected ToolStripValueTextBox()
    {
    }

    /// <summary>
    /// 未設定状態にする。
    /// </summary>
    public void SetEmpty()
    {
        this.Text = string.Empty;
    }

    /// <summary>
    /// テキスト更新。
    /// </summary>
    protected abstract void UpdateText();

    /// <summary>
    /// 値更新。
    /// </summary>
    protected abstract bool UpdateValue();

    /// <summary>
    /// 値変更ハンドラ。
    /// </summary>
    protected virtual void OnValueChanged( EventArgs e )
    {
        if( _invokeEvent )
        {
            if( this.ValueChanged != null )
            {
                this.ValueChanged( this, e );
            }
        }
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override bool IsInputKey( Keys keyData )
    {
        if( ( keyData & Keys.Alt ) != Keys.Alt )
        {
            Keys keys = keyData & Keys.KeyCode;
            if( keys == Keys.Return )
            {
                return true;
            }
        }
        return base.IsInputKey( keyData );
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override void OnLostFocus( EventArgs e )
    {
        // 値更新
        bool changed = UpdateValue();

        // テキスト更新
        UpdateText();

        // イベント発行
        if( changed )
        {
            OnValueChanged( EventArgs.Empty );
        }

        base.OnLostFocus( e );
    }

    /// <summary>
    /// オーバーライド。
    /// OnLostFocus と異なり、値変更状態にかかわらず
    /// ValueChanged イベントを発生します。
    /// </summary>
    protected override void OnKeyDown( KeyEventArgs e )
    {
        switch( e.KeyCode )
        {
            case Keys.Return:
            {
                // 値更新
                UpdateValue();

                // テキスト更新
                UpdateText();

                // 全選択
                SelectAll();

                // イベント発行
                // 常にValueChanged イベントを発生します。
                OnValueChanged( EventArgs.Empty );

                break;
            }
            default:
            base.OnKeyDown( e );
            break;
        }
    }

    private void InitializeComponent()
    {
        //this.SuspendLayout();
        //this.ResumeLayout( false );

    }
}

/// <summary>
/// 文字列テキストボックスクラス。
/// </summary>
public class ToolStripStringTextBox : ToolStripValueTextBox
{
    private string _value = string.Empty;

    /// <summary>
    /// 値。
    /// </summary>
    public string Value
    {
        get { return _value; }
        set
        {
            if (_value != value)
            {
                _value = value;
                UpdateText();
                OnValueChanged(EventArgs.Empty);
            }
        }
    }

    /// <summary>
    /// コンストラクタ。
    /// </summary>
    public ToolStripStringTextBox()
    {
    }

    /// <summary>
    /// Valueプロパティ設定。
    /// </summary>
    public void SetValue(string value)
    {
        _invokeEvent = false;
        {
            this.Value = value;
        }
        _invokeEvent = true;
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override void UpdateText()
    {
        this.Text = _value;
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override bool UpdateValue()
    {
        if (_value != this.Text)
        {
            _value = this.Text;
            return true;
        }

        return false;
    }
}

/// <summary>
/// 数値テキストボックスクラス。
/// </summary>
public abstract class ToolStripNumberTextBox : ToolStripValueTextBox
{
    /// <summary>
    /// コンストラクタ。
    /// </summary>
    protected ToolStripNumberTextBox() { }

    /// <summary>
    /// 入力可能文字かどうか。
    /// </summary>
    protected bool IsEnableInputChar( char ch )
    {
        // マイナス
        if( ch == '-' && IsEnableInputMinus() )
        {
            return true;
        }
        // 数字か編集キー（VK_BACK=0x08）
        else if( ( '0' <= ch && ch <= '9' ) || ch == 0x08 )
        {
            return true;
        }
        return false;
    }

    /// <summary>
    /// マイナスが入力可能かどうか。
    /// </summary>
    protected bool IsEnableInputMinus()
    {
        string text = this.Text;

        // 未入力
        if( text == string.Empty )
        {
            return true;
        }

        // キャレットが先頭
        if( this.SelectionStart == 0 )
        {
            // 非選択状態
            if( this.SelectionLength == 0 )
            {
                if( text.IndexOf( '-' ) == -1 )
                {
                    return true;
                }
            }
            // 選択状態
            else
            {
                if( text.IndexOf( '-', this.SelectionLength ) == -1 )
                {
                    return true;
                }
            }
        }
        return false;
    }
}

/// <summary>
/// 整数値テキストボックスクラス。
/// </summary>
[DefaultProperty( "Value" )]
[ToolboxBitmap( typeof( TextBox ) )]
public class ToolStripIntTextBox : ToolStripNumberTextBox
{
    internal const int _defValue = 0;
    internal const int _defMinValue = -99999;
    internal const int _defMaxValue = 99999;
    internal const int _defMaxLength = 6;

    private int _value = _defValue;
    private int _minValue = _defMinValue;
    private int _maxValue = _defMaxValue;

    /// <summary>
    /// コンストラクタ。
    /// </summary>
    public ToolStripIntTextBox()
    {
        this.MaxLength = _defMaxLength;
    }

    /// <summary>
    /// 値。
    /// </summary>
    [DefaultValue( _defValue )]
    [Description( "値。" )]
    public int Value
    {
        get { return _value; }
        set
        {
            if( _value != value )
            {
                _value = value;
                UpdateText();
                OnValueChanged( EventArgs.Empty );
            }
        }
    }

    /// <summary>
    /// 最小値。
    /// </summary>
    [DefaultValue( _defMinValue )]
    [Description( "最小値。" )]
    public int MinValue
    {
        get { return _minValue; }
        set { _minValue = value; }
    }

    /// <summary>
    /// 最大値。
    /// </summary>
    [DefaultValue( _defMaxValue )]
    [Description( "最大値。" )]
    public int MaxValue
    {
        get { return _maxValue; }
        set { _maxValue = value; }
    }

    /// <summary>
    /// 再定義。
    /// </summary>
    [DefaultValue( _defMaxLength )]
    public new int MaxLength
    {
        get { return base.MaxLength; }
        set { base.MaxLength = value; }
    }

    /// <summary>
    /// Valueプロパティ設定。
    /// </summary>
    public void SetValue( int value )
    {
        _invokeEvent = false;
        {
            this.Value = value;
        }
        _invokeEvent = true;
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override void UpdateText()
    {
        this.Text = _value.ToString();
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override bool UpdateValue()
    {
        int prev = _value;
        try
        {
            _value = int.Parse( this.Text );
            if( _value < _minValue ) { _value = _minValue; }
            else if( _value > _maxValue ) { _value = _maxValue; }
        }
        catch( SystemException )
        {
            _value = _minValue;
        }
        return _value != prev;
    }
}

/// <summary>
/// 実数値テキストボックスクラス。
/// </summary>
[DefaultProperty( "Value" )]
[ToolboxBitmap( typeof( TextBox ) )]
public class ToolStripFloatTextBox : ToolStripNumberTextBox
{
    internal const float _defValue = 0;
    internal const float _defMinValue = -99999;
    internal const float _defMaxValue = 99999;
    internal const string _defFormat = "f6";
    internal const int _defMaxLength = 13;

    private float _value = _defValue;
    private float _minValue = _defMinValue;
    private float _maxValue = _defMaxValue;
    private string _format = _defFormat;

    /// <summary>
    /// コンストラクタ。
    /// </summary>
    public ToolStripFloatTextBox()
    {
        this.MaxLength = _defMaxLength;
    }

    /// <summary>
    /// 値。
    /// </summary>
    [DefaultValue( _defValue )]
    [Description( "値。" )]
    public float Value
    {
        get { return _value; }
        set
        {
            if( _value != value )
            {
                _value = value;
                UpdateText();
                OnValueChanged( EventArgs.Empty );
            }
        }
    }

    /// <summary>
    /// 最小値。
    /// </summary>
    [DefaultValue( _defMinValue )]
    [Description( "最小値。" )]
    public float MinValue
    {
        get { return _minValue; }
        set { _minValue = value; }
    }

    /// <summary>
    /// 最大値。
    /// </summary>
    [DefaultValue( _defMaxValue )]
    [Description( "最大値。" )]
    public float MaxValue
    {
        get { return _maxValue; }
        set { _maxValue = value; }
    }

    /// <summary>
    /// 書式指定フォーマット。
    /// </summary>
    [DefaultValue( _defFormat )]
    [Description( "書式指定フォーマット。" )]
    public string Format
    {
        get { return _format; }
        set
        {
            _format = value;
            UpdateText();
        }
    }

    /// <summary>
    /// 再定義。
    /// </summary>
    [DefaultValue( _defMaxLength )]
    public new int MaxLength
    {
        get { return base.MaxLength; }
        set { base.MaxLength = value; }
    }

    /// <summary>
    /// Valueプロパティ設定。
    /// </summary>
    public void SetValue( float value )
    {
        _invokeEvent = false;
        {
            this.Value = value;
        }
        _invokeEvent = true;
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override void UpdateText()
    {
        this.Text = _value.ToString( _format );
    }

    /// <summary>
    /// オーバーライド。
    /// </summary>
    protected override bool UpdateValue()
    {
        float prev = _value;
        try
        {
            _value = float.Parse( this.Text );
            if( _value < _minValue ) { _value = _minValue; }
            else if( _value > _maxValue ) { _value = _maxValue; }
        }
        catch( SystemException )
        {
            _value = _minValue;
        }
        return _value != prev;
    }

    /// <summary>
    /// 小数点が入力可能かどうか。
    /// </summary>
    private bool IsEnableInputPeriod()
    {
        // 入力済みじゃない
        return ( this.Text.IndexOf( '.' ) == -1 );
    }
}



#endregion
