﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Controls
{
    using Utility;
    using LECore;
    /// <summary>
    /// 選択アイテムの位置に任意のコントロールを表示して、
    /// 値を編集可能なListViewコントロールです。
    ///
    /// </summary>
    public class InPlaceEditListView : UIListView
    {
        #region --------- デザイナ生成コード ---------
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            components = new System.ComponentModel.Container();
        }
        #endregion
        #endregion --------- デザイナ生成コード ---------

        #region --------- フィールド ---------

        // 編集に使用されるコントロール
        Control       _editingControl;
        // 編集対象アイテム
        ListViewItem  _editItem;
        // 複数選択時の編集対象アイテム
        List<ListViewItem> _multiEditItems = new List<ListViewItem>();

        // 編集対象サブアイテム
        int           _editSubItem;
        // 選択前のクリックか？
        bool _isEditStartClick;

        #endregion --------- フィールド ---------

        #region --------- 公開イベント ---------
        public event SubItemEventHandler SubItemClicked;
        public event SubItemEventHandler SubItemBeginEditing;
        public event SubItemEndEditingEventHandler SubItemEndEditing;
        public event EventHandler BeforeSubItemEndEditing;
        public event EventHandler AfterSubItemEndEditing;
        public Func<string, ListViewItem, bool> CheckNodeSubItemEditingChanged;
        public Func<int, bool> IsEditTargetSubItemIdxValid;

        public Func<Control, string> GetEditResultTextFunc;

        #endregion --------- 公開イベント ---------


        #region イベント監視関連定数
        /// <summary>
        /// WM_NOTIFY メッセージ パラメータ解釈用構造体
        /// </summary>
        private struct NMHDR
        {
            public IntPtr HwndFrom;
            public Int32  IdFrom;
            public Int32  Code;

            public NMHDR( IntPtr hwndFrom, Int32  idFrom, Int32  code )
            {
                HwndFrom =  hwndFrom;
                IdFrom   =  idFrom;
                Code     = code;
            }
        }
        private const int HDN_FIRST = -300;
        private const int HDN_BEGINDRAG = (HDN_FIRST-10);
        private const int HDN_ITEMCHANGINGA = (HDN_FIRST-0);
        private const int HDN_ITEMCHANGINGW = (HDN_FIRST-20);
        #endregion イベント監視関連定数

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InPlaceEditListView()
            :base()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            base.FullRowSelect      = true;
            base.View               = View.Details;
            base.AllowColumnReorder = true;

            // デフォルトのラベル編集はOFFにします。
            base.LabelEdit = false;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        /// <summary>
        /// ウインドウ・プロシージャ
        /// コントロールによる編集を中断すべき、メッセージを監視します。
        /// </summary>
        /// <param name="msg"></param>
        protected override void	WndProc(ref	Message	msg)
        {
            switch (msg.Msg)
            {
                    // 以下のメッセージを監視して...
                case LECore.Win32.WM.WM_VSCROLL:
                case LECore.Win32.WM.WM_HSCROLL:
                case LECore.Win32.WM.WM_SIZE:
                    // 編集を中断
                    EndEditing(false);
                    break;
                case LECore.Win32.WM.WM_NOTIFY:
                    // リスト行のサイズ、順番変更を起こしうる WM_NOTIFY を監視して...
                    NMHDR h = (NMHDR)Marshal.PtrToStructure(msg.LParam, typeof(NMHDR));
                    if ( h.Code == HDN_BEGINDRAG ||
                         h.Code == HDN_ITEMCHANGINGA ||
                         h.Code == HDN_ITEMCHANGINGW)
                    {
                        // 編集を中断
                        EndEditing(false);
                    }
                    break;
            }

            base.WndProc(ref msg);
        }


        #region サブアイテムの情報取得関連
        /// <summary>
        /// 行の順番配列を取得します。
        /// 行の並び替えを考慮するために必要となります。
        /// </summary>
        /// <returns></returns>
        public int[]      GetColumnOrder()
        {
            return ListViewHelper.GetColumnOrderArray(this);
        }

        /// <summary>
        /// サブアイテムの領域矩形を取得します。
        /// </summary>
        public Rectangle GetSubItemBounds( ListViewItem Item, int subItemIdx )
        {
            int[]     order         = GetColumnOrder();
            // 不正なケースをチェックします。
            if( subItemIdx >= order.Length || Item == null )
            {
                return Rectangle.Empty;
            }


            Rectangle lviBounds     = Item.GetBounds(ItemBoundsPortion.Entire);
            int	    subItemX      = lviBounds.Left;
            int       i             = 0;

            for( i = 0; i < order.Length; i++ )
            {
                ColumnHeader    col = this.Columns[order[i]];
                if ( col.Index == subItemIdx )
                {
                    break;
                }
                subItemX += col.Width;
            }

            return new Rectangle( subItemX, lviBounds.Top, this.Columns[order[i]].Width, lviBounds.Height);
        }
        #endregion サブアイテムの情報取得関連

        #region キー関連：オーバーライド
        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            if (e.KeyCode == Keys.Up || e.KeyCode == Keys.Down)
            {
                // 選択状態を記憶します
                _multiEditItems.Clear();
                foreach (ListViewItem SelectedItem in this.SelectedItems)
                {
                    _multiEditItems.Add(SelectedItem);
                }
            }
        }
        #endregion

        #region マウスハンドラ：オーバーライド
        /// <summary>
        /// MouseUp
        /// </summary>
        protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseDown(e);

            var item = this.GetItemAt(e.X, e.Y);

            this._isEditStartClick = item != null && item.Selected; // 既に選択されているものをクリックしている。
            if (_isEditStartClick)
            {
                // 選択状態を記憶します（MouseDown直後にキャンセルされてしまうので）。
                _multiEditItems.Clear();
                foreach (ListViewItem SelectedItem in this.SelectedItems)
                {
                    _multiEditItems.Add(SelectedItem);
                }
            }
        }

        /// <summary>
        /// MouseUp
        /// </summary>
        protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseUp(e);

            if (this._isEditStartClick)
            {
                bool isHandled = TryEditSubitemAt_(new Point(e.X, e.Y));
                if (isHandled)
                {
                    // MouseDown で変更された選択状態を復元します。
                    foreach (var item in _multiEditItems)
                    {
                        if (item.Selected != true)
                        {
                            item.Selected = true;
                        }
                    }
                }
            }
        }

        ///<summary>
        /// クリックした点でサブアイテム選択をこころみます。
        ///</summary>
        private bool TryEditSubitemAt_( Point p )
        {
            ListViewItem item;
            int idx = ListViewHelper.GetSubItemAt(this, p.X, p.Y, out item);
            if (idx >= 0)
            {
                if ((IsEditTargetSubItemIdxValid == null || IsEditTargetSubItemIdxValid(idx))
                    && SubItemClicked != null)
                {
                    var newSubItemEventArgs = new SubItemEventArgs(item, idx);
                    NotifySubItemClicked_(newSubItemEventArgs);

                    return true;
                }
            }

            return false;
        }
        #endregion

        #region 編集関連

        /// <summary>
        /// コールバック呼び出し処理中か？（多重呼び出しを避けるためのフラグです）
        /// </summary>
        bool _isCallBackFunctionCalled;

        /// <summary>
        /// RAII形式でフラグ設定をカプセル化したクラス。
        /// </summary>
        private class FlagBlock : IDisposable
        {
            Action<bool> _set;

            public FlagBlock(Action<bool> set)
            {
                _set = set;
                _set(true);
            }

            public void Dispose()
            {
                _set(false);
            }
        }

        #region イベント通知関連

        /// <summary>
        /// サブアイテムの編集が開始されたことを通知します。
        /// </summary>
        protected void NotifySubItemBeginEditing_(SubItemEventArgs e)
        {
            if (SubItemBeginEditing != null && _isCallBackFunctionCalled == false)
            {
                using (FlagBlock flagBlock = new FlagBlock((flag) => this._isCallBackFunctionCalled = flag))
                {
                    SubItemBeginEditing(this, e);
                }
            }
        }

        /// <summary>
        /// サブアイテムの編集が終了したことを通知します。
        /// </summary>
        protected void NotifySubItemEndEditing_(SubItemEndEditingEventArgs e)
        {
            if (SubItemEndEditing != null && _isCallBackFunctionCalled == false)
            {
                using (FlagBlock flagBlock = new FlagBlock((flag) => this._isCallBackFunctionCalled = flag))
                {
                    SubItemEndEditing(this, e);
                }
            }
        }

        /// <summary>
        /// サブアイテムがクリックされたことを通知します。
        /// </summary>
        protected void NotifySubItemClicked_(SubItemEventArgs e)
        {
            if (SubItemClicked != null && _isCallBackFunctionCalled == false)
            {
                using (FlagBlock flagBlock = new FlagBlock((flag) => this._isCallBackFunctionCalled = flag))
                {
                    SubItemClicked(this, e);
                }
            }
        }

        /// <summary>
        /// サブアイテムの編集を開始することを通知します。
        /// </summary>
        protected void NotifyBeofreSubItemEdit_()
        {
            if (BeforeSubItemEndEditing != null && _isCallBackFunctionCalled == false)
            {
                using (FlagBlock flagBlock = new FlagBlock((flag) => this._isCallBackFunctionCalled = flag))
                {
                    BeforeSubItemEndEditing(this, null);
                }
            }
        }

        /// <summary>
        /// サブアイテムの編集を終了することを通知します。
        /// </summary>
        protected void NotifyAfterSubItemEdit_()
        {
            if (AfterSubItemEndEditing != null && _isCallBackFunctionCalled == false)
            {
                using (FlagBlock flagBlock = new FlagBlock((flag) => this._isCallBackFunctionCalled = flag))
                {
                    AfterSubItemEndEditing(this, null);
                }
            }
        }
        #endregion イベント通知関連

        #region 編集用コントロールに設定するイベントハンドラ
        /// <summary>
        /// イベントハンドラ：Leave
        /// </summary>
        private void Event_EditControl_Leave(object sender, EventArgs e)
        {
            // 編集用コントロールからフォーカスが失われます。
            EndEditing(true);
        }

        /// <summary>
        /// イベントハンドラ：KeyPress
        /// </summary>
        private void Event_EditControl_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
        {
            switch (e.KeyChar)
            {
                case (char)(int)Keys.Escape:
                {
                    EndEditing(false);
                    break;
                }

                case (char)(int)Keys.Enter:
                {
                    EndEditing(true);
                    break;
                }
            }
        }
        #endregion 編集用コントロールに設定するイベントハンドラ

        /// <summary>
        /// 指定セルにおいて、コントロールを使用した編集を開始します。
        /// </summary>
        /// <param name="c">編集に使用するコントロール</param>
        /// <param name="Item">リストビュー項目</param>
        /// <param name="SubItem">サブアイテム番号</param>
        public void StartEditing
            (
            Control          c,
            ListViewItem     item,
            int              subItemIdx
            )
        {
            // デフォルトのラベル編集機能との併用は考慮されていません。OFFにして利用します。
            Ensure.Operation.True(!base.LabelEdit);

            NotifySubItemBeginEditing_(new SubItemEventArgs(item, subItemIdx));

            Rectangle rectSubItem = GetSubItemBounds(item, subItemIdx);

            // ----------- 左端、右端について、位置とサイズを調整します。
            if (rectSubItem.X < 0)
            {
                rectSubItem.Width += rectSubItem.X;
                rectSubItem.X = 0;
            }
            if (rectSubItem.X + rectSubItem.Width > this.Width)
            {
                rectSubItem.Width = this.Width - rectSubItem.Left;
            }

            // ----------- リストビューローカル座標系から、リストビューの親コントロールの座標系へ
            rectSubItem.Offset(this.Left, this.Top);

            // 編集用のコントロールの親と、ListView自身の親が異なる場合、
            // それらの位置の差を加算します。
            if (this.Parent != null && c.Parent != null &&
                !object.ReferenceEquals(this.Parent, c.Parent))
            {
                Point origin = new Point(0, 0);
                Point lvOrigin = this.Parent.PointToScreen(origin);
                Point ctlOrigin = c.Parent.PointToScreen(origin);
                rectSubItem.Offset(lvOrigin.X - ctlOrigin.X, lvOrigin.Y - ctlOrigin.Y);
            }

            //  ----------- 位置を設定して、表示します。
            c.Bounds = rectSubItem;
            c.Text = item.SubItems[subItemIdx].Text;
            c.Visible = true;
            c.BringToFront();
            c.Focus();
            c.LostFocus += new EventHandler(Event_EditControl_Leave);
            c.KeyPress += new KeyPressEventHandler(Event_EditControl_KeyPress);

            // ----------- 編集コントロールの設定、イベントハンドラの設定
            _editingControl = c;

            _editItem = item;
            _editSubItem = subItemIdx;
        }

        /// <summary>
        /// 編集を終了します。
        /// </summary>
        /// <param name="acceptChanges">変更を使用する場合は、trueを指定します。</param>
        public void EndEditing( bool acceptChanges )
        {
            Control editingControl = _editingControl;
            ListViewItem editItem = _editItem;
            int editSubItem = _editSubItem;

            List<ListViewItem> multiEditItems = new List<ListViewItem>(_multiEditItems);

            if (editingControl == null)
            {
                return;
            }

            // 多重に終了処理が実行されないように内部メンバをリセットしておく。
            {
                _editingControl = null;
                _editItem = null;
                _editSubItem = -1;

                _multiEditItems.Clear();
            }

            string editResultText = (GetEditResultTextFunc == null) ?
                editingControl.Text : GetEditResultTextFunc(editingControl);

            string resultStr = acceptChanges ?
                editResultText :	                    // 編集結果
                editItem.SubItems[editSubItem].Text;	// 編集前の元の文字列

            if(CheckNodeSubItemEditingChanged == null || CheckNodeSubItemEditingChanged(editResultText, editItem))
            {
                // 設定
                editItem.SubItems[editSubItem].Text = resultStr;

                // 編集終了イベントを通知（選択アイテム全体について通知します）
                {
                    NotifyBeofreSubItemEdit_();

                    foreach (var multiEditItem in multiEditItems)
                    {
                        NotifySubItemEndEditing_(new SubItemEndEditingEventArgs(multiEditItem, editSubItem, resultStr, !acceptChanges));
                    }

                    NotifyAfterSubItemEdit_();
                }
            }

            // イベントハンドラのリセットなど、後始末をします。
            editingControl.Leave -= new EventHandler(Event_EditControl_Leave);
            editingControl.KeyPress -= new KeyPressEventHandler(Event_EditControl_KeyPress);
            editingControl.Visible = false;
        }

        #endregion
    }

    #region デリゲート宣言
    /// <summary>
    /// SubItem 用イベントハンドラ
    /// </summary>
    public delegate void SubItemEventHandler( object sender, SubItemEventArgs e );
    /// <summary>
    /// SubItem 編集イベントハンドラ
    /// </summary>
    public delegate void SubItemEndEditingEventHandler( object sender, SubItemEndEditingEventArgs e );

    /// <summary>
    /// イベントハンドラ 用のパラメータ型です。
    /// </summary>
    public class SubItemEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SubItemEventArgs( ListViewItem item, int subItem )
        {
            _subItemIndex = subItem;
            _item = item;
        }

        private int _subItemIndex = -1;
        private ListViewItem _item = null;

        /// <summary>
        /// サブアイテム番号
        /// </summary>
        public int SubItem
        {
            get { return _subItemIndex; }
        }

        /// <summary>
        /// リストビューアイテム
        /// </summary>
        public ListViewItem Item
        {
            get { return _item; }
        }
    }

    /// <summary>
    /// 編集終了イベントハンドラ 用のパラメータ型です。
    /// </summary>
    public class SubItemEndEditingEventArgs : SubItemEventArgs
    {
        private string _text = string.Empty;
        private bool _cancel = true;

        public SubItemEndEditingEventArgs( ListViewItem item, int subItem, string display, bool cancel )
            :
            base( item, subItem )
        {
            _text = display;
            _cancel = cancel;
        }

        public string DisplayText
        {
            get { return _text; }
            set { _text = value; }
        }

        public bool Cancel
        {
            get { return _cancel; }
            set { _cancel = value; }
        }
    }

    #endregion デリゲート宣言
}


