﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Data;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    using Forms.ToolWindows;
    using LECore.Structures;
    using LECore.Structures.Core;
    using src.Controls;
    using System.Collections.Generic;

    /// <summary>
    /// Fvec2EditUpDownPanel の概要の説明です。
    /// </summary>
    public class Fvec3EditUpDownPanel : System.Windows.Forms.UserControl
        , IAnimationMarkUsable
    {
        private IContainer components;
        private UINumericUpDown _nudX;
        private UINumericUpDown _nudY;
        private UINumericUpDown _nudZ;
        private src.Controls.BindAnimationLabel _lblX;
        private src.Controls.BindAnimationLabel _lblY;
        private src.Controls.BindAnimationLabel _lblZ;
        private readonly int _InitialDecimalPlaces;
        private readonly Color _InitialForeColor;
        private src.Controls.AnimationMark _pnlMarkX;
        private src.Controls.AnimationMark _pnlMarkY;
        private src.Controls.AnimationMark _pnlMarkZ;

        public event FVec3EditEventHandler OnFVec3Edit = null;

        bool _roundByStepValue = false;
        bool _useAnimationMark = false;

        public Fvec3EditUpDownPanel()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            _InitialDecimalPlaces = this._nudX.DecimalPlaces;
            _InitialForeColor = this._nudX.ForeColor;

            // AnimationMarkの持つコンテキストメニューを関連付けます
            _lblX.BindAnimationMark(_pnlMarkX);
            _lblY.BindAnimationMark(_pnlMarkY);
            _lblZ.BindAnimationMark(_pnlMarkZ);
        }

        public FVec3 FVec3Value
        {
            get { return ConvertToFVec3_(); }
            set
            {
                SafeSetToNumericUpDown_( _nudX, value.X );
                SafeSetToNumericUpDown_( _nudY, value.Y );
                SafeSetToNumericUpDown_( _nudZ, value.Z );
            }
        }

        /// <summary>
        /// ステップ値で丸めるか？
        /// </summary>
        public bool RoundByStepValue
        {
            get { return _roundByStepValue; }
            set { _roundByStepValue = value; }
        }

        /// <summary>
        /// 増加値
        /// </summary>
        public float Increment
        {
            get
            {
                Debug.Assert( _nudX.Increment == _nudY.Increment && _nudX.Increment == _nudZ.Increment );
                return Decimal.ToSingle( _nudX.Increment );
            }

            set
            {
                _nudX.Increment = Convert.ToDecimal( value );
                _nudY.Increment = Convert.ToDecimal( value );
                _nudZ.Increment = Convert.ToDecimal( value );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Color TextBoxBackColor
        {
            set
            {
                _nudX.BackColor = value;
                _nudY.BackColor = value;
                _nudZ.BackColor = value;
            }
        }

        /// <summary>
        /// X が有効か
        /// </summary>
        public bool XEnabled
        {
            get { return _nudX.Enabled; }
            set
            {
                _nudX.Enabled = value;
            }
        }

        /// <summary>
        /// Y が有効か
        /// </summary>
        public bool YEnabled
        {
            get { return _nudY.Enabled; }
            set
            {
                _nudY.Enabled = value;
            }
        }

        /// <summary>
        /// Z が有効か
        /// </summary>
        public bool ZEnabled
        {
            get { return _nudZ.Enabled; }
            set
            {
                _nudZ.Enabled = value;
            }
        }

        /// <summary>
        /// アニメーションマークのリスト
        /// </summary>
        public AnimationMark[] AnimationMarkArray
        {
            get
            {
                List<AnimationMark> list = new List<AnimationMark>();
                list.Add(_pnlMarkX);
                list.Add(_pnlMarkY);
                list.Add(_pnlMarkZ);

                return list.ToArray();
            }
        }

        FVec3 ConvertToFVec3_()
        {
            return new FVec3(
                Decimal.ToSingle( _nudX.Value ),
                Decimal.ToSingle( _nudY.Value ),
                Decimal.ToSingle( _nudZ.Value ) );
        }

        void SafeSetToNumericUpDown_( UINumericUpDown udCtrl, float fval )
        {
            Decimal decVal = Convert.ToDecimal( fval );

            decVal = Math.Min( decVal, udCtrl.Maximum );
            decVal = Math.Max( decVal, udCtrl.Minimum );

            udCtrl.Value = decVal;

            // 小数点の調整
            udCtrl.AdjustDecimalPlace(_InitialDecimalPlaces);

            udCtrl.ForeColor = (udCtrl.DecimalPlaces == _InitialDecimalPlaces) ? _InitialForeColor : Color.RoyalBlue;
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            _pnlMarkX.SendMessageToViewManager = handler;
            _pnlMarkY.SendMessageToViewManager = handler;
            _pnlMarkZ.SendMessageToViewManager = handler;
        }

        /// <summary>
        /// コンテキストメニューを追加します。
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            _pnlMarkX.RegistContextMenu(name, image, handler);
            _pnlMarkY.RegistContextMenu(name, image, handler);
            _pnlMarkZ.RegistContextMenu(name, image, handler);
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute attr)
        {
            Debug.Assert(attr.NumSubAttribute == 3);

            _pnlMarkX.SetTargetAttribute(attr.FindSubAttributeByIdx(0));
            _pnlMarkY.SetTargetAttribute(attr.FindSubAttributeByIdx(1));
            _pnlMarkZ.SetTargetAttribute(attr.FindSubAttributeByIdx(2));
        }
        public void SetTargetAttribute(IAnmAttribute[] attrs)
        {
            _pnlMarkX.ClearTargetAttribute();
            _pnlMarkY.ClearTargetAttribute();
            _pnlMarkZ.ClearTargetAttribute();

            foreach (IAnmAttribute attr in attrs)
            {
                Debug.Assert(attr.NumSubAttribute == 3);

                _pnlMarkX.AddTargetAttribute(attr.FindSubAttributeByIdx(0));
                _pnlMarkY.AddTargetAttribute(attr.FindSubAttributeByIdx(1));
                _pnlMarkZ.AddTargetAttribute(attr.FindSubAttributeByIdx(2));
            }
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            _pnlMarkX.UpdateMarkColor();
            _pnlMarkY.UpdateMarkColor();
            _pnlMarkZ.UpdateMarkColor();
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {
            _pnlMarkX.UpdateContextMenu();
            _pnlMarkY.UpdateContextMenu();
            _pnlMarkZ.UpdateContextMenu();
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get { return _useAnimationMark; }
            set
            {
                _useAnimationMark = value;
                _pnlMarkX.Visible = value;
                _pnlMarkY.Visible = value;
                _pnlMarkZ.Visible = value;
                _lblX.UseContextMenu = value;
                _lblY.UseContextMenu = value;
                _lblZ.UseContextMenu = value;
            }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark()
        {
            UpdateMarkColor();
            UpdateContextMenu();
        }

        #endregion

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this._nudZ = new LayoutEditor.Controls.UINumericUpDown();
            this._nudX = new LayoutEditor.Controls.UINumericUpDown();
            this._nudY = new LayoutEditor.Controls.UINumericUpDown();
            this._pnlMarkX = new LayoutEditor.src.Controls.AnimationMark();
            this._pnlMarkY = new LayoutEditor.src.Controls.AnimationMark();
            this._pnlMarkZ = new LayoutEditor.src.Controls.AnimationMark();
            this._lblX = new LayoutEditor.src.Controls.BindAnimationLabel();
            this._lblY = new LayoutEditor.src.Controls.BindAnimationLabel();
            this._lblZ = new LayoutEditor.src.Controls.BindAnimationLabel();
            ((System.ComponentModel.ISupportInitialize)(this._nudZ)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudX)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudY)).BeginInit();
            this.SuspendLayout();
            //
            // _nudZ
            //
            this._nudZ.DecimalPlaces = 2;
            this._nudZ.Increment = new decimal(new int[] {
            1,
            0,
            0,
            131072});
            this._nudZ.Location = new System.Drawing.Point(209, 3);
            this._nudZ.Maximum = new decimal(new int[] {
            10000,
            0,
            0,
            0});
            this._nudZ.Minimum = new decimal(new int[] {
            10000,
            0,
            0,
            -2147483648});
            this._nudZ.Name = "_nudZ";
            this._nudZ.Size = new System.Drawing.Size(62, 19);
            this._nudZ.TabIndex = 5;
            this._nudZ.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this._nudZ.ValueChanged += new System.EventHandler(this.Event_NudX_ValueChanged);
            //
            // _nudX
            //
            this._nudX.DecimalPlaces = 2;
            this._nudX.ForeColor = System.Drawing.SystemColors.WindowText;
            this._nudX.Increment = new decimal(new int[] {
            1,
            0,
            0,
            131072});
            this._nudX.Location = new System.Drawing.Point(28, 3);
            this._nudX.Maximum = new decimal(new int[] {
            10000,
            0,
            0,
            0});
            this._nudX.Minimum = new decimal(new int[] {
            10000,
            0,
            0,
            -2147483648});
            this._nudX.Name = "_nudX";
            this._nudX.RightToLeft = System.Windows.Forms.RightToLeft.No;
            this._nudX.Size = new System.Drawing.Size(62, 19);
            this._nudX.TabIndex = 3;
            this._nudX.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this._nudX.ValueChanged += new System.EventHandler(this.Event_NudX_ValueChanged);
            //
            // _nudY
            //
            this._nudY.DecimalPlaces = 2;
            this._nudY.Increment = new decimal(new int[] {
            1,
            0,
            0,
            131072});
            this._nudY.Location = new System.Drawing.Point(117, 3);
            this._nudY.Maximum = new decimal(new int[] {
            10000,
            0,
            0,
            0});
            this._nudY.Minimum = new decimal(new int[] {
            10000,
            0,
            0,
            -2147483648});
            this._nudY.Name = "_nudY";
            this._nudY.Size = new System.Drawing.Size(62, 19);
            this._nudY.TabIndex = 4;
            this._nudY.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this._nudY.ValueChanged += new System.EventHandler(this.Event_NudX_ValueChanged);
            //
            // _pnlMarkX
            //
            this._pnlMarkX.Location = new System.Drawing.Point(21, 3);
            this._pnlMarkX.Name = "_pnlMarkX";
            this._pnlMarkX.Size = new System.Drawing.Size(5, 19);
            this._pnlMarkX.TabIndex = 0;
            this._pnlMarkX.Visible = false;
            //
            // _pnlMarkY
            //
            this._pnlMarkY.BackColor = System.Drawing.SystemColors.Control;
            this._pnlMarkY.Location = new System.Drawing.Point(110, 3);
            this._pnlMarkY.Name = "_pnlMarkY";
            this._pnlMarkY.Size = new System.Drawing.Size(5, 19);
            this._pnlMarkY.TabIndex = 0;
            this._pnlMarkY.Visible = false;
            //
            // _pnlMarkZ
            //
            this._pnlMarkZ.BackColor = System.Drawing.SystemColors.Control;
            this._pnlMarkZ.Location = new System.Drawing.Point(202, 3);
            this._pnlMarkZ.Name = "_pnlMarkZ";
            this._pnlMarkZ.Size = new System.Drawing.Size(5, 19);
            this._pnlMarkZ.TabIndex = 0;
            this._pnlMarkZ.Visible = false;
            //
            // _lblX
            //
            this._lblX.Location = new System.Drawing.Point(2, 7);
            this._lblX.Name = "_lblX";
            this._lblX.Size = new System.Drawing.Size(19, 16);
            this._lblX.TabIndex = 0;
            this._lblX.Text = "X :";
            this._lblX.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this._lblX.UseAnimationMark = false;
            this._lblX.UseContextMenu = false;
            //
            // _lblY
            //
            this._lblY.Location = new System.Drawing.Point(91, 7);
            this._lblY.Name = "_lblY";
            this._lblY.Size = new System.Drawing.Size(19, 16);
            this._lblY.TabIndex = 1;
            this._lblY.Text = "Y :";
            this._lblY.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this._lblY.UseAnimationMark = false;
            this._lblY.UseContextMenu = false;
            //
            // _lblZ
            //
            this._lblZ.Location = new System.Drawing.Point(183, 7);
            this._lblZ.Name = "_lblZ";
            this._lblZ.Size = new System.Drawing.Size(19, 16);
            this._lblZ.TabIndex = 2;
            this._lblZ.Text = "Z :";
            this._lblZ.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this._lblZ.UseAnimationMark = false;
            this._lblZ.UseContextMenu = false;
            //
            // Fvec3EditUpDownPanel
            //
            this.Controls.Add(this._pnlMarkX);
            this.Controls.Add(this._pnlMarkZ);
            this.Controls.Add(this._pnlMarkY);
            this.Controls.Add(this._nudZ);
            this.Controls.Add(this._nudX);
            this.Controls.Add(this._nudY);
            this.Controls.Add(this._lblX);
            this.Controls.Add(this._lblY);
            this.Controls.Add(this._lblZ);
            this.Name = "Fvec3EditUpDownPanel";
            this.Size = new System.Drawing.Size(277, 25);
            ((System.ComponentModel.ISupportInitialize)(this._nudZ)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudX)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudY)).EndInit();
            this.ResumeLayout(false);

        }
        #endregion

        private void Event_NudX_ValueChanged( object sender, System.EventArgs e )
        {
            if( OnFVec3Edit != null )
            {
                string name = String.Empty;
                if( sender == _nudX ) { name = "X"; }
                if( sender == _nudY ) { name = "Y"; }
                if( sender == _nudZ ) { name = "Z"; }
                OnFVec3Edit( this, new FVec3EditEventArgs( name, FVec3Value ) );
            }
        }
    }

    #region OnFVec3EditEvent
    public delegate void FVec3EditEventHandler( object sender, FVec3EditEventArgs args );

    /// <summary>
    /// イベントパラメータ型
    /// </summary>
    public class FVec3EditEventArgs : EventArgs
    {
        readonly FVec3     _val;
        readonly string    _name = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FVec3EditEventArgs( string name, FVec3 val )
        {
            _name = name;
            _val = val;
        }

        /// <summary>
        ///
        /// </summary>
        public FVec3 Value
        {
            get { return _val; }
        }

        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return _name; }
        }
    }
    #endregion OnFVec3EditEvent
}

