﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.Reflection;

namespace LECore.Structures
{
    using LECoreInterface;
    using System.Collections.Generic;
    using LECore.Structures.Nsrif.Attributes;
    using System.IO;

    /// <summary>
    ///
    /// </summary>
    internal class PlatformDetailCafe : IPlatformDetail
    {
        static private readonly string _NwRootPath;
        static private readonly string _NwToolsRootPath;
        static public readonly string _NwFontToolsRootPath;
        static private readonly bool _OnNintendoSdk;

        static PlatformDetailCafe()
        {
            // テスト：特定のフォルダ構成が検出された場合は処理を切り替えます。
            string executingFileName = Assembly.GetExecutingAssembly().Location;
            string layoutEditorFolderParentPath = Path.GetDirectoryName(Path.GetDirectoryName(executingFileName));
            if (layoutEditorFolderParentPath.Contains("Tools\\Graphics"))
            {
                if (layoutEditorFolderParentPath.EndsWith("Tools\\Graphics"))
                {
                    // パッケージフォルダ構成での動作
                    _NwRootPath = Path.GetDirectoryName(Path.GetDirectoryName(layoutEditorFolderParentPath));
                    _NwToolsRootPath = layoutEditorFolderParentPath;
                }
                else
                {
                    // リポジトリフォルダ構成での動作
                    // 親フォルダをたどって基点を見つけ、Toolフォルダを決定する。
                    string parentDir = layoutEditorFolderParentPath;
                    try
                    {
                        while (!parentDir.EndsWith("Programs"))
                        {
                            parentDir = Path.GetDirectoryName(parentDir);
                        }
                    }
                    catch
                    {
                        // 想定外のフォルダ構成だった。
                        // 適当な値を設定しておく。
                        parentDir = layoutEditorFolderParentPath;
                    }

                    _NwRootPath = Path.Combine(parentDir, "Iris");
                    _NwToolsRootPath = Path.Combine(_NwRootPath, "Tools\\Graphics");
                }

                _NwFontToolsRootPath = _NwToolsRootPath;
                _OnNintendoSdk = true;
            }
            else
            {
                _NwRootPath = System.Environment.GetEnvironmentVariable("NW4F_ROOT");

                _NwToolsRootPath = string.IsNullOrEmpty(_NwRootPath) ? string.Empty : Path.Combine(_NwRootPath, "Tool");
                _NwFontToolsRootPath = _NwToolsRootPath;
                _OnNintendoSdk = false;
            }
        }

        /// <summary>
        /// プレビュー前(バイナリコンバート前)の準備
        /// </summary>
        public bool PrepareBeforeBinaryConvert(bool forPCViewer, string tempDirPath)
        {
            // 何もしません
            return true;
        }

        /// <summary>
        /// プレビュー前のフォント変換
        /// </summary>
        public bool PrepareBinaryFontForPreview(bool forPCViewer, string tempDirPath)
        {
            // 何もしません
            return true;
        }

        /// <summary>
        /// PCビューアの画面サイズ制限が有効か
        /// </summary>
        public bool PCViewerScreenSizeLimitEnabled { get { return false; } }

        /// <summary>
        // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
        // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
        // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
        /// </summary>
        public bool SmallSizeFontProblemWorkaroundEnabled { get { return true; } }

        /// <summary>
        /// リニアガンマ編集環境が有効かどうか
        /// </summary>
        public bool IsLinearGammaSettingEnabled
        {
            get { return true; }
        }

        /// <summary>
        /// ツールルートパス
        /// </summary>
        public string NwToolsRootPath
        {
            get { return _NwToolsRootPath; }
        }

        /// <summary>
        /// レイアウトツールルートパス
        /// </summary>
        public string LayoutToolsRootPath
        {
            get { return Path.Combine(NwToolsRootPath, _OnNintendoSdk ? "LayoutTools" : "LytTool"); }
        }

        /// <summary>
        /// フォントツールルートパス
        /// </summary>
        public string NwFontToolsRootPath
        {
            get { return _NwFontToolsRootPath; }
        }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        public string LayoutConverterPath
        {
            get { return Path.Combine(AppConstants.LayoutToolsPath, _OnNintendoSdk ? @"LayoutConverter.exe" : @"LayoutConverter\NW4F_LayoutConverter.exe"); }
        }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        public string LayoutConverterForPCViewerPath { get { return LayoutConverterPath; } }

        /// <summary>
        /// フォントコンバータのパス
        /// </summary>
        public string FontConverterConsolePath
        {
            get { return Path.Combine(AppConstants.NwToolsRootPath, @"FontConverter\FontConverterConsole.exe"); }
        }

        /// <summary>
        /// PCビューアのパス
        /// </summary>
        public string PCLayoutViewerPath
        {
            get { return Path.Combine(_OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, AppConstants.ViewerSubFolderName + @"\Win32\" + AppConstants.PCViewerFileName); }
        }

        /// <summary>
        /// 実機ビューア起動バッチのパス
        /// </summary>
        public string RunCafeViewerBatPath
        {
            get { return Path.Combine(_OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, AppConstants.ViewerSubFolderName + @"\Cafe\run_viewer.bat"); }
        }

        /// <summary>
        /// Cafeビューアスタータのパス
        /// </summary>
        public string ViewerStarterPath
        {
            get { return Path.Combine(_OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, AppConstants.ViewerSubFolderName + @"\Cafe\host\preview_starter.exe"); }
        }

        /// <summary>
        /// プレビュー一時ファイルパスを取得します。
        /// </summary>
        public string PreviewTemporaryPath
        {
            get { return Path.Combine(AppConstants.NwToolsRootPath, "LayoutEditor\\preview_temp"); }
        }

        /// <summary>
        /// UI に表示するプラットフォーム名を取得します。
        /// </summary>
        public string UIPlatformName
        {
            get
            {
                // 今の実装だと常にCafeになりますが、
                // このプラグインを継承して設定ファイルを読むプラグインを実装すればCafeUi2dに分岐できます。
                return AppConstants.IsGfxMode ? "CafeUi2d" : "Cafe";
            }
        }

        /// <summary>
        /// プラットフォーム名を取得します。
        /// </summary>
        public string PlatformName
        {
            get { return "Cafe"; }
        }

        /// <summary>
        /// サンプルデータパスです。
        /// </summary>
        public string NwDefaultSampleRoot
        {
            get
            {
                if(_OnNintendoSdk)
                {
                    return AppConstants.GetNintendoSDKDefaultSamplePath(_NwRootPath);
                }
                else
                {
                    if (!string.IsNullOrEmpty(AppConstants.NwToolsRootPath))
                    {
                        return Path.Combine(AppConstants.NwToolsRootPath, "..\\SampleData\\Layout\\flyt");
                    }
                    else
                    {
                        return string.Empty;
                    }
                }
            }
        }

        /// <summary>
        /// 実機通信に必要な環境変数設定
        /// </summary>
        public IEnumerable<string> GetTargetComEnvList()
        {
            yield return "CAFE_ROOT";
            yield break;
        }

        /// <summary>
        /// アイコンを取得します。
        /// </summary>
        public System.Drawing.Icon GetIcon()
        {
            return LEImageResMgr.GetManifestResourceIco("LayoutEditor.ico");
        }

        /// <summary>
        ///
        /// </summary>
        public float GetBPP(TexImagePixelFmt PixelFmt)
        {
            switch (PixelFmt)
            {
                case TexImagePixelFmt.BC4L:
                case TexImagePixelFmt.BC4A:

                case TexImagePixelFmt.BC1:
                case TexImagePixelFmt.RGB5A1:
                case TexImagePixelFmt.RGB8:
                    return 0.5f;
                case TexImagePixelFmt.L8:
                case TexImagePixelFmt.A8:

                case TexImagePixelFmt.BC2:
                case TexImagePixelFmt.BC3:
                case TexImagePixelFmt.RGBA4:

                case TexImagePixelFmt.BC5:
                    return 1.0f;
                case TexImagePixelFmt.LA4:
                case TexImagePixelFmt.LA8:
                    return 2;
                case TexImagePixelFmt.RGB565:
                case TexImagePixelFmt.RGB565_INDIRECT:
                    return 2;
                case TexImagePixelFmt.RGBA8:
                    return 4;
                // 以下は使われていないフォーマット
                case TexImagePixelFmt.A4:
                case TexImagePixelFmt.L4:
                case TexImagePixelFmt.HILO8:
                case TexImagePixelFmt.ETC1:
                case TexImagePixelFmt.ETC1A4:
                default:
                    return 0;
            }
        }

        /// <summary>
        /// 変換テーブル
        /// </summary>
        private NW4F.LayoutBinaryConverter.TexConv.TexelFormat ConvetTexConvTexelFmt_(TexImagePixelFmt fmt)
        {
            switch (fmt)
            {
                case TexImagePixelFmt.A4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.A4;
                case TexImagePixelFmt.A8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.A8;
                case TexImagePixelFmt.BC1: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC1;
                case TexImagePixelFmt.BC2: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC2;
                case TexImagePixelFmt.BC3: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC3;
                case TexImagePixelFmt.BC4A: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC4A;
                case TexImagePixelFmt.BC4L: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC4L;
                case TexImagePixelFmt.BC5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC5;
                case TexImagePixelFmt.L4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.L4;
                case TexImagePixelFmt.L8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.L8;
                case TexImagePixelFmt.LA4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.LA4;
                case TexImagePixelFmt.LA8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.LA8;
                case TexImagePixelFmt.RGB565: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565;
                case TexImagePixelFmt.RGB565_INDIRECT: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565_INDIRECT;
                case TexImagePixelFmt.RGBA8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGBA8;
                default: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565;
            }
        }

        /// <summary>
        /// UIに表示する文字列を取得します。
        /// </summary>
        public string GetPixcelFormatString(TexImagePixelFmt fmt)
        {
            string result = fmt.ToString();
            switch (fmt)
            {
                case TexImagePixelFmt.RGB565:
                    result += string.Format(" : ({0}_RGB)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.RGB565_INDIRECT:
                    result += string.Format(" : ({0}_UVA)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.RGB8:
                case TexImagePixelFmt.RGB5A1:
                    result += string.Format(" : ({0})", LECoreStringResMgr.Get("TEXTURE_FMT_CONVERTTO_", "BC1"));
                    break;
                case TexImagePixelFmt.RGBA4:
                    result += string.Format(" : ({0})", LECoreStringResMgr.Get("TEXTURE_FMT_CONVERTTO_", "BC2"));
                    break;
                case TexImagePixelFmt.BC1:
                    result += " : (RGB+1A)";
                    break;
                case TexImagePixelFmt.BC2:
                    result += " : (RGB+4A)";
                    break;
                case TexImagePixelFmt.BC3:
                    result += " : (RGB+8A)";
                    break;
                case TexImagePixelFmt.BC4A:
                    result += " : (A)";
                    break;
                case TexImagePixelFmt.BC4L:
                    result += " : (I)";
                    break;
                case TexImagePixelFmt.BC5:
                    result += " : (I+A)";
                    break;
                case TexImagePixelFmt.RGBA8:
                    result += string.Format(" : ({0}_RGB+8A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.L4:
                case TexImagePixelFmt.L8:
                    result += string.Format(" : ({0}_I)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.A4:
                case TexImagePixelFmt.A8:
                    result += string.Format(" : ({0}_A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.LA4:
                case TexImagePixelFmt.LA8:
                    result += string.Format(" : ({0}_I+A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            return result;
        }

        public int CalcNativeTextureSize(int width, int height, TexImagePixelFmt fmt)
        {
            // DLL に丸ごと切り出します。
            // DLL に切り出されたら、この ifdef は不要になります。
#if NW_LYT_TOOLS_PLATFORM_CAFE
            return NW4F.LayoutBinaryConverter.GX2TextureWriter.GetGX2TextureSize(width, height, ConvetTexConvTexelFmt_(fmt));
#else
            return 0;
#endif
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<TexImagePixelFmt> GetGrayScaleFormatList(bool hasAlpha)
        {
            if (hasAlpha)
            {
                yield return TexImagePixelFmt.BC4A;
                yield return TexImagePixelFmt.BC5;
                yield return TexImagePixelFmt.A8;
                yield return TexImagePixelFmt.LA8;
                yield return TexImagePixelFmt.BC3;
                yield return TexImagePixelFmt.RGB565_INDIRECT;
            }
            else
            {
                yield return TexImagePixelFmt.BC1;
                yield return TexImagePixelFmt.BC4L;
                yield return TexImagePixelFmt.L8;
                yield return TexImagePixelFmt.RGB565;
                yield return TexImagePixelFmt.RGB8;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<TexImagePixelFmt> GetFormatList(bool hasAlpha, bool subFormat)
        {

            if (hasAlpha)
            {
                if (!subFormat)
                {
                    yield return TexImagePixelFmt.BC1;
                    yield return TexImagePixelFmt.BC2;
                    yield return TexImagePixelFmt.BC3;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.RGBA8;
                    yield return TexImagePixelFmt.RGB565_INDIRECT;

                    yield return TexImagePixelFmt.Unknown;

                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA4;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGB5A1;
                    yield return TexImagePixelFmt.RGBA4;
                }
                else
                {
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGBA8;
                }
            }
            else
            {
                if (!subFormat)
                {
                    yield return TexImagePixelFmt.BC1;
                    yield return TexImagePixelFmt.BC4L;

                    yield return TexImagePixelFmt.Unknown;

                    yield return TexImagePixelFmt.L8;
                    yield return TexImagePixelFmt.RGB565;
                    yield return TexImagePixelFmt.RGB8;
                }
                else
                {
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGBA8;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public TexImagePixelFmt GetDefaultTextureFormat(
            bool isGrayScale, bool intendedToBeAlphaFormat, TexImagePixelFmt nw4cTgaFormat)
        {
            // フォーマット初期値を決定します。
            TexImagePixelFmt fmt;
            if (isGrayScale)
            {
                if (intendedToBeAlphaFormat)
                {
                    fmt = TexImagePixelFmt.BC5;
                }
                else
                {
                    fmt = TexImagePixelFmt.BC4L;
                }
            }
            else
            {
                if (intendedToBeAlphaFormat)
                {
                    fmt = TexImagePixelFmt.BC3;
                }
                else
                {
                    fmt = TexImagePixelFmt.BC1;
                }
            }
            return fmt;
        }

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        public IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoColor(int stageNo)
        {
            yield return new TextureCombineModeDesc(AttrCombineMode.Replace, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_REPLACE"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Modulate, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_MODULATE"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Add, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADD"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Subtract, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_SUBTRACT"));

            yield return new TextureCombineModeDesc(AttrCombineMode.AddSigned, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADDSIGNED"));
            yield return new TextureCombineModeDesc(AttrCombineMode.AddMultDMP, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADDMULT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.MultAddDMP, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_MULTADD"));

            yield return new TextureCombineModeDesc(AttrCombineMode.Overlay, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_OVERLAY"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Indirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_INDIRECT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.BlendIndirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_BLENDINDIRECT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.EachIndirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_EACHINDIRECT"));
        }

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        public IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoAlpha()
        {
            yield return new TextureCombineModeDesc(AttrCombineMode.Replace, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ALPHA_MAX"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Modulate, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ALPHA_MIN"));
        }

        /// <summary>
        /// 利用可能なステージ数を取得します。
        /// </summary>
        public int GetNumAvailableStages(ITevStages tevStages)
        {
            return LayoutEditorCore.PlatformDetail.MaxTevStageCount;
        }

        /// <summary>
        /// 非２べき乗テクスチャでミラーやリピートが可能か？
        /// </summary>
        public bool CheckTextureMirrorRepeatEnabled(TexImagePixelFmt nw4cTgaFormat, int size)
        {
            return true;
        }

        /// <summary>
        /// エディタでのテクスチャフォーマット変更を許すか？
        /// </summary>
        public bool TextureFormatEditEnabled { get { return true; } }

        /// <summary>
        /// ウィンドウペインのカスタムフレームサイズ設定が有効かどうか
        /// </summary>
        public bool CustomWindowFrameSizeEnabled { get { return true; } }

        /// <summary>
        /// インダイレクトテクスチャが有効か
        /// </summary>
        public bool IndirectTextureEnabled { get { return true; } }

        /// <summary>
        ///
        /// </summary>
        public bool MultiScreenEnabled { get { return false; } }

        /// <summary>
        /// 詳細なマテリアルの編集は可能か？
        /// </summary>
        public bool LowLevelCombinerSettingsEnabled { get { return false; } }

        /// <summary>
        /// 最大TEVステージ数を取得します。
        /// </summary>
        public int MaxTevStageCount { get { return 2; } }

        /// <summary>
        /// TEVコンスタントカラーの数を取得します。
        /// </summary>
        public int TevConstantColorCount { get { return 0; } }

        /// <summary>
        /// プラットフォームに対応しているか？
        /// </summary>
        public bool IsAcceptableBntxPlatform(string platform)
        {
            return platform != null && platform.TrimEnd() == "Cafe";
        }
    }
}
