﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Runtime.InteropServices;
using System.Text;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using System.Collections.Generic;
using LECore.Structures;
using LECore.Structures.Core;
using System.Drawing;
using System.Reflection;
using LECore.Save_Load;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.InGameEditing.Communication;


namespace LECore
{
    /// <summary>
    /// ビューア実行クラスです。
    /// </summary>
    public class ViewerExecuter
    {
        //----------------------------------------------------------
        const string _ViewerCommandStringPrefix = "ViewerCtrlCmd_:";

        //----------------------------------------------------------

        ViewerPreviewParam _previewParam;
        static Process _pcViewerProcess = null;
        static AutoFilteredFontState _sAutoFilteredFontState = new AutoFilteredFontState();
        const int _MaxRetryCountForPCViewerLaunchCheck = 20;

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ViewerExecuter()
        {
        }


        /// <summary>
        ///
        /// </summary>
        public static void DisposeStaticResources()
        {
            if (_pcViewerProcess != null)
            {
                _pcViewerProcess.CloseMainWindow();
            }
        }

        /// <summary>
        /// ビューア転送スレッドでのエラーイベント
        /// </summary>
        public static event EventHandler ErrorOnViewerThread
        {
            add
            {
                lock (_ErrorOnViewerThreadLocker)
                {
                    _ErrorOnViewerThread += value;
                }
            }
            remove
            {
                lock (_ErrorOnViewerThreadLocker)
                {
                    _ErrorOnViewerThread -= value;
                }
            }
        }

        private static event EventHandler _ErrorOnViewerThread;
        private static object _ErrorOnViewerThreadLocker = new object();

        /// <summary>
        /// プレビューパラメータを設定します。
        /// </summary>
        static private bool SavePreviewTemporary_(
            ISubScene targetSubScene,
            ISubScene partsSubScene,
            ViewerPreviewParam previewParam,
            out string error)
        {
            error = string.Empty;
            try
            {
                // 不正なアニメーションがないかチェックし、削除結果をユーザに通知します。
                LayoutEditorCore.RepoertInactiveAnimations(targetSubScene);

                FileIOUserScriptHandler.StopUserScriptCall = true;

                ExportOption option = new ExportOption() { ConvertInvalidTextboxToNull = true };
                if (GlobalTime.Inst.UseTargetFrameSectionRangeAsPlayRange) { option.UseTargetFrameSectionRangeAsPlayRange = true; }
                if (GlobalTime.Inst.AutoAdjustPlayRange) { option.AutoAdjustPlayRange = true; }
                if (previewParam.IsAutoReload)
                {
                    // 値を変えない。
                    option.Frame = GlobalTime.Inst.Time;
                    option.AfterExport = ExportOption.AfterExportOption.RetriveValues;
                }
                else if (previewParam.Mode == ViewerPreviewParam.PreviewMode.Animation)
                {
                    // 0 フレーム目で評価した値を出力する。
                    option.Frame = 0;
                    option.AfterExport = ExportOption.AfterExportOption.ResetFrame;
                }
                else
                {
                    // 現在の時刻の値に設定しなおす。
                    option.Frame = GlobalTime.Inst.Time;
                }

                bool bTempSaveResult = LayoutEditorCore.ExportToFileAll(targetSubScene, previewParam.GetTemporaryLayoutDirName(), option);

                // 部品レイアウトの指定があれば、プレビュー用に一時保存する。
                bool isNeededToSavePartsLayout = partsSubScene != null && partsSubScene != targetSubScene;
                if (isNeededToSavePartsLayout)
                {
                    bTempSaveResult &= LayoutEditorCore.ExportToFileAll(partsSubScene, previewParam.GetTemporaryPartsLayoutDirName(), option);
                }

                if (bTempSaveResult)
                {
                    // 部品ペインがあれば、参照する部品レイアウトも一時フォルダにコピーします。
                    // 部品設定していないレイアウトを配置している場合もあるので、すべてをコピー対象にしておきます。
                    var partsFiles = LECore.LayoutEditorCore.Scene.EnumrateLayoutFiles().ToList();

                    // 一時的に保存された部品ファイルがあればそちらを優先的に利用するように、部品ファイルパスを変更する。
                    if (isNeededToSavePartsLayout)
                    {
                        string tempPartsFileName = previewParam.PartsLayoutFileNameWithoutExt + AppConstants.LayoutFileExt;
                        int partsPathIndex = partsFiles.FindIndex((filePath) => filePath.EndsWith(tempPartsFileName));
                        if (partsPathIndex != -1)
                        {
                            string tempPartsFilePath = previewParam.GetTemporaryPartsLayoutDirName() + AppConstants.LayoutFileExt;
                            partsFiles[partsPathIndex] = tempPartsFilePath;
                        }
                    }

                    List<string> referencedPartsFilePaths = new List<string>();
                    CopyPartsFileIfReferenced_(partsFiles, previewParam.InputDir, targetSubScene, referencedPartsFilePaths);

                    // 同名ファイルがあればエラー
                    if (referencedPartsFilePaths.Distinct().Any(x => string.Compare(Path.GetFileNameWithoutExtension(x), previewParam.LayoutFileNameWithoutExt.ToLower(), true) == 0))
                    {
                        error = string.Format(LECoreStringResMgr.Get("PREVIEW_SAME_NAME_PARTS"), previewParam.LayoutFileNameWithoutExt);
                        bTempSaveResult = false;
                    }
                }

                return bTempSaveResult;
            }
            finally
            {
                FileIOUserScriptHandler.StopUserScriptCall = false;
            }
        }

        /// <summary>
        /// シーンが参照している部品レイアウトをすべてコピーします。
        /// </summary>
        private static void CopyPartsFileIfReferenced_(IEnumerable<string> partsFiles, string dstDir, ISubScene subScene, List<string> referencedPartsFilePaths)
        {
            foreach (var partsFile in partsFiles)
            {
                string partsLayoutName = Path.GetFileName(partsFile);

                var referencingPanes = subScene.GetPaneSet(
                        (pane) =>
                            pane.PaneKind == PaneKind.Parts &&
                            pane.IPartsLayout.IsReferencingLayout(partsLayoutName));

                if (referencingPanes.Any())
                {
                    SceneHelper.CopyPartsLayoutFile(partsFile, dstDir, referencedPartsFilePaths);
                    var partsScene = LECore.LayoutEditorCore.Scene.PartsSubScenes.FirstOrDefault((p) => p.FilePath == partsFile);

                    // IsReferencingLayout でも再帰的な検索は行っている。
                    // 仮に pane 以下に含まれる部品の情報と Scene.PartsSubScenes に含まれる情報に違いがあれば、コピーされるものに違いがでる。
                    if (partsScene != null && partsScene.IsLoaded)
                    {
                        CopyPartsFileIfReferenced_(partsFiles, dstDir, partsScene.SubScene, referencedPartsFilePaths);
                    }
                }
            }
        }

        //----------------------------------------------------------
        void ShowPCViewerWindowToTop_()
        {
            var viewerWndProc = SearchPCViewerProcess_();
            if (viewerWndProc != null)
            {
                try
                {
                    LECore.Win32.User32.SetForegroundWindow(viewerWndProc.MainWindowHandle);
                }
                catch
                {
                }
            }
        }

        //----------------------------------------------------------
        string CreateArchieverArg_(string archiverArg, string previewTargetDir)
        {
            var arg = "-D \"" + _previewParam.GetBinaryOutputDirName() + "\" " + archiverArg + " \"" + Path.Combine(previewTargetDir, "layout.arc") + "\" . --keep-ftxb --keep-shader-variation-xml";

            // --tile-optimize
            {
                arg += " --tile-optimize " + _previewParam.PreviewTileOptimize;
                if (_previewParam.PreviewTileOptimize == "size_auto")
                {
                    arg += " --tile-size-threshold " + _previewParam.PreviewTileSizeThreshold.ToString();
                }
            }

            // --shader-cache-path
            arg += string.Format(" --shader-cache-path \"{0}\"", Util.TemporaryFileUtil.ShaderCacheRoot);

            return arg;
        }

        //----------------------------------------------------------
        void DoExecuteArchiver_(string archiverArg)
        {
            var errorBuilder = new StringBuilder();
            string error = "";
            var result = ProcessUtil.ProcessStart(GetArchiverPath_(), archiverArg, null, (s, e) => errorBuilder.Append(e.Data), out error, workingDirectory: _previewParam.GetBinaryOutputDirName());
            if (!result.HasValue || result.Value != 0)
            {
                var builder = new StringBuilder();
                builder.AppendLine(string.Format(LECoreStringResMgr.Get("ARCHIVER_ERROR_BADRESULTCODE"), result.HasValue ? result.Value.ToString() : string.Empty));
                if (!string.IsNullOrEmpty(error))
                {
                    builder.AppendLine(error);
                }
                builder.Append(errorBuilder.ToString());
                ShowErrorDialogAsync_(builder.ToString());
            }
        }

        //----------------------------------------------------------
        // メッセージ通知
        //----------------------------------------------------------
        /// <summary>
        /// Htcs経由で接続するビューアへの転送通知
        /// </summary>
        private bool SendMessageToHtcsViewer_(string protocol, string peerType, string archiverArg, string[] messageArgs)
        {
            // PCだったらビューアを自動転送時以外は最前面に
            if (_previewParam.Kind == ViewerPreviewParam.ViewerKind.PC)
            {
                // オートリロードでは最前面にしない。
                if (!_previewParam.IsAutoReload)
                {
                    ShowPCViewerWindowToTop_();
                }
            }

            string previewTargetDir = _previewParam.GetArcFilePath();
            {
                if (!Directory.Exists(previewTargetDir))
                {
                    Directory.CreateDirectory(previewTargetDir);
                }
            }

            if (!ExecutePreArchiverCommand(_previewParam.GetBinaryOutputDirName()))
            {
                return false;
            }

            using (var portMapper = new ConnectionManager())
            {
                portMapper.Start(p => p.Equals(protocol));

                // アーカイバーの実行
                DoExecuteArchiver_(CreateArchieverArg_(archiverArg, previewTargetDir));

                // 転送
                if (HtcsPreviewStarter.Start(portMapper, protocol, peerType, messageArgs, _previewParam.ConnectionRetryDuration) != 0)
                {
                    ShowErrorDialogAsync_(LECoreStringResMgr.Get("CONVERTER_ERROR_VIEWER_TIMEOUT_RUNVIEWER_HTCS"));
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ビューアへのデータ送信処理を行います。
        /// </summary>
        bool TransferToViewer_()
        {
            Debug.Assert(AppConstants.IsGfxMode);

            if (AppConstants.IsGfxMode)
            {
                // TODO: 空白を含むパスが正しく扱えないので要素ごとに分ける
                string option = _previewParam.GetPreviewNotifyCommand();
                string[] options = option.Split(' ');

                // プラグインが読み込んだプラットフォーム設定を適用します。
                string protocolPc;
                string pcPeerType;
                string protocolTarget;
                string targetPeerType;
                GetProtocolName_(_previewParam, out protocolPc, out pcPeerType, out protocolTarget, out targetPeerType);

                // PCでもTargetでもアーカイブの作成先は一緒なので、
                // HtcsPreviewStarter内の処理でディレクトリを作成する
                switch (_previewParam.Kind)
                {
                    case ViewerPreviewParam.ViewerKind.PC:
                        return SendMessageToHtcsViewer_(protocolPc, pcPeerType, GetArchiverAdditionalArguments(), options);
                    case ViewerPreviewParam.ViewerKind.Target:
                        return SendMessageToHtcsViewer_(protocolTarget, targetPeerType, GetArchiverAdditionalArguments(), options);
                    default: Ensure.Operation.True(false); break;
                }
            }

            return false;
        }

        /// <summary>
        /// ビューアーに任意のコマンド文字列を送信します。
        /// </summary>
        public static bool SendCommandToViewer(string command, ViewerPreviewParam previewParam)
        {
            previewParam.ViewerCommand = command;

            if (AppConstants.IsGfxMode)
            {
                EnqueueParam(previewParam);
                return true;
            }

            return false;
        }

        //----------------------------------------------------------
        // ビューア起動
        //----------------------------------------------------------

        /// <summary>
        /// 実機ビューア起動
        /// </summary>
        public static void LaunchTargetViewer(string lauchOption)
        {
            string filePath = GetTargetViewerLauncherPath_();
            if (!File.Exists(filePath))
            {
                ShowErrorDialogAsync_(LECore.LECoreStringResMgr.Get("PREVIEW_ERROR_CANTFIND_LAUNCHBAT", filePath));
                return;
            }

            if (!AppConstants.CheckTargetComEnviromentValid())
            {
                ShowErrorDialogAsync_(LECore.LECoreStringResMgr.Get("SYSTEM_TARGETCOMENV_WARNING", string.Join(", ", AppConstants.GetTargetComEnvList().ToList())));
                return;
            }

            try
            {
                Process proc = new Process();
                proc.StartInfo.FileName = filePath;
                if (RlytConverter.FloatColorWrite)
                {
                    proc.StartInfo.Arguments = "-hdr_color_buffer ";
                }
                proc.StartInfo.Arguments += lauchOption;
                proc.StartInfo.CreateNoWindow = false;
                proc.StartInfo.UseShellExecute = true;
                proc.Start();
            }
            catch (Exception exp)
            {
                ShowErrorDialogAsync_(exp.Message);
            }
        }

        /// <summary>
        /// PCビューアを起動します。
        /// </summary>
        bool LaunchPCViewer_(string lauchOption)
        {
            // それらしい名前で既存のウインドウを探す。
            _pcViewerProcess = SearchPCViewerProcess_();

            // 無ければ作る。
            if (_pcViewerProcess == null)
            {
                if(!File.Exists(_previewParam.PcViewerPath))
                {
                    ShowErrorDialogAsync_(LECore.LECoreStringResMgr.Get("VIEWERLAUNCH_ERROR") + Environment.NewLine + _previewParam.PcViewerPath);
                    return false;
                }

                _pcViewerProcess = new Process();
                _pcViewerProcess.StartInfo.FileName = _previewParam.PcViewerPath;
                _pcViewerProcess.StartInfo.CreateNoWindow = false;

                if (RlytConverter.FloatColorWrite)
                {
                    // LayoutViewer が最後の引数をレイアウトシーン名として扱っているため、オプション引数は前に接続する。
                    _pcViewerProcess.StartInfo.Arguments = "-hdr_color_buffer " + lauchOption;
                }
                else
                {
                    _pcViewerProcess.StartInfo.Arguments = lauchOption;
                }
                _pcViewerProcess.StartInfo.WorkingDirectory = Path.GetDirectoryName(_pcViewerProcess.StartInfo.FileName);
                _pcViewerProcess.EnableRaisingEvents = true;
                _pcViewerProcess.Exited += new EventHandler((sender, e) => _pcViewerProcess = null);

                try
                {
                    _pcViewerProcess.Start();

                    if (_previewParam.Mode != ViewerPreviewParam.PreviewMode.PCViewerScreenCapture)
                    {
                        // ウインドウができるまで待機します。
                        int retryCount = 0;
                        while (SearchPCViewerProcess_() == null && retryCount < _MaxRetryCountForPCViewerLaunchCheck)
                        {
                            Thread.Sleep(100);
                            retryCount++;
                        }

                        // 接続待ち時間を 5 秒にする
                        _previewParam.ConnectionRetryDuration = 5000;
                    }
                    else
                    {
                        // キャプチャモードではプロセスの終了を待機します。
                        while (!_pcViewerProcess.WaitForExit(1000))
                        {
                            if (requestViewerThreadEnd)
                            {
                                break;
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    _pcViewerProcess = null;
                    ShowErrorDialogAsync_(LECore.LECoreStringResMgr.Get("VIEWERLAUNCH_ERROR") + Environment.NewLine + e.Message);
                }
            }

            return _pcViewerProcess != null;
        }

        /// <summary>
        /// 実機ビューアーランチャーのパスを取得します。
        /// </summary>
        /// <remarks>UI スレッドから呼ぶこと</remarks>
        static string GetTargetViewerLauncherPath_()
        {
            // プラグインが読み込んだプラットフォーム設定を適用します。
            string path = AppConstants.RunCafeViewerBatPath;

            var prefPf = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            if (prefPf != null && prefPf.PlatformPreference != null)
            {
                path = Path.Combine(AppConstants.NwToolsRootPath, prefPf.PlatformPreference.TargetViewerRunnnerPath);
            }

            return path;
        }

        /// <summary>
        /// アーカイバーのパスを取得します。
        /// </summary>
        /// <remarks>現在はプラットフォームに依存しないので UI スレッドからもアクセス可能</remarks>
        string GetArchiverPath_()
        {
            // プラグインが読み込んだプラットフォーム設定を適用します。
            string path = Path.Combine(AppConstants.NwToolsRootPath, "LayoutTools\\LayoutArchiver.exe");

            if (_previewParam.PlatformPreference != null)
            {
                path = Path.Combine(AppConstants.NwToolsRootPath, _previewParam.PlatformPreference.ArchiverPath);
            }

            return path;
        }

        /// <summary>
        /// プロトコル名を取得します。
        /// </summary>
        /// <param name="protocolPc">PC用プロトコル名です</param>
        /// <param name="protocolTarget">実機用プロトコル名です</param>
        static void GetProtocolName_(ViewerPreviewParam previewParam, out string protocolPc, out string pcPeerType, out string protocolTarget, out string targetPeerType)
        {
            // プラグインが読み込んだプラットフォーム設定を適用します。
            if (previewParam.PlatformPreference != null)
            {
                protocolPc = previewParam.PlatformPreference.PcTransferProtocol;
                pcPeerType = previewParam.PlatformPreference.PcPeerType;
                protocolTarget = previewParam.PlatformPreference.TargetTransferProtocol;
                targetPeerType = previewParam.PlatformPreference.TargetPeerType;
            }
            else
            {
                protocolPc = "HTCS";
                pcPeerType = null;
                protocolTarget = "HTCS";
                targetPeerType = null;
            }

            pcPeerType = string.IsNullOrEmpty(pcPeerType) ? "UnknownPeerType" : pcPeerType;
            targetPeerType = string.IsNullOrEmpty(targetPeerType) ? "UnknownPeerType" : targetPeerType;
        }

        /// <summary>
        /// プラットフォーム名を取得します。
        /// </summary>
        /// <returns>プラットフォーム名です。</returns>
        string GetPlatformName_()
        {
            string archiverPlatform = "Win";
            if (_previewParam.PlatformPreference != null)
            {
                archiverPlatform = _previewParam.PlatformPreference.ArchiverPlatformName;
            }

            return archiverPlatform;
        }

        /// <summary>
        /// PC 用の Archiver の追加引数を指定します。
        /// </summary>
        string GetArchiverAdditionalArguments()
        {
            if (_previewParam.PlatformPreference != null)
            {
                return _previewParam.PlatformPreference.ArchiverAdditionalArguments ?? string.Empty;
            }

            return string.Empty;
        }

        /// <summary>
        /// PreArchiverCommand の追加引数を指定します。
        /// </summary>
        string GetPreArchiverCommandAdditionalArguments()
        {
            if (_previewParam.PlatformPreference != null)
            {
                return _previewParam.PlatformPreference.PreArchiverCommandAdditionalArguments ?? string.Empty;
            }

            return string.Empty;
        }

        /// <summary>
        /// アーカイバー実行前コマンドを実行します。
        /// </summary>
        bool ExecutePreArchiverCommand(string sourceDirectory)
        {
            var command = _previewParam.PreArchiverCommandPath;
            if (string.IsNullOrEmpty(command))
            {
                return true;
            }

            command = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, command);

            // 第一引数はディレクトリ
            var args = "\"" + sourceDirectory + "\"";
            var preArchiveCommandAdditionalArguments = GetPreArchiverCommandAdditionalArguments();
            if (!string.IsNullOrEmpty(preArchiveCommandAdditionalArguments))
            {
                args += " "+ preArchiveCommandAdditionalArguments;
            }

            var outputBuilder = new StringBuilder();
            string error;
            int? result = ProcessUtil.ProcessStart(command, args, (s, e) => outputBuilder.Append(e.Data), (s, e) => outputBuilder.Append(e.Data), out error);
            if (!result.HasValue || result != 0)
            {
                var builder = new StringBuilder();
                builder.AppendLine(LECoreStringResMgr.Get("PREARCHIVER_ERROR"));
                if (!string.IsNullOrEmpty(error))
                {
                    builder.AppendLine(error);
                }
                builder.Append(outputBuilder.ToString());
                ShowErrorDialogAsync_(builder.ToString());

                return false;
            }

            return true;
        }

        /// <summary>
        /// キャプチャのために、アーカイバーを実行します。
        /// </summary>
        void ExecArchiverForCapture_()
        {
            try
            {
                var source = Path.Combine(_previewParam.TempDir, "arc");

                Process proc = new Process();

                string dstArcPath = Path.Combine(AppConstants.NwToolsRootPath, "LayoutViewer\\Cafe\\data\\content\\");
                if (Directory.Exists(dstArcPath))
                {
                    Directory.Delete(dstArcPath, true);
                }
                Directory.CreateDirectory(dstArcPath);

                if (!ExecutePreArchiverCommand(source))
                {
                    return;
                }

                proc.StartInfo.FileName = GetArchiverPath_();
                proc.StartInfo.UseShellExecute = false;
                proc.StartInfo.CreateNoWindow = true;
                proc.StartInfo.Arguments = string.Format("{0} -D \"{1}\" \"{2}\" \".\"", GetArchiverAdditionalArguments(), source, Path.Combine(dstArcPath, "layout.arc"));
                proc.EnableRaisingEvents = true;
                proc.Start();

                if (!proc.WaitForExit(3000))
                {
                    ShowErrorDialogAsync_(LECoreStringResMgr.Get("ARCHIVER_ERROR_TIMEOUT"));
                }

                if (proc.ExitCode != 0)
                {
                    ShowErrorDialogAsync_(LECoreStringResMgr.Get("ARCHIVER_ERROR_BADRESULTCODE", proc.ExitCode));
                }
            }
            catch (System.Exception ex)
            {
                ShowErrorDialogAsync_(LECoreStringResMgr.Get("ARCHIVER_ERROR_BADRESULTCODE", ex.ToString()));
            }
        }

        /// <summary>
        /// ビューアへのデータ送信処理を行います。
        /// </summary>
        void LaunchPCViewerForCapture_()
        {
            // 複数のレイアウトの中から、表示するレイアウト名を指定します。
            string option = _previewParam.GetPreviewNotifyCommandForCapture();
            Debug.Assert(_previewParam.Kind == ViewerPreviewParam.ViewerKind.PC);

            if (_pcViewerProcess != null)
            {
                _pcViewerProcess.CloseMainWindow();
                _pcViewerProcess.WaitForExit(1000);
                _pcViewerProcess = null;
            }

            LaunchPCViewer_(option);
        }

        //----------------------------------------------------------
        // すでにあるビューアを探す
        //----------------------------------------------------------

        /// <summary>
        /// ビューアプロセスか？
        /// </summary>
        private static bool IsDefaultPCViewerProcess_(Process proc)
        {
            return IsPCViewerProcess_(proc, "Layout") && IsPCViewerProcess_(proc, "Viewer") && !proc.MainWindowTitle.Contains("Visual Studio");
        }

        /// <summary>
        /// ビューアプロセスか？
        /// </summary>
        private static bool IsPCViewerProcess_(Process proc, string hintString)
        {
            return
                !string.IsNullOrEmpty(proc.MainWindowTitle) &&
                proc.MainWindowTitle.IndexOfAny(Path.GetInvalidFileNameChars()) == -1 &&
                proc.MainWindowTitle.IndexOfAny(Path.GetInvalidPathChars()) == -1 &&
                Path.GetFileNameWithoutExtension(proc.MainWindowTitle).Contains(hintString);
        }

        /// <summary>
        ///
        /// </summary>
        private Process SearchPCViewerProcess_()
        {
            Process viewerWndProc = null;

            // カスタムPCビューアが指定されていれば探します。
            if (!string.IsNullOrEmpty(_previewParam.CustomPCViewerWindowTitleHint))
            {
                viewerWndProc = Process.GetProcesses().FirstOrDefault((p) => IsPCViewerProcess_(p, _previewParam.CustomPCViewerWindowTitleHint));
            }

            // 発見できなかった場合は、デフォルトのPCビューアを探します。
            if (viewerWndProc == null)
            {
                // x86 からは x64 プロセスの MainModule が取れない
                if (_previewParam.SearchPCViewerByTitle || !Environment.Is64BitProcess)
                {
                    viewerWndProc = Process.GetProcesses().FirstOrDefault((p) => IsDefaultPCViewerProcess_(p));
                }
                else
                {
                    var path = Path.GetFullPath(_previewParam.PcViewerPath).ToLowerInvariant();
                    var processName = Path.GetFileNameWithoutExtension(path).ToLowerInvariant();
                    var processes = Process.GetProcesses();
                    viewerWndProc = Process.GetProcesses().FirstOrDefault(p =>
                        {
                            try
                            {
                                return
                                    // 一つ目の条件によって、例外を減らす意図がある。
                                    p.ProcessName.ToLowerInvariant() == processName
                                    // 二つ目の条件で例外が頻発すると遅くなる。
                                    && p.MainModule.FileName.ToLowerInvariant() == path
                                    // ウインドウが生成されていないといけない
                                    && p.MainWindowHandle != IntPtr.Zero;
                            }
                            catch
                            {
                                return false;
                            }
                        }
                    );
                }
            }

            return viewerWndProc;
        }

        //----------------------------------------------------------
        // 処理本体
        //----------------------------------------------------------
        /// <summary>
        /// プレビュー
        /// </summary>
        static public bool RunPreviewer(
            ViewerPreviewParam previewSetting,
            ISubScene targetSubScene,
            string layoutFileName,
            ISubScene partsSubScene,
            string partsLayoutFileName,
            ViewerPreviewParam.TransferDestination destination,
            ViewerPreviewParam.PreviewMode mode)
        {
            try
            {
                // ツールが発見できない場合は処理を行いません。
                if (!AppConstants.IsToolEnviromentValid)
                {
                    ShowDialogByID_(previewSetting.ErrorDialogOwner, "SYSTEM_NW4F_ROOT_WARNING");
                    return false;
                }

                // 実機ビューアでは、SDK環境設定を確認する
                if (previewSetting.Kind == ViewerPreviewParam.ViewerKind.Target && !AppConstants.CheckTargetComEnviromentValid())
                {
                    ShowDialog_(previewSetting.ErrorDialogOwner, LECore.LECoreStringResMgr.Get("SYSTEM_TARGETCOMENV_WARNING", string.Join(", ", AppConstants.GetTargetComEnvList().ToList())));
                    return false;
                }

                //// パラメータを初期化
                previewSetting.DestinationScreen = destination;
                previewSetting.InitPreviewMode(targetSubScene, mode);
                previewSetting.LayoutFileNameWithoutExt = layoutFileName;
                previewSetting.PartsLayoutFileNameWithoutExt = partsLayoutFileName;
                previewSetting.LayoutSize = targetSubScene.BackGround.ScreenSize.AsSizeF.ToSize();
                previewSetting.BackgroundColor = targetSubScene.BackGround.BackGroundColor;
                if (targetSubScene.BackGround.BackgroundImage != null)
                {
                    previewSetting.BackgroundImagePath = targetSubScene.BackGround.BackgroundImage.FilePath;
                }
                else
                {
                    previewSetting.BackgroundImagePath = null;
                }

                // 情報の収集、フォント毎に使っている語句を収集していく。
                ISubScene[] filterTargetSubSceneSet = partsSubScene != null ? new ISubScene[] { targetSubScene, partsSubScene } : new ISubScene[] { targetSubScene };
                previewSetting.FontUsages = AutoFilteredFontUtil.CreateFontUsageData(filterTargetSubSceneSet);

                previewSetting.ConvertedImageFileNames = AutoFilteredFontUtil.FileNamesOfImageFontWithConvertSettings(filterTargetSubSceneSet)
                    .Concat(AutoFilteredFontUtil.FileNamesOfImageFontWithConvertSettingsInFcpx(filterTargetSubSceneSet)).Distinct().ToArray();

                // 一時フォルダの掃除
                previewSetting.CleanupTemporaryPath(true);

                // 一時ファイルを保存します。
                string error;
                bool result = SavePreviewTemporary_(targetSubScene, partsSubScene, previewSetting, out error);
                if (!result)
                {
                    if (!string.IsNullOrEmpty(error))
                    {
                        ShowDialog_(previewSetting.ErrorDialogOwner, error);
                    }
                    else
                    {
                        ShowDialogByID_(previewSetting.ErrorDialogOwner, "LAYOUT_ERROR_FAIL_SAVETEMP");
                    }
                    return false;
                }

                EnqueueParam(previewSetting);
            }
            catch (Exception e)
            {
                DbgConsole.WriteLine(e.ToString());
                ShowDialogByID_(previewSetting.ErrorDialogOwner, "LAYOUT_ERROR_FAIL_SAVETEMP");

                return false;
            }

            return true;
        }

        // ビューア転送スレッドの初期化
        public static void Initialize()
        {
            Util.TemporaryFileUtil.Initialize();

            viewerThread = new Thread(new ThreadStart(ViewerThreadMain))
            {
                CurrentCulture = Thread.CurrentThread.CurrentCulture,
                CurrentUICulture = Thread.CurrentThread.CurrentUICulture,
            };

            viewerThread.Start();
        }

        // ビューア転送スレッドの終了
        public static void Cleanup()
        {
            requestViewerThreadEnd = true;
            lock (viewerThreadLocker)
            {
                Monitor.Pulse(viewerThreadLocker);
            }
            viewerThread.Join();
            Util.TemporaryFileUtil.Destroy();
        }

        // ビューア転送スレッド
        private static Thread viewerThread;

        // ビューア転送スレッドとの同期用のオブジェクト
        internal static object viewerThreadLocker = new object();

        // 未送信の項目
        private static List<ViewerPreviewParam> waitingParams = new List<ViewerPreviewParam>();

        // 送信済みの項目
        private static Queue<ViewerPreviewParam> sendedParam = new Queue<ViewerPreviewParam>();

        // ビューア転送スレッドの終了要求がされたか
        private static volatile bool requestViewerThreadEnd = false;

        /// <summary>
        /// プレビューのキャンセル
        /// </summary>
        static public void CancelPreview()
        {
            lock (viewerThreadLocker)
            {
                // 自動転送用のものは削除する
                var items = waitingParams.ToArray();
                waitingParams.Clear();
                foreach (var item in items)
                {
                    if (item.IsAutoReload)
                    {
                        Util.TemporaryFileUtil.DeleteViewerTempDirectory(item.TempDir);
                    }
                    else
                    {
                        waitingParams.Add(item);
                    }
                }
            }
        }

        // 送信要求の追加
        private static void EnqueueParam(ViewerPreviewParam param)
        {
            lock (viewerThreadLocker)
            {
                if (param.IsAutoReload)
                {
                    // 末尾の自動転送は削除
                    if (waitingParams.Count > 0 &&
                        waitingParams[waitingParams.Count - 1].IsAutoReload)
                    {
                        Util.TemporaryFileUtil.DeleteViewerTempDirectory(waitingParams[waitingParams.Count - 1].TempDir);
                        waitingParams.RemoveAt(waitingParams.Count - 1);
                    }
                }

                waitingParams.Add(param);
                param.PushedToViewerThread = true;
                Monitor.Pulse(viewerThreadLocker);
            }
        }

        /// <summary>
        /// 処理準備ができるまで待機する。
        /// </summary>
        private static bool IsNeededToWaitForReady_(ViewerPreviewParam lastPreviewSetting)
        {
            if (requestViewerThreadEnd)
            {
                return false;
            }

            if (HtcsPreviewStarter.ErrorAync)
            {
                return false;
            }

            if (waitingParams.Count > 0)
            {
                bool condition = HtcsPreviewStarter.IsWaitingToReceiveResponse &&
                   waitingParams.First().PlatformPreference == lastPreviewSetting.PlatformPreference &&
                   waitingParams.First().Kind == lastPreviewSetting.Kind;

                if (!condition)
                {
                    return false;
                }
            }

            return true;
        }

        static void CleanUpTemporary_(Queue<ViewerPreviewParam> tasksToBeCleanedUpTemporaryQueue)
        {
            // 不要な情報を削除
            // 接続先の切り替えがなければ一つ残せば十分。
            // きわめて短時間に接続先の切り替えを繰り返すとエラーになる可能性はある。
            while (tasksToBeCleanedUpTemporaryQueue.Count() > 10)
            {
                var param = tasksToBeCleanedUpTemporaryQueue.Dequeue();
                Util.TemporaryFileUtil.DeleteViewerTempDirectory(param.TempDir);
            }
        }

        /// <summary>
        /// エラー報告
        /// </summary>
        private static void ReportError_()
        {
            ShowErrorDialogAsync_(LECoreStringResMgr.Get("ERROR_VIEWER_ACK_ERROR"));

            try
            {
                // メインスレッドを止めないためにロックを一時解除
                Monitor.Exit(viewerThreadLocker);
                lock (_ErrorOnViewerThreadLocker)
                {
                    _ErrorOnViewerThread?.Invoke(null, new EventArgs());
                }
            }
            finally
            {
                // 再ロック
                Monitor.Enter(viewerThreadLocker);
            }
        }

        /// <summary>
        /// 次のプレビューパラメーターをキューから取る。
        /// </summary>
        static ViewerPreviewParam DequeueParam_()
        {
            Debug.Assert(waitingParams.Count > 0);

            ViewerPreviewParam previewSetting = waitingParams.First();
            waitingParams.RemoveAt(0);

            DbgConsole.WriteLine(previewSetting.TempDir);

            return previewSetting;
        }

        /// <summary>
        /// ビューアー制御コマンドを送る。
        /// </summary>
        static bool SendViewerCommand_(ViewerPreviewParam previewSetting)
        {
            string protocol;
            string peerType;
            string protocolTarget;
            string targetPeerType;
            GetProtocolName_(previewSetting, out protocol, out peerType, out protocolTarget, out targetPeerType);
            protocol = previewSetting.Kind == ViewerPreviewParam.ViewerKind.PC ? protocol : protocolTarget;
            peerType = previewSetting.Kind == ViewerPreviewParam.ViewerKind.PC ? peerType : targetPeerType;

            using (var portMapper = new ConnectionManager())
            {
                portMapper.Start(p => p.Equals(protocol));

                // 転送
                string command = _ViewerCommandStringPrefix + " " + previewSetting.ViewerCommand;
                string[] commands = command.Split(' ');
                if (HtcsPreviewStarter.Start(portMapper, protocol, peerType, commands, previewSetting.ConnectionRetryDuration) != 0)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// プレビューを実行する
        /// </summary>
        static bool DoPreview_(ViewerPreviewParam previewSetting)
        {
            bool anyError = false;

            // 自動フィルタファイルを作成。
            _sAutoFilteredFontState.ClearMessages();
            AutoFilteredFontUtil.Create(previewSetting.FontUsages, _sAutoFilteredFontState, previewSetting);

            // フォント生成のエラーダイアログ表示
            var errors = _sAutoFilteredFontState.GetMessages(true).ToArray();
            if (errors.Any())
            {
                var errorBuilder = new StringBuilder();
                errorBuilder.AppendLine(LECoreStringResMgr.Get("WARNING_FONT_RESOURCE_CREATION_FOR_VIEWER"));
                foreach (var error in errors)
                {
                    errorBuilder.AppendLine("    " + error);
                }
                ShowErrorDialogAsync_(errorBuilder.ToString());

                anyError = true;
            }

            // 一時ファイルに、ScalableFont 用の調整をします。
            AdjustPreviewFlytForScalableFont_(previewSetting.FontUsages, previewSetting.InputDir, _sAutoFilteredFontState);

            // TODO: PlatformDetail への依存を消したい。現在は値は不変な名で問題ない
            // PC ビューアーでは、一時ファイルに、astc テクスチャ 用の調整をします（表示がサポートされていないので）。
            var replaceAstcToBc3 = previewSetting.Kind == ViewerPreviewParam.ViewerKind.PC || !LayoutEditorCore.PlatformDetail.IsGpuEncodingAvailable;
            var replaceBc7ToBc3 = !LayoutEditorCore.PlatformDetail.IsGpuEncodingAvailable;
            if (replaceAstcToBc3 || replaceBc7ToBc3)
            {
                AdjustPreviewFlytForTexture_(previewSetting.InputDir, replaceAstcToBc3, replaceBc7ToBc3);
            }

            // コンバイナエディタからの更新処理
            if (!string.IsNullOrEmpty(previewSetting.CombinerReplaceFile) &&
               !string.IsNullOrEmpty(previewSetting.CombinerReceivedTempFile))
            {
                AdjustPreviewFlytForCombinerEditor_(previewSetting.InputDir, previewSetting.CombinerReplaceFile, previewSetting.CombinerReceivedTempFile);
            }

            // プレビューの実行。
            ViewerExecuter viewerExecuter = new ViewerExecuter();
            if (!viewerExecuter.RunPreviewer_(previewSetting, previewSetting.ConvertedImageFileNames))
            {
                anyError = true;
            }

            return anyError;
        }

        /// <summary>
        /// Viewer転送スレッドメイン
        /// </summary>
        private static void ViewerThreadMain()
        {
            ViewerPreviewParam previewSetting = null;
            Queue<ViewerPreviewParam> tasksToBeCleanedUpTemporaryQueue = new Queue<ViewerPreviewParam>();
            bool anyError = false;

            while (true)
            {
                // 前回の送信に対する結果の非同期受信や UI スレッドからの送信要求を受け付ける
                lock (viewerThreadLocker)
                {
                    // 条件を満たされるまで待つ
                    while (IsNeededToWaitForReady_(previewSetting))
                    {
                        CleanUpTemporary_(tasksToBeCleanedUpTemporaryQueue);
                        // Pulse されるまで待つ
                        Monitor.Wait(viewerThreadLocker);
                    }

                    // 要求に応じてスレッドを終了する
                    if (requestViewerThreadEnd)
                    {
                        break;
                    }

                    // エラーを報告する
                    if (HtcsPreviewStarter.ErrorAync)
                    {
                        HtcsPreviewStarter.ErrorAync = false;
                        if (!anyError)
                        {
                            ReportError_();
                        }

                        // もう一度要求を確認する
                        continue;
                    }

                    // エラーフラグの初期化
                    // パラメータの設定
                    anyError = false;
                    previewSetting = DequeueParam_();
                }

                // 送信
                try
                {
                    // 種類によって処理を分岐します。
                    if (string.IsNullOrEmpty(previewSetting.ViewerCommand))
                    {
                        anyError = DoPreview_(previewSetting);
                        tasksToBeCleanedUpTemporaryQueue.Enqueue(previewSetting);
                    }
                    else
                    {
                        anyError = SendViewerCommand_(previewSetting);
                    }
                }
                catch (Exception e)
                {
                    Debug.Assert(false, "Unhandled Exception");
                    ShowErrorDialogAsync_(e.Message);
                    anyError = true;
                }

                // エラー通知
                if (anyError)
                {
                    lock (_ErrorOnViewerThreadLocker)
                    {
                        _ErrorOnViewerThread?.Invoke(null, new EventArgs());
                    }
                }

                previewSetting.OnFinished();
            }
        }

        /// <summary>
        /// スケーラブルフォントのために、プレビュー用 flyt を 調整します。
        /// </summary>
        private static void AdjustPreviewFlytForScalableFont_(IEnumerable<AutoFilteredFontUtil.FontCharUsageData> usages, string previewInputDir, AutoFilteredFontState autoFilteredFontState)
        {
            // 自動フィルタフォントが実際に利用されている、スケーラブルフォントか判定します。
            Func<AutoFilteredFontEntry, bool> isAutoFilteredFontEntryUsedScFont_ = (entry) =>
            {
                if (!entry.IsSrcScalableFont())
                {
                    return false;
                }

                string fontName = Path.GetFileNameWithoutExtension(entry.DstFontFilePath);
                return usages.Any((usage) => usage.FontName == fontName);
            };

            // 実際に使われているスケーラブルフォントが一つも含まれない場合は中断。
            if (!autoFilteredFontState.Entry.Any((font) => isAutoFilteredFontEntryUsedScFont_(font)))
            {
                return;
            }

            // 使っているのだけ保存するように変更する。
            // 中間ファイルを調整して、ttf フォントを bffnt を指すようにする。
            // 実際には、ttf フォント をまったく処理していないファイルでも処理する必要があります。
            foreach (var flytFile in Directory.GetFiles(previewInputDir, "*.flyt", SearchOption.AllDirectories))
            {
                FontManager dummyFontMgr = new FontManager();

                string content = File.ReadAllText(flytFile);

                foreach (AutoFilteredFontEntry fontEntry in autoFilteredFontState.Entry.Where((ent) => ent.IsSrcScalableFont()))
                {
                    // フォントファイル名を変更します。
                    {
                        string pattern = string.Format("<fontFile path=\".+?{0}\" name=\"{1}\"", Path.GetFileName(fontEntry.SrcFontFilePath), Path.GetFileNameWithoutExtension(fontEntry.DstFontFilePath));
                        string newPath = string.Format("<fontFile path=\"{0}\"", fontEntry.DstFontFilePath);

                        Regex rgx = new Regex(pattern);
                        content = rgx.Replace(content, newPath, 1);
                    }

                    // 差し替えたフォントを使っている、テキストボックスペインのフォントサイズを変更します。
                    using (var newFont = FontFactory.BuildFontForPreview(Path.GetFileName(fontEntry.SrcFontFilePath), fontEntry.LinearDstFontFilePath ?? fontEntry.DstFontFilePath))
                    {
                        // フォントを使っているテキストに、サイズ調整を行う。
                        // 当該フォントを使っているテキストボックスの XML を fontSize や fontSizeOriginal を含めてキャプチャする 正規表現
                        string patternTextBox = string.Format("(<textBox font=\"{0}\".+? )<fontSize x=\"(.+?)\" y=\"(.+?)\"(.+?)<fontSizeOriginal x=\"(.+?)\" y=\"(.+?)\"", Path.GetFileNameWithoutExtension(fontEntry.DstFontFilePath));

                        // マッチした部分 XML について、計算を施し fontSize の値を差し替えて戻すハンドラ
                        Func<Match, string> matchEvaluator = (match) =>
                        {
                            float fontW = float.Parse(match.Groups[2].Value);
                            float fontH = float.Parse(match.Groups[3].Value);
                            float fontOrigW = float.Parse(match.Groups[5].Value);
                            float fontOrigH = float.Parse(match.Groups[6].Value);

                            string originalXML1 = match.Groups[1].Value;
                            string originalXML2 = match.Groups[4].Value;

                            fontW *= (float)newFont.Width / fontOrigW;
                            fontH *= (float)newFont.Height / fontOrigH;

                            return string.Format("{0}<fontSize x=\"{1}\" y=\"{2}\"{3}<fontSizeOriginal x=\"{4}\" y=\"{5}\"", originalXML1, fontW, fontH, originalXML2, fontOrigW, fontOrigH);
                        };

                        content = new Regex(patternTextBox, RegexOptions.Singleline).Replace(content, new MatchEvaluator(matchEvaluator));
                    }
                }

                File.WriteAllText(flytFile, content);
            }
        }

        /// <summary>
        /// プレビュー用 flyt 中のテクスチャフォーマットを調整します。
        /// </summary>
        private static void AdjustPreviewFlytForTexture_(string previewInputDir, bool replaceAstcToBc3, bool replaceBc7toBc3)
        {
            if (!replaceAstcToBc3 && !replaceBc7toBc3)
            {
                return;
            }

            // 中間ファイルを調整して、astc テクスチャを利用しないように修正します。
            foreach (var flytFile in Directory.GetFiles(previewInputDir, "*.flyt", SearchOption.AllDirectories))
            {
                string content = File.ReadAllText(flytFile);

                // テクスチャのフォーマットを変更します。
                {
                    string astcPattern = string.Format("format=\"ASTC.+?\"");
                    string bc7Pattern = string.Format("format=\"BC7\"");
                    string newFormat = string.Format("format=\"BC3\"");
                    string pattern = replaceAstcToBc3 ? astcPattern : "";
                    if (replaceBc7toBc3)
                    {
                        if (string.IsNullOrEmpty(pattern))
                        {
                            pattern = bc7Pattern;
                        }
                        else
                        {
                            pattern = "(" + pattern + ")|(" + bc7Pattern + ")";
                        }
                    }

                    Regex rgx = new Regex(pattern);
                    content = rgx.Replace(content, newFormat);
                }

                File.WriteAllText(flytFile, content);
            }
        }

        /// <summary>
        /// プレビュー用 flyt 中のコンバイナエディタから受信した コンバイナファイルの書き換えを行います。
        /// </summary>
        private static void AdjustPreviewFlytForCombinerEditor_(string previewInputDir, string replaceFile, string receivedTempFile)
        {
            // 中間ファイルを調整して、コンバイナファイルをテンポラリ用のものを利用するように修正します。
            foreach (var flytFile in Directory.GetFiles(previewInputDir, "*.flyt", SearchOption.AllDirectories))
            {
                // 相対パスへの変換(※同一ドライブにレイアウトと コンバイナファイルが存在する場合は相対パスで保存がされています)
                string replaceRelativePath = LECore.Win32.Utility.GetRelativePath(flytFile, replaceFile);

                string content = File.ReadAllText(flytFile);

                string pattern = string.Format("<CombinerUserShader fileName=\"{0}\" /", replaceRelativePath);
                string replace = string.Format("<CombinerUserShader fileName=\"{0}\" />", receivedTempFile);

                string replaceStrings = content.Replace(pattern, replace);
                File.WriteAllText(flytFile, replaceStrings);
            }
        }

        /// <summary>
        /// ビューアを実行します。
        /// </summary>
        bool RunPreviewer_(ViewerPreviewParam previewParam, IEnumerable<string> convertedImageFileNames)
        {
            _previewParam = previewParam;

            return RunPreviewerImpl_(convertedImageFileNames);
        }

        /// <summary>
        /// プレビューの実行をします。
        /// </summary>
        bool RunPreviewerImpl_(IEnumerable<string> convertedImageFileNames)
        {
            try
            {
                // 実機プレビューのためのデータ転送中であることを示す、
                // ダイアログを表示します。
                if (_previewParam.ProgressDialog != null)
                {
                    _previewParam.ProgressDialog.Show();
                    _previewParam.ProgressDialog.Refresh();
                }

                //-----------------------------------------------------------------
                // バイナリデータを準備します。
                //-----------------------------------------------------------------
                ViewerBinaryCreator previewBinaryCreator = new ViewerBinaryCreator();
                if (!previewBinaryCreator.PrepareBinaryDataForPreview(_previewParam, convertedImageFileNames))
                {
                    return false;
                }

                if (_previewParam.Mode == ViewerPreviewParam.PreviewMode.PCViewerScreenCapture)
                {
                    if (_previewParam.CaptureFrameCount < GlobalTime.Inst.AnimPlayStartTime)
                    {
                        ShowErrorDialogAsync_(LECore.LECoreStringResMgr.Get("INVALID_CAPTUREFRAME_ERROR", _previewParam.CaptureFrameCount, GlobalTime.Inst.AnimPlayStartTime));
                    }

                    if (AppConstants.IsGfxMode)
                    {
                        // アーカイーバーを実行します。（bntx の統合を行うために実行が必要です）
                        ExecArchiverForCapture_();
                    }

                    LaunchPCViewerForCapture_();
                }
                else
                {
                    //-----------------------------------------------------------------
                    // ビューアを起動します。
                    //-----------------------------------------------------------------
                    if (_previewParam.Kind == ViewerPreviewParam.ViewerKind.PC)
                    {
                        if(!LaunchPCViewer_(_previewParam.GetLaunchPCViewerCommand()))
                        {
                            return false;
                        }
                    }

                    //-----------------------------------------------------------------
                    // バイナリデータをビューアに転送します。
                    //-----------------------------------------------------------------
                    if (!TransferToViewer_())
                    {
                        return false;
                    }
                }
            }
            catch (Exception e)
            {
                ShowErrorDialogAsync_(e.Message);
            }
            finally
            {
                if( _previewParam.ProgressDialog != null )
                {
                    _previewParam.ProgressDialog.Close();
                }
            }

            return true;
        }

        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        private static void ShowDialogByID_(IWin32Window owner, string msgID)
        {
            ShowDialog_(owner, LECoreStringResMgr.Get(msgID));
        }

        /// <summary>
        ///
        /// </summary>
        private static void ShowDialog_(IWin32Window owner, string message)
        {
            if (owner == null)
            {
                return;
            }

            MessageBox.Show(
                owner,
                message,
                LECoreStringResMgr.Get("LAYOUT_WARNING_DLG_TITLE"),
                MessageBoxButtons.OK);
        }

        /// <summary>
        /// エラーダイアログを表示します。
        /// </summary>
        private static void ShowErrorDialogAsync_(string message)
        {
            // PREVIEW_ERROR_DLG_TITLE "MCS 通信エラー - NW4R LayoutEditor"
            string title = LECoreStringResMgr.Get("PREVIEW_ERROR_DLG_TITLE");
            LayoutEditorCore.MsgReporter.OnShowMessageDialogAsync(title, message);
        }
    }
}
