﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;

namespace LECore.Structures
{
    /// <summary>
    /// 水平位置
    /// </summary>
    public enum HorizontalLocation
    {
        Left,
        Center,
        Right
    }

    /// <summary>
    /// 垂直位置
    /// </summary>
    public enum VerticalLocation
    {
        Top,
        Center,
        Bottom
    }


    /// <summary>
    /// 2次元ベクタ(int)
    /// </summary>
    public struct IVec2
    {
        int _x;
        int _y;

        public int X
        {
            get{ return  _x;}
            set{ _x = value;}
        }

        public int Y
        {
            get{ return  _y;}
            set{ _y = value;}
        }



        public IVec2( int x, int y )
        {
            _x = x;
            _y = y;
        }

        public override string ToString()
        {
            return string.Format( "[{0},{1}]", _x, _y );
        }

        public IEnumerable<int> AsIEnumerabe()
        {
            yield return _x;
            yield return _y;
        }
    }

    /// <summary>
    /// 3次元ベクタ(int)
    /// </summary>
    public struct IVec3
    {
        int _x;
        int _y;
        int _z;

        public int X
        {
            get{ return  _x;}
            set{ _x = value;}
        }

        public int Y
        {
            get{ return  _y;}
            set{ _y = value;}
        }

        public int Z
        {
            get{ return  _z;}
            set{ _z = value;}
        }


        public IVec3( int x, int y, int z )
        {
            _x = x;
            _y = y;
            _z = z;
        }

        public override string ToString()
        {
            return string.Format( "[{0},{1},{2}]", _x, _y, _z );
        }

        public IEnumerable<int> AsIEnumerable()
        {
            yield return _x;
            yield return _y;
            yield return _z;
        }
    }

    /// <summary>
    /// 2次元ベクタ(float)
    /// </summary>
    public struct FVec2
    {
        float _x;
        float _y;

        public float X
        {
            get{ return  _x;}
            set{ _x = value;}
        }

        public float Y
        {
            get{ return  _y;}
            set{ _y = value;}
        }

        public FVec2( float x, float y )
        {
            _x = x;
            _y = y;
        }

        public FVec2( int x, int y )
            : this( Convert.ToSingle( x ), Convert.ToSingle( y ) )
        {

        }

        public FVec2( decimal x, decimal y )
            :this( Convert.ToSingle( x ), Convert.ToSingle( y ) )
        {

        }

        public FVec2( FVec2 vec2 )
        {
            _x = vec2.X;
            _y = vec2.Y;
        }

        public FVec2( FVec3 vec3 )
        {
            _x = vec3.X;
            _y = vec3.Y;
        }

        public FVec2( Point p )
        {
            _x = p.X;
            _y = p.Y;
        }

        public FVec2( PointF p )
        {
            _x = p.X;
            _y = p.Y;
        }

        public FVec2( Size   sz )
        {
            _x = sz.Width;
            _y = sz.Height;
        }

        public FVec2( IVec3   p )
        {
            _x = p.X;
            _y = p.Y;
        }

        public void Scale( float scale )
        {
            _x *= scale;
            _y *= scale;
        }

        static public FVec2 operator + ( FVec2 p0, FVec2 p1 )
        {
            return new FVec2( p0.X + p1.X, p0.Y + p1.Y );
        }

        static public FVec2 operator - ( FVec2 p0, FVec2 p1 )
        {
            return new FVec2( p0.X - p1.X, p0.Y - p1.Y );
        }

        static public FVec2 operator *(FVec2 p, float s)
        {
            return new FVec2( p.X * s, p.Y * s );
        }

        static public FVec2 operator *(float s, FVec2 p)
        {
            return p * s;
        }

        /// <summary>
        /// ベクトルの長さを所得します。
        /// </summary>
        public float Length
        {
            get{ return (float)Math.Sqrt( _x * _x + _y * _y );}
        }

        static public bool operator == ( FVec2 p0, FVec2 p1 )
        {
            return p0.X == p1.X && p0.Y == p1.Y;
        }

        static public bool operator != ( FVec2 p0, FVec2 p1 )
        {
            return !(p0 == p1);
        }

        public override bool Equals( object obj )
        {
            if (obj is FVec2)
            {
                FVec2 another = (FVec2) obj;
                return another == this;
            }
            return false;
        }

        public override int GetHashCode()
        {
            return (int)this.X ^ (int)this.Y;
        }


        public PointF AsPointF
        {
            get{ return new PointF( _x, _y );}
            set{ _x = value.X;_y = value.Y;}
        }

        public SizeF AsSizeF
        {
            get { return new SizeF( _x, _y ); }
            set { _x = value.Width; _y = value.Height; }
        }

        public IEnumerable<float> AsIEnumerable()
        {
            yield return _x;
            yield return _y;
        }

        public bool IsZero
        {
            get { return this.X == 0.0f && this.Y == 0.0f; }
        }

        public void Set( float x, float y )
        {
           _x = x;
           _y = y;
        }

        public void Set( FVec2 src )
        {
            _x = src.X;
            _y = src.Y;
        }

        public override string ToString()
        {
            return string.Format( "[{0:f2},{1:f2}]", _x, _y );
        }

        static public readonly FVec2 Empty = new FVec2( 0.0f, 0.0f );
        static public readonly FVec2 One   = new FVec2( 1.0f, 1.0f );

        static public readonly FVec2 Up    = new FVec2(  0.0f,  1.0f );
        static public readonly FVec2 Down  = new FVec2(  0.0f, -1.0f );
        static public readonly FVec2 Left  = new FVec2( -1.0f,  0.0f );
        static public readonly FVec2 Right = new FVec2(  1.0f,  0.0f );
    }

    /// <summary>
    /// ３次元ベクタ(float)
    /// </summary>
    public struct FVec3
    {
        float _x;
        float _y;
        float _z;

        public float X
        {
            get{ return  _x;}
            set{ _x = value;}
        }

        public float Y
        {
            get{ return  _y;}
            set{ _y = value;}
        }

        public float Z
        {
            get{ return  _z;}
            set{ _z = value;}
        }

        public FVec2 FVec2
        {
            get{ return new FVec2( this.X, this.Y ); }
        }

        public FVec3( PointF v )
        {
            _x = v.X;
            _y = v.Y;
            _z = 0.0f;
        }

        public FVec3(SizeF v)
        {
            _x = v.Width;
            _y = v.Height;
            _z = 0.0f;
        }

        public FVec3( float x, float y, float z )
        {
            _x = x;
            _y = y;
            _z = z;
        }

        public FVec3( decimal x, decimal y, decimal z )
            :this( Convert.ToSingle( x ), Convert.ToSingle( y ), Convert.ToSingle( z )  )
        {

        }

        public FVec3( FVec2 vec2 )
        {
            _x = vec2.X;
            _y = vec2.Y;
            _z = 0.0f;
        }

        public FVec3( FVec3 vec3 )
        {
            _x = vec3.X;
            _y = vec3.Y;
            _z = vec3.Z;
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します。
        /// </summary>
        public float this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    case 1:
                        return this.Y;
                    case 2:
                        return this.Z;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    case 1:
                        this.Y = value;
                        break;
                    case 2:
                        this.Z = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        public FVec3 Scale(float factor)
        {
            _x *= factor;
            _y *= factor;
            _z *= factor;

            return new FVec3(this);
        }

        public bool IsZero
        {
            get { return this.X == 0.0f && this.Y == 0.0f && this.Z == 0.0f; }
        }

        public FVec3 Scale(FVec3 scale)
        {
            _x *= scale.X;
            _y *= scale.Y;
            _z *= scale.Z;

            return new FVec3(this);
        }

        static public bool operator ==( FVec3 p0, FVec3 p1 )
        {
            return p0.X == p1.X && p0.Y == p1.Y && p0.Z == p1.Z;
        }

        static public bool operator !=( FVec3 p0, FVec3 p1 )
        {
            return !( p0 == p1 );
        }

        public override bool Equals( object obj )
        {
            if( obj is FVec3 )
            {
                FVec3 another = (FVec3)obj;
                return another == this;
            }
            return false;
        }

        public void Set(FVec3 src)
        {
            _x = src.X;
            _y = src.Y;
            _z = src.Z;
        }

        public override int GetHashCode()
        {
            return (int)this.X ^ (int)this.Y ^ (int)this.Z;
        }

        public static FVec3 operator + ( FVec3 lhs, FVec3 rhs )
        {
            return new FVec3( lhs.X + rhs.X, lhs.Y + rhs.Y, lhs.Z + rhs.Z );
        }

        public static FVec3 operator - ( FVec3 lhs, FVec3 rhs )
        {
            return new FVec3( lhs.X - rhs.X, lhs.Y - rhs.Y, lhs.Z - rhs.Z );
        }

        public override string ToString()
        {
            return string.Format( "[{0:f2},{1:f2},{2:f2}]", _x, _y, _z );
        }

        static public readonly FVec3 Empty = new FVec3( 0.0f, 0.0f, 0.0f );
        static public readonly FVec3 One   = new FVec3( 1.0f, 1.0f, 1.0f );

        public IEnumerable<float> AsIEnumerable()
        {
            yield return _x;
            yield return _y;
            yield return _z;
        }
    }

    /// <summary>
    /// ４次元ベクタ(float)
    /// </summary>
    public struct FVec4
    {
        float _x;
        float _y;
        float _z;
        float _w;

        public float X
        {
            get { return _x; }
            set { _x = value; }
        }

        public float Y
        {
            get { return _y; }
            set { _y = value; }
        }

        public float Z
        {
            get { return _z; }
            set { _z = value; }
        }

        public float W
        {
            get { return _w; }
            set { _w = value; }
        }

        public FVec2 FVec2
        {
            get { return new FVec2(this.X, this.Y); }
        }

        public FVec4(PointF v)
        {
            _x = v.X;
            _y = v.Y;
            _z = 0.0f;
            _w = 0.0f;
        }

        public FVec4(SizeF v)
        {
            _x = v.Width;
            _y = v.Height;
            _z = 0.0f;
            _w = 0.0f;
        }

        public FVec4(float x, float y, float z)
        {
            _x = x;
            _y = y;
            _z = z;
            _w = 0.0f;
        }

        public FVec4(float x, float y, float z, float w)
        {
            _x = x;
            _y = y;
            _z = z;
            _w = w;
        }

        public FVec4(decimal x, decimal y, decimal z)
            : this(Convert.ToSingle(x), Convert.ToSingle(y), Convert.ToSingle(z))
        {
        }

        public FVec4(decimal x, decimal y, decimal z, decimal w)
            : this(Convert.ToSingle(x), Convert.ToSingle(y), Convert.ToSingle(z), Convert.ToSingle(w))
        {
        }

        public FVec4(FVec2 vec2)
        {
            _x = vec2.X;
            _y = vec2.Y;
            _z = 0.0f;
            _w = 0.0f;
        }

        public FVec4(FVec3 vec3)
        {
            _x = vec3.X;
            _y = vec3.Y;
            _z = vec3.Z;
            _w = 0.0f;
        }

        public FVec4(FVec4 vec4)
        {
            _x = vec4.X;
            _y = vec4.Y;
            _z = vec4.Z;
            _w = vec4.W;
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します。
        /// </summary>
        public float this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    case 1:
                        return this.Y;
                    case 2:
                        return this.Z;
                    case 3:
                        return this.W;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    case 1:
                        this.Y = value;
                        break;
                    case 2:
                        this.Z = value;
                        break;
                    case 3:
                        this.W = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        public FVec4 Scale(float factor)
        {
            _x *= factor;
            _y *= factor;
            _z *= factor;
            _w *= factor;

            return new FVec4(this);
        }

        public bool IsZero
        {
            get { return this.X == 0.0f && this.Y == 0.0f && this.Z == 0.0f && this.W == 0.0f; }
        }

        public FVec4 Scale(FVec3 scale)
        {
            _x *= scale.X;
            _y *= scale.Y;
            _z *= scale.Z;

            return new FVec4(this);
        }

        public FVec4 Scale(FVec4 scale)
        {
            _x *= scale.X;
            _y *= scale.Y;
            _z *= scale.Z;
            _w *= scale.W;

            return new FVec4(this);
        }

        static public bool operator ==(FVec4 p0, FVec4 p1)
        {
            return p0.X == p1.X && p0.Y == p1.Y && p0.Z == p1.Z && p0.W == p1.W;
        }

        static public bool operator !=(FVec4 p0, FVec4 p1)
        {
            return !(p0 == p1);
        }

        public override bool Equals(object obj)
        {
            if (obj is FVec4)
            {
                FVec4 another = (FVec4)obj;
                return another == this;
            }
            return false;
        }

        public void Set(FVec4 src)
        {
            _x = src.X;
            _y = src.Y;
            _z = src.Z;
            _w = src.W;
        }

        public override int GetHashCode()
        {
            return (int)this.X ^ (int)this.Y ^ (int)this.Z ^ (int)this.W;
        }

        public static FVec4 operator +(FVec4 lhs, FVec4 rhs)
        {
            return new FVec4(lhs.X + rhs.X, lhs.Y + rhs.Y, lhs.Z + rhs.Z, lhs.W + rhs.W);
        }

        public static FVec4 operator -(FVec4 lhs, FVec4 rhs)
        {
            return new FVec4(lhs.X - rhs.X, lhs.Y - rhs.Y, lhs.Z - rhs.Z, lhs.W - rhs.W);
        }

        public override string ToString()
        {
            return string.Format("[{0:f2},{1:f2},{2:f2},{3:f3}]", _x, _y, _z, _w);
        }

        static public readonly FVec4 Empty = new FVec4(0.0f, 0.0f, 0.0f, 0.0f);
        static public readonly FVec4 One = new FVec4(1.0f, 1.0f, 1.0f, 1.0f);

        public IEnumerable<float> AsIEnumerable()
        {
            yield return _x;
            yield return _y;
            yield return _z;
            yield return _w;
        }
    }

    /// <summary>
    /// 3 × 4 行列を表します。
    /// </summary>
    public class Matrix34
    {
        //-----------------------------------------------------------------
        private const int MatrixRowCount = 3;
        private const int MatrixColumnCount = 4;
        private readonly float[,] m = new float[MatrixRowCount, MatrixColumnCount]
            {
                { 1f, 0f, 0f, 0f },
                { 0f, 1f, 0f, 0f },
                { 0f, 0f, 1f, 0f },
            };

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Matrix34()
        {
        }

        //-----------------------------------------------------------------
        // 行列の成分の取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したインデックス位置の成分へのアクセスを行うインデクサです。
        /// </summary>
        /// <param name="row">アクセスする成分の行の 0 から始まるインデックスです。</param>
        /// <param name="column">アクセスする成分の列の 0 から始まるインデックスです。</param>
        /// <returns>指定したインデックス位置の成分です。</returns>
        public float this[int row, int column]
        {
            get
            {
                return this.m[row, column];
            }

            set
            {
                this.m[row, column] = value;
            }
        }

        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <remarks>
        /// 4 行目の成分が (0, 0, 0, 1) であるとして演算します。
        /// </remarks>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix34 operator *(Matrix34 lhs, Matrix34 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            // RevolutionSDK 準拠 mtx.c C_MTXConcat
            Matrix34 mtx = new Matrix34();

            mtx[0, 0] = (lhs[0, 0] * rhs[0, 0]) + (lhs[0, 1] * rhs[1, 0]) + (lhs[0, 2] * rhs[2, 0]);
            mtx[0, 1] = (lhs[0, 0] * rhs[0, 1]) + (lhs[0, 1] * rhs[1, 1]) + (lhs[0, 2] * rhs[2, 1]);
            mtx[0, 2] = (lhs[0, 0] * rhs[0, 2]) + (lhs[0, 1] * rhs[1, 2]) + (lhs[0, 2] * rhs[2, 2]);
            mtx[0, 3] = (lhs[0, 0] * rhs[0, 3]) + (lhs[0, 1] * rhs[1, 3]) + (lhs[0, 2] * rhs[2, 3]) + lhs[0, 3];

            mtx[1, 0] = (lhs[1, 0] * rhs[0, 0]) + (lhs[1, 1] * rhs[1, 0]) + (lhs[1, 2] * rhs[2, 0]);
            mtx[1, 1] = (lhs[1, 0] * rhs[0, 1]) + (lhs[1, 1] * rhs[1, 1]) + (lhs[1, 2] * rhs[2, 1]);
            mtx[1, 2] = (lhs[1, 0] * rhs[0, 2]) + (lhs[1, 1] * rhs[1, 2]) + (lhs[1, 2] * rhs[2, 2]);
            mtx[1, 3] = (lhs[1, 0] * rhs[0, 3]) + (lhs[1, 1] * rhs[1, 3]) + (lhs[1, 2] * rhs[2, 3]) + lhs[1, 3];

            mtx[2, 0] = (lhs[2, 0] * rhs[0, 0]) + (lhs[2, 1] * rhs[1, 0]) + (lhs[2, 2] * rhs[2, 0]);
            mtx[2, 1] = (lhs[2, 0] * rhs[0, 1]) + (lhs[2, 1] * rhs[1, 1]) + (lhs[2, 2] * rhs[2, 1]);
            mtx[2, 2] = (lhs[2, 0] * rhs[0, 2]) + (lhs[2, 1] * rhs[1, 2]) + (lhs[2, 2] * rhs[2, 2]);
            mtx[2, 3] = (lhs[2, 0] * rhs[0, 3]) + (lhs[2, 1] * rhs[1, 3]) + (lhs[2, 2] * rhs[2, 3]) + lhs[2, 3];

            return mtx;
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(Matrix34 src)
        {
            for (int i = 0; i < MatrixRowCount; i++)
            {
                for (int j = 0; j < MatrixColumnCount; j++)
                {
                    this.m[i, j] = src.m[i, j];
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SharpDX.Matrix ToSharpDXMatrix34()
        {
            SharpDX.Matrix dst = SharpDX.Matrix.Identity;

            dst.M11 = this.m[0, 0];
            dst.M21 = this.m[0, 1];
            dst.M31 = this.m[0, 2];
            dst.M41 = this.m[0, 3];

            dst.M12 = this.m[1, 0];
            dst.M22 = this.m[1, 1];
            dst.M32 = this.m[1, 2];
            dst.M42 = this.m[1, 3];

            dst.M13 = this.m[2, 0];
            dst.M23 = this.m[2, 1];
            dst.M33 = this.m[2, 2];
            dst.M43 = this.m[2, 3];

            return dst;
        }

        /// <summary>
        ///
        /// </summary>
        public void Set(SharpDX.Matrix src)
        {
            this.m[0, 0] = src.M11;
            this.m[0, 1] = src.M21;
            this.m[0, 2] = src.M31;
            this.m[0, 3] = src.M41;

            this.m[1, 0] = src.M12;
            this.m[1, 1] = src.M22;
            this.m[1, 2] = src.M32;
            this.m[1, 3] = src.M42;

            this.m[2, 0] = src.M13;
            this.m[2, 1] = src.M23;
            this.m[2, 2] = src.M33;
            this.m[2, 3] = src.M43;
        }
    }

    /// <summary>
    /// RGBAカラー
    /// </summary>
    public struct RGBAColor
    {
        byte      _r;
        byte      _g;
        byte      _b;
        byte      _a;


        public RGBAColor( Color src )
            :this( src.R, src.G, src.B, src.A )
        {

        }

        public RGBAColor( RGBAColor src )
            :this( src.R, src.G, src.B, src.A )
        {

        }

        public RGBAColor( byte r, byte g, byte b, byte a )
        {
            _r = r;
            _g = g;
            _b = b;
            _a = a;
        }

        public RGBAColor( int r, int g, int b, int a )
        {
            _r = (byte)r;
            _g = (byte)g;
            _b = (byte)b;
            _a = (byte)a;
        }

        public Color AsColor
        {
            get
            {
                return Color.FromArgb( _a, _r, _g, _b );
            }
        }

        public RGBAColor RGBA
        {
            set
            {
                this._r = value.R;
                this._g = value.G;
                this._b = value.B;
                this._a = value.A;
            }
        }

        public byte R
        {
            get{ return _r;}
            set{ _r = value;}
        }

        public byte G
        {
            get{ return _g;}
            set{ _g = value;}
        }

        public byte B
        {
            get{ return _b;}
            set{ _b = value;}
        }

        public byte A
        {
            get{ return _a;}
            set{ _a = value;}
        }

        static public bool operator ==( RGBAColor p0, RGBAColor p1 )
        {
            return p0.R == p1.R && p0.G == p1.G && p0.B == p1.B && p0.A == p1.A;
        }

        static public bool operator !=( RGBAColor p0, RGBAColor p1 )
        {
            return !( p0 == p1 );
        }

        /// <summary>
        /// HSV補正されたカラーを取得します。
        /// </summary>
        public static RGBAColor GetHSVTweakedColor(float hOffset, float sScale, float bScale, RGBAColor color)
        {
            HSV hsvColor = HSV.FromRGB(new RGB(color.R, color.G, color.B));
            hsvColor.H = hsvColor.H + (int)hOffset;
            hsvColor.S = (int)(hsvColor.S * sScale);
            hsvColor.V = (int)(hsvColor.V * bScale);
            RGB rgb = RGB.FromHSV(hsvColor);
            return new RGBAColor(Math.Min(255, rgb.R), Math.Min(255, rgb.G), Math.Min(255, rgb.B), color.A);
        }

        public override bool Equals( object obj )
        {
            if( obj is RGBAColor )
            {
                RGBAColor another = (RGBAColor)obj;
                return another == this;
            }
            return false;
        }

        public override int GetHashCode()
        {
            return (int)this.R ^ (int)this.G ^ (int)this.B ^ (int)this.A;
        }

        public override string ToString()
        {
            return string.Format( "[{0},{1},{2},{3}]", _r, _g, _b, _a );
        }

        static public readonly RGBAColor Black         = new RGBAColor( 0, 0, 0, 255 );
        static public readonly RGBAColor Transparence  = new RGBAColor( 0, 0, 0, 0 );
        static public readonly RGBAColor White         = new RGBAColor( 255, 255, 255, 255 );

        public IEnumerable<byte> AsIEnumerable()
        {
            yield return _r;
            yield return _g;
            yield return _b;
            yield return _a;
        }
    }

    /// <summary>
    /// 浮動小数点数カラー
    /// </summary>
    public class FloatColor
    {
        public static readonly FloatColor Empty;

        public float _r = 0.0f;
        public float _g = 0.0f;
        public float _b = 0.0f;
        public float _a = 0.0f;
        public bool _byte_convert = false;

        public float R
        {
            get { return _r; }
            set { _r = value; }
        }

        public float G
        {
            get { return _g; }
            set { _g = value; }
        }

        public float B
        {
            get { return _b; }
            set { _b = value; }
        }

        public float A
        {
            get { return _a; }
            set { _a = value; }
        }

        public bool ByteConvert
        {
            get { return _byte_convert; }
            set { _byte_convert = value; }
        }

        public FloatColor()
        {
        }

        public FloatColor(float r, float g, float b, float a) : this()
        {
            _r = r;
            _g = g;
            _b = b;
            _a = a;
            ByteConvert = false;
        }

        public FloatColor(float r, float g, float b, float a, bool byteConvert) : this()
        {
            _r = r;
            _g = g;
            _b = b;
            _a = a;
            ByteConvert = byteConvert;
        }


        public FloatColor(Color color) : this()
        {
            FromSystemColor(color);
            ByteConvert = true;
        }

        public FloatColor(RGBAColor color) : this()
        {
            FromRGBAColor(color);
            ByteConvert = true;
        }

        public FloatColor(FloatColor color) : this()
        {
            _r = color.R;
            _g = color.G;
            _b = color.B;
            _a = color.A;
            ByteConvert = color.ByteConvert;
        }

        static public bool operator ==(FloatColor p0, FloatColor p1)
        {
            return p0.R == p1.R && p0.G == p1.G && p0.B == p1.B && p0.A == p1.A && p0.ByteConvert == p1.ByteConvert;
        }

        static public bool operator !=(FloatColor p0, FloatColor p1)
        {
            return !(p0 == p1);
        }

        public override bool Equals(object obj)
        {
            if (obj is FloatColor)
            {
                FloatColor another = (FloatColor)obj;
                return another == this;
            }
            return false;
        }

        public override int GetHashCode()
        {
            return (int)this.R ^ (int)this.G ^ (int)this.B ^ (int)this.A;
        }

        public void FromSystemColor(Color color)
        {
            _r = (float)color.R / 255.0f;
            _g = (float)color.G / 255.0f;
            _b = (float)color.B / 255.0f;
            _a = (float)color.A / 255.0f;
            ByteConvert = true;
        }

        public void FromRGBAColor(RGBAColor color)
        {
            FromSystemColor(color.AsColor);
            ByteConvert = true;
        }

        public Color ToSystemColor()
        {
            return ToRGBAColor().AsColor;
        }

        public RGBAColor ToRGBAColor()
        {
            byte a = (byte)(Math.Min((int)(A * 255.0f), 255));
            byte r = (byte)(Math.Min((int)(R * 255.0f), 255));
            byte g = (byte)(Math.Min((int)(G * 255.0f), 255));
            byte b = (byte)(Math.Min((int)(B * 255.0f), 255));

            return new RGBAColor(r, g, b, a);
        }

        static public readonly FloatColor Black         = new FloatColor(0.0f, 0.0f, 0.0f, 1.0f, true);
        static public readonly FloatColor Transparence  = new FloatColor(0.0f, 0.0f, 0.0f, 0.0f, true);
        static public readonly FloatColor White         = new FloatColor(1.0f, 1.0f, 1.0f, 1.0f, true);
    }

    public delegate void LEPatamaterChanged( object sender, EventArgs args );

    public static class RectangleHelper
    {
        public static FVec2 CalcCenter(RectangleF rect)
        {
            return new FVec2((rect.Left +  rect.Right) / 2.0f, (rect.Top +  rect.Bottom) / 2.0f);
        }
    }

    /// <summary>
    /// 文字毎のアニメーションの簡単設定
    /// </summary>
    public struct EasySettingOfPerCharTransform
    {
        public bool easySettingMoveFromLeft;
        public bool easySettingCalcEndFrame;
        public float easySettingBeginFrame;
        public float easySettingEndFrame;
        public float easySettingStopFrameOfFirstChar;
        public float easySettingStartFrameOfLastChar;
    }

    //public enum SceneModifyType
    // SceneModifyType は SceneModifyEventArgs.Kind に 統合されました。
}

namespace LECore.Structures.LECoreInterface
{
    /// <summary>
    /// テクスチャピクセルフォーマット
    /// </summary>
    public enum TexImagePixelFmt
    {
        L4,
        A4,
        L8,
        A8,
        LA4,
        LA8,
        HILO8,
        RGB565,
        RGB8,
        RGB5A1,
        RGBA4,
        RGBA8,
        ETC1,
        ETC1A4,
        BC1,
        BC2,
        BC3,
        BC4L,
        BC4A,
        BC5,
        BC7,
        RGB565_INDIRECT,

        ASTC4x4,
        ASTC5x4,
        ASTC5x5,
        ASTC6x5,
        ASTC6x6,
        ASTC8x5,
        ASTC8x6,
        ASTC8x8,
        ASTC10x5,
        ASTC10x6,
        ASTC10x8,
        ASTC10x10,
        ASTC12x10,
        ASTC12x12,

        R10G10B10A2,

        Unknown,
    }

    /// <summary>
    /// デフォルトテクスチャーフォーマット設定
    /// </summary>
    public class DefaultTextureFormat
    {
        public TexImagePixelFmt GrayScale { get; set; } = TexImagePixelFmt.BC4L;
        public TexImagePixelFmt GrayScaleWithAlpha { get; set; } = TexImagePixelFmt.BC5;
        public TexImagePixelFmt RGBColor { get; set; } = TexImagePixelFmt.BC1;
        public TexImagePixelFmt RGBColorWithAlpha { get; set; } = TexImagePixelFmt.BC3;
    }

    /// <summary>
    ///
    /// </summary>
    public static class TexImagePixelFmtHelper
    {
        static public bool IsIntencityFormat(this TexImagePixelFmt fmt)
        {
            return fmt == TexImagePixelFmt.L4 || fmt == TexImagePixelFmt.L8 || fmt == TexImagePixelFmt.BC4L;
        }

        static public bool IsAlphaFormat(this TexImagePixelFmt fmt)
        {
            return fmt == TexImagePixelFmt.A4 || fmt == TexImagePixelFmt.A8 || fmt == TexImagePixelFmt.BC4A;
        }

        static public bool IsIntencityAlphaFormat(this TexImagePixelFmt fmt)
        {
            return fmt == TexImagePixelFmt.LA4 || fmt == TexImagePixelFmt.LA8 || fmt == TexImagePixelFmt.BC5;
        }
    }

    /// <summary>
    /// テクスチャソース
    /// </summary>
    public enum TextureSourceType
    {
        File,
        Dynamic
    }

    /// <summary>
    /// アニメーションループ種類
    /// </summary>
    public enum AnimationLoopType
    {
        OneTime,
        Loop,
        Wiat
    }

    /// <summary>
    /// ドロップシャドウ効果のブレンド種類
    /// </summary>
    public enum DropShadowBlendMode
    {
        Normal,
        Multiply,
        Addition,
        Subtraction,
    }

    /// <summary>
    /// 角丸効果のブレンド種類
    /// </summary>
    public enum ProceduralShapeEffectBlendMode
    {
        Normal,
        Multiply,
        Addition,
        Subtraction,
    }

    /// <summary>
    /// 角丸効果の影タイプ
    /// </summary>
    public enum ProceduralShapeShadowType
    {
        Weak,
        Normal,
        Strong,
    }

}
