﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace LECore.Structures
{
    using Core;

    /// <summary>
    /// ペインアニメーションアトリビュート
    /// </summary>
    public interface IPaneAttribute
    {
        IAnmAttribute TransIAttr { get; }
        IAnmAttribute SizeIAttr { get; }
        IAnmAttribute ScaleIAttr { get; }
        IAnmAttribute RotAngIAttr { get; }
        IAnmAttribute TransparencyIAttr { get; }
        IAnmAttribute VisibleIAttr { get; }
    }

    /// <summary>
    /// 部品倍率の影響
    /// </summary>
    public enum PartsMagnifyInfluence
    {
        None,
        ScaleMagnify,
        AdjustToPartsBound
    }

    /// <summary>
    /// 拡張ユーザデータを持っている
    /// </summary>
    public interface IUserDataHolder
    {
        /// <summary>
        /// アニメーションが有効かどうか取得します。
        /// </summary>
        bool AnimationEnabled { get; }

        /// <summary>
        /// 上書きデータかどうか取得します。
        /// </summary>
        bool Overwrite { get; }

        /// <summary>
        /// 拡張ユーザデータ配列を取得します。
        /// </summary>
        IUserDataElement[] UserDataElementSet { get; }

        /// <summary>
        /// 上書き元の拡張ユーザデータを取得します。
        /// </summary>
        /// <param name="name">データ名</param>
        /// <returns>
        /// 上書き元の拡張ユーザデータを返します。
        /// 上書き元の拡張ユーザデータが見つからないときはnullを返します。
        /// </returns>
        IUserDataElement FindOriginalData(string name);
    }

    /// <summary>
    /// PaneModify クラス システムコア外部公開用インタフェース
    /// </summary>
    public interface IPane :
        IDrawable,
        IHierarchyNode,
        ICollidable
    {
        FVec3 Size { get;}

        float Width { get;}
        float Height { get;}

        float RenderingWidth { get; }
        float RenderingHeight { get; }


        FVec3 RotAng { get;}
        FVec3 Scale { get;}
        FVec3 Trans { get;}

        float DrawOffsetX { get;}
        float DrawOffsetY { get;}

        byte Transparency { get;}


        string PaneName { get;}
        string UserCommentString { get;}
        string UserData { get;}


        IAnmAttribute IAttributeRef { get;}
        IPaneAttribute IPaneAttribute { get;}

        ITextBox ITextBox { get;}
        IPicture IPicture { get;}
        ILEWindow ILEWindow { get;}
        IPartsLayout IPartsLayout { get; }
        ICapture ICapture { get; }

        IAlignment IAlignment { get; }
        IScissor IScissor { get; }

        IMaterial[] IMaterial { get;}
        IMaterial[] UsedMaterials { get; }
        IEnumerable<TexCoord4> TexCoords { get; }

        IMask IMask { get; }
        IDropShadow IDropShadow { get; }

        HorizontalLocation BasePosTypeH { get;}
        VerticalLocation BasePosTypeV { get;}

        HorizontalLocation ParentBasePosTypeH { get; }
        VerticalLocation ParentBasePosTypeV { get; }

        bool Visible { get;}
        bool IsSelected { get;}

        bool Locked { get;}
        bool Hidden { get;}
        bool IsReadOnlyLocked { get; }
        bool AvoidPaneTreeCompression { get; }
        bool ExtUserDataAnimEnabled { get; }

        bool InfluenceChildrenTransparency { get;}
        bool LocationAdjust { get;}

        bool IsAlignmentHorizontalMarginEnabled { get; }

        bool IsAlignmentVerticalMarginEnabled { get; }

        float AlignmentHorizontalMargin { get; }

        float AlignmentVerticalMargin { get; }

        bool IsAlignmentIgnore { get; }

        PartsMagnifyInfluence PartsMagnifyInfluence { get; }

        PaneSizeMode PaneSizeMode { get; }

        // TODO:派生ベースにリファクタクリングをし、
        // 本プロパティを廃止したい。
        PaneKind PaneKind { get;}

        IPaneExParamater IPaneExParamater { get;}

        ISubScene OwnerSubScene { get; }
        IUserDataHolder IUserDataHolder { get; }

        // 警告文字列
        string WarningMsg { get; }
    }



    /// <summary>
    /// ペイン拡張パラメータ
    /// </summary>
    public interface IPaneExParamater : IDrawable, IDisposable
    {
        IPane OwnerPane { get; }
        void UpdateMatarialName();
        // シーン登録時の初期化
        void OnJoinSceneInitialize();

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// OnJoinSceneInitialize との違い：
        /// ファイル読み込み時、OnJoinSceneInitialize は、アニメーション読み込み前に呼ばれます（flyt 読み込み完了時点でシーン登録される）。
        /// </summary>
        void FirstTimeInitialize();

        IEnumerable<TexCoord4> TexCoords { get; }

        bool PaneBoundDrawEnabled { get; }

        // 警告文字列
        string WarningMsg { get; }
    }

    /// <summary>
    /// ペイン拡張パラメータ：ヘルパ
    /// </summary>
    public static class PaneExParamaterHelper
    {
        /// <summary>
        /// 持ちぬしシーンを取得します。
        /// </summary>
        public static ISubScene GetOwnerSubScene(this IPaneExParamater paramater)
        {
            if (paramater == null || paramater.OwnerPane == null)
            {
                return null;
            }

            return paramater.OwnerPane.OwnerSubScene;
        }

        /// <summary>
        /// スクリーンサイズを取得します。
        /// </summary>
        public static FVec2 GetBackGroundScreenSize(this IPaneExParamater paramater)
        {
            if (paramater == null || paramater.OwnerPane == null || paramater.OwnerPane.OwnerSubScene == null)
            {
                return FVec2.Empty;
            }

            return paramater.OwnerPane.OwnerSubScene.BackGround.ScreenSize;
        }
    }

    /// <summary>
    /// サイズ変更可能状態
    /// </summary>
    [Flags]
    public enum PaneSizeMode
    {
        Fixed = 0,
        HEnabled = 1,
        VEnabled = 2,
        All = HEnabled | VEnabled,
    }

    /// <remarks/>
    [Flags]
    public enum PaneKind
    {
        // None = 0x0,
        Picture = 0x1,
        Textbox = 0x2,
        Window  = 0x4,
        Parts   = 0x8,
        Bounding = 0x10,
        Null     = 0x20,
        Capture  = 0x40,
        Alignment = 0x80,
        Scissor = 0x100,
    }
}
