﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Text;
using System.IO;
using System.Linq;
using System.Resources;
using System.Windows.Forms;

namespace LECore.Structures
{
    using Core;
    using LECoreInterface;

    using NW4C.Tga;
    using Structures;
    using Save_Load;
    using System.Threading.Tasks;

    /// <summary>
    ///
    /// </summary>
    internal class SrcBitmapHolder : IDisposable
    {
        static Bitmap _invalidTextureBmp = LEImageResMgr.GetManifestResourcePng("Invalid_Image.png");

        public string FilePath { get; private set; }
        public Bitmap GDIBitmap { get; private set; }
        public Bitmap ColorGDIBitmap { get; private set; }
        public Bitmap AlphaGDIBitmap { get; private set; }
        public bool HasSrcImageAlpha { get; private set; }
        public bool IsSrcImageGlayScale { get; private set; }
        public bool IsValid { get; private set; }
        public DateTime UpdateTime { get; private set; }

        public Size ActualTextureSize { get; private set; }


        public TexImagePixelFmt InitialFormat { get; private set; }

        public List<TextureImage> References { get; private set; } = new List<TextureImage>();

        /// <summary>
        /// 生成
        /// </summary>
        public SrcBitmapHolder(string filePath, Bitmap normal, Bitmap color, Bitmap alpha, bool srcImageHasAlphaBit, bool srcImageIsGrayScale, TexImagePixelFmt initialFormat, Size actualTextureSize)
        {
            this.Update(filePath, normal, color, alpha, srcImageHasAlphaBit, srcImageIsGrayScale, initialFormat, actualTextureSize);
        }

        /// <summary>
        /// 生成
        /// </summary>
        static public SrcBitmapHolder CreateInvalid(string filePath)
        {
            SrcBitmapHolder invalid = new SrcBitmapHolder(filePath, _invalidTextureBmp, _invalidTextureBmp, _invalidTextureBmp, true, false, TexImagePixelFmt.BC3, _invalidTextureBmp.Size);
            return invalid;
        }

        public void UpdateAsInvalide(string filePath)
        {
            this.Update(filePath, _invalidTextureBmp, _invalidTextureBmp, _invalidTextureBmp, true, false, TexImagePixelFmt.BC3, _invalidTextureBmp.Size);
        }

        /// <summary>
        /// 破棄
        /// </summary>
        public void Dispose()
        {
            if (object.ReferenceEquals(this.GDIBitmap, _invalidTextureBmp))
            {
                return;
            }
            else
            {
                if (this.GDIBitmap != null) { this.GDIBitmap.Dispose(); this.GDIBitmap = null; }
                if (this.ColorGDIBitmap != null) { this.ColorGDIBitmap.Dispose(); this.ColorGDIBitmap = null; }
                if (this.AlphaGDIBitmap != null) { this.AlphaGDIBitmap.Dispose(); this.AlphaGDIBitmap = null; }
            }
        }

        /// <summary>
        /// 内容を更新する。
        /// </summary>
        public void Update(string filePath, Bitmap normal, Bitmap color, Bitmap alpha, bool srcImageHasAlphaBit, bool srcImageIsGrayScale, TexImagePixelFmt initialFormat, Size actualTextureSize)
        {
            this.Dispose();

            this.FilePath = filePath;
            this.GDIBitmap = normal;
            this.ColorGDIBitmap = color;
            this.AlphaGDIBitmap = alpha;

            this.HasSrcImageAlpha = srcImageHasAlphaBit;
            this.IsSrcImageGlayScale = srcImageIsGrayScale;
            this.IsValid = true;
            this.InitialFormat = initialFormat;
            this.UpdateTime = DateTime.Now;
            this.ActualTextureSize = actualTextureSize;
        }
    }

    /// <summary>
    /// テクスチャ画像生成
    /// </summary>
    public static class TextureImageFactory
    {

        #region private

        /// <summary>
        /// エラーメッセージをメッセージボックスで表示します。
        /// </summary>
        static void ShowNgMessageBox_( string resIDStr, params object[] args )
        {
            // 現状は対応していないので、警告メッセージを表示します。
            string title = LECoreStringResMgr.Get( "LECORE_TEXMGR_TITLE_NGTGAFMT" );
            string msg = LECoreStringResMgr.Get( resIDStr );

            // MessageBox.Show( msg, title );

            LayoutEditorCore.MsgReporter.ReportError( title, msg, args );
        }

        /// <summary>
        /// TextureFormat to TexImagePixelFmt
        /// </summary>
        static TexImagePixelFmt ConvertToPixelFormat_( NW4C.Tga.TextureFormat nw4rFormat )
        {
            switch( nw4rFormat )
            {
            case TextureFormat.L4:      return TexImagePixelFmt.L4;
            case TextureFormat.A4:      return TexImagePixelFmt.A4;
            case TextureFormat.L8:      return TexImagePixelFmt.L8;
            case TextureFormat.A8:      return TexImagePixelFmt.A8;
            case TextureFormat.La4:     return TexImagePixelFmt.LA4;
            case TextureFormat.La8:     return TexImagePixelFmt.LA8;
            case TextureFormat.Hilo8:   return TexImagePixelFmt.HILO8;
            case TextureFormat.Rgb565:  return TexImagePixelFmt.RGB565;
            case TextureFormat.Rgb8:    return TexImagePixelFmt.RGB8;
            case TextureFormat.Rgb5_a1: return TexImagePixelFmt.RGB5A1;
            case TextureFormat.Rgba4:   return TexImagePixelFmt.RGBA4;
            case TextureFormat.Rgba8:   return TexImagePixelFmt.RGBA8;
            case TextureFormat.Etc1:    return TexImagePixelFmt.ETC1;
            case TextureFormat.Etc1_a4: return TexImagePixelFmt.ETC1A4;
            case TextureFormat.Unknown: return TexImagePixelFmt.Unknown;
            default:
                Debug.Assert( false );
                return TexImagePixelFmt.RGB565;
            }
        }

        static void GetBitmapByteArrayAsReadOnly_(Bitmap bmp, out byte[] bytes, out int stride)
        {
            BitmapData bmpd = bmp.LockBits(
                new Rectangle(Point.Empty, bmp.Size), ImageLockMode.ReadOnly, bmp.PixelFormat);

            bytes = new byte[bmpd.Stride * bmpd.Height];
            stride = bmpd.Stride;

            System.Runtime.InteropServices.Marshal.Copy(bmpd.Scan0, bytes, 0, bytes.Length);
            bmp.UnlockBits(bmpd);
        }

        /// <summary>
        /// アルファイメージビットマップを生成します。
        /// </summary>
        static Bitmap MakeColorImageBitmap_(Bitmap srcBitmap)
        {
            Debug.Assert(srcBitmap != null);
            if (Bitmap.IsAlphaPixelFormat(srcBitmap.PixelFormat))
            {
                return srcBitmap.Clone(new Rectangle(0,0, srcBitmap.Width, srcBitmap.Height), PixelFormat.Format24bppRgb) as Bitmap;
            }
            else
            {
                return srcBitmap;
            }
        }

        /// <summary>
        /// アルファイメージビットマップを生成します。
        /// </summary>
        static Bitmap MakeAlphaImageBitmap_(Bitmap srcBitmap)
        {
            Debug.Assert(srcBitmap != null);

            Bitmap alphaBmp = new Bitmap(srcBitmap.Width, srcBitmap.Height, PixelFormat.Format32bppArgb);
            BitmapData dstBmpd = alphaBmp.LockBits(
                new Rectangle(Point.Empty, alphaBmp.Size), ImageLockMode.WriteOnly, alphaBmp.PixelFormat);

            int srcStride;
            byte[] colValues;
            GetBitmapByteArrayAsReadOnly_(srcBitmap, out colValues, out srcStride);

            int bytes = dstBmpd.Stride * dstBmpd.Height;
            byte[] argbValues = new byte[bytes];

            int dstStride = dstBmpd.Stride;
            int srcWidth = srcBitmap.Width;
            int bytePerPixcel = Bitmap.IsAlphaPixelFormat(srcBitmap.PixelFormat) ? 4 : 3;
            int alphaChanelOffset = Bitmap.IsAlphaPixelFormat(srcBitmap.PixelFormat) ? 3 : 0;
            Parallel.For(0, srcBitmap.Height, y =>
            {
                int dstBaseIdx = y * dstStride;
                int srcBaseIdx = y * srcStride;
                for (int x = 0; x < srcWidth; x++)
                {
                    int dstIdx = dstBaseIdx + x * 4;
                    byte val = colValues[srcBaseIdx + x * bytePerPixcel + alphaChanelOffset];
                    argbValues[dstIdx + 0] = val;
                    argbValues[dstIdx + 1] = val;
                    argbValues[dstIdx + 2] = val;

                    argbValues[dstIdx + 3] = 255;
                }
            });

            System.Runtime.InteropServices.Marshal.Copy(argbValues, 0, dstBmpd.Scan0, bytes);
            alphaBmp.UnlockBits(dstBmpd);

            return alphaBmp;
        }

        /// <summary>
        /// 2つのBitmapからARGB形式Bitmapを作成します。
        /// </summary>
        static public Bitmap MakeARGBBitmapFromTwoBitmap(Bitmap colorBmp, Bitmap alphaBmp)
        {
            Bitmap argbBmp = new Bitmap(colorBmp.Width, colorBmp.Height, PixelFormat.Format32bppArgb);
            BitmapData dstBmpd = argbBmp.LockBits(
                new Rectangle(Point.Empty, argbBmp.Size), ImageLockMode.WriteOnly, argbBmp.PixelFormat);

            int colStride;
            byte[] colValues;
            GetBitmapByteArrayAsReadOnly_(colorBmp, out colValues, out colStride);
            int alpStride;
            byte[] alpValues;
            GetBitmapByteArrayAsReadOnly_(alphaBmp, out alpValues, out alpStride);

            int colBytePerPixcel = Bitmap.IsAlphaPixelFormat(colorBmp.PixelFormat) ? 4 : 3;
            int alpBytePerPixcel = Bitmap.IsAlphaPixelFormat(alphaBmp.PixelFormat) ? 4 : 3;

            int dstBytes = dstBmpd.Stride * dstBmpd.Height;
            byte[] argbValues = new byte[dstBytes];
            int colorIdxOffset = Bitmap.IsAlphaPixelFormat(colorBmp.PixelFormat) ? 1 : 0;
            for (int y = 0; y < colorBmp.Height; y++)
            {
                int baseIdx = y * dstBmpd.Stride;
                int baseColIdx = y * colStride;
                int baseAlpIdx = y * alpStride;
                for (int x = 0; x < colorBmp.Width; x++)
                {
                    int idx = baseIdx + x * 4;
                    int colIdx = baseColIdx + x * colBytePerPixcel + colorIdxOffset;
                    int alpIdx = baseAlpIdx + x * alpBytePerPixcel;

                    argbValues[idx + 0] = colValues[colIdx + 0];
                    argbValues[idx + 1] = colValues[colIdx + 1];
                    argbValues[idx + 2] = colValues[colIdx + 2];
                    argbValues[idx + 3] = alpValues[alpIdx + 0];
                }
            }

            System.Runtime.InteropServices.Marshal.Copy(argbValues, 0, dstBmpd.Scan0, dstBytes);
            argbBmp.UnlockBits(dstBmpd);

            return argbBmp;
        }

        /// <summary>
        /// バイト値を逆転します。
        /// </summary>
        static byte InvertByte_(byte src)
        {
            return (byte)(255 - src);
        }

        /// <summary>
        /// Tgaファイルが有効か判定します。
        /// </summary>
        static bool IsTgaValid_( string filePath )
        {
            //---------------- ファイルが存在するか確認
            if( !File.Exists( filePath ) )
            {
                // ファイルが読み込めませんでした。
                ShowNgMessageBox_( "LECORE_SYS_ERROR_FILELOAD", filePath );
                return false;
            }

            //---------------- 拡張子の確認
            TGABitmapLoader loder = new TGABitmapLoader();
            if( !loder.IsValidFileFmtExt( filePath ) )
            {
                // ファイル拡張子が不正です。
                ShowNgMessageBox_( "LECORE_TEXMGR_ERROR_NGFILEEXT", filePath );
                return false;
            }

            //---------------- ファイル名の有効性の確認。
            if( !SaveLoadHelper.IsValidStringForFileName( Path.GetFileNameWithoutExtension( filePath ) ) )
            {
                // ファイル名が不正です。
                ShowNgMessageBox_( "LECORE_SYS_ERROR_NGFILENAME", filePath );
                return false;
            }

            //---------------- 付加情報付きの場合。
            TgaReader tgaReader = new TgaReader();
            using( FileStream inputStream = new FileStream( filePath, FileMode.Open, FileAccess.Read ) )
            {
                // 付加情報付き以外なら...
                if (!TgaReader.CheckAdditionalInformation(inputStream))
                {
                    // LayoutEditorによるフォーマット変更が禁止されている場合、
                    // 通常の TGA は不正ファイルとしてあつかう。
                    if (!LayoutEditorCore.PlatformDetail.TextureFormatEditEnabled)
                    {
                        ShowNgMessageBox_("LECORE_TEXMGR_ERROR_NGNW4CTGA", filePath);
                        return false;
                    }

                    TGABitmapLoader.TGAHeader tgaHeader;
                    bool bGetHeader = loder.LoadTgaHeaderFromFile(filePath, out tgaHeader);
                    Debug.Assert(bGetHeader);

                    //---------------- ビットデプスの確認。
                    // 24 ビット、もしくは32ビット(アルファ付き)のみ対応します。
                    if (tgaHeader.BitDepth != 24 && tgaHeader.BitDepth != 32)
                    {
                        ShowNgMessageBox_("LECORE_TEXMGR_ERROR_TGABITDEPTH", filePath);
                        return false;
                    }

                    //---------------- 画像サイズの確認(TODO:)
                }
            }

            return true;
        }

        #endregion private

        /// <summary>
        /// 付加情報付きTGAと通常TGAの両方をサポートしています。
        /// </summary>
        static void InitializeTextureImage_(TextureImage texImage, string filePath)
        {
            if (!IsTgaValid_(filePath))
            {
                //---------------- 不正なファイルの場合
                texImage.InitializeAsInvalid(filePath);
                return;
            }

            using (FileStream inputStream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                TGABitmapLoader loder = new TGABitmapLoader();
                Bitmap bmp = null;
                Bitmap colorBitmap;
                Bitmap alphaBitmap;

                bmp = loder.LoadFromFile(filePath);
                if (bmp == null)
                {
                    //---------------- 不正なファイルの場合
                    texImage.InitializeAsInvalid(filePath);
                    return;
                }

                // カラーチャンネル用画像を生成します。
                colorBitmap = MakeColorImageBitmap_(bmp);

                // アルファフォーマットかどうか？
                // アルファチャンネルを持つか、付加情報でアルファ付きであることが
                // 指定されている。
                bool sourceHadAlpha = loder.LastLoadedTgaHeader.BitDepth == 32;
                bool intendedToBeAlphaFormat = sourceHadAlpha;
                bool isGrayScale = TGABitmapLoader.CheckGrayScaleBmp(colorBitmap);

                // 付加情報があれば、アルファ付きフォーマットが指定されているか確認する
                TexImagePixelFmt nw4cTgaFormat = TexImagePixelFmt.Unknown;
                Size textureActualSize = colorBitmap.Size;
                if (TgaReader.CheckAdditionalInformation(inputStream))
                {
                    TgaReader tgaReader = new TgaReader();
                    tgaReader.Read(inputStream);
                    intendedToBeAlphaFormat = tgaReader.HasAlpha;
                    nw4cTgaFormat = ConvertToPixelFormat_(tgaReader.TextureFormat);

                    // GPUネイティブフォーマットの都合によって付加されたパディングを含むサイズ
                    textureActualSize = tgaReader.ColorBitmap.Size;
                }

                // アルファ画像を作っておく。
                // ソースにアルファチャンネルが無い場合は、カラー情報からアルファ画像を作ります。
                alphaBitmap = MakeAlphaImageBitmap_(bmp);

                // フォーマット初期値を決定します。
                TexImagePixelFmt defaultFmt = LayoutEditorCore.PlatformDetail.GetDefaultTextureFormat(isGrayScale, intendedToBeAlphaFormat, nw4cTgaFormat);

                if (texImage.SrcBitmapHolder == null)
                {
                    SrcBitmapHolder holder = new SrcBitmapHolder(filePath, bmp, colorBitmap, alphaBitmap, sourceHadAlpha, isGrayScale, defaultFmt, textureActualSize);
                    texImage.Initialize(filePath, holder, colorBitmap.Size, false);
                }
                else
                {
                    texImage.SrcBitmapHolder.Update(filePath, bmp, colorBitmap, alphaBitmap, sourceHadAlpha, isGrayScale, defaultFmt, textureActualSize);

                    // 参照箇所も更新
                    foreach (var texture in texImage.SrcBitmapHolder.References)
                    {
                        texture.UpdateState();
                    }
                }
            }
        }

        /// <summary>
        /// ITextureImage の状態を最新のファイルの情報に更新します。
        /// </summary>
        static public bool UpdateTextureImage(ITextureImage iTexImg, string newFilePath)
        {
            var texImg = iTexImg as TextureImage;
            if (texImg == null)
            {
                return false;
            }

            var oldHolder = texImg.SrcBitmapHolder;

            var newName = Path.GetFileNameWithoutExtension(newFilePath);

            // リロード後も、現在のフォーマットを維持するように更新前の状態を保存しておく。
            TexImagePixelFmt textureFmt = texImg.PixelFmt;

            var bmpHolder = _srcBitmapHolders.Find(holder =>
                (string.Compare(holder.FilePath, newFilePath, StringComparison.OrdinalIgnoreCase) == 0));

            // SrcBitmapHolder を差し替え
            texImg.SetSrcBitmapHolder(bmpHolder);

            if (oldHolder != null)
            {
                if (oldHolder.References.Count == 0)
                {
                    oldHolder.Dispose();
                    _srcBitmapHolders.Remove(oldHolder);

                    DbgConsole.WriteLine("TEX_CNT={0}", _srcBitmapHolders.Count);
                }
            }

            // ソースのファイルが更新されているかもしれないので bmpHolder があっても初期化
            InitializeTextureImage_(texImg, newFilePath);

            if (bmpHolder == null)
            {
                _srcBitmapHolders.Add(texImg.SrcBitmapHolder);
            }

            // 以前の設定が妥当なものであれば再設定する。
            if (!texImg.PixelFmtIsFixed)
            {
                texImg.PixelFmt = textureFmt;
            }

            return true;
        }

        private static readonly List<SrcBitmapHolder> _srcBitmapHolders = new List<SrcBitmapHolder>();

        /// <summary>
        /// 背景画像など、特殊な用途で ITextureImage を生成する。
        /// </summary>
        static public ITextureImage BuildTextureImageFromTga(string filePath)
        {
            return BuildTextureImageFromTga(null, filePath);
        }

        /// <summary>
        /// TgaからTextureImageを生成します。
        /// </summary>
        static public ITextureImage BuildTextureImageFromTga(ITextureMgr creator, string filePath)
        {
            var bmpHolder = _srcBitmapHolders.Find( holder =>
                (string.Compare(holder.FilePath, filePath, StringComparison.OrdinalIgnoreCase) == 0));

            if (bmpHolder == null)
            {
                TextureImage texImage = new TextureImage();
                InitializeTextureImage_(texImage, filePath);

                _srcBitmapHolders.Add(texImage.SrcBitmapHolder);

                return texImage;
            }
            else
            {
                TextureImage texImage = new TextureImage();
                if (bmpHolder.IsValid)
                {
                    texImage.Initialize(filePath, bmpHolder, bmpHolder.ColorGDIBitmap.Size, false);
                }
                else
                {
                    texImage.InitializeAsInvalid(bmpHolder.FilePath);
                }

                return texImage;
            }
        }

        /// <summary>
        /// テクスチャを破棄します。
        /// </summary>
        static public void DisposeTexture(ITextureMgr creator, ITextureImage texture)
        {
            // 登録済みか検索します
            var bmpHolder = (texture as TextureImage).SrcBitmapHolder;

            if (bmpHolder != null)
            {
                (texture as TextureImage).SetSrcBitmapHolder(null);

                if (bmpHolder.References.Count == 0)
                {
                    bmpHolder.Dispose();
                    _srcBitmapHolders.Remove(bmpHolder);

                    DbgConsole.WriteLine("TEX_CNT={0}", _srcBitmapHolders.Count);
                }
            }else{
                (texture as TextureImage).Dispose();
                Debug.Assert(false);
            }
        }
    }
}
