﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;


using System.Drawing;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
using System.IO;

using System.Runtime.InteropServices;

using System.Windows.Forms;

namespace LECore.Structures
{
    using Structures;

    #region Bitmap ローダ
    /// <summary>
    /// 各種ファイルからBitmapを読み込むクラスが実装するインタフェース
    /// </summary>
    public interface IBitmapLoader
    {
        Bitmap LoadFromFile( string filePath );
        bool IsValidFileFmtExt( string filePath );
    }

    /// <summary>
    /// ファイルからBitmapを読み込みます。
    /// </summary>
    public class TGABitmapLoader :
        IBitmapLoader
    {

        #region TGAHeader
        /// <summary>
        /// TGA ファイルヘッダ
        /// 参考：
        /// http://astronomy.swin.edu.au/~pbourke/dataformats/tga/
        /// http://www.openspc2.org/format/TGA/
        /// </summary>
        public struct TGAHeader
        {
            /// <summary>定数</summary>
            public const int HFlipShift = 4; // TexelFmt メンバの水平方向格納フラグを取り出すマスク

            /// <summary>定数</summary>
            public const int VFlipShift = 5; // TexelFmt メンバの垂直方向格納フラグを取り出すマスク



            /// <summary>フィールド</summary>
            public readonly byte IdSize;          // IDフィールド長
            public readonly byte ColorMapExist;   // カラーマップ有り無し(0 = 無し、1 = 有り)
            public readonly byte ImageFmt;        // 画像フォーマット
            //   0 = イメージ無し、
            //   1 = インデックスカラー（256色、
            //   2 = フルカラー、
            //   3 = 白黒
            //   9 = Runlength encoded color-mapped images.
            //  10 = Runlength encoded RGB images.
            //  11 = Compressed, black and white images.
            //  32 = Compressed color-mapped data, using Huffman, Delta, and runlength encoding.
            //  33 = Compressed color-mapped data, using Huffman, Delta, and runlength encoding.  4-pass quadtree-type process.

            public readonly short ColorMapEntry;   // カラーマップエントリ
            public readonly short ColorMapLength;  // カラーマップエントリ
            public readonly byte ColorMapBitSize; // カラーマップエントリサイズ

            public readonly short PosX;             // X座標
            public readonly short PosY;             // Y座標
            public readonly short Width;            // 幅
            public readonly short Hegiht;           // 高さ

            public readonly byte BitDepth;         // 色深度
            public readonly byte TexelFmt;         // テクセルフォーマット
            // 0-3 : 属性                             // 各ピクセルあたりの、アトリビュートビット数
            // 4: 水平方向格納( 0 = 左 => 右 )
            // 5: 垂直方向格納( 0 = 上 => 下 )

            public TGAHeader( BinaryReader binReader )
            {
                IdSize = binReader.ReadByte();
                ColorMapExist = binReader.ReadByte();
                ImageFmt = binReader.ReadByte();

                ColorMapEntry = binReader.ReadInt16();
                ColorMapLength = binReader.ReadInt16();
                ColorMapBitSize = binReader.ReadByte();

                PosX = binReader.ReadInt16();
                PosY = binReader.ReadInt16();
                Width = binReader.ReadInt16();
                Hegiht = binReader.ReadInt16();

                BitDepth = binReader.ReadByte();
                TexelFmt = binReader.ReadByte();
            }

            /// <summary>
            /// 圧縮フォーマットか？
            /// </summary>
            public bool IsCompressed
            {
                get { return ImageFmt > 3; }
            }

            static readonly public TGAHeader Empty = new TGAHeader();
        }
        #endregion TGAHeader

        // 最後にロードしたファイルのヘッダ情報
        TGAHeader _lastLoadedTgaHeader = TGAHeader.Empty;

        /// <summary>
        /// 最後にロードしたファイルのヘッダ情報を取得します。
        /// </summary>
        public TGAHeader LastLoadedTgaHeader
        {
            get { return _lastLoadedTgaHeader; }
        }

        #region プライベートメソッド
        /// <summary>
        /// TGA ビット深度情報( byte ) を PixelFormat 列挙子に変換します。
        /// </summary>
        PixelFormat ConvertBitDepthToPixelFormat_( int bitDepth )
        {
            switch( bitDepth )
            {
                case 8: return PixelFormat.Format8bppIndexed;
                case 16: return PixelFormat.Format16bppArgb1555;
                case 24: return PixelFormat.Format24bppRgb;
                case 32: return PixelFormat.Format32bppArgb;
                default: Debug.Assert( false ); return PixelFormat.Format32bppArgb;
            }
        }

        /// <summary>
        /// カラーパレットエントリを抽出します。
        /// </summary>
        Color ExtractColorEntry_( byte ColorMapBitSize, byte[] src, int pos )
        {
            switch( ColorMapBitSize )
            {
                case 16: return Color.FromArgb( 255, src[pos + 0], src[pos + 1], src[pos + 2] );
                case 24: return Color.FromArgb( 255, src[pos + 2], src[pos + 1], src[pos + 0] );
                case 32: return Color.FromArgb( 255, src[pos + 2], src[pos + 1], src[pos + 0] );
                default: Debug.Assert( false ); return Color.DarkGray;
            }
        }

        /// <summary>
        /// パレット情報を初期化します。
        /// </summary>
        void MakeColorPltt_( ColorPalette dstPltt, TGAHeader tgaHeader, byte[] src )
        {
            int pos = tgaHeader.IdSize;

            for( int colorEntryIdx = 0 ;
                colorEntryIdx < tgaHeader.ColorMapLength ;
                colorEntryIdx++ )
            {
                dstPltt.Entries[colorEntryIdx] =
                    ExtractColorEntry_( tgaHeader.ColorMapBitSize, src, pos );
                pos += tgaHeader.ColorMapBitSize / 8;
            }

        }

        /// <summary>
        /// 画像へのポインタを取得します。
        /// </summary>
        IntPtr GetImagePtr_(TGAHeader tgaHeader, byte[] buffer, out GCHandle gcHandle)
        {
            GCHandle srcHandle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

            long offsetOfPlttData = tgaHeader.ColorMapLength * tgaHeader.ColorMapBitSize / 8;
            long srcImageDataPos = (long)srcHandle.AddrOfPinnedObject() + offsetOfPlttData;

            if (CheckNeedToCreateAlignedImage_(tgaHeader))
            {
                int dstStride = GetBmpDataStride_(tgaHeader);
                int srcStride = (tgaHeader.Width * tgaHeader.BitDepth / 8);

                // バイトサイズ制限を考慮したバッファにコピーしなおす
                byte[] temp = new byte[dstStride * tgaHeader.Hegiht];
                gcHandle = GCHandle.Alloc(temp, GCHandleType.Pinned);

                for (int y = 0; y < tgaHeader.Hegiht; y++)
                {
                    int offs = dstStride * y;
                    int srcOffs = srcStride * y;

                    IntPtr srcPtr = (IntPtr)(srcImageDataPos + srcOffs);
                    Marshal.Copy(srcPtr, temp, offs, srcStride);
                }

                srcHandle.Free();
                return (IntPtr)((long)gcHandle.AddrOfPinnedObject());
            }
            else
            {
                gcHandle = srcHandle;
                return (IntPtr)srcImageDataPos;
            }
        }

        /// <summary>
        /// ストライド調整をした画像を作る必要があるか？
        /// </summary>
        bool CheckNeedToCreateAlignedImage_(TGAHeader tgaHeader)
        {
            int originalStride = (tgaHeader.BitDepth / 8 * tgaHeader.Width);
            int neededStride = (originalStride + 0x3) & ~0x3;
            return originalStride != neededStride;
        }

        /// <summary>
        /// Bitmap コンストラクタ引数である、stride を計算します。
        /// </summary>
        int GetBmpDataStride_( TGAHeader tgaHeader )
        {
            // ピクセルあたりのバイト数 * 画像の幅
            // Bitmap クラスの仕様に合わせ 4バイトに整列します。
            int stride = (tgaHeader.BitDepth / 8 * tgaHeader.Width);
            stride = (stride + 0x3) & ~0x3;
            return stride;
        }

        /// <summary>
        /// Bitmap コンストラクタ引数である、stride を計算します。
        /// </summary>
        int GetAdjustedWidth_(TGAHeader tgaHeader)
        {
            int stride = GetBmpDataStride_(tgaHeader);
            int res = stride / tgaHeader.BitDepth * 8;
            return res;
        }

        /// <summary>
        /// ランレングス圧縮を展開する。
        /// </summary>
        private byte[] UnCompressRunlength_(byte[] srcBuffer, int bitDepth)
        {
            Ensure.Argument.True(srcBuffer.Length > 0);
            var dstBuffer = new List<byte>();

            uint mask = ~((uint)0x1 << 7);
            uint numBytePreTexcel = (byte)(bitDepth / 8);
            long position = 0;
            while (position < srcBuffer.Length)
            {
                // 最上位ビットは記法を定義するフラグ。
                int type = srcBuffer[position] >> 7;

                // 進めるバイト数。(0は無いというルールで、+1した値が本当の値。最大値は 128。)
                uint step = Math.Min(128, (mask & srcBuffer[position]) + (uint)1);
                if (type == 0)
                {
                    // 非圧縮区間
                    for (int i = 0; i < step; i++)
                    {
                        var basePos = position + 1 + i * numBytePreTexcel;
                        for (int j = 0; j < numBytePreTexcel; j++)
                        {
                            dstBuffer.Add(srcBuffer[basePos + j]);
                        }
                    }

                    position += 1 + step * numBytePreTexcel;
                }
                else
                {
                    // 圧縮区間
                    List<byte> value = new List<byte>();
                    for (int j = 0; j < numBytePreTexcel; j++)
                    {
                        value.Add(srcBuffer[position + 1 + j]);
                    }

                    for (int i = 0; i < step; i++)
                    {
                        dstBuffer.AddRange(value);
                    }

                    position += 1 + numBytePreTexcel;
                }
            }

            return dstBuffer.ToArray();
        }

        #endregion プライベートメソッド

        #region IBitmapLoader メンバ

        /// <summary>
        ///
        /// </summary>
        public bool LoadTgaHeaderFromFile( string filePath, out TGAHeader tgaHeader )
        {
            using( FileStream inputStream = new FileStream( filePath, FileMode.Open, FileAccess.Read ) )
            {
                try
                {
                    tgaHeader = new TGAHeader( new BinaryReader( inputStream ) );
                    _lastLoadedTgaHeader = tgaHeader;
                    return true;
                }
                catch( Exception e )
                {
                    MessageBox.Show( e.Message );
                }
            }
            tgaHeader = TGAHeader.Empty;
            _lastLoadedTgaHeader = tgaHeader;
            return false;
        }

        /// <summary>
        /// グレースケールBMPか判定します。
        /// 適当にサンプリングした点の色が、すべてのモノトーンか調査する。
        /// </summary>
        public static bool CheckGrayScaleBmp(Bitmap bmp)
        {
            Random rnd = new Random();
            int stepY = Math.Max(bmp.Height >> 3, 1);
            int stepX = Math.Max(bmp.Width >> 3, 1);

            int tempStepX = stepX;
            int tempStepY = stepY;
            for (int y = 0; y < bmp.Height; y += tempStepY)
            {
                for (int x = 0; x < bmp.Width; x += tempStepX)
                {
                    Color c = bmp.GetPixel(x, y);
                    if (!(c.R == c.G && c.G == c.B))
                    {
                        return false;
                    }

                    // 乱数でサンプル点をすこしずらす。
                    tempStepX = stepX + rnd.Next(0, 3);
                }
                tempStepY = stepY + rnd.Next(0, 3);
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public Bitmap LoadFromFile( string filePath )
        {
            if (!IsValidFileFmtExt(filePath))
            {
                return null;
            }

            using( FileStream inputStream = new FileStream( filePath, FileMode.Open, FileAccess.Read ) )
            {
                try
                {
                    TGAHeader tgaHeader = new TGAHeader( new BinaryReader( inputStream ) );
                    _lastLoadedTgaHeader = tgaHeader;

                    // TGA ヘッダ+ID field を読み飛ばした位置に設定。
                    int offsetToImg = 18 + tgaHeader.IdSize;
                    inputStream.Seek(offsetToImg, SeekOrigin.Begin);

                    long bufferLen = inputStream.Length - offsetToImg;
                    byte[] buffer = new byte[bufferLen];
                    inputStream.Read(buffer, 0, (int)bufferLen);

                    //
                    // 圧縮形式のTGA
                    //
                    if( tgaHeader.IsCompressed )
                    {
                        // 10 == Runlength encoded RGB images.
                        if (tgaHeader.ImageFmt == 10)
                        {
                            // ファイル終端に NintendoWare の 拡張情報があれば、破棄します。
                            for (int i = 0; i < buffer.Length - 8; i++)
                            {
                                // 拡張情報で、必ず先頭に配置されるブロックの識別子、nw4X_tfm(X は RやC)という文字列を探す
                                if (buffer[i] == 'n' && buffer[i + 1] == 'w' && buffer[i + 2] == '4' &&
                                    buffer[i + 4] == '_' && buffer[i + 5] == 't' && buffer[i + 6] == 'f' && buffer[i + 7] == 'm')
                                {
                                    var subBuffer = new byte[i];
                                    Buffer.BlockCopy(buffer, 0, subBuffer, 0, i);

                                    buffer = subBuffer;
                                    break;
                                }
                            }

                            // TRUEVISION-XFILE.という文字列を判定してTarga 2.0 フッタ３ が存在するなら破棄します。
                            if( buffer.Length > 7 &&
                                buffer[buffer.Length - 7] == 'X' &&
                                buffer[buffer.Length - 6] == 'F' &&
                                buffer[buffer.Length - 5] == 'I' &&
                                buffer[buffer.Length - 4] == 'L' &&
                                buffer[buffer.Length - 3] == 'E' &&
                                buffer[buffer.Length - 2] == '.' &&
                                buffer[buffer.Length - 1] == 0)
                            {
                                // フッタのバイトサイズです。
                                // 詳しくはフォーマット仕様書を参照してください。
                                int newLength = buffer.Length - (4 + 4 + 16 + 1 + 1);
                                var subBuffer = new byte[newLength];
                                Buffer.BlockCopy(buffer, 0, subBuffer, 0, newLength);

                                buffer = subBuffer;
                            }

                            buffer = UnCompressRunlength_(buffer, tgaHeader.BitDepth);
                        }
                        else
                        {
                            return null;
                        }
                    }

                    GCHandle gcHandle;
                    IntPtr srcImagePtr = GetImagePtr_(tgaHeader, buffer, out gcHandle);
                    Bitmap bmp = new Bitmap(
                        tgaHeader.Width,
                        tgaHeader.Hegiht,
                        GetBmpDataStride_( tgaHeader ),
                        ConvertBitDepthToPixelFormat_( tgaHeader.BitDepth ),
                        srcImagePtr);
                    gcHandle.Free();

                    // カラーマップ（パレット）が存在するなら、パレット情報を設定します。
                    if( tgaHeader.ColorMapExist == 1 )
                    {
                        ColorPalette pltt = bmp.Palette;
                        MakeColorPltt_( pltt, tgaHeader, buffer );
                        bmp.Palette = pltt;
                    }

                    // 水平方向フリップを適用します。
                    bmp.RotateFlip( RotateFlipType.RotateNoneFlipY );

                    // フリップビットに対応して、絵を水平反転します。
                    if (((tgaHeader.TexelFmt >> TGAHeader.HFlipShift) & 0x1) != 0x0)
                    {
                        bmp.RotateFlip(RotateFlipType.RotateNoneFlipX);
                    }

                    // フリップビットに対応して、絵を垂直反転します。
                    if (((tgaHeader.TexelFmt >> TGAHeader.VFlipShift) & 0x1) != 0x0)
                    {
                        bmp.RotateFlip(RotateFlipType.RotateNoneFlipY);
                    }

                    return bmp;
                }
                catch( Exception e )
                {
                    MessageBox.Show( e.Message );
                    return null;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsValidFileFmtExt( string filePath )
        {
            return Path.GetExtension( filePath ) == ".tga" ||
                   Path.GetExtension( filePath ) == ".TGA";
        }

        #endregion IBitmapLoader メンバ
    }

    #endregion Bitmap ローダ
}
