﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Diagnostics;
using System.IO;


namespace LECore.Structures
{
    using LECoreInterface;
    using System.Drawing.Drawing2D;

    /// <summary>
    /// Texture 画像 を表すクラスです。
    ///
    /// 実際にレンダリングする際に必要となる、
    /// 描画APIのためのリソースハンドルなども本クラスが管理します。
    /// </summary>
    internal class TextureImage
        : ITextureImage,
          IDisposable
    {
        string _textureName = string.Empty;

        SrcBitmapHolder              _srcBitmapHolder;

        TexImagePixelFmt             _pixelFmt         = TexImagePixelFmt.RGB565;
        bool                         _pixelFmtIsFixed = false;
        Size                         _textureSize = new Size();
        int                          _nativeTextureDataSize = 0;
        int                          _nativeTextureDataSizeNX = 0;

        #region -------------- プロパティ --------------
        /// <summary>
        /// 変更時イベント：現在はTexturemMgrクラスへの通知にためにだけ使用されています。
        /// </summary>
        public event EventHandler       OnChanged;

        #endregion -------------- プロパティ --------------

        #region -------------- プロパティ --------------
        public TexImagePixelFmt    PixelFmt
        {
            get { return LayoutEditorCore.PlatformDetail.TextureFormatEditEnabled ? _pixelFmt : _srcBitmapHolder.InitialFormat; }
            set
            {
                if( !PixelFmtIsFixed )
                {
                    _pixelFmt = value;

                    UpdateNativeTextureDataSize_();

                    NotifyChanged_();
                }else{
                    // フォーマットが固定されている場合は変更できません。
                    Debug.Assert( false );
                }
            }
        }

        public SrcBitmapHolder SrcBitmapHolder { get { return _srcBitmapHolder; } }

        #endregion -------------- プロパティ --------------

        /// <summary>
        /// アルファフォーマットか判定します。
        /// </summary>
        static bool IsAlphaFmt_( TexImagePixelFmt fmt )
        {
            switch (fmt)
            {
                case TexImagePixelFmt.A4:
                case TexImagePixelFmt.A8:
                case TexImagePixelFmt.LA4:
                case TexImagePixelFmt.LA8:
                case TexImagePixelFmt.RGB5A1:
                case TexImagePixelFmt.RGBA4:
                case TexImagePixelFmt.RGBA8:
                case TexImagePixelFmt.ETC1A4:
                    return true;
            }
            return false;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureImage(){}

        /// <summary>
        /// 初期化します。
        /// </summary>
        public void Initialize(
            string filePath,
            SrcBitmapHolder holder,
            Size textureSize,
            bool bFixPixelFmt )
        {
            SetSrcBitmapHolder(holder);

            _textureSize = textureSize;
            _pixelFmt = holder.InitialFormat;
            _pixelFmtIsFixed = bFixPixelFmt;

            this.UpdateState();
        }

        public void SetSrcBitmapHolder(SrcBitmapHolder holder)
        {
            if (_srcBitmapHolder == holder)
            {
                return;
            }

            if (_srcBitmapHolder != null)
            {
                _srcBitmapHolder.References.Remove(this);
            }

            if (holder != null)
            {
                holder.References.Add(this);
            }

            _srcBitmapHolder = holder;
        }

        /// <summary>
        /// テクスチャ更新時に呼び出します。
        /// </summary>
        public void UpdateState()
        {
            _textureName = Path.GetFileNameWithoutExtension(_srcBitmapHolder.FilePath);
            _textureSize = _srcBitmapHolder.GDIBitmap.Size;

            UpdateNativeTextureDataSize_();
        }

        private void UpdateNativeTextureDataSize_()
        {
            _nativeTextureDataSize = LayoutEditorCore.PlatformDetail.CalcNativeTextureSize(_srcBitmapHolder.ActualTextureSize.Width, _srcBitmapHolder.ActualTextureSize.Height, this.PixelFmt);
            _nativeTextureDataSizeNX = LayoutEditorCore.PlatformDetail.CalcNativeTextureSizeNX(_srcBitmapHolder.ActualTextureSize.Width, _srcBitmapHolder.ActualTextureSize.Height, this.PixelFmt, ConvertTilingFlags_(TileSettingParams.TileOptimize), TileSettingParams.TileSizeThreshold);
        }

        private uint ConvertTilingFlags_(string tilingFlags)
        {
            if (tilingFlags.ToLower().Equals("performance"))
            {
                return 0;
            }
            else if (tilingFlags.ToLower().Equals("size"))
            {
                return 2;
            }
            else if (tilingFlags.ToLower().Equals("size_auto"))
            {
                return 3;
            }

            return 0;
        }

        /// <summary>
        /// 不正なファイルとして、初期化します。
        /// </summary>
        public void InitializeAsInvalid(string filePath)
        {
            if (SrcBitmapHolder == null)
            {
                SrcBitmapHolder holder = SrcBitmapHolder.CreateInvalid(filePath);

                Size texSize = holder.GDIBitmap.Size;

                Initialize(filePath, holder, texSize, false);
            }
            else
            {
                SrcBitmapHolder.UpdateAsInvalide(filePath);

                // 参照箇所を更新
                foreach (var texture in SrcBitmapHolder.References)
                {
                    texture.UpdateState();
                }
            }
        }

        #region ITextureImage メンバ

        public TextureSourceType SourceType
        {
            get { return TextureSourceType.File; }
        }

        public string Name
        {
            get
            {
                return _textureName;
            }
        }

        public string FilePath { get { return _srcBitmapHolder != null ? _srcBitmapHolder.FilePath : string.Empty; } }
        public Bitmap GDIBitmap
        {
            get { return _srcBitmapHolder.GDIBitmap; }
        }

        /// <summary>
        /// ビット深度
        /// </summary>
        private float _BPP
        {
            get
            {
                return LECore.LayoutEditorCore.PlatformDetail.GetBPP(this.PixelFmt);
            }
        }

        public Bitmap ColorGDIBitmap { get { return _srcBitmapHolder.ColorGDIBitmap; } }
        public Bitmap AlphaGDIBitmap
        {
            get
            {
                switch (PixelFmt)
                {
                    case TexImagePixelFmt.BC4A:
                    case TexImagePixelFmt.BC5:
                    case TexImagePixelFmt.A4:
                    case TexImagePixelFmt.A8:
                    case TexImagePixelFmt.LA4:
                    case TexImagePixelFmt.LA8:
                        // 明度＋アルファフォーマットなら(ソース画像によらず)必ずアルファ画像を返す。
                        return _srcBitmapHolder.AlphaGDIBitmap;
                    default:
                        // それ以外のフォーマットでは、ソース画像がアルファを持っていた場合だけアルファ画像を返す。
                        return this.SrcImageHasAlphaBit ? _srcBitmapHolder.AlphaGDIBitmap : null;
                }
            }
        }

        public Size TextureSize
        {
            get { return _textureSize; }
        }

        public int TextureDataSize
        {
            get { return (int)Math.Floor(_BPP * _textureSize.Width * _textureSize.Height); }
        }

        public int NativeTextureDataSize
        {
            get { return _nativeTextureDataSize; }
        }

        public int NativeTextureDataSizeNX
        {
            get { return _nativeTextureDataSizeNX; }
        }

        ///
        public IVec3 Size
        {
            get { return new IVec3( _textureSize.Width, _textureSize.Height, 0 ); }
        }

        ///
        public IVec3 ActualSize
        {
            get { return new IVec3(_srcBitmapHolder.GDIBitmap.Width, _srcBitmapHolder.GDIBitmap.Height, 0); }
        }

        ///
        public bool SrcImageHasAlphaBit
        {
            get { return _srcBitmapHolder.HasSrcImageAlpha; }
        }

        ///
        public bool SrcImageIsGrayScale
        {
            get { return _srcBitmapHolder.IsSrcImageGlayScale; }
        }

        public bool PixelFmtIsFixed { get { return _pixelFmtIsFixed; } internal set { _pixelFmtIsFixed = value; } }
        public bool Valid { get { return _srcBitmapHolder != null && _srcBitmapHolder.IsValid; } }

        public DateTime UpdateTime { get { return _srcBitmapHolder != null ? _srcBitmapHolder.UpdateTime : DateTime.MinValue; } }

        public bool IsHidingFromList
        {
            get { return false; }
        }

        public ICaptureTexture ICaptureTexture
        {
            get { return null; }
        }

        #endregion

        #region IDisposable メンバ

        public void Dispose()
        {
        }

        #endregion

        /// <summary>
        /// 変更イベントを通知します。
        /// </summary>
        void NotifyChanged_()
        {
            if( OnChanged != null )
            {
                OnChanged( this, null );
            }
        }
    }

    /// <summary>
    /// Texture 画像 を表すクラスです。
    ///
    /// 実際にレンダリングする際に必要となる、
    /// 描画APIのためのリソースハンドルなども本クラスが管理します。
    /// </summary>
    public class TileSettingParams
    {
        #region -------------- プロパティ --------------

        public static string TileOptimize { get; set; }
        public static int TileSizeThreshold { get; set; }

        #endregion -------------- プロパティ --------------
    }
}
