﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.IO;

namespace LECore.Structures
{
    //----------------------------------------------------------
    // IParamaterPane
    //----------------------------------------------------------

    /// <summary>
    /// コントロールパラメータです。
    /// </summary>
    internal class ControlParamaterBase
    {
        private readonly string _name;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="name">非null,空文字列の名前です。</param>
        public ControlParamaterBase(string name)
        {
            Ensure.Argument.StringNotEmpty(name);
            this._name = name;
        }

        /// <summary>
        /// パラメータの名前です。
        /// </summary>
        public string Name { get { return this._name; } }

        /// <summary>
        /// 持ち主です。
        /// </summary>
        public ControlSettings Owner { get; set; }
    }

    /// <summary>
    /// ペインパラメータです。
    /// </summary>
    internal class ParamaterPane : ControlParamaterBase, IParamaterPane
    {
        private string _paneName;

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ParamaterPane(string name, string paneName, bool isRequired)
            : base(name)
        {
            this._paneName = paneName;
            this.IsRequired = isRequired;
            this._PaneReference = null;
        }

        /// <summary>
        /// ペイン名です。
        /// </summary>
        public string PaneName
        {
            get { return
                this._PaneReference != null && this._PaneReference.OwnerSubScene != null
                ? this._PaneReference.PaneName : this._paneName; }
            set
            {
                if (this._paneName != value)
                {
                    this._paneName = value;
                    this._PaneReference = null;

                    if (this.Owner != null)
                    {
                        ResetReference();
                        this.Owner.RaiseModifyEvent();
                    }
                }
            }
        }

        /// <summary>
        /// 参照をリセットする。
        /// </summary>
        public void ResetReference()
        {
            if (this.Owner != null && this.Owner.Owner != null)
            {
                this._PaneReference = this.Owner.GetOwnerSubScene().FindPaneByName(this._paneName);
            }
        }

        /// <summary>
        /// 設定が必須かどうかです。
        /// </summary>
        public bool IsRequired { get; private set; }

        /// <summary>
        /// ペインへの参照です。
        /// </summary>
        private IPane _PaneReference { get; set; }
    }

    //----------------------------------------------------------
    // IParamaterAnimaiton
    //----------------------------------------------------------

    /// <summary>
    /// アニメーションパラメータです。
    /// </summary>
    internal class ParamaterAnimaiton : ControlParamaterBase, IParamaterAnimaiton
    {
        private string _tagName;

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ParamaterAnimaiton(string name, string tagName, bool isRequired)
            : base(name)
        {
            this._tagName = tagName;
            this.IsRequired = isRequired;
            this._AnimFrameSectionReference = null;
        }

        /// <summary>
        /// アニメーション区間タグの名前です。
        /// </summary>
        public string TagName
        {
            get { return
                this._AnimFrameSectionReference != null && this._AnimFrameSectionReference.Owner != null
                ? this._AnimFrameSectionReference.Name : this._tagName; }
            set
            {
                if (this._tagName != value)
                {
                    this._tagName = value;
                    this._AnimFrameSectionReference = null;

                    if (this.Owner != null)
                    {
                        ResetReference();
                        this.Owner.RaiseModifyEvent();
                    }
                }
            }
        }

        /// <summary>
        /// 参照をリセットする。
        /// </summary>
        public void ResetReference()
        {
            if (this.Owner != null && this.Owner.Owner != null)
            {
                this._AnimFrameSectionReference = this.Owner.GetOwnerSubScene().IAnimFrameSectionSet.FindAnimFrameSectionByName(this._tagName);
            }
        }

        /// <summary>
        /// 設定が必須かどうかです。
        /// </summary>
        public bool IsRequired { get; private set; }

        /// <summary>
        /// ペインへの参照です。
        /// </summary>
        private IAnimFrameSection _AnimFrameSectionReference { get; set; }
    }

    //----------------------------------------------------------
    // PartsPropaertySettings
    //----------------------------------------------------------

    /// <summary>
    /// 部品上書きプロパティの設定です。
    /// </summary>
    internal class PartsPropaertySettings : IPartsPropaertySettings
    {
        private PaneKind _paneKind = PaneKind.Null;
        private string _description = string.Empty;
        private string _targetName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPropaertySettings()
        {
        }

        /// <summary>
        /// 種類です。
        /// </summary>
        public PaneKind PaneKind
        {
            get
            {
                if (TargetPane != null)
                {
                    return this.TargetPane.PaneKind;
                }
                return _paneKind;
            }

            internal set
            {
                if (TargetPane != null)
                {
                    if(TargetPane.PaneKind != value)
                    {
                        TargetPane = null;
                    }
                }
                _paneKind = value;
            }
        }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        public string TargetName
        {
            get
            {
                if (TargetPane != null)
                {
                    return this.TargetPane.PaneName;
                }
                return _targetName;
            }

            internal set
            {
                if (TargetPane != null)
                {
                    if (TargetPane.PaneName != value)
                    {
                        TargetPane = null;
                    }
                }
                _targetName = value;
            }
        }

        /// <summary>
        /// 説明です。
        /// </summary>
        public string Description { get; internal set; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        internal IPane TargetPane
        {
            get;
            set;
        }
    }

    /// <summary>
    /// 設定用一時クラスです。
    /// </summary>
    public class PartsPropaertySettingsSource : IPartsPropaertySettings
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPropaertySettingsSource(IPane source, string desc)
            : this(source.PaneName, source.PaneKind, desc)
        {
            this.TargetPane = source;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPropaertySettingsSource(string target, PaneKind kind, string desc)
        {
            this.TargetName = target;
            this.PaneKind = kind;
            this.Description = desc;
        }

        /// <summary>
        /// 種類です。
        /// </summary>
        public PaneKind PaneKind { get; set; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// 説明です。
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        public IPane TargetPane { get; set; }
    }

    //----------------------------------------------------------
    // PartsSettings
    //----------------------------------------------------------

    /// <summary>
    /// 部品プリセット設定です。
    /// </summary>
    internal class DerivativePartsSettings : IDerivativePartsSettings
    {
        private readonly List<IPartsPropaerty> _partsPropaerties = new List<IPartsPropaerty>();

        //----------------------------------------------------------

        /// <summary>
        /// 名前
        /// </summary>
        public string Name { get; internal set; }

        /// <summary>
        /// 説明
        /// </summary>
        public string Descriptions { get; internal set; }

        /// <summary>
        /// 部品のレイアウトです。
        /// </summary>
        public ISubScene PartsSubScene { get; internal set; }

        /// <summary>
        /// 上書きパラメータ列です。
        /// </summary>
        public IEnumerable<IPartsPropaerty> PartsPropaerties
        {
            get { return this._partsPropaerties; }
        }

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DerivativePartsSettings()
        {
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        public void Initialize(ISubScene originalScene)
        {
            Ensure.Operation.True(this.PartsSubScene == null);

            // シーンを複製
            SubScene sceneClone = Scene.Instance.CreateSubSceneInstance();

            // フォントやテクスチャ類をコピーする
            SubSceneHelper.CopySubSceneTextureAndFont(originalScene, sceneClone);

            // ペインの複製
            foreach (Pane pane in originalScene.IPaneArray)
            {
                // プロパティを優先して使用する。
                Pane newPane = null;
                newPane = pane.CloneWithoutAnimation();
                sceneClone.AddPane(newPane);
            }

            // キャプチャテクスチャをコピーする
            SubSceneHelper.CopySubSceneCaptureTexture(originalScene, sceneClone);

            this.PartsSubScene = sceneClone;
        }

        /// <summary>
        /// プロパティを初期化します。
        /// </summary>
        public void InitializePartsProperties(IEnumerable<IPartsPropaerty> partsPropaerties)
        {
            Ensure.Operation.True(this.PartsSubScene != null);
            Ensure.Operation.True(_partsPropaerties.Count == 0);

            foreach (var prop in partsPropaerties)
            {
                // 上書きパラメータが初期化済み
                if (prop.Paramater != null)
                {
                    // 自身に所属するペインのパラメータでなければ無視します。
                    Ensure.Operation.ObjectNotNull(prop.Paramater.OwnerPane);
                    if (this.PartsSubScene.IPaneArray.FirstOrDefault((pane) => object.ReferenceEquals(pane, prop.Paramater.OwnerPane)) == null)
                    {
                        continue;
                    }
                }

                _partsPropaerties.Add(prop);
            }
        }

        /// <summary>
        /// プロパティを一度だけ初期化します。
        /// </summary>
        public void InitializePartsProperties(
            ISubScene originalScene, IEnumerable<IPartsPropaerty> partsPropaerties)
        {
            Ensure.Operation.True(this.PartsSubScene == null);
            Ensure.Operation.True(_partsPropaerties.Count == 0);

            // シーンを複製
            SubScene sceneClone = Scene.Instance.CreateSubSceneInstance();

            // プロパティを複製
            // TODO:変更されているか、されていないか？を判定できないか？
            foreach (var prop in partsPropaerties)
            {
                PartsPropaerty tempProperty = new PartsPropaerty(
                    prop.PaneKind,
                    prop.TargetName,
                    prop.Paramater != null ? prop.Paramater.OwnerPane.CloneWithoutAnimation().IPaneExParamater : null,
                    prop.Option);
                _partsPropaerties.Add(tempProperty);

                if(prop.PaneKind == PaneKind.Textbox && prop.Paramater != null)
                {
                    sceneClone.ILEFontManager.TryRegisterFont((prop.Paramater as ITextBox).ILEFont);
                }
            }

            // フォントやテクスチャ類をコピーする
            SubSceneHelper.CopySubSceneTextureAndFont(originalScene, sceneClone);

            // ペインの複製
            foreach (Pane pane in originalScene.IPaneArray)
            {
                // プロパティを優先して使用する。
                Pane newPane = null;
                var paneProp = _partsPropaerties.FirstOrDefault(
                    (prop) => prop.TargetName == pane.PaneName);
                if (paneProp != null && paneProp.Paramater != null)
                {
                    newPane = paneProp.Paramater.OwnerPane as Pane;
                }
                else
                {
                    newPane = pane.CloneWithoutAnimation();
                }

                sceneClone.AddPane(newPane);
            }

            // キャプチャテクスチャをコピーする
            SubSceneHelper.CopySubSceneCaptureTexture(originalScene, sceneClone);

            this.PartsSubScene = sceneClone;
        }
    }

    /// <summary>
    /// パーツ設定です。
    /// <remarks>
    /// 中間ファイのControlタグに対応します。
    /// 一度初期化したら、のちの更新は
    /// PaneName、TagNameに対してのみとなります。
    /// </remarks>
    /// </summary>
    internal class PartsSettings : IPartsSettings
    {
        //----------------------------------------------------------
        private string _name = string.Empty;
        //private string _uiName = string.Empty;
        private string _descriptionName = string.Empty;
        private string _description = string.Empty;
        private readonly List<IPartsPropaertySettings> _overridePorperties = new List<IPartsPropaertySettings>();

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsSettings(ISubScene owner)
        {
            this.Owner = owner;
            this.BasePartsName = string.Empty;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 名前です。
        /// </summary>
        public string Name { get { return this._name; } }

        /// <summary>
        /// 短い説明名称です。
        /// </summary>
        public string DescriptionName { get { return this._descriptionName; } }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        public string Description { get { return this._description; } }

        /// <summary>
        /// 上書きプロパティです。
        /// </summary>
        public IEnumerable<IPartsPropaertySettings> OverridePorperties { get { return this._overridePorperties; } }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        public ISubScene Owner { get; private set; }

        /// <summary>
        /// 初期状態かどうかです。
        /// </summary>
        public bool IsInitialState
        {
            get { return string.IsNullOrEmpty(_descriptionName); }
        }

        /// <summary>
        /// 基底レイアウトの名前です。
        /// <remarks>
        /// 派生部品の場合だけ設定されます。
        /// </remarks>
        /// </summary>
        public string BasePartsName { get; private set; }

        /// <summary>
        /// 基底レイアウトの最終変更時間です。
        /// </summary>
        public DateTime BasePartsLastModify { get; private set; }

        /// <summary>
        /// 基底レイアウトのハッシュ値 (SHA1) です。
        /// </summary>
        public string BasePartsHashValue { get; private set; }

        /// <summary>
        /// 元のサイズです。
        /// </summary>
        public FVec3 OriginalSize { get; private set; }

        /// <summary>
        /// 更新イベントです。
        /// </summary>
        public event Action OnChanged;

        //----------------------------------------------------------

        /// <summary>
        /// 初期化します。
        /// <remarks>
        /// 必ず一度だけ呼び出します。
        /// </remarks>
        /// </summary>
        public void Initialize(
            string name,
            string descriptionName,
            string description,
            FVec3 originalSize)
        {
            Ensure.Argument.True(!string.IsNullOrEmpty(name) || !string.IsNullOrEmpty(descriptionName));

            this.DoReset_();

            this._name = name;
            this._descriptionName = descriptionName;
            this._description = description;

            this.OriginalSize = originalSize;

            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 説明を設定します。
        /// </summary>
        public void SetDescriptions(
            string descriptionName,
            string description)
        {
            this._descriptionName = descriptionName;
            this._description = description;

            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 上書きするペインを設定します。
        /// リセットする場合は、空のペイン列を指定します。
        /// </summary>
        public void SetOverrideProperties(IEnumerable<PartsPropaertySettingsSource> src)
        {
            this._overridePorperties.Clear();

            foreach (PartsPropaertySettingsSource srcSetting in src)
            {
                PartsPropaertySettings newSetting = new PartsPropaertySettings();

                if (srcSetting.TargetPane != null)
                {
                    newSetting.TargetPane = srcSetting.TargetPane;
                }
                else
                {
                    newSetting.PaneKind = srcSetting.PaneKind;
                    newSetting.TargetName = srcSetting.TargetName;
                }

                newSetting.Description = srcSetting.Description;

                this._overridePorperties.Add(newSetting);
            }

            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 派生部品として初期化します。
        /// </summary>
        /// <param name="basePartsName"></param>
        public void InitializeAsDelivativeParts(ISubScene basePartsSubScene, string basePartsName)
        {
            SubScene dstSubScene = this.Owner as SubScene;
            try
            {
                dstSubScene.BeginMassiveModify();
                // 全く同一でなければ、まず複製します。
                if (!Object.ReferenceEquals(dstSubScene, basePartsSubScene))
                {
                    // テクスチャ
                    foreach (ITextureImage baseTex in basePartsSubScene.ITextureMgr.ITextureImageSet)
                    {
                        dstSubScene.ITextureMgr.TryRegisterTextureImage(baseTex);
                    }

                    // フォント
                    foreach (ILEFont baseFont in basePartsSubScene.ILEFontManager.ILEFontSet)
                    {
                        dstSubScene.ILEFontManager.TryRegisterFont(baseFont);
                    }

                    // ペイン
                    SubSceneHelper.ClonePanes(dstSubScene, basePartsSubScene);

                    // キャプチャテクスチャ
                    // コピー時にコピー先のペインを参照するためペインのコピー処理の後に実行する。
                    SubSceneHelper.CopySubSceneCaptureTexture(basePartsSubScene, dstSubScene);

                    // グループ
                    foreach (LEGroup baseGroup in basePartsSubScene.ILEGroupMgr.GroupSet)
                    {
                        var groupMember = dstSubScene.GetPaneSet((pane) => baseGroup.ContainsPaneName(pane.PaneName)).ToArray();
                        (dstSubScene.ILEGroupMgr as LEGroupMgr).MakeNewGroup(baseGroup.GrouprName, groupMember);
                    }

                    // 区間タグ
                    AnimFrameSectionSet dstTagSet = dstSubScene.IAnimFrameSectionSet as AnimFrameSectionSet;
                    foreach (AnimFrameSection baseAfs in basePartsSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                    {
                        var newAfs = dstTagSet.AddNewFrameSection(baseAfs.Name, baseAfs.Comment, baseAfs.StartFrame, baseAfs.EndFrame) as AnimFrameSection;
                        var newwAfsBbs = newAfs.BinarySettings as AnimFrameSectionBinarySettings;
                        var baseAfsBbs = baseAfs.BinarySettings;
                        newwAfsBbs.Set(baseAfsBbs.Active, baseAfsBbs.FileName, baseAfsBbs.LoopType, baseAfsBbs.TargetAnimKind);

                        var dstGroups = dstSubScene.ILEGroupMgr.GroupSet.Where(
                            (group) => Array.Exists(baseAfs.TargetGroup, (baseGroup) => baseGroup.GrouprName == group.GrouprName)).ToArray();
                        newAfs.SetTargetGroup(dstGroups, baseAfs.BindAnimationRecursively);
                    }
                }

                // 読み取り専用に設定します。
                {
                    // ペイン
                    foreach (Pane dstPane in dstSubScene.PaneSet)
                    {
                        dstPane.IsReadOnlyLocked = true;
                    }

                    // テクスチャ
                    foreach (ITextureImage dstTex in dstSubScene.ITextureMgr.ITextureImageSet)
                    {
                        (dstSubScene.ITextureMgr as TextureMgr).SetReadOnlyState(dstTex, true);
                    }

                    // フォント
                    foreach (LEFont dstFont in dstSubScene.ILEFontManager.ILEFontSet)
                    {
                        dstSubScene.FontManager.SetReadOnlyState(dstFont, true);
                    }

                    // グループ
                    foreach (LEGroup dstGroup in dstSubScene.ILEGroupMgr.GroupSet)
                    {
                        dstSubScene.LEGroupMgr.SetReadOnlyState(dstGroup, true);
                    }

                    // 区間タグ
                    foreach (AnimFrameSection dstAfs in dstSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                    {
                        (dstSubScene.IAnimFrameSectionSet as AnimFrameSectionSet).SetReadOnlyState(dstAfs, true);
                    }
                    (dstSubScene.IAnimFrameSectionSet as AnimFrameSectionSet).ModeReadOnly = true; // 分割/結合ボタンを無効化
                }

                // 基底部品情報
                this.BasePartsName = basePartsName;

                IPartsSubScene baseParts = Scene.Instance.FindPartsSubSceneByFileName(basePartsName);
                if (baseParts != null && File.Exists(baseParts.FilePath))
                {
                    // 保存時の最終更新時間
                    this.BasePartsLastModify = File.GetLastWriteTime(baseParts.FilePath);
                    this.BasePartsHashValue = FileUpdateHelper.ComputeSha1FromFlytAndFlan(baseParts.FilePath);
                }
            }
            finally
            {
                dstSubScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// 参照を解決します。
        /// </summary>
        /// <param name="subScene"></param>
        public void ResolveReference(ISubScene subScene)
        {
            this._overridePorperties.ForEach(
                (prop) => (prop as PartsPropaertySettings).TargetPane = subScene.FindPaneByName(prop.TargetName));

            this._overridePorperties.Sort((lhs, rhs) => PaneHelper.ComparePaneByTreeOrder(
                (lhs as PartsPropaertySettings).TargetPane, (rhs as PartsPropaertySettings).TargetPane));
        }

        /// <summary>
        /// リセットします。
        /// </summary>
        public void Reset()
        {
            this.DoReset_();
            this.SetOverrideProperties(new PartsPropaertySettingsSource[0]);

            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 更新イベントを送信します。
        /// </summary>
        public void RaiseModifyEvent()
        {
            if (this.OnChanged != null) { this.OnChanged(); }
        }

        //----------------------------------------------------------

        /// <summary>
        /// リセットをします。
        /// </summary>
        private void DoReset_()
        {
            this._name = string.Empty;
            this._descriptionName = string.Empty;
            this._description = string.Empty;
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class ControlSettingsSet : IControlSettingsSet
    {
        readonly List<IControlSettings> _contorolSettings = new List<IControlSettings>();
        readonly ISubScene _Owner = null;

        /// <summary>
        /// 更新イベントです。
        /// </summary>
        public event Action OnChanged;

        /// <summary>
        ///
        /// </summary>
        public ControlSettingsSet(ISubScene scene)
        {
            _Owner = scene;
        }

        /// <summary>
        ///
        /// </summary>
        public void CloneAdd(IControlSettings srcItem)
        {
            ControlSettings newCtrl = new ControlSettings(this);
            newCtrl.InitializeNoEvent(srcItem.Name, srcItem.UIName, srcItem.DescriptionName, srcItem.Description, srcItem.Panes, srcItem.Animations);

            (srcItem.IUserDataHolder as UserDataHolder).CloneCopyTo(newCtrl.IUserDataHolder as UserDataHolder);

            _contorolSettings.Add(newCtrl);
            RaiseModifyEvent();
        }

        /// <summary>
        ///
        /// </summary>
        public void Remove(IControlSettings newItem)
        {
            _contorolSettings.RemoveAll((ctrl) => ctrl.DescriptionName == newItem.DescriptionName);
            RaiseModifyEvent();
        }

        /// <summary>
        ///
        /// </summary>
        public void Clear()
        {
            _contorolSettings.Clear();
            RaiseModifyEvent();
        }

        public ISubScene Owner { get { return _Owner; } }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<IControlSettings> Items
        {
            get { return _contorolSettings; }
        }

        /// <summary>
        /// 更新イベントを送信します。
        /// </summary>
        public void RaiseModifyEvent()
        {
            if (this.OnChanged != null) { this.OnChanged(); }
        }
    }


    /// <summary>
    ///
    /// </summary>
    internal class ControlSettings : IControlSettings
    {
        //----------------------------------------------------------
        private string _name = string.Empty;
        private string _uiName = string.Empty;
        private string _descriptionName = string.Empty;
        private string _description = string.Empty;
        private readonly List<IParamaterPane> _panes = new List<IParamaterPane>();
        private readonly List<IParamaterAnimaiton> _animations = new List<IParamaterAnimaiton>();

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ControlSettings(IControlSettingsSet owner)
        {
            this.Owner = owner;

            // 拡張ユーザデータ
            var userData = new UserDataHolder((SubScene)this.GetOwnerSubScene());
            userData.OnChanged = new Action(() => this.RaiseModifyEvent());
            this.IUserDataHolder = userData;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 名前です。
        /// </summary>
        public string Name { get { return this._name; } }

        /// <summary>
        /// 名前です。
        /// </summary>
        public string UIName { get { return this._uiName; } }

        /// <summary>
        /// 短い説明名称です。
        /// </summary>
        public string DescriptionName { get { return this._descriptionName; } }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        public string Description { get { return this._description; } }

        /// <summary>
        /// Readonlyのペインパラメータ列です。
        /// </summary>
        public IEnumerable<IParamaterPane> Panes { get { return this._panes; } }

        /// <summary>
        /// Readonlyのアニメーションパラメータ列です。
        /// </summary>
        public IEnumerable<IParamaterAnimaiton> Animations { get { return this._animations; } }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        public IControlSettingsSet Owner { get; private set; }

        /// <summary>
        /// 拡張ユーザデータ
        /// </summary>
        public IUserDataHolder IUserDataHolder { get; private set; }


        //----------------------------------------------------------

        /// <summary>
        /// 初期化します。
        /// <remarks>
        /// 必ず一度だけ呼び出します。
        /// </remarks>
        /// </summary>
        public void Initialize(
            string name,
            string uiName,
            string descriptionName,
            string description,
            IEnumerable<IParamaterPane> panes,
            IEnumerable<IParamaterAnimaiton> animations)
        {
            InitializeNoEvent(name, uiName, descriptionName, description, panes, animations);
            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 初期化します。
        /// <remarks>
        /// 必ず一度だけ呼び出します。
        /// </remarks>
        /// </summary>
        public void InitializeNoEvent(
            string name,
            string uiName,
            string descriptionName,
            string description,
            IEnumerable<IParamaterPane> panes,
            IEnumerable<IParamaterAnimaiton> animations)
        {
            Ensure.Argument.StringNotEmpty(name);
            Ensure.Argument.NotNull(panes);
            Ensure.Argument.NotNull(animations);

            this.DoReset_();

            this._name = name;
            this._uiName = uiName;
            this._descriptionName = descriptionName;
            this._description = description;

            foreach (var pane in panes)
            {
                this._panes.Add(new ParamaterPane(pane.Name, pane.PaneName, pane.IsRequired));
            }

            foreach (var animation in animations)
            {
                this._animations.Add(new ParamaterAnimaiton(animation.Name, animation.TagName, animation.IsRequired));
            }

            this._panes.ForEach((pane) => (pane as ParamaterPane).Owner = this);
            this._animations.ForEach((anim) => (anim as ParamaterAnimaiton).Owner = this);

            this.ResetReference();
        }

        /// <summary>
        /// 参照関係をリセットします。
        /// </summary>
        public void ResetReference()
        {
            this._panes.ForEach((pane) => (pane as ParamaterPane).ResetReference());
            this._animations.ForEach((anim) => (anim as ParamaterAnimaiton).ResetReference());
        }

        /// <summary>
        /// 説明を設定します。
        /// </summary>
        public void SetDescriptions(
            string descriptionName,
            string description)
        {
            this._descriptionName = descriptionName;
            this._description = description;

            this.RaiseModifyEvent();
        }

        /// <summary>
        /// リセットします。
        /// </summary>
        public void Reset()
        {
            this.DoReset_();
            this.RaiseModifyEvent();
        }

        /// <summary>
        /// 更新イベントを送信します。
        /// </summary>
        public void RaiseModifyEvent()
        {
            (this.Owner as ControlSettingsSet).RaiseModifyEvent();
        }

        //----------------------------------------------------------

        /// <summary>
        /// リセットをします。
        /// </summary>
        private void DoReset_()
        {
            this._panes.ForEach((pane) => (pane as ParamaterPane).Owner = null);
            this._animations.ForEach((anim) => (anim as ParamaterAnimaiton).Owner = null);

            this._panes.Clear();
            this._animations.Clear();

            this._name = string.Empty;
            this._uiName = string.Empty;
            this._descriptionName = string.Empty;
            this._description = string.Empty;

            (this.IUserDataHolder as UserDataHolder).RemoveUserDataElementAll();
        }
    }
}
