﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Numerics;



namespace LECore.Structures
{
    using Core;
    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Util;

    /// <summary>
    /// ペインのアニメーションアトリビュート
    /// アプリケーション領域固有の 具象アトリビュートクラス。
    ///
    /// </summary>
    internal class PaneAttribute :
        AnmAttribute,
        IPaneAttribute
    {
        #region 定数
        static readonly AnmAttrDescripter SelfAttrDesc =
            new AnmAttrDescripter(AttributeType.Combined, "pane", null, null);

        #endregion 定数

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="name"></param>
        public PaneAttribute(Pane ownerPane, int x, int y, int w, int h)
            : base(ownerPane,
                  new[] {
                      new AnmAttrDescripter(AttributeType.FloatVec3, "pos", null, new FVec3( x, y, 0.0f )),
                      new AnmAttrDescripter(AttributeType.FloatVec2, "size", null, new FVec2( w, h )),
                      new AnmAttrDescripter(AttributeType.FloatVec2, "scale", null, new FVec2( 1.0f, 1.0f )),
                      new AnmAttrDescripter(AttributeType.FloatVec3, "rotate", null, new FVec3( 0.0f, 0.0f, 0.0f )),
                      new AnmAttrDescripter(AttributeType.Byte, "transparency", null, (byte)255),
                      new AnmAttrDescripter(AttributeType.Bool, "visible", null, true)},
                  SelfAttrDesc)
        {
        }

        #region サブアトリビュート取得
        public AnmAttribute  TransAttr
        {
            get{ return FindAttributeByIdx( 0 );}
        }

        public AnmAttribute  SizeAttr
        {
            get{ return FindAttributeByIdx( 1 );}
        }

        public AnmAttribute  ScaleAttr
        {
            get{ return FindAttributeByIdx( 2 );}
        }

        public AnmAttribute  RotAngAttr
        {
            get{ return FindAttributeByIdx( 3 );}
        }

        public AnmAttribute TransparencyAttr
        {
            get { return FindAttributeByIdx( 4 ); }
        }

        public AnmAttribute  VisibleAttr
        {
            get{ return FindAttributeByIdx( 5 );}
        }
        #endregion サブアトリビュート取得

        #region IPaneAttribute
        public IAnmAttribute  TransIAttr
        {
            get{ return TransAttr;}
        }

        public IAnmAttribute  SizeIAttr
        {
            get{ return SizeAttr;}
        }

        public IAnmAttribute  ScaleIAttr
        {
            get{ return ScaleAttr;}
        }

        public IAnmAttribute  RotAngIAttr
        {
            get{ return RotAngAttr;}
        }

        public IAnmAttribute TransparencyIAttr
        {
            get { return TransparencyAttr; }
        }

        public IAnmAttribute  VisibleIAttr
        {
            get{ return VisibleAttr;}
        }
        #endregion IPaneAttribute
    }

    /// <summary>
    /// 拡張ユーザデータ保持クラス
    /// </summary>
    internal class UserDataHolder : AnmAttribute, IUserDataHolder
    {
        #region 定数

        static readonly AnmAttrDescripter SelfAttrDesc =
            new AnmAttrDescripter( AttributeType.Combined, "UserData", null, null) ;

        #endregion 定数

        List<UserDataElement> _userDataElementSet = new List<UserDataElement>();

        private SubScene _ownerSubScene;

        /// <summary>
        /// コンストラクタです。
        /// ペインの拡張ユーザーデータ用に初期設定を行います。
        /// </summary>
        /// <param name="ownerPane">オーナーペイン</param>
        public UserDataHolder( Pane ownerPane )
            :base( ownerPane, null, SelfAttrDesc )
        {
        }

        // TODO: _ownerSubScene は使われていないので削除
        /// <summary>
        /// コンストラクタです。
        /// 部品の拡張ユーザーデータ用に初期設定を行います。
        /// </summary>
        /// <param name="ownerSubScene">オーナーサブシーン</param>
        public UserDataHolder( SubScene ownerSubScene )
            :base( null, null, SelfAttrDesc )
        {
            _ownerSubScene = ownerSubScene;
        }

        public UserDataHolder()
            :base( null, null, SelfAttrDesc )
        {
        }

        /// <summary>
        /// 持ち主ペインを取得します。（ペインが持ち主ではない場合 null が返ります）
        /// </summary>
        private IPane OwnerPane_
        {
            get { return this.OwnerNode as Pane; }
        }

        /// <summary>
        /// 変更時通知
        /// </summary>
        public Action OnChanged { get; set; }

        /// <summary>
        /// アニメーションが有効かどうか取得します。
        /// </summary>
        public bool AnimationEnabled
        {
            get
            {
                var ownerPane = this.OwnerPane_;
                return (ownerPane != null) ? ownerPane.ExtUserDataAnimEnabled : false;
            }
        }

        /// <summary>
        /// 上書きデータかどうか取得します。
        /// </summary>
        public bool Overwrite
        {
            get
            {
                if (this.OwnerNode != null)
                {
                    Pane ownerPane = (Pane)this.OwnerNode;
                    return (ownerPane.OwnerSubScene != null && ownerPane.OwnerSubScene.IPartsLayout != null);
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// 拡張ユーザデータ配列を取得します。
        /// </summary>
        public IUserDataElement[] UserDataElementSet
        {
            get { return _userDataElementSet.ToArray(); }
        }

        /// <summary>
        /// 上書き元の拡張ユーザデータを取得します。
        /// </summary>
        /// <param name="name">名前</param>
        /// <returns>
        /// 上書き元の拡張ユーザデータを返します。
        /// 上書き元の拡張ユーザデータが見つからないときはnullを返します。
        /// </returns>
        public IUserDataElement FindOriginalData(string name)
        {
            if (this.OwnerNode == null)
            {
                return null;
            }

            Pane ownerPane = (Pane)this.OwnerNode;
            if (ownerPane.OwnerSubScene == null || ownerPane.OwnerSubScene.IPartsLayout == null)
            {
                return null;
            }

            IPane originalPane = ownerPane.OwnerSubScene.IPartsLayout.GetOriginalPane(ownerPane.PaneName);
            if (originalPane == null)
            {
                return null;
            }

            return UserDataElementHelper.FindByName(name, originalPane.IUserDataHolder.UserDataElementSet);
        }

        /// <summary>
        /// ユーザ情報項目を追加します。
        /// </summary>
        public bool AddUserDataElement(string name, UserDataKind kind, bool overwrite, object value)
        {
            if (UserDataElementHelper.CheckParamatersValid(name, kind, value))
            {
                if (UserDataElementHelper.FindByName(name, _userDataElementSet) == null)
                {
                    UserDataElement data = new UserDataElement(name, kind, overwrite, value);
                    _userDataElementSet.Add(data);

                    // アニメーションならAnmAttributeに項目を追加
                    if (data.UserDataAttribute != null)
                    {

                        data.UserDataAttribute.OwnerNode = this;
                        _subAttrArray.Add(data.UserDataAttribute);

                        NotifyModifyEvent(this, (int)EventKind.AttributeModify);
                    }

                    if (OnChanged != null) OnChanged();
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 番号を指定してユーザー情報項目を追加します。
        /// </summary>
        /// <param name="element">ユーザー情報項目</param>
        /// <param name="elementIndex">エレメント番号</param>
        /// <param name="attributeIndex">アトリビュート番号</param>
        public bool InsertUserDataElement(string name, UserDataKind kind, bool overwrite, object value, int elementIndex, int attributeIndex)
        {
            if (UserDataElementHelper.CheckParamatersValid(name, kind, value))
            {
                if (UserDataElementHelper.FindByName(name, _userDataElementSet) == null)
                {
                    UserDataElement data = new UserDataElement(name, kind, overwrite, value);
                    _userDataElementSet.Insert(elementIndex, data);

                    // アニメーションならAnmAttributeに項目を追加 (TODO: attributeIndex == -1にならないようにする)
                    if (data.UserDataAttribute != null && attributeIndex != -1) {
                        data.UserDataAttribute.OwnerNode = this;
                        _subAttrArray.Insert(attributeIndex, data.UserDataAttribute);
                        NotifyModifyEvent(this, (int)EventKind.AttributeModify);
                    }

                    if (OnChanged != null) OnChanged();
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// ユーザ情報項目を削除します。
        /// </summary>
        public bool RemoveUserDataElement(string name)
        {
            UserDataElement data = UserDataElementHelper.FindByName(name, _userDataElementSet) as UserDataElement;
            if (data != null)
            {
                _userDataElementSet.Remove(data);

                // アニメーションデータならAnmAttributeから項目を削除
                if (data.IAnmAttribute != null)
                {
                    this._subAttrArray.Remove(data.IAnmAttribute);
                    NotifyModifyEvent(this, (int)EventKind.AttributeModify);
                }

                if (OnChanged != null) OnChanged();
                return true;
            }

            return false;
        }

        /// <summary>
        /// ユーザー情報項目を追加します。
        /// </summary>
        /// <param name="element">ユーザー情報項目</param>
        public void AddUserDataElement(UserDataElement element)
        {
            _userDataElementSet.Add(element);

            if (element.UserDataAttribute != null) {
                _subAttrArray.Add(element.UserDataAttribute);
                NotifyModifyEvent(this, (int)EventKind.AttributeModify);
            }

            if (OnChanged != null) OnChanged();
        }

        /// <summary>
        /// 番号を指定してユーザー情報項目を追加します。
        /// </summary>
        /// <param name="element">ユーザー情報項目</param>
        /// <param name="elementIndex">エレメント番号</param>
        /// <param name="attributeIndex">アトリビュート番号</param>
        public void InsertUserDataElement(UserDataElement element, int elementIndex, int attributeIndex)
        {
            _userDataElementSet.Insert(elementIndex, element);

            // TODO: attributeIndex == -1にならないようにする
            if (element.UserDataAttribute != null && attributeIndex != -1) {
                _subAttrArray.Insert(attributeIndex, element.UserDataAttribute);
                NotifyModifyEvent(this, (int)EventKind.AttributeModify);
            }

            if (OnChanged != null) OnChanged();
        }

        /// <summary>
        /// ユーザー情報項目を削除します。
        /// </summary>
        /// <param name="element">ユーザー情報項目</param>
        public void RemoveUserDataElement(UserDataElement element)
        {
            _userDataElementSet.Remove(element);

            if (element.UserDataAttribute != null) {
                _subAttrArray.Remove(element.UserDataAttribute);
                NotifyModifyEvent(this, (int)EventKind.AttributeModify);
            }

            if (OnChanged != null) OnChanged();
        }

        /// <summary>
        /// ユーザ情報項目をすべて削除します。
        /// </summary>
        public bool RemoveUserDataElementAll()
        {
            _userDataElementSet.Clear();
            _subAttrArray.Clear();
            NotifyModifyEvent(this, (int)EventKind.AttributeModify);

            if (OnChanged != null) OnChanged();
            return true;
        }

        /// <summary>
        /// ユーザー情報項目を復元します。
        /// </summary>
        public void RestoreUserDataElement(UserDataElement[] elements)
        {
            _userDataElementSet.Clear();
            _subAttrArray.Clear();

            foreach (UserDataElement element in elements)
            {
                _userDataElementSet.Add(element);

                if (element.UserDataAttribute != null) {
                    _subAttrArray.Add(element.UserDataAttribute);
                }
            }

            NotifyModifyEvent(this, (int)EventKind.AttributeModify);

            if (OnChanged != null) OnChanged();
        }

        /// <summary>
        /// ユーザ情報項目の番号を変更します。
        /// </summary>
        public bool ChangeUserDataIndex(string name, int newIndex)
        {
            UserDataElement data = UserDataElementHelper.FindByName(name, _userDataElementSet) as UserDataElement;
            if (data != null)
            {
                int lengthAfterRemoving = _userDataElementSet.Count - 1;

                if (newIndex >= 0 && newIndex <= lengthAfterRemoving)
                {
                    _userDataElementSet.Remove(data);
                    _userDataElementSet.Insert(newIndex, data);
                    NotifyModifyEvent(this, (int)EventKind.AttributeModify);
                    if (OnChanged != null) OnChanged();
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// ユーザ情報項目のパラメータを設定します。
        /// </summary>
        public bool SetUserDataPamaters(string name, string newName, bool overwrite, UserDataKind kind, object value, object baseValue)
        {
            if (UserDataElementHelper.CheckParamatersValid(newName, kind, value))
            {
                if (name == newName ||
                    UserDataElementHelper.FindByName(newName, _userDataElementSet) == null)
                {
                    UserDataElement data = UserDataElementHelper.FindByName(name, _userDataElementSet) as UserDataElement;


                    if (data != null)
                    {
                        if (UserDataElementHelper.IsInvalidModificationOfKind(data.UserDataKind, kind))
                        {
                            return false;
                        }

                        data.Set(newName, kind, overwrite, value, baseValue);
                        NotifyModifyEvent(this, (int)EventKind.AttributeModify);
                        if (OnChanged != null) OnChanged();
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public void CloneCopyTo(UserDataHolder dst)
        {
            dst.RemoveUserDataElementAll();
            foreach (var ud in this.UserDataElementSet)
            {
                dst.AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
            }
        }
    }

    /// <summary>
    /// ペインです。
    /// 具体的なペインの基底クラスです。
    /// </summary>
    internal class Pane :
        LEDataNode,
        IPane,
        IDisposable
    {
        #region -------------- 定数 --------------
        // 描画にしようする最小のスケール値
        const float _MinScaleForDraw = 0.01f;
        #endregion -------------- 定数 --------------

        internal class NullPaneExData : IPaneExParamater
        {
            readonly IPane _owner = null;

            public NullPaneExData(IPane owner)
            {
                _owner = owner;
            }

            public IPane OwnerPane
            {
                get { return _owner; }
            }

            public void UpdateMatarialName()
            {

            }

            public void OnJoinSceneInitialize(){}

            /// <summary>
            /// 編集対象に設定される直前の初期化処理
            /// </summary>
            public void FirstTimeInitialize()
            {
                // 何もしません。
            }

            public IEnumerable<TexCoord4> TexCoords
            {
                get { yield break; }
            }

            public string WarningMsg
            {
                get { return string.Empty; }
            }

            /// <summary>
            /// 境界を書くかどうか
            /// </summary>
            public bool PaneBoundDrawEnabled
            {
                get { return true; }
            }

            public void Draw(IRenderer renderer, DrawableOption option){}

            public void Dispose(){}
        }

        #region -------------- フィールド --------------
        // ペインを表現する矩形（テスト的に）
        readonly PaneAttribute        _attribute    = null;

        // ペインが選択状態か
        bool             _isSelected   = false;
        // 隠されている状態か？
        bool             _isLocked     = false;
        bool             _isHidden     = false;
        bool             _isReadOnlyLocked = false;
        bool             _avoidPaneTreeCompression = false;
        bool             _extUserDataAnimEnabled = false;

        // 自身の透明度が、子供の透明度に影響を与えるか。
        bool _influenceChildrenTransparency = false;
        // 位置調整演算の対象にするかどうか？
        bool             _locationAdjust = false;

        bool _isAlignmentHorizontalMarginEnabled = false;
        bool _isAlignmentVerticalMarginEnabled = false;
        float _alignmentHorizontalMargin = 0.0f;
        float _alignmentVerticalMargin = 0.0f;
        bool _isAlignmentIgnore = false;

        // 部品サイズ調整を無視するかどうか？
        PartsMagnifyInfluence _partsMagnifyInfluence = PartsMagnifyInfluence.ScaleMagnify;

        //
        // IHierarchyNode 関連
        //
        // 親ノード
        IHierarchyNode   _parentNode      = null;
        // 子供ノード
        List<IHierarchyNode>        _childrenNodeSet = new List<IHierarchyNode>();

        // ペイン中心に関する設定
        HorizontalLocation      _basePosTypeH = HorizontalLocation.Center;
        VerticalLocation        _basePosTypeV = VerticalLocation.Center;
        HorizontalLocation _parentBasePosTypeH = HorizontalLocation.Center;
        VerticalLocation _parentBasePosTypeV = VerticalLocation.Center;


        string                  _paneName;

        // TextBox-PaneModify や Picture-PaneModify の情報
        IPaneExParamater		  _paneExParamater;

        // マスクの情報
        IMask _mask = null;

        // ドロップシャドウの情報
        IDropShadow _dropShadow = null;

        // コメント文字列：ランタイムデータに反映されません。
        string                   _userCommentString = string.Empty;
        // ユーザ情報文字列：ランタイムデータに反映されます。
        string                   _userDataString    = string.Empty;

        ISubScene                _ownerSubScene = null;
        FVec3 _sizeMagnify = new FVec3(1.0f, 1.0f, 1.0f);
        readonly UserDataHolder _userDataHolder = null;

        #endregion // -------------- フィールド --------------

        #region  -------------- 公開イベント --------------
        /// <summary>
        /// ペイン更新時に呼び出されるイベントです。
        /// </summary>
        public delegate void OnPainModifyHandler( Pane sender, SceneModifyEventArgs args );
        public event    OnPainModifyHandler   OnPainModify;

        #endregion // -------------- 公開イベント --------------

        #region --------------- プロパティ ---------------

        /// <summary>
        /// サイズ倍率です。
        /// </summary>
        public FVec3 SizeMagnify
        {
            get { return _sizeMagnify; }
            set { _sizeMagnify = value; }
        }

        #endregion // --------------- プロパティ ---------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Pane()
            : base( null, PaneHelper.PaneNodeName )
        {
            _attribute    = new PaneAttribute( this, 0, 0, 30, 40 );
            _paneExParamater = new NullPaneExData(this);
            this.SizeMagnify = FVec3.One;
            _mask = new Mask(this);
            _dropShadow = new DropShadow(this);

            _userDataHolder = new UserDataHolder(this);
            _userDataHolder.OnChanged = new Action(() => NotifyChangeToScene_());
        }

        /// <summary>
        /// リソースの開放を行います。
        /// </summary>
        public void Dispose()
        {
            _attribute.Dispose();

            if( _paneExParamater != null )
            {
                _paneExParamater.Dispose();
            }
        }

        /// <summary>
        /// シーン登録時の初期化処理
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            if( this.IPaneExParamater != null )
            {
                this.IPaneExParamater.OnJoinSceneInitialize();
            }
        }

        /// <summary>
        /// 平行移動します。
        /// </summary>
        public void Offset( PointF trans )
        {
            PointF loc = new PointF( this.Trans.X, this.Trans.Y );

            // 親ペインローカル系のベクトルに変換
            Matrix mtxInvPane = this.ParentMtx;
            mtxInvPane.Invert();
            trans = MathUtil.MtxTransformVec( mtxInvPane, trans );


            loc = MathUtil.AddVec( loc, trans );

            this.Trans = new FVec3( loc.X, loc.Y, 0.0f );
        }

        /// <summary>
        /// マテリアル名からマテリアルを検索します。
        /// </summary>
        public Material FindMaterialByName( string name )
        {
            foreach( Material material in this.Material )
            {
                if( material.MaterialName == name )
                {
                    return material;
                }
            }
            return null;
        }

        #region ペイン生成

        /// <summary>
        /// ペイン拡張情報を設定します。
        /// </summary>
        public void BindPaneExData( IPaneExParamater paneExParamaters )
        {
            if( _paneExParamater != null )
            {
                _paneExParamater.Dispose();
            }

            _paneExParamater = paneExParamaters;
        }

        // マスクの設定
        public void BindPaneMask(IMask mask)
        {
            _mask = mask;
        }

        // ドロップシャドウの設定
        public void BindPaneDropShadow(IDropShadow dropShadow)
        {
            _dropShadow = dropShadow;
        }

        #endregion ペイン生成

        #region IDrawable 実装

        /// <summary>
        /// ペインを描画します。
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            // 隠された状態であれば、描画をしません。
            if( Hidden )
            {
                return;
            }

            // スケール値が有効な状態でなければ描画しません。
            if( !ValidScaleForDraw )
            {
                return;
            }

            renderer.PushMtx();
            renderer.Scale(1.0f, -1.0f);

            //----------------------------------
            // ペイン透明度に影響を受けない描画
            int currentTransparency = renderer.Transparency;
            renderer.Transparency = 255;

            if( _isSelected )
            {
                renderer.Color = option.SystemRed;
                RendererHelper.DrawCrossWithoutTransform(renderer, PointF.Empty, 20.0f);
            }
            else if( Locked )
            {
                renderer.Color = option.SystemBlue;
            }
            else
            {
                renderer.Color = option.SystemGreen;
            }

            ISubScene subScene = this.OwnerSubScene;

            if (subScene != null && this == subScene.CurrentPane)
            {
                renderer.Color = option.SystemWhite;
            }

            renderer.PushMtx();
            renderer.Trans( DrawOffsetX, DrawOffsetY );

            // 境界枠の描画
            if( !option.CheckActive( DrawableOptionFlag.IgnoreBVDrawing ) )
            {
                if (_isSelected
                    || _paneExParamater == null
                    || _paneExParamater.PaneBoundDrawEnabled
                    || (subScene != null && this == subScene.CurrentPane))
                {
                    renderer.DrawRectangle(0, 0, 0, this.RenderingWidth, this.RenderingHeight);
                }
            }

            renderer.Transparency = currentTransparency;

            //----------------------------------
            // ペイン透明度に影響を受ける描画
            if( Visible )
            {
                if( _paneExParamater != null && !(_paneExParamater is NullPaneExData) )
                {
                    _paneExParamater.Draw( renderer, option );
                }
                else
                {
                    if( !option.CheckActive( DrawableOptionFlag.IgnoreDummyObjectDrawing ) )
                    {
                        renderer.DrawRectangle(0, 0, 0, this.RenderingWidth, this.RenderingHeight);
                    }
                }
            }

            renderer.PopMtx();

#if DEBUG
            // ロックされている場合は
            // デバックビルドでは、デバック文字列を表示します。
            if( Locked )
            {
                renderer.DbgDrawString( "Locked", LayoutEditorCore.GuiFont, 0.0f, 0.0f );
                renderer.DrawRectangle(DrawOffsetX, DrawOffsetY, 0, this.RenderingWidth, this.RenderingHeight);
            }
#endif
            renderer.PopMtx();
        }
        #endregion

        #region ICollidable メンバ

        /// <summary>
        /// 境界矩形を返します。
        /// AABB(Axis Aligned Bounding Box)です。
        /// 今後、OBBにする必要があるかもしれません。
        /// </summary>
        public RectangleF BoundingVolumeInWorld
        {
            get{ return PaneHelper.GetBoundingVolumeInWorld( this ); }
        }

        public bool Contains( FVec2    posInScene )
        {
//            // ワールド行列とローカル変換行列を乗算します。
//            Matrix mtx = this.WorldMtx;
//            // 描画されるペイン右上位置への変換
//            mtx.Translate( DrawOffsetX, DrawOffsetY );
//            // ペイン中心への変換
//            mtx.Translate( Size.X * 0.5f, Size.Y * 0.5f );
//
//            // 逆行列を求めます。
//            mtx.Invert();
//
//            // 入力点を逆行列で変換します、本ペインが軸整列し、
//            // ペインの中心が原点である座標系に、入力点が変換されます。
//            posInScene.AsPointF = MathUtil.MtxTransformPoint( mtx, posInScene.AsPointF );
//
//
//            return (posInScene.X < this.Size.X * 0.5f &&
//                    posInScene.X > -this.Size.X * 0.5f) &&
//                   (posInScene.Y < this.Size.Y * 0.5f &&
//                    posInScene.Y > -this.Size.Y * 0.5f);
              RectangleF bv = BoundingVolumeInWorld;

              return ( bv.Left <= posInScene.X && bv.Right >= posInScene.X ) &&
                     ( bv.Bottom >= posInScene.Y && bv.Top <= posInScene.Y ); // TODO:Top <=> Bottom
        }


        #endregion

        #region IHierarchyNode メンバ
        /// <summary>
        /// ペインのX位置
        /// </summary>
        public float X
        {
            get{ return this.Trans.X;}
            set{ this.Trans = new FVec3( value, Trans.Y, Trans.Z );}
        }

        public float XInWorld
        {
            get { return WorldMtx.OffsetX; }
        }

        public float XInWorld4x4
        {
            get { return WorldMtx4x4.M41; }
        }

        /// <summary>
        /// ペインのY位置
        /// </summary>
        public float Y
        {
            get{ return Trans.Y;}
            set{ Trans = new FVec3( Trans.X, value, Trans.Z );}
        }

        public float YInWorld
        {
            get { return WorldMtx.OffsetY; }
        }

        public float YInWorld4x4
        {
            get { return WorldMtx4x4.M42; }
        }

        /// <summary>
        /// ペインのZ位置
        /// </summary>
        public float Z
        {
            get{ return Trans.Z;}
            set{ Trans = new FVec3( Trans.X, Trans.Y, value ); }
        }

        public float ZInWorld4x4
        {
            get { return WorldMtx4x4.M43; }
        }

        /// <summary>
        /// 回転角X
        /// </summary>
        public float RotAngleX
        {
            get { return RotAng.X; }
            set { RotAng = new FVec3(value, RotAng.Y, RotAng.Z); }
        }

        /// <summary>
        /// 回転角Y
        /// </summary>
        public float RotAngleY
        {
            get { return RotAng.Y; }
            set { RotAng = new FVec3(RotAng.X, value, RotAng.Z); }
        }

        /// <summary>
        /// 回転角Z
        /// </summary>
        public float RotAngleZ
        {
            get{ return RotAng.Z;}
            set{ RotAng = new FVec3( RotAng.X, RotAng.Y, value );}
        }

        /// <summary>
        /// 親ノード
        /// </summary>
        public IHierarchyNode Parent
        {
            get{ return _parentNode;}
            set
            {
               _parentNode = value;
            }
        }

        /// <summary>
        /// 子ノード
        /// </summary>
        public IHierarchyNode[] Children
        {
            get{ return _childrenNodeSet.ToArray();}
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        public Matrix34 LocalMtx34
        {
            get
            {
                var mR = SharpDX.Matrix.RotationX(MathUtil.DegToRad(RotAng.X)) * SharpDX.Matrix.RotationY(MathUtil.DegToRad(RotAng.Y)) * SharpDX.Matrix.RotationZ(MathUtil.DegToRad(RotAng.Z));
                var mS = SharpDX.Matrix.Scaling(Scale.X, Scale.Y, Scale.Z);
                var mT = SharpDX.Matrix.Translation(this.GetParentDrawOffsetX() + Trans.X, this.GetParentDrawOffsetY() + Trans.Y, Trans.Z);

                var m = mS * mR * mT;

                Matrix34 mRes = new Matrix34();
                mRes.Set(m);

                return mRes;
            }
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        public Matrix LocalMtx
        {
            get
            {
                PointF vTrans = new PointF(
                    this.GetParentDrawOffsetX() + this.Trans.X,
                    this.GetParentDrawOffsetY() + this.Trans.Y);

                return PaneHelper.CalcLocalMatrix(
                    vTrans,
                    RotAngleZ,
                    Scale.FVec2.AsPointF );
            }
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        public Matrix4x4 LocalMtx4x4
        {
            get
            {
                Matrix4x4 trans = Matrix4x4.CreateTranslation(new Vector3(
                    this.GetParentDrawOffsetX() + this.Trans.X,
                    this.GetParentDrawOffsetY() + this.Trans.Y,
                    this.Trans.Z));

                Matrix4x4 rotateX = Matrix4x4.CreateRotationX((float)(RotAngleX / 180 * Math.PI));
                Matrix4x4 rotateY = Matrix4x4.CreateRotationY((float)(RotAngleY / 180 * Math.PI));
                Matrix4x4 rotateZ = Matrix4x4.CreateRotationZ((float)(RotAngleZ / 180 * Math.PI));

                Matrix4x4 scale = Matrix4x4.CreateScale(new Vector3(
                    Scale.FVec2.AsPointF.X,
                    Scale.FVec2.AsPointF.Y,
                    1.0f));

                return PaneHelper.CalcLocalMatrix4x4(trans, rotateX, rotateY, rotateZ, scale);
            }
        }

        /// <summary>
        /// ワールド変換行列を計算します。
        /// </summary>
        public Matrix WorldMtx
        {
            get
            {
                Matrix mtxWorld = this.ParentMtx;
                mtxWorld.Multiply( this.LocalMtx );

                return mtxWorld;
            }
        }

        /// <summary>
        /// ワールド変換行列を計算します。
        /// </summary>
        public Matrix4x4 WorldMtx4x4
        {
            get
            {
                Matrix4x4 mtxWorld = this.ParentMtx4x4;
                mtxWorld = Matrix4x4.Multiply(this.LocalMtx4x4, mtxWorld);

                return mtxWorld;
            }
        }

        /// <summary>
        /// 親行列を計算する
        /// </summary>
        /// <param name="overPartsHierarchy">パーツを超えてルートまで行列を計算するかどうか</param>
        /// <returns></returns>
        Matrix CalculateParentMatrix(bool overPartsHierarchy)
        {
            IHierarchyNode parent = this.Parent;

            ArrayList mtxSet = new ArrayList();
            while (parent != null)
            {
                mtxSet.Add(parent.LocalMtx);

                if (parent.Parent == null)
                {
                    // 親が null になっていたらパーツペイン経由で親のサブシーンのペインをたどる
                    IPane pane = parent as IPane;
                    if (overPartsHierarchy &&
                        pane.OwnerSubScene.IPartsLayout != null)
                    {
                        parent = pane.OwnerSubScene.IPartsLayout.OwnerPane;
                    }
                    else
                    {
                        parent = null;
                    }
                }
                else
                {
                    parent = parent.Parent;
                }
            }

            Matrix mtxWorld = new Matrix();
            for (int i = mtxSet.Count - 1; i >= 0; i--)
            {
                Matrix mtx = mtxSet[i] as Matrix;
                mtxWorld.Multiply(mtx);
            }

            return mtxWorld;
        }

        /// <summary>
        /// 親ペインのワールド変換行列
        /// </summary>
        public Matrix ParentMtx
        {
            get
            {
                return CalculateParentMatrix(false);
            }
        }

        /// <summary>
        /// パーツも考慮した親ペインのワールド変換行列
        /// </summary>
        public Matrix ParentMtxOverPartsHierarchy
        {
            get
            {
                return CalculateParentMatrix(true);
            }
        }


        /// <summary>
        /// 親ペインのワールド変換行列
        /// </summary>
        public Matrix4x4 ParentMtx4x4
        {
            get
            {
                IHierarchyNode parent = this.Parent;

                ArrayList mtxSet = new ArrayList();
                while (parent != null)
                {
                    mtxSet.Add(parent.LocalMtx4x4);
                    parent = parent.Parent;
                }

                Matrix4x4 mtxWorld = Matrix4x4.Identity;
                for (int i = mtxSet.Count - 1; i >= 0; i--)
                {
                    Matrix4x4 mtx = (Matrix4x4)mtxSet[i];
                    mtxWorld = Matrix4x4.Multiply(mtx, mtxWorld);
                }
                return mtxWorld;
            }
        }

        /// <summary>
        /// スケールが妥当な値か判定します。
        /// </summary>
        public bool ValidScaleForDraw
        {
            get
            {
                return Math.Abs( Scale.X ) >= _MinScaleForDraw &&
                        Math.Abs( Scale.Y ) >= _MinScaleForDraw &&
                        Math.Abs( Scale.Z ) >= _MinScaleForDraw;
            }
        }

        /// <summary>
        /// 子供を持つか判定します。
        /// </summary>
        public bool HasChildren { get { return _childrenNodeSet.Count > 0; } }




        /// <summary>
        /// 子ノードのパラメータを親ノードの座標空間に変換します。
        ///
        /// 本メソッドは、親子登録の前に実行されるメソッドです。
        /// </summary>
        /// <param name="parentNode"></param>
        /// <param name="childNode"></param>
        void TranslateChildTransToParentSpace_
        (
            IHierarchyNode parentNode,
            IHierarchyNode childNode
        )
        {
            // 本メソッドは、親子登録の前に実行されるメソッドです。
            Debug.Assert( !childNode.IsChildrenOf( parentNode ) );

            PointF newChildPos = new PointF( childNode.XInWorld, childNode.YInWorld );

            Matrix invW = parentNode.WorldMtx;

            IPane parentPane = parentNode as IPane;
            IPane childPane = childNode as IPane;
            if (parentPane != null && childPane != null)
            {
                invW.Translate(
                    childPane.GetParentDrawOffsetX(parentPane),
                    childPane.GetParentDrawOffsetY(parentPane));
            }

            if( invW.IsInvertible )
            {
                invW.Invert();
                newChildPos = MathUtil.MtxTransformPoint( invW, newChildPos );
            }

            childNode.X = newChildPos.X;
            childNode.Y = newChildPos.Y;

            // 簡易的に回転を調整します。
            childNode.RotAngleZ -= parentNode.RotAngleZ;
        }

        /// <summary>
        /// 子ノードのパラメータを親ノードの座標空間から、
        /// ワールド座標空間に変換します。
        ///
        /// 本メソッドは、親子階層リセットの前に実行されるメソッドです。
        ///
        /// </summary>
        /// <param name="parentNode"></param>
        /// <param name="childNode"></param>
        void TranslateChildTransToWorldSpace_
            (
            IHierarchyNode parentNode,
            IHierarchyNode childNode
            )
        {
            PointF newChildPos = new PointF( childNode.XInWorld, childNode.YInWorld );


            childNode.X = newChildPos.X;
            childNode.Y = newChildPos.Y;

            // 簡易的に回転を調整します。
            childNode.RotAngleZ += parentNode.RotAngleZ;
        }


        void AddHierarchy_( IHierarchyNode childNode, bool adjustPosition )
        {
            Debug.Assert( childNode != null );
            Debug.Assert( childNode.IsValidForChildOf( this ) );

            if( childNode.IsValidForChildOf( this ) )
            {
                if (Scene.CurrentSubScene != null)
                {
                    Scene.CurrentSubScene.BeginMassiveModify();
                }

                // すでに親が存在するなら、一旦親との関係を解除します。
                if( childNode.Parent != null )
                {
                    // 必要ならばワールド系に戻す。
                    if( adjustPosition )
                    {
                        TranslateChildTransToWorldSpace_( childNode.Parent, childNode );
                    }
                    childNode.Parent.RemoveChildNode( childNode );
                }

                // 座標を親のローカル系に変更します。
                if( adjustPosition )
                {
                    TranslateChildTransToParentSpace_( this, childNode );
                }

                childNode.Parent = this;
                if( !_childrenNodeSet.Contains( childNode ) )
                {
                    _childrenNodeSet.Add( childNode );
                }
                NotifyHierarchyChangeToScene_( SceneModifyEventArgs.Kind.HierarchyModify );

                if (Scene.CurrentSubScene != null)
                {
                    Scene.CurrentSubScene.EndMassiveModify();
            }
        }
        }

        /// <summary>
        /// 子供ペインを追加します。
        /// 現在の位置を、親のローカル座標系に変換するため、見た目の位置が変化しません。
        /// </summary>
        /// <param name="childNode"></param>
        public void AddChildPane( Pane childNode )
        {
            AddHierarchy_( childNode, true );
        }

        /// <summary>
        /// 子供ノードを設定します。
        /// 子ノードは親のローカル系で定義されているものと仮定して、変換を行わずにそのまま登録します。
        /// </summary>
        public void AddChildNodeByLoaclCoordinate( IHierarchyNode childNode )
        {
            AddHierarchy_( childNode, false );
        }

        /// <summary>
        /// 子供ペインの順番を変更します。
        /// 描画順に対応しています。
        /// </summary>
        public int ChangeChildOrder( Pane child, int newIndex )
        {
             Debug.Assert( child.IsChildrenOf( this ) );

            int currentIndex = _childrenNodeSet.IndexOf( child );

            // 挿入し、現在のインスタンスを削除します。
            _childrenNodeSet.Insert( newIndex, child );

            bool bMoveFromFront = currentIndex >= newIndex;
            if( bMoveFromFront )
            {
                _childrenNodeSet.RemoveAt( currentIndex + 1);
            }
            else
            {
                _childrenNodeSet.RemoveAt( currentIndex );
            }
            NotifyHierarchyChangeToScene_( SceneModifyEventArgs.Kind.HierarchyModify );

            return ( bMoveFromFront ) ? newIndex + 1 : newIndex;
        }

        /// <summary>
        /// 子供ノードを削除します。
        /// </summary>
        void RemoveChild_( IHierarchyNode childNode, bool adjustPosition  )
        {
            if( _childrenNodeSet.Contains( childNode ) )
            {
                // 座標をワールド系に戻します。
                if( adjustPosition )
                {
                    TranslateChildTransToWorldSpace_( childNode.Parent, childNode );
                }

                childNode.Parent = null;
                _childrenNodeSet.Remove( childNode );
            }
        }

        /// <summary>
        /// 子供ノードを削除します。
        /// </summary>
        public void RemoveChildNode( IHierarchyNode childNode )
        {
            RemoveChild_( childNode, false );
            NotifyHierarchyChangeToScene_( SceneModifyEventArgs.Kind.HierarchyModify );
        }

        /// <summary>
        /// ノードの番号を取得します。
        /// </summary>
        public int IndexOf( IHierarchyNode childNode )
        {
            return _childrenNodeSet.IndexOf( childNode );
        }


        /// <summary>
        /// 親子階層構造を破壊します。ペインを消去する際などに使用してください。
        /// </summary>
        public void DestroyHierarchyConnection()
        {
            // 親が存在すれば、自分を、親の子供から消去する。
            if( this.Parent != null )
            {
                this.Parent.RemoveChildNode( this );
            }

            // 自分の子供をすべて消去する
            IHierarchyNode[] children = this.Children;
            foreach( IHierarchyNode child in children )
            {
                this.RemoveChildNode( child );
            }
        }

        /// <summary>
        /// 指定したノードの子供ノードか判定します。
        /// </summary>
        /// <param name="candidate"></param>
        /// <returns></returns>
        public bool IsChildrenOf( IHierarchyNode candidate )
        {
            // candidate の子供に含まれていないか？
            IHierarchyNode[] children = candidate.Children;
            foreach( IHierarchyNode child in children )
            {
                if( child == this )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 子ノードとして相応しいか判定します。
        /// </summary>
        /// <param name="parentCandidate"></param>
        /// <returns></returns>
        public bool IsValidForChildOf( IHierarchyNode parentCandidate )
        {
            if( parentCandidate != null )
            {
                // 部品ペインには子供の設定を許さない。
                if (parentCandidate is IPane)
                {
                    if ((parentCandidate as IPane).PaneKind == PaneKind.Parts)
                    {
                        return false;
                    }
                }

                // 親候補の子供に含まれていないか？
                {
                    // 親候補の親を次々にたどりながら...
                    while( parentCandidate != null )
                    {
                        // 親候補の親として自分自身が発見されたら...
                        if( parentCandidate == this )
                        {
                            // 循環参照が発生するケース
                            // 不正である。
                            return false;
                        }
                        parentCandidate = parentCandidate.Parent;
                    }
                    // 妥当である
                    return true;
                }
            }
            // 不正である。
            return false;
        }

        #endregion IHierarchyNode メンバ

        #region IPane メンバ

        /// <summary>
        /// ペインが選択された状態か、取得設定をします。
        /// </summary>
        public bool IsSelected
        {
            get{ return _isSelected;}
            set{ _isSelected = value;}
        }

        /// <summary>
        /// ペインのもつアトリビュートへの参照を取得します。
        /// </summary>
        public IAnmAttribute IAttributeRef
        {
            get{ return _attribute;}
        }

        /// <summary>
        /// ペインのもつアトリビュートへの参照を取得します。
        /// </summary>
        public IPaneAttribute IPaneAttribute
        {
            get{ return  _attribute;}
        }

        /// <summary>
        /// ペインのもつアトリビュートへの変更可能な参照を取得します。
        /// </summary>
        public PaneAttribute PaneAttrRef
        {
            get{ return  _attribute;}
        }

        /// <summary>
        /// ペインの幅
        /// </summary>
        public float Width
        {
            get
            {
                return _attribute.SizeAttr.GetAsFVec2().X;
            }
            set{ _attribute.SizeAttr.SetValue( new FVec2( value, this.Height ) );}
        }

        /// <summary>
        /// ペインの高さ
        /// </summary>
        public float Height
        {
            get
            {
                return _attribute.SizeAttr.GetAsFVec2().Y;
            }
            set{ _attribute.SizeAttr.SetValue( new FVec2( this.Width, value ) );}
        }

        /// <summary>
        /// ペインの幅
        /// </summary>
        public float RenderingWidth
        {
            get
            {
                var mag = (this.PartsMagnifyInfluence == PartsMagnifyInfluence.None) ? 1.0f : SizeMagnify.X;
                return Width * mag;
            }
        }

        /// <summary>
        /// ペインの高さ
        /// </summary>
        public float RenderingHeight
        {
            get
            {
                var mag = (this.PartsMagnifyInfluence == PartsMagnifyInfluence.None) ? 1.0f : SizeMagnify.Y;
                return Height * mag;
            }
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        public float DrawOffsetX
        {
            get
            {
                switch( _basePosTypeH )
                {
                    case HorizontalLocation.Left: return 0.0f;
                    case HorizontalLocation.Center: return -RenderingWidth * 0.5f;
                    case HorizontalLocation.Right: return -RenderingWidth;
                    default: Debug.Assert( false ); return 0.0f;
                }
            }
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        public float DrawOffsetY
        {
            get
            {
                switch( _basePosTypeV )
                {
                    case VerticalLocation.Top: return 0.0f;
                    case VerticalLocation.Center: return -RenderingHeight * 0.5f;
                    case VerticalLocation.Bottom: return -RenderingHeight;
                    default: Debug.Assert( false ); return 0.0f;
                }
            }
        }

        /// <summary>
        /// ベースポジション。アフィン変換中心(水平)
        /// </summary>
        public HorizontalLocation      BasePosTypeH
        {
            get
            {
                if (this.PaneKind == PaneKind.Parts) { return HorizontalLocation.Center; }
                return _basePosTypeH;
            }

            set
            {
                if (this.PaneKind == PaneKind.Parts) { return; }

                _basePosTypeH = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// ベースポジション。アフィン変換中心(垂直)
        /// </summary>
        public VerticalLocation        BasePosTypeV
        {
            get
            {
                if (this.PaneKind == PaneKind.Parts) { return VerticalLocation.Center; }
                return _basePosTypeV;
            }

            set
            {
                if (this.PaneKind == PaneKind.Parts) { return; }

                _basePosTypeV = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public HorizontalLocation ParentBasePosTypeH
        {
            get { return (this.Parent as IPane).IsRootPane() ? HorizontalLocation.Center : _parentBasePosTypeH; }
            set { _parentBasePosTypeH = value; NotifyChangeToScene_(); }
        }

        /// <summary>
        ///
        /// </summary>
        public VerticalLocation ParentBasePosTypeV
        {
            get { return (this.Parent as IPane).IsRootPane() ? VerticalLocation.Center : _parentBasePosTypeV; }
            set { _parentBasePosTypeV = value; NotifyChangeToScene_(); }
        }

        //
        // public object Item;
        // letter Picture に固有の情報を格納する。
        /// <summary>
        /// TextBoxペインの情報を取得します。
        /// </summary>
        public TextBox TextBox
        {
            get { return _paneExParamater as TextBox; }
        }

        /// <summary>
        /// TextBoxペインの情報を取得します。
        /// 外部モジュール公開インタフェース
        /// </summary>
        public ITextBox ITextBox
        {
            get { return _paneExParamater as ITextBox; }
        }

        /// <summary>
        /// Pictureペインの情報を取得します。
        /// </summary>
        public IPicture IPicture
        {
            get { return _paneExParamater as IPicture; }
        }

        /// <summary>
        /// Windowペインの情報を取得します。
        /// </summary>
        public ILEWindow ILEWindow
        {
            get { return _paneExParamater as ILEWindow; }
        }

        /// <summary>
        /// Captureペインの情報を取得します。
        /// </summary>
        public ICapture ICapture
        {
            get { return _paneExParamater as ICapture; }
        }

        /// <summary>
        /// Captureペインの情報を取得します。
        /// </summary>
        public IAlignment IAlignment
        {
            get { return _paneExParamater as IAlignment; }
        }

        /// <summary>
        /// Scissorペインの情報を取得します。
        /// </summary>
        public IScissor IScissor
        {
            get { return _paneExParamater as IScissor; }
        }

        /// <summary>
        /// Pictureペインの情報を取得します。
        /// セーブロードで使用される。
        /// </summary>
        public Picture Picture
        {
            get { return _paneExParamater as Picture; }
        }

        /// <summary>
        /// 部品ペインが持つ、部品レイアウト情報です。
        /// </summary>
        public IPartsLayout IPartsLayout
        {
            get { return _paneExParamater as IPartsLayout; }
        }

        /// <summary>
        /// マテリアルを取得します
        /// マテリアルを持たない場合は長さゼロの配列をかえします。
        /// </summary>
        public IMaterial[] IMaterial
        {
            get
            {
                IMaterial[] result = null;
                if( this.ITextBox != null )
                {
                    result = new IMaterial[]
                    {
                        this.ITextBox.IMaterial
                    };
                }
                else if( this.IPicture != null )
                {
                    result = new IMaterial[]
                    {
                        this.IPicture.IMaterial
                    };
                }
                else if( this.ILEWindow != null )
                {
                    int numMat = Enum.GetValues( typeof( LEWindowPartsID ) ).Length;
                    result = new IMaterial[numMat];

                    for( int i = 0; i < result.Length; i++ )
                    {
                        result[i] = ILEWindow.GetMaterial( ILEWindow.IntToPartsID( i ) );
                    }
                }
                else
                {
                    result = new IMaterial[0];
                }
                return result;
            }
        }

        public virtual IMaterial[] UsedMaterials
        {
            get
            {
                var materials = IMaterial;
                if (this.ILEWindow != null)
                {
                    return Enum.GetValues(typeof(LEWindowPartsID)).OfType<LEWindowPartsID>()
                        .Select(x => materials[LEWindow.GetInternalWindowPartsID(this.ILEWindow, x)])
                        .Distinct().ToArray();
                }
                else
                {
                    return materials;
                }
            }
        }

        /// <summary>
        /// マテリアルを取得します
        /// （Nullもありえます。）
        /// </summary>
        public Material[] Material
        {
            get
            {
                IMaterial[]     srcMatSet = IMaterial;
                Material[]      dstMatSet = new Material[srcMatSet.Length];
                Array.Copy( srcMatSet, dstMatSet, srcMatSet.Length );
                return dstMatSet;
            }
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get
            {
                if (this.IPaneExParamater == null)
                {
                    yield break;
                }

                foreach (var texCoord in this.IPaneExParamater.TexCoords)
                {
                    yield return texCoord;
                }
            }
        }

        /// <summary>
        /// コメント
        /// </summary>
        public string UserCommentString
        {
            get{ return _userCommentString;}
            set
            {
                Debug.Assert( value != null );
                if( value != _userCommentString )
                {
                    _userCommentString = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// ユーザ情報
        ///
        /// TODO:最終的には、
        /// さまざまな型(int, float, bool など)の情報を格納できるようになる。
        /// </summary>
        public string UserData
        {
            get{ return _userDataString;}
            set
            {
                Debug.Assert( value != null );
                if( value != _userDataString )
                {
                    _userDataString = value;
                    NotifyChangeToScene_();
                }
            }
        }


        /// <summary>
        /// 位置
        /// </summary>
        public FVec3 Trans
        {
            get
            {
                return _attribute.TransAttr.GetAsFVec3();
            }
            set
            {
                _attribute.TransAttr.SetValue( value );
            }
        }

        /// <summary>
        /// 回転角度
        /// </summary>
        public FVec3 RotAng
        {
            get
            {
                return _attribute.RotAngAttr.GetAsFVec3();
            }
            set{ _attribute.RotAngAttr.SetValue( value );}
        }

        /// <summary>
        /// スケール
        /// </summary>
        public FVec3 Scale
        {
            get
            {
                FVec2 scale = _attribute.ScaleAttr.GetAsFVec2();

                return new FVec3( scale.X , scale.Y , 1.0f );
            }
            set
            {
                _attribute.ScaleAttr.SetValue( new FVec2( value.X, value.Y ) );
            }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public FVec3 Size
        {
            get
            {
                FVec2   p = _attribute.SizeAttr.GetAsFVec2();
                return new FVec3( p.X, p.Y, 0.0f );
            }
            set
            {
                _attribute.SizeAttr.SetValue( new FVec2( value.X, value.Y ) );
            }
        }

        /// <summary>
        /// 表示/非表示
        /// </summary>
        public bool Visible
        {
            get
            {
                object   flag = null;
                _attribute.VisibleAttr.GetValue( out flag );
                return (bool)flag;
            }
            set
            {
                _attribute.VisibleAttr.SetValue( value );
            }
        }

        /// <summary>
        /// 自身の透明度が、子供の透明度に影響を与えるか。
        /// </summary>
        public bool InfluenceChildrenTransparency
        {
            get { return _influenceChildrenTransparency; }
            set
            {
                if( _influenceChildrenTransparency != value )
                {
                    _influenceChildrenTransparency = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 位置調整処理の対象とするかどうか？
        /// </summary>
        public bool LocationAdjust
        {
            get { return _locationAdjust; }
            set
            {
                if( _locationAdjust != value )
                {
                    _locationAdjust = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public bool IsAlignmentHorizontalMarginEnabled
        {
            get { return _isAlignmentHorizontalMarginEnabled; }
            set
            {
                if (_isAlignmentHorizontalMarginEnabled != value)
                {
                    _isAlignmentHorizontalMarginEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public bool IsAlignmentVerticalMarginEnabled
        {
            get { return _isAlignmentVerticalMarginEnabled; }
            set
            {
                if (_isAlignmentVerticalMarginEnabled != value)
                {
                    _isAlignmentVerticalMarginEnabled = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public float AlignmentHorizontalMargin
        {
            get { return _alignmentHorizontalMargin; }
            set
            {
                if (_alignmentHorizontalMargin != value)
                {
                    _alignmentHorizontalMargin = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public float AlignmentVerticalMargin
        {
            get { return _alignmentVerticalMargin; }
            set
            {
                if (_alignmentVerticalMargin != value)
                {
                    _alignmentVerticalMargin = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public bool IsAlignmentIgnore
        {
            get { return _isAlignmentIgnore; }
            set
            {
                if (_isAlignmentIgnore != value)
                {
                    _isAlignmentIgnore = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 部品サイズ調整を無視するかどうか？
        /// </summary>
        public PartsMagnifyInfluence PartsMagnifyInfluence
        {
            get
            {
                return _partsMagnifyInfluence;
            }

            set
            {
                if (_partsMagnifyInfluence != value)
                {
                    _partsMagnifyInfluence = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// サイズが変更可能かを取得します。
        /// </summary>
        public PaneSizeMode PaneSizeMode
        {
            get
            {
                if (this.PaneKind == PaneKind.Parts)
                {
                    return PaneSizeMode.Fixed;
                }

                if (this.PaneKind == PaneKind.Window)
                {
                    if (this.ILEWindow.WindowKind != WindowKind.Around)
                    {
                        return PaneSizeMode.HEnabled;
                    }
                }

                return PaneSizeMode.All;
            }
        }

        /// <summary>
        /// ペインの種類
        /// </summary>
        public PaneKind PaneKind
        {
            get
            {
                if( _paneExParamater is ITextBox )
                {
                    return PaneKind.Textbox;
                }
                else if( _paneExParamater is IPicture )
                {
                    return PaneKind.Picture;
                }
                else if( _paneExParamater is ILEWindow )
                {
                    return PaneKind.Window;
                }
                else if( _paneExParamater is Bounding )
                {
                    return PaneKind.Bounding;
                }
                else if (_paneExParamater is PartsLayout)
                {
                    return PaneKind.Parts;
                }
                else if (_paneExParamater is Capture )
                {
                    return PaneKind.Capture;
                }
                else if (_paneExParamater is Alignment)
                {
                    return PaneKind.Alignment;
                }
                else if (_paneExParamater is Scissor)
                {
                    return PaneKind.Scissor;
                }
                else
                {
                    return PaneKind.Null;
                }
            }
        }

        /// <summary>
        /// ペイン名前
        /// </summary>
        public string PaneName
        {
            get{ return _paneName;}
            set
            {
                // ルートペインの名前は、変更させない。
                if (this.IsRootPane())
                {
                    return;
                }

                // 変更があれば...
                if( _paneName != value )
                {
                    _paneName   = value;
                    Description = value;

                    // マテリアル名も変更
                    if( IPaneExParamater != null )
                    {
                        IPaneExParamater.UpdateMatarialName();
                    }
                    NotifyChangeToScene_();
                    // ペイン名が変わることによりキャプチャテクスチャの名前が変わるため TextureMgr も更新する。
                    NotifyChangeToSceneImpl_(SceneModifyEventArgs.Kind.TextureManager);
                }
            }
        }

        /// <summary>
        /// 透明度を取得設定します。
        /// </summary>
        public byte Transparency
        {
            get { return _attribute.TransparencyAttr.GetAsByte();}
            set { _attribute.TransparencyAttr.SetValue( value );}
        }

        /// <summary>
        /// 編集が禁止された状態かどうかを取得、設定します。
        /// </summary>
        public bool   Locked
        {
            get{ return _isLocked || this.IsReadOnlyLocked;}
            set
            {
                _isLocked = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 隠された状態かどうかを取得、設定します。
        /// </summary>
        public bool Hidden
        {
            get{ return _isHidden;}
            set
            {
                _isHidden = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 読み取り専用か取得します。
        /// </summary>
        public bool IsReadOnlyLocked
        {
            get { return _isReadOnlyLocked; }
            internal set
            {
                _isReadOnlyLocked = value;

                // アニメーションが設定されたカーブは読み取り専用に設定します。
                var animatedAttrs = PaneAnimationHelper.GetAllActiveAnmAttribute(this).Where((attr) => attr.HasAnyKey());
                foreach(var animatedAttr in animatedAttrs)
                {
                    (animatedAttr.ICurrentAnimationCurve as AnmCurve).IsReadOnlyLocked = value;
                }
            }
        }

        /// <summary>
        /// ペイン追加情報を取得します。
        /// </summary>
        public IPaneExParamater IPaneExParamater
        {
            get
            {
                return _paneExParamater;
            }
        }

        // マスクの取得
        public IMask IMask
        {
            get
            {
                return _mask;
            }
        }

        // ドロップシャドウの取得
        public IDropShadow IDropShadow
        {
            get
            {
                return _dropShadow;
            }
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get { return (this.IPaneExParamater != null) ? this.IPaneExParamater.WarningMsg : string.Empty; }
        }

        /// <summary>
        /// 持ち主サブシーンを設定します。
        /// </summary>
        public ISubScene OwnerSubScene
        {
            get { return _ownerSubScene; }
            set { _ownerSubScene = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public IUserDataHolder IUserDataHolder
        {
            get { return _userDataHolder; }
        }

        /// <summary>
        ///
        /// </summary>
        public UserDataHolder UserDataHolder
        {
            get { return _userDataHolder; }
        }

        /// <summary>
        /// ノード圧縮禁止か取得します。
        /// </summary>
        public bool AvoidPaneTreeCompression
        {
            get { return _avoidPaneTreeCompression; }
            internal set
            {
                _avoidPaneTreeCompression = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 拡張ユーザー情報のアニメーションを有効にするか取得します。
        /// </summary>
        public bool ExtUserDataAnimEnabled
        {
            get { return _extUserDataAnimEnabled; }
            internal set
            {
                _extUserDataAnimEnabled = value;
                NotifyChangeToScene_();
            }
        }

        #endregion IPane メンバ

        #region イベント通知関連
        /// <summary>
        /// ペインの変更をシーンに通知します。
        /// （なお、各種Dlg(View)の再描画は、
        /// シーン更新イベントがきっかけとなります。）
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyChangeToSceneImpl_( SceneModifyEventArgs.Kind.PaneModify );
        }

        void NotifyChangeToSceneImpl_( SceneModifyEventArgs.Kind kind )
        {
            if( OnPainModify != null )
            {
                SceneModifyEventArgs args = new SceneModifyEventArgs( kind, this );
                OnPainModify( this, args );
            }
        }

        /// <summary>
        /// 階層の変更をシーンに通知します。
        /// </summary>
        void NotifyHierarchyChangeToScene_( SceneModifyEventArgs.Kind kind )
        {
            if( OnPainModify != null )
            {
                SceneModifyEventArgs args =
                    new SceneModifyEventArgs( kind, this );
                OnPainModify( this, args );
            }
        }

        /// <summary>
        /// オーバーライド
        /// </summary>
        public override void NotifyModifyEvent(LEDataNode sender, int kind)
        {
            try
            {
                // 配下で、アトリビュート変更があったら
                if((EventKind)kind == EventKind.AttributeModify)
                {
                    // タグの登録を行います。
                    if (this.IsInSeparateModeWithoutOwner())
                    {
                        SubSceneHelper.AddAllAnimTagsToAttribute(this.OwnerSubScene, sender);
                    }
                }

                SceneModifyEventArgs.Kind sceneEventKind = (SceneModifyEventArgs.Kind)kind;
                NotifyChangeToSceneImpl_(sceneEventKind);
            }
            catch (System.Exception ex)
            {
                DbgConsole.WriteLine("Unknown event kind.{0}", ex.ToString());
            }
        }
        #endregion イベント通知関連
    }
}
