﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;

namespace LECore.Structures
{
    using Core;
    using Nsrif.Attributes;

    #region ＰＥデータ

    #region PEData

    /// <summary>
    /// ＰＥデータクラス。外部公開インタフェース
    /// </summary>
    public interface IPEData
    {
        /// <summary>
        /// 描画モード。
        /// </summary>
        AttrTransparencyMode DrawMode {get;}


        /// <summary>
        /// アルファコンペア。
        /// </summary>
        PEACompare ACompare {get;}


        /// <summary>
        /// Ｚコンペア。
        /// </summary>
        PEZCompare ZCompare {get;}


        /// <summary>
        /// ブレンド。
        /// </summary>
        PEBlend Blend {get;}


        /// <summary>
        /// 設定状態。
        /// </summary>
        PEState State {get;}
    }

    /// <summary>
    /// ＰＥデータクラス。
    /// </summary>
    internal sealed class PEData :
        LEDataNode,
        IPEData
    {

        public const string PEDataNodeName = "PEData";

        private AttrTransparencyMode _drawMode = AttrTransparencyMode.Opa;
        private readonly PEACompare  _acompare;
        private readonly PEZCompare  _zcompare;
        private readonly PEBlend     _blend;

        /// <summary>不透明設定</summary>
        public static PEData Opa  = new PEData( null );
        /// <summary>抜き設定</summary>
        public static PEData Edge = new PEData( null );
        /// <summary>半透明設定</summary>
        public static PEData Xlu  = new PEData( null );

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEData( LEDataNode owner )
            :base( owner, PEDataNodeName )
        {
            _acompare = new PEACompare();
            _zcompare = new PEZCompare();
            _blend    = new PEBlend();
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEData( LEDataNode owner, PEData source) : this( owner )
        {
            Set(source);
        }

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static PEData()
        {
            // 不透明設定
            Opa._drawMode           = AttrTransparencyMode.Opa;
            Opa._acompare.Comp0     = AttrCompareFunc.Always;
            Opa._acompare.Ref0      = 0;
            Opa._acompare.Op        = AttrAlphaOp.And;
            Opa._acompare.Comp1     = AttrCompareFunc.Always;
            Opa._acompare.Ref1      = 0;
            Opa._zcompare.Enable    = true;
            Opa._zcompare.Func      = AttrCompareFunc.Lequal;
            Opa._zcompare.Update    = true;
            Opa._zcompare.BeforeTex = true;
            Opa._blend.Type         = AttrBlendType.None;
            Opa._blend.SrcFactor    = AttrBlendFactor.SrcAlpha;
            Opa._blend.DstFactor    = AttrBlendFactor.InvSrcAlpha;
            Opa._blend.LogicOp      = AttrLogicOp.Copy;

            // 抜き設定
            Edge._drawMode           = AttrTransparencyMode.Opa;
            Edge._acompare.Comp0     = AttrCompareFunc.Gequal;
            Edge._acompare.Ref0      = 128;
            Edge._acompare.Op        = AttrAlphaOp.And;
            Edge._acompare.Comp1     = AttrCompareFunc.Lequal;
            Edge._acompare.Ref1      = 255;
            Edge._zcompare.Enable    = true;
            Edge._zcompare.Func      = AttrCompareFunc.Lequal;
            Edge._zcompare.Update    = true;
            Edge._zcompare.BeforeTex = false;
            Edge._blend.Type         = AttrBlendType.None;
            Edge._blend.SrcFactor    = AttrBlendFactor.SrcAlpha;
            Edge._blend.DstFactor    = AttrBlendFactor.InvSrcAlpha;
            Edge._blend.LogicOp      = AttrLogicOp.Copy;

            // 半透明設定
            Xlu._drawMode           = AttrTransparencyMode.Xlu;
            Xlu._acompare.Comp0     = AttrCompareFunc.Always;
            Xlu._acompare.Ref0      = 0;
            Xlu._acompare.Op        = AttrAlphaOp.And;
            Xlu._acompare.Comp1     = AttrCompareFunc.Always;
            Xlu._acompare.Ref1      = 0;
            Xlu._zcompare.Enable    = true;
            Xlu._zcompare.Func      = AttrCompareFunc.Lequal;
            Xlu._zcompare.Update    = false;
            Xlu._zcompare.BeforeTex = true;
            Xlu._blend.Type         = AttrBlendType.Blend;
            Xlu._blend.SrcFactor    = AttrBlendFactor.SrcAlpha;
            Xlu._blend.DstFactor    = AttrBlendFactor.InvSrcAlpha;
            Xlu._blend.LogicOp      = AttrLogicOp.Copy;
        }
        #endregion

        #region アトリビュート
        /// <summary>
        /// 描画モード。
        /// </summary>
        public AttrTransparencyMode DrawMode
        {
            get { return _drawMode;  }
            set { _drawMode = value; NotifyModifyEvent( this, EventKind.Modify ); }
        }

        /// <summary>
        /// アルファコンペア。
        /// </summary>
        public PEACompare ACompare
        {
            get { return _acompare; }
        }

        /// <summary>
        /// Ｚコンペア。
        /// </summary>
        public PEZCompare ZCompare
        {
            get { return _zcompare; }
        }

        /// <summary>
        /// ブレンド。
        /// </summary>
        public PEBlend Blend
        {
            get { return _blend; }
        }

        /// <summary>
        /// 設定状態。
        /// </summary>
        public PEState State
        {
            get
            {
                if      (this.Equals(PEData.Opa))  { return PEState.Opa;    }
                else if (this.Equals(PEData.Edge)) { return PEState.Edge;   }
                else if (this.Equals(PEData.Xlu))  { return PEState.Xlu;    }
                else                               { return PEState.Custom; }
            }
        }
        #endregion

        #region アトリビュートテキスト
        /// <summary>
        /// 描画モードテキスト。
        ///
        /// MEMO：UIDataBinder が UIレイヤのモジュールであるため、廃止しました。
        /// </summary>
//        public string DrawModeText
//        {
//            get { return UIDataBinder.GetAttributeText(_drawMode); }
//        }

        /// <summary>
        /// 設定状態テキスト。
        /// </summary>
        public string StateText
        {
            get
            {
                switch (this.State)
                {
                    case PEState.Opa:  return StringResMgr.Get("MAT_PESTATE_OPA");
                    case PEState.Edge: return StringResMgr.Get("MAT_PESTATE_EDGE");
                    case PEState.Xlu:  return StringResMgr.Get("MAT_PESTATE_XLU");
                    default:           return StringResMgr.Get("TAG_CUSTOM");
                }
            }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEData src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _drawMode = src._drawMode;
            _acompare.Set(src._acompare);
            _zcompare.Set(src._zcompare);
            _blend.Set(src._blend);

            NotifyModifyEvent( this, EventKind.Modify );
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEData src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _drawMode == src._drawMode      &&
                _acompare.Equals(src._acompare) &&
                _zcompare.Equals(src._zcompare) &&
                _blend.Equals(src._blend);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion
    }
    #endregion

    #region PEState
    /// <summary>
    /// ＰＥ設定状態。
    /// </summary>
    public enum PEState
    {
        /// <summary>不透明設定。</summary>
        Opa = 0,
        /// <summary>抜き設定。</summary>
        Edge,
        /// <summary>半透明設定。</summary>
        Xlu,
        /// <summary>加算合成設定。</summary>
        Add,
        /// <summary>乗算合成設定。</summary>
        Mul,
        /// <summary>減算合成設定。</summary>
        Sub,
        /// <summary>カスタム設定。</summary>
        Custom,
    }
    #endregion

    #region PEACompare
    /// <summary>
    /// ＰＥアルファコンペアクラス。
    /// </summary>
    public sealed class PEACompare
    {
        private AttrCompareFunc _comp0 = AttrCompareFunc.Always;
        private int             _ref0  = 0;
        private AttrAlphaOp     _op    = AttrAlphaOp.Or;
        private AttrCompareFunc _comp1 = AttrCompareFunc.Always;
        private int             _ref1  = 0;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEACompare() {}
        #endregion

        #region アトリビュート
        /// <summary>
        /// 比較方法０。
        /// </summary>
        public AttrCompareFunc Comp0
        {
            get { return _comp0;  }
            set { _comp0 = value; }
        }

        /// <summary>
        /// 参照値０。
        /// </summary>
        public int Ref0
        {
            get { return _ref0;  }
            set { _ref0 = value; }
        }

        /// <summary>
        /// 演算方法。
        /// </summary>
        public AttrAlphaOp Op
        {
            get { return _op;  }
            set { _op = value; }
        }

        /// <summary>
        /// 比較方法１。
        /// </summary>
        public AttrCompareFunc Comp1
        {
            get { return _comp1;  }
            set { _comp1 = value; }
        }

        /// <summary>
        /// 参照値１。
        /// </summary>
        public int Ref1
        {
            get { return _ref1;  }
            set { _ref1 = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEACompare src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _comp0 = src._comp0;
            _ref0  = src._ref0;
            _op    = src._op;
            _comp1 = src._comp1;
            _ref1  = src._ref1;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEACompare src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _comp0 == src._comp0 &&
                _ref0  == src._ref0  &&
                _op    == src._op    &&
                _comp1 == src._comp1 &&
                _ref1  == src._ref1;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEACompare);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion
    }
    #endregion

    #region PEZCompare
    /// <summary>
    /// ＰＥＺコンペアクラス。
    /// </summary>
    public sealed class PEZCompare
    {
        private bool            _enable    = true;
        private AttrCompareFunc _func      = AttrCompareFunc.Lequal;
        private bool            _update    = true;
        private bool            _beforeTex = false;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEZCompare() {}

        #endregion

        #region アトリビュート
        /// <summary>
        /// 有効フラグ。
        /// </summary>
        public bool Enable
        {
            get { return _enable;  }
            set { _enable = value; }
        }

        /// <summary>
        /// 比較方法。
        /// </summary>
        public AttrCompareFunc Func
        {
            get { return _func;  }
            set { _func = value; }
        }

        /// <summary>
        /// 更新フラグ。
        /// </summary>
        public bool Update
        {
            get { return _update;  }
            set { _update = value; }
        }

        /// <summary>
        /// テクスチャ処理前フラグ。
        /// </summary>
        public bool BeforeTex
        {
            get { return _beforeTex;  }
            set { _beforeTex = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEZCompare src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _enable    = src._enable;
            _func      = src._func;
            _update    = src._update;
            _beforeTex = src._beforeTex;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEZCompare src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _enable    == src._enable &&
                _func      == src._func   &&
                _update    == src._update &&
                _beforeTex == src._beforeTex;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEZCompare);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion

    }
    #endregion

    #region PEBlend
    /// <summary>
    /// ＰＥブレンドクラス。
    /// </summary>
    public sealed class PEBlend
    {
        private AttrBlendType   _type      = AttrBlendType.None;
        private AttrBlendFactor _srcFactor = AttrBlendFactor.SrcAlpha;
        private AttrBlendFactor _dstFactor = AttrBlendFactor.InvSrcAlpha;
        private AttrLogicOp     _logicOp   = AttrLogicOp.Copy;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEBlend() {}

        #endregion

        #region アトリビュート
        /// <summary>
        /// タイプ。
        /// </summary>
        public AttrBlendType Type
        {
            get { return _type;  }
            set { _type = value; }
        }

        /// <summary>
        /// ソース引数。
        /// </summary>
        public AttrBlendFactor SrcFactor
        {
            get { return _srcFactor;  }
            set { _srcFactor = value; }
        }

        /// <summary>
        /// ディスティネーション引数。
        /// </summary>
        public AttrBlendFactor DstFactor
        {
            get { return _dstFactor;  }
            set { _dstFactor = value; }
        }

        /// <summary>
        /// 論理演算。
        /// </summary>
        public AttrLogicOp LogicOp
        {
            get { return _logicOp;  }
            set { _logicOp = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEBlend src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _type      = src._type;
            _srcFactor = src._srcFactor;
            _dstFactor = src._dstFactor;
            _logicOp   = src._logicOp;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEBlend src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _type      == src._type      &&
                _srcFactor == src._srcFactor &&
                _dstFactor == src._dstFactor &&
                _logicOp   == src._logicOp;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEBlend);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion

    }
    #endregion

    #endregion
}
