﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using LECore.Structures.Core.Command;

namespace LECore.Structures
{
    /// <summary>
    /// 編集されうるファイル(LECore モジュール外には公開されない)
    /// </summary>
    internal class LECoreDocument
    {

        private string _filePath = AppConstants.InitialFileName;

        private DateTime _lastSavedTime;
        private DateTime _lastReloadedTime;

        private bool _forceDirty;

        private SubSceneCmdQueue.InternalState _subSceneStateAtLastSaved;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal LECoreDocument(string filePath)
        {
            _filePath = filePath;
        }

        /// <summary>
        /// サブシーンが更新されたか？
        /// </summary>
        public bool CheckUpdated(ISubScene subScene)
        {
            if (this._forceDirty)
            {
                return true;
            }

            return _subSceneStateAtLastSaved.IsOwnerSubSceneChanged(subScene.CmdQueueInternalState);
        }

        /// <summary>
        /// 名前付けされたか？
        /// </summary>
        public bool IsFileSaved
        {
            get { return FilePath != AppConstants.InitialFileName; }
        }

        /// <summary>
        /// 最後に保存されたファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get { return _filePath; }
        }

        /// <summary>
        /// 最後の保存時間
        /// </summary>
        public DateTime LastSavedTime
        {
            get { return _lastSavedTime; }
        }

        /// <summary>
        /// 最後に最新に更新された時間
        /// </summary>
        public DateTime LastReloadedTime
        {
            get { return _lastReloadedTime; }
        }

        /// <summary>
        /// 最新状態になったことを設定します。
        /// </summary>
        public void SetStateReloaded(DateTime time)
        {
            _lastReloadedTime = time;
        }

        /// <summary>
        /// セーブされた状態に設定します。
        /// saveFileName は絶対パスを指定する必要があります。
        /// </summary>
        public void SetStateSaved(ISubScene subScene, string saveFileName, DateTime time)
        {
            _filePath = saveFileName;
            _lastSavedTime = time;

            ResetModifyState(subScene);
        }

        ///
        public void ResetModifyState(ISubScene subScene)
        {
            // コマンドキューの状態の取得します。
            _subSceneStateAtLastSaved = subScene.CmdQueueInternalState;
            this._forceDirty = false;
        }

        ///
        public void ForceModifyStateDirty()
        {
            this._forceDirty = true;
        }
    }
}
