﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    /// <summary>
    /// Exporter inportetが実装するインタフェース
    /// </summary>
    public interface IFileInfoHolder
    {
        /// <summary>
        /// 出力ファイルの拡張子を取得します(短縮)。
        /// </summary>
        string[] GetFileExtensionShort();

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        string[] GetFileExtensionLong();

        /// <summary>
        /// ファイルパスを調整する。
        /// </summary>
        string TrimFilePath(string originalOutPath);

        /// <summary>
        /// FileIO イベントを抑止する
        /// </summary>
        bool StopFileIOEvent { get; }
    }

    /// <summary>
    /// セーブ・ロード関連のヘルパ
    /// </summary>
    static public class SaveLoadHelper
    {
        static readonly Regex _RegexInvalidFileNameStr = new Regex( "[^0-9a-zA-Z!#$%&'()+.=\\[\\]^_@`{}~ -]" );

        /// <summary>
        /// ファイルダイアログ用の、フィルタ文字列を生成します。
        /// </summary>
        static public string GetFliterString(IEnumerable<IFileInfoHolder> fileInfos)
        {
            string resultFilter = string.Empty;

            List<string> extLongList = new List<string>();
            List<string> extShortList = new List<string>();
            foreach(var fileInfo in fileInfos)
            {
                extLongList.AddRange(fileInfo.GetFileExtensionLong());
                extShortList.AddRange(fileInfo.GetFileExtensionShort());
            }

            string[] extLongSet = extLongList.ToArray();
            string[] extShortSet = extShortList.ToArray();
            Debug.Assert( extLongSet.Length == extShortSet.Length );

            for( int i = 0 ; i < extLongSet.Length ; i++ )
            {
                string filter = string.Format( "{0} (*.{1})|*.{1}|", extLongSet[i], extShortSet[i] );
                resultFilter += filter;
            }
            // all 全て   を最後に追加
            resultFilter += "all (*.*)|*.*";


            return resultFilter;
        }

        /// <summary>
        /// アスキー文字のみからなる、文字列か判定します。
        /// </summary>
        static public bool IsValidStringForFileName( string fileName )
        {
            return !_RegexInvalidFileNameStr.IsMatch( fileName );
        }
    }
}
