﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections.Generic;

namespace LECore.Structures.Core
{
    using TexWrap = LECore.Structures.Nsrif.Attributes.AttrTexWrap;
    using TexFilterMin = LECore.Structures.Nsrif.Attributes.AttrTexFilterMin;
    using TexFilterMag = LECore.Structures.Nsrif.Attributes.AttrTexFilterMag;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures.LECoreInterface;

    /// <summary>
    /// 描画モード
    /// </summary>
    public sealed class IRenderOptionFlag
    {
        public const uint DrawOutline    = 0x00000001; // アウトライン描画を行います。
        public const uint DrawDebugInfo  = 0x00000002; // デバック描画を有効にします。
    }

    /// <summary>
    /// レンダラヘルパ
    /// </summary>
    public static class RendererHelper
    {
        /// <summary>
        ///
        /// </summary>
        public static void SetTextureState(this IRenderer renderer, Bitmap bmp, int stageIdx)
        {
            renderer.SetTextureState(bmp, stageIdx, AttrCombineMode.Modulate, AttrCombineMode.Replace, FVec2.One, 0.0f, 4, RendererTextrureFormat.ARGB);
        }

        /// <summary>
        ///
        /// </summary>
        public static void SetTextureState(this IRenderer renderer, Bitmap bmp, int stageIdx, RendererTextrureFormat fmt)
        {
            renderer.SetTextureState(bmp, stageIdx, AttrCombineMode.Modulate, AttrCombineMode.Replace, FVec2.One, 0.0f, 4, fmt);
        }

        /// <summary>
        ///
        /// </summary>
        public static void SetDetailedCombinerState(this IRenderer renderer, Color[] colors, ITevStage[] tevStage, int stageCount)
        {
            renderer.SetDetailedCombinerState(colors, tevStage, stageCount);
        }

        /// <summary>
        ///
        /// </summary>
        public static void DrawImage(this IRenderer renderer, Bitmap bmp, FVec3 imgPos, FVec2 imgSize, TexCoord4[] coords)
        {
            renderer.SetTextureState(bmp, 0);
            renderer.DrawImage(imgPos, imgSize, coords);
        }

        /// <summary>
        ///
        /// </summary>
        public static void DrawImage(this IRenderer renderer, Bitmap bmp, FVec3 imgPos, FVec2 imgSize, TexCoord4[] coords, Color[] vtxColors)
        {
            renderer.SetTextureState(bmp, 0);
            renderer.DrawImage(imgPos, imgSize, coords, vtxColors);
        }

        /// <summary>
        /// 矩形を描きます。
        /// </summary>
        static public void DrawRectangleWithoutTransform( IRenderer renderer, PointF center, float size )
        {
            PointF[] rectPos = { new PointF(), new PointF(size, size) };

            renderer.TransformPointsToScene( rectPos );

            PointF sizeInScene = new PointF(rectPos[1].X - rectPos[0].X, rectPos[1].Y - rectPos[0].Y);


            renderer.DrawRectangle(center.X - sizeInScene.X * 0.5f, center.Y - sizeInScene.Y * 0.5f, 0, sizeInScene.X, sizeInScene.Y);

        }

        /// <summary>
        /// 矩形を描きます。
        /// </summary>
        static public void DrawRectangleWithoutTransform( IRenderer renderer, RectangleF rect )
        {
            PointF[]	rectPos = {
                new PointF( rect.Left, rect.Top ),
                new PointF( rect.Right, rect.Top ),
                new PointF( rect.Right, rect.Bottom ),
                new PointF( rect.Left, rect.Bottom )
            };


            renderer.DrawRectangle(rect.Left, rect.Top, 0, rect.Width, rect.Height);

        }


        /// <summary>
        /// 十字を描画します。
        /// </summary>
        static public void DrawCrossWithoutTransform( IRenderer renderer, PointF p, float size )
        {
            PointF[] ps = { p };
            renderer.DrawLine(
                              new PointF( ps[0].X, ps[0].Y - size * 0.5f ),
                              new PointF( ps[0].X, ps[0].Y + size * 0.5f ) );

            renderer.DrawLine(
                              new PointF( ps[0].X - size * 0.5f, ps[0].Y ),
                              new PointF( ps[0].X + size * 0.5f, ps[0].Y ) );
        }

        /// <summary>
        /// 文字列を描画します。
        ///
        /// 座標変換処理が内部で行われるため、パフォーマンスに不安があります。
        /// とりあえずはこのままにしておき、将来リファクタリングすることとします。
        /// DirectX描画が導入されれば、その必要もないかもしれません。
        ///
        /// メモ：座標系の差異を吸収するために、Y座標スケールに -1.0f を乗じている。
        /// そのために、通常の文字列描画を行うと、文字がさかさまに描画されてしまう。
        ///
        /// </summary>
        static public void DrawStringWithoutTransform( IRenderer renderer, Font font, Color color, string str, float px, float py )
        {
            // 現在のカレント行列で、描画位置を計算します。
            var temp = new PointF[]{new PointF(px, py)};
            renderer.TransformPointsToView(temp);

            // 行列を単位行列に設定し、文字を描画します。
            renderer.PushMtx();
            renderer.IdentityMtx();

            renderer.Color = color;

            // 左上原、Y軸下向きの座標系に変換して利用します。
            float halfW = renderer.ViewportSize.Width * 0.5f;
            float halfH = renderer.ViewportSize.Height * 0.5f;
            renderer.DrawString(str, font, temp[0].X + halfW, -temp[0].Y + halfH);

            renderer.PopMtx();
        }

        /// <summary>
        /// 値を min/max でクランプします。
        /// </summary>
        /// <param name="value"></param>
        /// <param name="min"></param>
        /// <param name="max"></param>
        /// <returns></returns>
        static public float Clamp(float value, float min, float max)
        {
            return Math.Min(Math.Max(value, min), max);
        }

        /// <summary>
        /// Degree を Radian へ変換します。
        /// </summary>
        /// <param name="degree"></param>
        /// <returns></returns>
        static public float ToRadian(float degree)
        {
            return degree / 180.0f * (float)Math.PI;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public enum RendererTextrureFormat
    {
        ARGB,
        A8FromRGB,
        A8FromARGB,
        BC1FromARGB,
        L8,
        LA8,
    }

    /// <summary>
    ///
    /// </summary>
    public static class RendererTextrureFormatHelper
    {
        public static RendererTextrureFormat GetRendererTextrureFormat(TexImagePixelFmt fmt, bool sourceImgHasAlpha)
        {
            if (fmt.IsIntencityFormat())
            {
                return Core.RendererTextrureFormat.L8;
            }
            else if (fmt.IsIntencityAlphaFormat())
            {
                return Core.RendererTextrureFormat.LA8;
            }
            else if (fmt.IsAlphaFormat())
            {
                return (sourceImgHasAlpha) ? Core.RendererTextrureFormat.A8FromARGB : Core.RendererTextrureFormat.A8FromRGB;
            }
            else if (fmt == TexImagePixelFmt.BC1 && sourceImgHasAlpha)
            {
                return Core.RendererTextrureFormat.BC1FromARGB;
            }

            return RendererTextrureFormat.ARGB;
        }
    }

    /// <summary>
    /// IRenderer の概要の説明です。
    ///
    /// 描画モジュールを抽象化するクラス。
    /// 複数の描画API(GDI+,OpenGL,DirectXなど)を
    /// アプリケーションから隠蔽します。
    ///
    ///
    /// 描画処理は Gof. Visitorパターンで処理されます。
    ///
    /// IRenderer は Visitor を 表現するインタフェースと考えることも出来ます。
    /// (Acceptor は IDrawable )
    ///
    /// </summary>
    public interface IRenderer : IDisposable
    {
        /// <summary>
        /// レンダラの初期化を行います。
        /// </summary>
        bool Initialize( object initParams );

        /// <summary>
        /// 描画処理の前処理を行います。
        /// すべての描画処理の前に呼び出す必要があります。
        /// </summary>
        /// <param name="initParams">前処理に必要な、パラメータ</param>
        void BeginRendering( object initParams );
        /// <summary>
        /// 描画後処理を行います。
        /// すべての描画処理の後に呼び出す必要があります。
        /// </summary>
        void EndRendering();

        /// <summary>
        /// IRenderOptionFlag で規定される、
        /// オプションフラグを設定します。
        /// </summary>
        uint OptionFlag   { get; set; }

        void EnableOptionFlag( uint flag );
        void DisableOptionFlag( uint flag );
        void ToggleOptionFlag( uint flag );

        /// <summary>
        /// 四角を描画します。
        /// </summary>
        /// <param name="x">位置</param>
        /// <param name="y">位置</param>
        /// <param name="w">サイズ</param>
        /// <param name="h">サイズ</param>
        void DrawRectangle( float x, float y, float z, float w, float h );
        void DrawRectangleOrtho(float x, float y, float z, float w, float h);
        void DrawPoint( PointF p );
        void DrawLine( Point p0, Point p1 );
        void DrawLine( PointF p0, PointF p1 );
        void FillRectangle( int x, int y, int z, int w, int h );
        void FillRectangle( float x, float y, float z, float w, float h );

        void DrawString( string str, System.Drawing.Font font, int x, int y );
        void DrawString( string str, System.Drawing.Font font, float x, float y );

        void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] UVs);
        void DrawImage(FVec3 imgPos, FVec2 imgSize, TexCoord4[] UVs, Color[] vtxColors);
        void TryUpdateTextureIfOld(Bitmap bmp, DateTime lastModifyTime, RendererTextrureFormat format);
        void TryRemoveTextureFromBitmap(Bitmap bmp, RendererTextrureFormat format);
        void SetTextureState(Bitmap bmp, int slotIdx, AttrCombineMode colorBlend, AttrCombineMode alphaBlend, FVec2 indirectScale, float indirectRotate, int indirectImageChunnelNum, RendererTextrureFormat format);
        void SetDetailedCombinerState(Color[] colors, ITevStage[] tevStage, int stageCount);

        void SetProceduralShapeState(float paneWidth, float paneHeight, float exp, float radius);
        void SetProceduralShapeEffectBlendState(
            ProceduralShapeEffectBlendMode innerStrokeBlendType,
            ProceduralShapeEffectBlendMode innerShadowBlendType,
            ProceduralShapeEffectBlendMode colorOverlayBlendType,
            ProceduralShapeEffectBlendMode gradationOverlayBlendType);
        void SetProceduralShapeInnerStrokeState(bool enabled, float innerStrokeSize, Color innerStrokeColor);
        void SetProceduralShapeInnerShadowState(bool enabled, float paneWidth, float paneHeight, float radius, float innerShadowSize, float angle, float distance, Color innerShadowColor, ProceduralShapeShadowType shadowType);
        void SetProceduralShapeColorOverlayState(bool enabled, Color color);
        void SetProceduralShapeGradationOverlayState(bool enabled, float[] controlPoints, Color[] colors, float angle);

        void DrawMesh(float[] vertices, float[] texCoords, int[] indices);

        void SetBlendMode(AttrBlendType colorBlendtype, AttrBlendFactor srcColor, AttrBlendFactor dstColor, AttrBlendOp colorOp, AttrBlendFactor srcAlpha, AttrBlendFactor dstAlpha, AttrBlendOp alphaOp);
        void SetAlphaCompare(bool enabled, AttrCompareFunc alphaCompare, float alpha);

        void SetPackedFontMode(bool ignoreBorder);

        void SetLineCap( LineCap start, LineCap end );

        void TransformPointsToView( PointF[] pointSet );
        void TransformPointsToScene(PointF[] pointSet);

        /// <summary>
        /// デバック情報描画用API
        /// デバック情報出力を無効化すると、描画が行われなくなります。
        /// </summary>
        void DbgDrawLine( PointF p0, PointF p1 );
        void DbgDrawRectangle( float x, float y, float z, float w, float h );
        void DbgDrawString( string str, System.Drawing.Font font, float x, float y );

        /// <summary>
        /// テクスチャ行列を設定する。
        /// </summary>
        void SetTextureMtx( int index, FVec2 trans, FVec2 scale, float rotate );
        void SetTextureProjectionState(int index, FVec2 imageSize, Matrix34 mxt);

        /// <summary>
        /// テクスチャサンプリング設定を行う。
        /// </summary>
        void SetTextureSamplingState(
                                     int idx,
                                     TexWrap wrapS, TexWrap wrapT,
                                     TexFilterMin minFilter, TexFilterMag magFilter, bool isSrgb);

        /// <summary>
        /// テクスチャサンプリング設定を行う。
        /// </summary>
        void SetTextureSamplingSrgbState(int idx, bool isSrgb);

        /// <summary>
        /// テクスチャサンプリング設定を行う。
        /// </summary>
        void GetMatrialColorBlend( out Color whiteBlendColor, out Color blackBlendColor );

        /// <summary>
        /// テクスチャサンプリング設定を行う。
        /// </summary>
        void SetMatrialColorBlend(FloatColor whiteBlendColor, FloatColor blackBlendColor, bool isThresholdingAlphaInterpolationEnabled);

        /// <summary>
        /// 行列スタックをPushします。
        /// </summary>
        void PushMtx();

        /// <summary>
        /// 行列スタックをPopします。
        /// </summary>
        void PopMtx();

        /// <summary>
        /// 行列スタックのカレント行列を正規行列に設定します。
        /// </summary>
        void IdentityMtx();

        /// <summary>
        /// 行列スタックのカレント行列に平行移動成分を乗算します。
        /// </summary>
        /// <param name="x">平行移動</param>
        /// <param name="y">平行移動</param>
        void Trans( int x, int y );
        void Trans( float x, float y );

        /// <summary>
        /// 行列スタックのカレント行列に平行移動成分を乗算します。
        /// </summary>
        /// <param name="x">平行移動</param>
        /// <param name="y">平行移動</param>
        /// <param name="z">平行移動</param>
        void Trans( int x, int y, int z );
        void Trans( float x, float y, float z );

        /// <summary>
        /// 行列スタックのカレント行列にスケール演算を乗算します。
        /// </summary>
        /// <param name="sx">スケール</param>
        /// <param name="sy">スケール</param>
        void Scale( float sx, float sy );

        /// <summary>
        /// 行列スタックのカレント行列にスケール演算を乗算します。
        /// </summary>
        /// <param name="sx">スケール</param>
        /// <param name="sy">スケール</param>
        /// <param name="sz">スケール</param>
        void Scale( float sx, float sy, float sz );

        /// <summary>
        /// ビュー変換を設定します。
        /// </summary>
        void SetViewTransform(float sx, float sy, float tx, float ty, float tz, float pz);

        /// <summary>
        /// 基準スクリーンサイズ（≒レイアウトサイズ）を設定します。
        /// </summary>
        void SetDefaultScreenSize(float width, float height);

        /// <summary>
        /// Bitmap に対応するテクスチャを開放します。
        /// </summary>
        void ClearBitmapTextureMap();

        /// <summary>
        /// カレント行列を設定、取得します。
        /// </summary>
        /// <param name="mtx"></param>
        Matrix CurrentMtx
        {
            get;
            set;
        }

        /// <summary>
        /// カレント行列を設定、取得します。
        /// </summary>
        /// <param name="mtx"></param>
        Matrix34 CurrentMtx34
        {
            get;
            set;
        }

        /// <summary>
        /// 現在の色
        /// </summary>
        Color Color
        {
           get;
           set;
        }

        /// <summary>
        /// ライン幅
        /// </summary>
        float LineWidth
        {
            get;
            set;
        }

        /// <summary>
        /// ライン破線スタイル
        /// </summary>
        DashStyle LineDashStyle
        {
            get;
            set;
        }

        /// <summary>
        /// 透明度
        /// </summary>
        int Transparency
            {
            get;
            set;
        }

        /// <summary>
        /// 矩形描画時の上端 X位置オフセット
        /// <remarks>
        /// GDIレンダラーはほとんど使われないはずなので対応していません。
        /// </remarks>
        /// </summary>
        float RectangleTopXOffset
        {
            get;
            set;
        }

        /// <summary>
        /// フォントの高さ
        /// </summary>
        float FontHeight
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        Bitmap Capture( DrawerForCapture drawer, Rectangle bounds, bool outputAlpha);

        /// <summary>
        /// リニア色空間でSRGBフェッチを行うかどうかを指定する
        /// </summary>
        bool LinearGammmaSRGBFetchEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間でSRGB書き込みを行うかどうかを指定する
        /// </summary>
        bool LinearGammmaSRGBWriteEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// リニア色空間の色パラメータモードを指定する
        /// </summary>
        bool LinearGammmaColorParamatersEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// レンダリング結果を画面に表示するかどうか
        /// </summary>
        bool NeedPresent
        {
            get;
            set;
        }

        /// <summary>
        /// 視野角（ラジアン）
        /// </summary>
        float FOV { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        float PerseNear { get; set; }

        /// <summary>
        /// ニア平面
        /// </summary>
        float PerseFar { get; set; }

        /// <summary>
        /// パースぺクティブ表示が有効
        /// </summary>
        bool PersepectiveRendering { get; set; }

        /// <summary>
        /// 深度テストが有効
        /// </summary>
        bool DepthWriteTestEnabled { get; set; }

        /// <summary>
        /// ビューポートのサイズ
        /// </summary>
        SizeF ViewportSize { get; }

        /// <summary>
        /// デバイスロストされたかどうか？
        /// </summary>
        bool IsDeviceLost { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public delegate void DrawerForCapture( Graphics gc);
}
