﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Linq;

namespace LECore.Structures.Core.Command
{
    using LECore.Structures.LECoreInterface;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;


    using FileFmtMaterialTexMap = LECore.Save_Load.RlytConverter.FileFmtMaterialTexMap;

    #region MementoBase<TInstanceType>
    /// <summary>
    ///
    /// </summary>
    internal abstract class MementoBase<TInstanceType> :
        ICommandMemento<TInstanceType>
    {
        static protected readonly AppData.Pane _DummyMementoPane = new AppData.Pane();
        static protected readonly PaneParamaterCopyOption _CopyOption;
        static protected readonly LECore.Save_Load.RlytConverter _RlytConverter;

        /// <summary>
        ///
        /// </summary>
        abstract protected ParamaterKind ActiveParamaterKind { get; }

        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static MementoBase()
        {
            _RlytConverter = new LECore.Save_Load.RlytConverter(string.Empty,LECore.LayoutEditorCore.MsgReporter );

            // テキストボックス関連のみをコピーするように設定します。
            _CopyOption = new PaneParamaterCopyOption();
        }

        protected void SetupCopyOption_()
        {
            _CopyOption.SetOption( ParamaterKind.All, false );
            _CopyOption.EnableOption( ActiveParamaterKind );
        }
        protected abstract void DoCopy_( TInstanceType dstInstance );

        public void CopyTo( TInstanceType dstInstance )
        {
            SetupCopyOption_();
            DoCopy_( dstInstance );
        }
    }
    #endregion

    #region UserDataMemento
    /// <summary>
    /// IPaneのユーザ情報のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class UserDataMemento : MementoBase<IUserDataHolder>
    {
        /// <summary>
        /// ファイル保存形式のユーザデータ
        /// </summary>
        readonly object[] _userData;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataMemento( AppData.IUserDataHolder src )
        {
            _userData = _RlytConverter.Convert_Item_UserDataEx(src.UserDataElementSet);
        }


        #region ICommandMementFactory<ITextBox> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.UserDataEx; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_(IUserDataHolder dst)
        {
            UserDataHolder dstHolder = dst as UserDataHolder;

            dstHolder.RemoveUserDataElementAll();
            foreach (UserDataElement ud in _RlytConverter.Convert_Pane_UserData(_userData))
            {
                dstHolder.AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
            }
        }
        #endregion
    }
    #endregion

    internal class PaneMemento : MementoBase<IPane>
    {
        /// <summary>
        /// ファイル保存形式のTextBox
        /// </summary>
        readonly FileFmt.Pane _filePane;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneMemento(AppData.IPane src)
        {
            _filePane = _RlytConverter.ConvertPaneToSerializable(src);
        }


        #region ICommandMementFactory<ITextBox> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.All; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_(IPane dstPane)
        {
            IPane tempPane = _RlytConverter.ConvertPaneToInternal(_filePane);

            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, tempPane);
        }
        #endregion
    }

    #region TextBoxMemento
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class TextBoxMemento : MementoBase<ITextBox>
    {
        /// <summary>
        /// ファイル保存形式のTextBox
        /// </summary>
        readonly FileFmt.TextBox _fileTextBox;


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextBoxMemento( AppData.ITextBox srcTextBox )
        {
            _fileTextBox = _RlytConverter.Convert_Item_TextBoxSelf( srcTextBox );
        }

        #region ICommandMementFactory<ITextBox> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            // PerCharTransformCurve はアニメーションデータとしてUndo/Redo処理されるので扱わない
            get { return ParamaterKind.TextBoxAll & ~ParamaterKind.PerCharTransformCurve; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( ITextBox dstTextBox )
        {
            ITextBox srcTextBox = _RlytConverter.Convert_Pane_TextBoxSelf( _DummyMementoPane, _fileTextBox );
            IPane dstPane = ( dstTextBox as AppData.TextBox ).OwnerPane;

            PaneParamaterPaster.PasteParamaters( _CopyOption, dstPane, _DummyMementoPane );
        }
        #endregion
    }
    #endregion

    #region CaptureMemento
    /// <summary>
    /// AppData.ICapture のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class CaptureMemento : MementoBase<ICapture>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CaptureMemento( AppData.ICapture srcCapture)
        {
            // CapturePane に対する処理は実質 CaptureTexture に対する処理となるため CaptureMemento では特に何も行いません。
        }

        #region ICommandMementFactory<ICapture> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.None; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( ICapture dstCapture )
        {
        }
        #endregion
    }
    #endregion

    #region MaskMemento
    internal class MaskMemento : MementoBase<IMask>
    {
        readonly FileFmt.Mask _mask;

        public MaskMemento(AppData.IMask srcMask)
        {
            _mask = _RlytConverter.Convert_Item_MaskSelf(srcMask);
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get
            {
                return ParamaterKind.MaskAll;
            }
        }

        protected override void DoCopy_(IMask dstMask)
        {
            IMask srcMask = _RlytConverter.Convert_Pane_MaskSelf(_DummyMementoPane, _mask);
            IPane dstPane = dstMask.OwnerPane;
            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
    }
    #endregion

    #region StateMachineMemento
    internal class StateMachineMemento : MementoBase<IStateMachine>
    {
        //readonly FileFmt.Mask _mask;

        public StateMachineMemento(AppData.IStateMachine src)
        {
            // TODO:
            //_mask = _RlytConverter.Convert_Item_MaskSelf(srcMask);
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get
            {
                return ParamaterKind.MaskAll;
            }
        }

        protected override void DoCopy_(IStateMachine dst)
        {
            // TODO:
            //IMask srcMask = _RlytConverter.Convert_Pane_MaskSelf(_DummyMementoPane, _mask);
            //IPane dstPane = dstMask.OwnerPane;
            //PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
    }
    #endregion

    #region DropShadowMemento
    internal class DropShadowMemento : MementoBase<IDropShadow>
    {
        readonly FileFmt.DropShadow _dropShadow;

        public DropShadowMemento(AppData.IDropShadow srcDropShadow)
        {
            _dropShadow = _RlytConverter.Convert_Item_DropShadowSelf(srcDropShadow);
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get
            {
                return ParamaterKind.DropShadowAll;
            }
        }

        protected override void DoCopy_(IDropShadow dstDropShadow)
        {
            IDropShadow srcDropShadow = _RlytConverter.Convert_Pane_DropShadowSelf(_DummyMementoPane, _dropShadow);
            IPane dstPane = dstDropShadow.OwnerPane;
            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
    }
    #endregion

    #region ProceduralShapeMemento
    internal class ProceduralShapeMemento : MementoBase<IProceduralShape>
    {
        readonly FileFmt.ProceduralShapeParam _proceduralShapeParam;

        public ProceduralShapeMemento(AppData.IProceduralShape srcProceduralShape)
        {
            _proceduralShapeParam = _RlytConverter.Convert_Item_ProceduralShapeSelf(srcProceduralShape);
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get
            {
                return ParamaterKind.ProceduralShapeAll;
            }
        }

        protected override void DoCopy_(IProceduralShape dstProceduralShape)
        {
            // パラメータのコピー時にピクチャペインとして扱ってほしいため、ダミーの Picture クラスのインスタンスを作成する。
            Picture pic = new AppData.Picture(_DummyMementoPane);

            _RlytConverter.Convert_Pane_ProceduralShapeSelf(_DummyMementoPane, _proceduralShapeParam);
            IPane dstPane = dstProceduralShape.OwnerPane;
            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
    }
    #endregion



    #region AlignmentMemento
    /// <summary>
    /// AppData.ICapture のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class AlignmentMemento : MementoBase<IAlignment>
    {
        readonly FileFmt.Alignment _fileAlignment;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AlignmentMemento(AppData.IAlignment src)
        {
            _fileAlignment = _RlytConverter.Convert_Item_AlignmentSelf(src);
        }

        #region ICommandMementFactory<ICapture> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.AlignmentAll; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_(IAlignment dst)
        {
            IAlignment src = _RlytConverter.Convert_Pane_AlignmentSelf(_DummyMementoPane, _fileAlignment);
            IPane dstPane = dst.OwnerPane;

            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
        #endregion
    }


    #endregion

    #region ScissorMemento
    /// <summary>
    /// AppData.IScissor のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class ScissorMemento : MementoBase<IScissor>
    {
        readonly FileFmt.Scissor _fileScissor;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ScissorMemento(AppData.IScissor src)
        {
            _fileScissor = _RlytConverter.Convert_Item_ScissorSelf(src);
        }

        #region ICommandMementFactory<IScissor> メンバ
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.ScissorAll; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_(IScissor dst)
        {
            IScissor src = _RlytConverter.Convert_Pane_ScissorSelf(_DummyMementoPane, _fileScissor);
            IPane dstPane = dst.OwnerPane;

            PaneParamaterPaster.PasteParamaters(_CopyOption, dstPane, _DummyMementoPane);
        }
        #endregion
    }
    #endregion

    #region PictureMemento
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class PictureMemento : MementoBase<IPicture>
    {
        /// <summary>
        /// ファイル保存形式のTextBox
        /// </summary>
        readonly FileFmt.Picture _filePicture;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PictureMemento( AppData.IPicture srcPicture )
        {
            _filePicture = _RlytConverter.Convert_Item_TextBoxSelf(srcPicture);
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.PictureAll; }
        }

        protected override void DoCopy_( IPicture dstInstance )
        {
            IPicture srcPicture = _RlytConverter.Convert_Pane_TextBoxSelf( _DummyMementoPane, _filePicture );
            IPane dstPane = ( dstInstance as AppData.Picture ).OwnerPane;

            PaneParamaterPaster.PasteParamaters( _CopyOption, dstPane, _DummyMementoPane );
        }

    }
    #endregion

    #region WindowMemento
    /// <summary>
    /// AppData.ILEWindowのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class WindowMemento : MementoBase<ILEWindow>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly IPane _paneChache;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public WindowMemento( AppData.ILEWindow srcWindow )
        {
            _paneChache = _RlytConverter.ConvertPaneToInternal(_RlytConverter.ConvertPaneToSerializable(srcWindow.OwnerPane));
        }

        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.Size | ParamaterKind.WindowAll; }
        }

        protected override void DoCopy_( ILEWindow dstInstance )
        {
            IPane dstPane = dstInstance.OwnerPane;
            PaneParamaterPaster.PasteParamaters( _CopyOption, dstPane, _paneChache );
            (dstInstance as AppData.LEWindow).OnWindowKindChangeUpdate();
        }

    }
    #endregion

    #region WindowMaterial
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class MaterialMemento : MementoBase<IMaterial>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly FileFmt.Material _fileMaterial;
        readonly List<TextureChangerMemento> _texChangerMementSet;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MaterialMemento( AppData.IMaterial srcMaterial )
        {
            _fileMaterial = _RlytConverter.Convert_ToFileMaterial( srcMaterial );

            // テクスチャーパターン
            if (_fileMaterial.texMap.Length > 0)
            {
                _texChangerMementSet = new List<TextureChangerMemento>();
                for (int i = 0; i < _fileMaterial.texMap.Length; i++)
                {
                    _texChangerMementSet.Add(new TextureChangerMemento(srcMaterial.GetMaterialTexMap(i).ITextureChanger as TextureChanger));
                }
            }
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.MaterialAll; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( IMaterial dstInstance )
        {
            IMaterial srcInstance = _RlytConverter.Convert_Pane_Material( _fileMaterial, false );
            MaterialParamaterPaster.CopyMaterialParamater( _CopyOption, dstInstance, srcInstance );

            // テクスチャーパターン
            for (int i = 0; i < _fileMaterial.texMap.Length; i++)
            {
                var dstTexMap = dstInstance.GetMaterialTexMap(i);
                if (dstTexMap != null)
                {
                    string currentName = dstTexMap.TexImgName;
                    _texChangerMementSet[i].CopyTo(dstTexMap.ITextureChanger as TextureChanger);

                    // テクスチャパターン設定をコピーした際、アニメーションが評価されるので元の値に戻します。
                    // 本来は、アニメーションが評価されない状況(例えば部品に対する上書き)での Undo /Redo で違和感のある結果になることを防ぎます。
                    (dstTexMap as MaterialTexMap).TexImgName = currentName;
                }
            }
        }

    }
    #endregion

    #region WindowMaterial
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class MaterialRVLMemento : MementoBase<IRevHWMaterial>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly FileFmt.Material_CTR _fileMaterial;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MaterialRVLMemento( AppData.IRevHWMaterial srcMaterial )
        {

            _fileMaterial = _RlytConverter.Convert_ToFileMaterial_Revo( srcMaterial );
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.MaterialDetailsAll; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( IRevHWMaterial dstInstance )
        {
            IRevHWMaterial srcInstance = _RlytConverter.Convert_Pane_RevMaterial( _fileMaterial );
            MaterialParamaterPaster.CopyMaterialDetailParamaters( _CopyOption, dstInstance, srcInstance, true );
        }
    }
    #endregion

    #region
    /// <summary>
    ///
    /// </summary>
    internal class TextureChangerMemento : MementoBase<TextureChanger>
    {
        ///
        List<string> texturePatternNames = new List<string>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureChangerMemento( TextureChanger srcInstance )
        {
            foreach( string name in srcInstance.ITextureSubSet.TextureSubSetNameSet )
            {
                this.texturePatternNames.Add( name);
            }
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.Material_Textures; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( TextureChanger dstInstance )
        {
            TextureSubSet textureSubSet = dstInstance.ITextureSubSet as TextureSubSet;
            textureSubSet.Clear();
            textureSubSet.AddTextures( this.texturePatternNames.ToArray());
            dstInstance.Evaluate();
        }
    }
    #endregion

    #region IMaterialTexMap
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class MaterialTexMapMemento : MementoBase<IMaterialTexMap>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly FileFmtMaterialTexMap _fileData;
        readonly FileFmt.TexMatrix _fileMtx;
        static readonly IMaterial _DummyMaterial;

        /// <summary>
        ///
        /// </summary>
        static MaterialTexMapMemento()
        {
            _DummyMaterial = new Material( null, "MaterialMemento_DummyMaterial" );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MaterialTexMapMemento(AppData.IMaterialTexMap srcInstance)
        {
            _fileData = _RlytConverter.Convert_ToFileTexMap(srcInstance);
            _fileMtx = _RlytConverter.Convert_ToFileTexMtx(srcInstance.ITexGen.ITexMtx);
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.Material_Textures; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( IMaterialTexMap dstInstance )
        {
            IMaterialTexMap srcInstance = _RlytConverter.Convert_MaterialTexMap( _DummyMaterial, _fileData );
            _RlytConverter.Convert_Pane_MaterialTexMtx(srcInstance.ITexGen.ITexMtx as AppData.TexMtx, _fileMtx);

            MaterialParamaterPaster.CopyAllMatareialTextureMap( dstInstance as MaterialTexMap, srcInstance );
        }
    }
    #endregion

#if false
    ///@@@
    #region IMaterialTexMap
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class TexGenMemento : MementoBase<TexGen>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly FileFmtMaterialTexMap _fileData;
        static readonly TexGen _DummyTexGen = null;

        /// <summary>
        ///
        /// </summary>
        static TexGenMemento()
        {
            _DummyTexGen = new TexGen();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexGenMemento( AppData.IMaterialTexMap srcInstace )
        {
            _fileData = _RlytConverter.Convert_ToFileTexCoordGen( srcInstace );
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.PictureAll; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( TexGen dstInstance )
        {
            IMaterialTexMap srcInstance = _RlytConverter.Convert_TexCoordGen( _DummyMaterial, _fileData );
            MaterialParamaterPaster.CopyAllMatareialTextureMap( dstInstance as MaterialTexMap, srcInstance );
        }
    }
    #endregion
#endif

    #region TexMtxMemento
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class TexMtxMemento : MementoBase<ITexMtx>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly FileFmt.TexMatrix _fileData;


        /// <summary>
        ///
        /// </summary>
        static TexMtxMemento()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexMtxMemento( AppData.ITexMtx srcInstace )
        {
            _fileData = _RlytConverter.Convert_ToFileTexMtx( srcInstace );
        }

        /// <summary>
        /// ActiveParamaterKind
        /// </summary>
        protected override ParamaterKind ActiveParamaterKind
        {
            get { return ParamaterKind.Material_Textures; }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void DoCopy_( AppData.ITexMtx dstInstance )
        {
            _RlytConverter.Convert_Pane_MaterialTexMtx( dstInstance as AppData.TexMtx, _fileData );
        }
    }
    #endregion

    #region LEGroup
    /// <summary>
    /// AppData.ITextBoxのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class LEGroupMemento : ICommandMemento<ILEGroup>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly string _name;
        readonly string _commnet;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEGroupMemento( AppData.ILEGroup srcInstace )
        {
            _name = srcInstace.GrouprName;
            _commnet = srcInstace.UserCommentString;
        }


        /// <summary>
        ///
        /// </summary>
        public void CopyTo( AppData.ILEGroup dstInstance )
        {
            ( dstInstance as LEGroup ).GrouprName = _name;
            ( dstInstance as LEGroup ).UserCommentString = _commnet;
        }
    }
    #endregion

    #region AnimFrameSection
    /// <summary>
    /// AppData.AnimFrameSectionのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class AnimFrameSectionMemento : ICommandMemento<IAnimFrameSection>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly AnimFrameSection _snapShotCopy;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimFrameSectionMemento( AppData.IAnimFrameSection srcInstace )
        {
            // _owner フィールドはコピーしません。
            _snapShotCopy = new AnimFrameSection();
            _snapShotCopy.Set(
                srcInstace.Name,
                srcInstace.Comment,
                srcInstace.StartFrame,
                srcInstace.EndFrame );

            _snapShotCopy.SetTargetGroup( srcInstace.TargetGroup,
                                    srcInstace.BindAnimationRecursively);

            (_snapShotCopy.BinarySettings as AnimFrameSectionBinarySettings).Set(
                srcInstace.BinarySettings.Active,
                srcInstace.BinarySettings.FileName,
                srcInstace.BinarySettings.LoopType,
                srcInstace.BinarySettings.TargetAnimKind );
        }


        /// <summary>
        ///
        /// </summary>
        public void CopyTo( AppData.IAnimFrameSection dstInstance )
        {
            ( dstInstance as AnimFrameSection ).Set(
                _snapShotCopy.Name,
                _snapShotCopy.Comment,
                _snapShotCopy.StartFrame,
                _snapShotCopy.EndFrame );

      AnimFrameSection dstFrameSection = dstInstance as AnimFrameSection;
      dstFrameSection.SetTargetGroup( _snapShotCopy.TargetGroup,
                                      _snapShotCopy.BindAnimationRecursively);

            (dstInstance.BinarySettings as AnimFrameSectionBinarySettings).Set(
                _snapShotCopy.BinarySettings.Active,
                _snapShotCopy.BinarySettings.FileName,
                _snapShotCopy.BinarySettings.LoopType,
                _snapShotCopy.BinarySettings.TargetAnimKind );

            ( dstInstance as AnimFrameSection ).NotifyAnimSectionTagChanged();
        }
    }
    #endregion

    #region AnimFrameSectionSet
    /// <summary>
    /// AppData.AnimFrameSectionのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class AnimFrameSectionSetMemento : ICommandMemento<IAnimFrameSectionSet>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        readonly IAnimFrameSection[] _snapShotCopy;
        readonly IAnimFrameSection[] targets = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimFrameSectionSetMemento( AppData.IAnimFrameSectionSet srcInstace )
        {
            this.targets = srcInstace.IAnimFrameSectionSet.ToArray();

            //
            _snapShotCopy = new IAnimFrameSection[ this.targets.Length ];
            for( int index = 0; index < this.targets.Length; index++ )
            {
                var newFrameSection = new AnimFrameSection();
                newFrameSection.Initialzie(srcInstace);

                _snapShotCopy[index] = newFrameSection;
                Copy( this.targets[index], _snapShotCopy[index]);
            }
        }

#if false
        /// <summary>
        /// セクションを登録します。
        /// </summary>
        IAnimFrameSection AddSectionToSet_( AnimFrameSectionSet sectionSet, IAnimFrameSection srcSection )
        {
            IAnimFrameSection newSection = sectionSet.AddNewFrameSection(
                                                                         srcSection.Name,
                                                                         srcSection.Comment,
                                                                         srcSection.StartFrame,
                                                                         srcSection.EndFrame );

            AnimFrameSection dstSection = newSection as AnimFrameSection;
            dstSection.SetTargetGroup( srcSection.TargetGroup,
                                       srcSection.BindAnimationRecursively);

            IAnimFrameSectionBinarySettings srcBinSetting = srcSection.BinarySettings;
            AnimFrameSectionBinarySettings dstBinSetting = newSection.BinarySettings as AnimFrameSectionBinarySettings;

            dstBinSetting.Set(
                              srcBinSetting.Active,
                              srcBinSetting.FileName,
                              srcBinSetting.LoopType,
                              srcBinSetting.TargetAnimKind );

            return newSection;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        private void Copy( IAnimFrameSection srcSection, IAnimFrameSection iDstSection)
        {
            AnimFrameSection dstSection = iDstSection as AnimFrameSection;

            dstSection.Set( srcSection.Name,
                            srcSection.Comment,
                            srcSection.StartFrame,
                            srcSection.EndFrame);

            dstSection.SetTargetGroup( srcSection.TargetGroup,
                                       srcSection.BindAnimationRecursively);

            (dstSection.BinarySettings as AnimFrameSectionBinarySettings).Set
                ( srcSection.BinarySettings.Active,
                  srcSection.BinarySettings.FileName,
                  srcSection.BinarySettings.LoopType,
                  srcSection.BinarySettings.TargetAnimKind );
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo( AppData.IAnimFrameSectionSet dstInstance )
        {
            AnimFrameSectionSet dstSectionSet = dstInstance as AnimFrameSectionSet;
            IAnimFrameSection[] dstArray = dstInstance.IAnimFrameSectionSet.ToArray();

            // 削除された区間タグを削除
            for( int i = 0 ; i < dstArray.Length ; i++ )
            {
                if( Array.IndexOf( this.targets, dstArray[i] ) == -1 )
                {
                    dstSectionSet.Remove( dstArray[i]);
                }
            }

            // 追加された区間タグを登録
            for( int i = 0 ; i < this.targets.Length ; i++ )
            {
                IAnimFrameSection dstSection = this.targets[i];
                if( dstSectionSet.CheckExistsByName( dstSection.Name) == false )
                {
                    Copy( _snapShotCopy[i], dstSection);
                    dstSectionSet.AddFrameSection( dstSection);
                }
            }

            // 再度配列を取得します。
            dstArray = dstSectionSet.IAnimFrameSectionSet.ToArray();
            Debug.Assert( this.targets.Length == dstArray.Length );

            // タグの順番を整列
            for( int i = 0 ; i < this.targets.Length ; i++ )
            {
                // 等しくないなら...
                if( dstArray[i].Name != this.targets[i].Name )
                {
                    // 等しい順番の、タグ区間を検索し、
                    IAnimFrameSection target = Array.Find
                        ( dstArray,
                          delegate( IAnimFrameSection section )
                              {
                                  return section.Name == this.targets[i].Name;
                              });

                    // 順番を同一に設定する。
                    bool result = dstSectionSet.SetTagIndex( target, i );
                    Debug.Assert( result );
                }
            }
        }
    }

    #endregion

    #region AnimFrameSection

    /// <summary>
    /// AppData.AnimFrameSectionのクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class TextureMgrMemento : ICommandMemento<ITextureMgr>
    {
        /// <summary>
        /// ファイル保存形式
        /// </summary>
        class TextureData
        {
            static protected readonly LECore.Save_Load.RlytConverter _RlytConverter = new LECore.Save_Load.RlytConverter(string.Empty,LECore.LayoutEditorCore.MsgReporter );

            public readonly string _name;
            public readonly string _filePath;
            public readonly TexImagePixelFmt _pixelFmt;
            public readonly TextureSourceType _sourceType;
            public readonly FileFmt.CaptureTexture _captureTextureInfo;

            public TextureData( ITextureImage src )
            {
                _name = src.Name;
                _filePath = src.FilePath;
                _pixelFmt = src.PixelFmt;
                _sourceType = src.SourceType;

                switch (_sourceType)
                {
                    case TextureSourceType.File:
                        // 基本情報にすべて含まれるため何も行わない
                        break;
                    case TextureSourceType.Dynamic:
                        // キャプチャテクスチャ専用のデータを中間ファイルフォーマットで保存する
                        _captureTextureInfo = _RlytConverter.Convert_SOCaptureTexture_(src.ICaptureTexture);
                        break;
                    default:
                        Debug.Assert(false);
                        break;
                }
            }

            /// <summary>
            /// AppData.CaptureTexture へ自身の持っているキャプチャテクスチャ情報をコピーします。
            /// </summary>
            /// <param name="captureTexture"></param>
            public void CopyCaptureTextureInfoToApp(AppData.ICaptureTexture captureTexture)
            {
                if (_captureTextureInfo != null)
                {
                    _RlytConverter.Convert_CaptureTexture_(captureTexture, _captureTextureInfo);
                }
            }
        }

        readonly List<TextureData> _snapShotCopy;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureMgrMemento( AppData.ITextureMgr srcInstace )
        {
            _snapShotCopy = new List<TextureData>();
            foreach( ITextureImage img in srcInstace.ITextureImageSet )
            {
                _snapShotCopy.Add( new TextureData( img ) );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo( AppData.ITextureMgr dstInstance )
        {
            TextureMgr dstMgr = dstInstance as TextureMgr;
            dstMgr.BeginUpdate();

            //---------------------------
            // 削除すべきテクスチャを検索して、削除します
            ITextureImage[] dstImgSet = dstInstance.ITextureImageSet.ToArray();
            foreach( ITextureImage img in dstImgSet )
            {
                bool bFound = _snapShotCopy.Exists( delegate( TextureData data )
                        {
                            switch(data._sourceType)
                            {
                                case TextureSourceType.File:
                                    return data._filePath == img.FilePath;
                                case TextureSourceType.Dynamic:
                                    return data._name == img.Name;
                                default:
                                    break;
                            }
                            return false;
                        } );

                if( !bFound )
                {
                    dstMgr.RemoveTexture( img );
                }
            }

            //---------------------------
            // すべてのテクスチャについて、
            foreach( TextureData textureData in _snapShotCopy )
            {
                // 必要があれば、テクスチャを登録します。
                if (dstMgr.FindITextureImageByName(textureData._name) == null)
                {
                    switch (textureData._sourceType)
                    {
                        case TextureSourceType.File:
                            dstMgr.RegisterITextureImageFromFile(textureData._filePath);
                            break;
                        case TextureSourceType.Dynamic:
                            {
                                FileFmt.CaptureTexture captureTextureInfo = textureData._captureTextureInfo;

                                // 基本的にあり得ない状態だけど念のためチェック
                                if (captureTextureInfo == null)
                                {
                                    Debug.Assert(false);
                                    break;
                                }

                                // 未登録のキャプチャテクスチャを発見した場合は登録します。
                                ITextureImage captureTexture = TextureMgrHelper.FindCaptureTextureByName(dstMgr, captureTextureInfo.paneName);

                                // すでに同名のテクスチャが存在しない場合のみ追加する
                                if (captureTexture == null)
                                {
                                    IPane pane = dstMgr.ISubScene.FindPaneByName(captureTextureInfo.paneName);
                                    bool isHidingFromList = captureTextureInfo.isHidingFromList;
                                    ITextureImage tex = dstMgr.RegisterCaptureTexture(pane, captureTextureInfo.paneName, isHidingFromList, (CaptureTextureUsage)captureTextureInfo.usage);

                                    // textureData に保存されていたキャプチャテクスチャ情報を復帰する
                                    textureData.CopyCaptureTextureInfoToApp(tex.ICaptureTexture);
                                }
                            }
                            break;
                    }
                }

                // フォーマットを設定します。
                ITextureImage img = dstMgr.FindITextureImageByName(textureData._name);
                if( img != null )
                {
                    if( img.PixelFmt != textureData._pixelFmt )
                    {
                        img.PixelFmt = textureData._pixelFmt;
                    }
                }
            }
            dstMgr.EndUpdate();
        }
    }

    #endregion

    #region CaptureTextureMemento
    /// <summary>
    /// AppData.ICaptureTexture のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class CaptureTextureMemento : ICommandMemento<ICaptureTexture>
    {
        /// <summary>
        /// ファイル保存形式の Capture
        /// </summary>
        readonly FileFmt.CaptureTexture _fileCapture;
        readonly ISubScene _targetScene;
        readonly string _ownerPaneName;
        static protected readonly LECore.Save_Load.RlytConverter _RlytConverter = new LECore.Save_Load.RlytConverter(string.Empty,LECore.LayoutEditorCore.MsgReporter );

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CaptureTextureMemento( AppData.ICaptureTexture srcCapture, ISubScene targetScene)
        {
            _fileCapture = _RlytConverter.Convert_SOCaptureTexture_( srcCapture );

            _targetScene = targetScene;
            if (srcCapture.OwnerPane != null)
            {
                _ownerPaneName = srcCapture.OwnerPane.PaneName;
            }
            else
            {
                _ownerPaneName = "";
            }
        }
        /// <summary>
        ///
        /// </summary>
        public void CopyTo( AppData.ICaptureTexture dstCapture)
        {
            // ファイルにはペイン名と Usage でシリアライズされるため、内部使用用の名前に変換します。
            string captureTextureName = TextureMgrHelper.CreateAutoGeneratedCaptureTextureName(_fileCapture.paneName, (AppData.CaptureTextureUsage)_fileCapture.usage);

            ISubScene subScene = SubSceneHelper.FindCaptureTextureManagedSubScene(_targetScene, captureTextureName);

            // 所属するサブシーンが見つからないケースはないはずだが、念のため null チェック
            if (subScene == null)
            {
                return;
            }

            ITextureImage tex = TextureMgrHelper.FindCaptureTextureByName(subScene.ITextureMgr, captureTextureName);

            if (tex == null)
            {
                return;
            }

            _RlytConverter.Convert_CaptureTexture_(tex.ICaptureTexture, _fileCapture);

            CaptureTexture captureTexture = tex.ICaptureTexture as CaptureTexture;
            if (_ownerPaneName.Length > 0)
            {
                captureTexture.OwnerPane = subScene.FindPaneByName(_ownerPaneName);
            }
            else
            {
                captureTexture.OwnerPane = null;
            }
        }
    }
    #endregion

    #region AnimFrameSection

    /// <summary>
    ///
    /// </summary>
    internal class FontManagerMemento : ICommandMemento<FontManager>
    {
        /// <summary>
        ///
        /// </summary>
        class Data
        {
            public readonly string name;
            public readonly string filePath;

            public Data( ILEFont font)
            {
                this.name = font.FontName;
                this.filePath = font.FontPath;
            }

            public string Name { get { return this.name; }}
            public string FilePath { get { return this.filePath; }}
        }

        readonly List<Data> snapShots;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FontManagerMemento( FontManager fontManager)
        {
            this.snapShots = new List<Data>();


            foreach( ILEFont font in fontManager.ILEFontSet )
            {
                this.snapShots.Add( new Data( font));
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo( FontManager fontManager)
        {
            //
            {
                ILEFont[] fonts = fontManager.ILEFontSet.ToArray();
                foreach( ILEFont font in fonts )
                {
                    bool found = this.snapShots.Exists
                        ( delegate( Data data )
                                {
                                    return data.Name == font.FontName;
                                } );

                    if( found == false )
                    {
                        fontManager.RemoveFont( font as LEFont);
                    }
                }
            }

            //
            {
                ILEFont font = null;
                foreach( Data data in this.snapShots )
                {
                    font = fontManager.FindFontByName( data.Name);
                    if( font != null )
                    {
                        fontManager.UpdateFont( font as LEFont, data.FilePath);
                    }
                    else
                    {
                        font = fontManager.CreateILEFont(data.Name, data.FilePath, null);
                    }
                }
            }
        }
    }

    #endregion

    #region PartsMement

    /// <summary>
    ///
    /// </summary>
    internal class ParamaterPaneMemento : ICommandMemento<IParamaterPane>
    {
        readonly ControlSettings owner;
        readonly string paneName;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ParamaterPaneMemento(IParamaterPane src)
        {
            this.owner = (src as ParamaterPane).Owner;
            this.paneName = src.PaneName;
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(IParamaterPane target)
        {
            var targetPane = this.owner.Panes.FirstOrDefault(
                (pane) => pane.Name == target.Name);
            if (targetPane != null)
            {
                targetPane.PaneName = this.paneName;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class ParamaterAnimationMemento : ICommandMemento<IParamaterAnimaiton>
    {
        readonly ControlSettings owner;
        readonly string tagName;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ParamaterAnimationMemento(IParamaterAnimaiton src)
        {
            this.owner = (src as ParamaterAnimaiton).Owner;
            this.tagName = src.TagName;
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(IParamaterAnimaiton target)
        {
            var targetAnim = this.owner.Animations.FirstOrDefault(
                (anim) => anim.Name == target.Name);
            if (targetAnim != null)
            {
                targetAnim.TagName = this.tagName;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class PartsSettingsMemento : ICommandMemento<PartsSettings>
    {
        readonly PartsSettings partsSettings;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PartsSettingsMemento(PartsSettings src)
        {
            this.partsSettings = new PartsSettings(src.Owner);
            if (!src.IsInitialState)
            {
                this.partsSettings.Initialize(
                    src.Name, src.DescriptionName, src.Description, src.OriginalSize);

                var propSources = src.OverridePorperties.Select(
                    (prop) => new PartsPropaertySettingsSource(
                        prop.TargetName,prop.PaneKind, prop.Description));
                this.partsSettings.SetOverrideProperties(propSources);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(PartsSettings target)
        {
            if (!this.partsSettings.IsInitialState)
            {
                target.Initialize(
                    this.partsSettings.Name,
                    this.partsSettings.DescriptionName,
                    this.partsSettings.Description,
                    this.partsSettings.OriginalSize);

                var propSources = this.partsSettings.OverridePorperties.Select(
                    (prop) => new PartsPropaertySettingsSource(
                        prop.TargetName, prop.PaneKind, prop.Description));
                target.SetOverrideProperties(propSources);
            }
            else
            {
                target.Reset();
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class ControlSettingsSetMemento : ICommandMemento<ControlSettingsSet>
    {
        readonly ControlSettingsSet _settingsSet;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ControlSettingsSetMemento(ControlSettingsSet src)
        {
            this._settingsSet = new ControlSettingsSet(src.Owner);
            this._settingsSet.Clear();

            foreach(var srcCtrl in  src.Items)
            {
                this._settingsSet.CloneAdd(srcCtrl);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(ControlSettingsSet target)
        {
            IControlSettingsSet src = this._settingsSet;
            target.Clear();

            foreach(var srcCtrl in  src.Items)
            {
                target.CloneAdd(srcCtrl);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class ControlSettingsMemento : ICommandMemento<ControlSettings>
    {
        readonly ControlSettings _settings;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ControlSettingsMemento(ControlSettings src)
        {
            this._settings = new ControlSettings(src.Owner);
            this._settings.InitializeNoEvent(src.Name, src.UIName, src.DescriptionName, src.Description, src.Panes, src.Animations);
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(ControlSettings target)
        {
            IControlSettings src = this._settings;
            target.Initialize(src.Name, src.UIName, src.DescriptionName, src.Description, src.Panes, src.Animations);
        }
    }

    internal class PartsLayoutMemento : ICommandMemento<PartsLayout>
    {
        // 現状の仕様で変化しうる部分だけを保存しています。
        readonly List<IPaneExParamater> partsPropaertyParamaters = new List<IPaneExParamater>();
        readonly List<PartsPropertyUsageOptions> partsPropaertyOptions = new List<PartsPropertyUsageOptions>();
        readonly FVec3 Magnify;
        readonly FVec3 SizeConstraint;
        readonly IPane ownerPane;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PartsLayoutMemento(PartsLayout src)
        {
            foreach(var param in src.PartsPropaerties)
            {
                if(param.PaneKind == PaneKind.Parts)
                {
                    partsPropaertyParamaters.Add(param.Paramater);
                }

                partsPropaertyOptions.Add(param.Option);
            }

            this.Magnify = src.Magnify;
            this.SizeConstraint = src.SizeConstraint;
            this.ownerPane = src.OwnerPane;
        }

        /// <summary>
        ///
        /// </summary>
        public void CopyTo(PartsLayout target)
        {
            int idxPart = 0;
            int idx = 0;

            target = this.ownerPane.IPartsLayout as PartsLayout;

            foreach (PartsPropaerty param in target.PartsPropaerties)
            {
                if (param.PaneKind == PaneKind.Parts)
                {
                    target.OverrideChildePartsPane(param.TargetName, partsPropaertyParamaters[idxPart]);
                    idxPart++;
                }

                target.ModifyOverwriteParamaters(param, partsPropaertyOptions[idx]);
                idx++;
            }

            target.Magnify = this.Magnify;
            target.SizeConstraint = this.SizeConstraint;
        }
    }

    #endregion PartsMement

    #region BackGround
    /// <summary>
    /// IPaneのユーザ情報のクローンを生成します。
    ///
    /// Undoコマンドが利用する、インスタンスのスナップショット
    /// として利用します。
    ///
    /// </summary>
    internal class BackGroundMement : ICommandMemento<AppData.BackGround>
    {
      /// <summary>
      /// ファイル保存形式のユーザデータ
      /// </summary>
      readonly AppData.BackGround _backGround;

      /// <summary>
      /// コンストラクタ
      /// </summary>
      public BackGroundMement(AppData.BackGround srcBackGround)
      {
        _backGround = new BackGround();
        _backGround.CopyFrom(srcBackGround);
      }

      #region ICommandMemento<BackGround> メンバ

      public void CopyTo(BackGround dstInstance)
      {
        // データを復元
        dstInstance.CopyFrom(_backGround);
      }

      #endregion
    }
    #endregion BackGround

    #region DocumentHeader
    internal class DocumentHeaderMement : ICommandMemento<IDocumentHeader>
    {
        string comment;
        public DocumentHeaderMement(IDocumentHeader header)
        {
            comment = header.Comment;
        }

        public void CopyTo(IDocumentHeader header)
        {
            header.Comment = comment;
        }
    }
    #endregion

}
