﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing.Drawing2D;
using System.Drawing;
using System.Linq;


namespace LECore.Structures
{
    using Save_Load;
    using SerializableFmt = LECore.Structures.SerializableObject.Lyt;
    using SerializableAnmFmt = LECore.Structures.SerializableObject.Lan;
    using LECore.Structures.Core;
    using Manipulator;

    /// <summary>
    /// Pane配列を複製します。
    ///
    /// </summary>
    public class PaneSetDupulicator
    {
        /// <summary>
        /// オブジェクトに設定するオプション情報です。
        /// </summary>
        [System.FlagsAttribute]
        public enum Option
        {
            None = 0x00000000, // オプション指定なし
            CopyAnimation = 0x00000001, // アニメーションをコピーする
            CopyHierarchy = 0x00000002, // 階層情報をコピーする
            ConvertTransToWorld = 0x00000004, // 親の失われたペインの平行移動成分をワールド系に変換する。
            CopyAll = CopyAnimation | CopyHierarchy // 全て
        }

        /// <summary>
        /// ペインセット複製の際に作成される、中間データです。
        /// </summary>
        public class IntermediateData
        {
            public readonly SerializableFmt.Pane[] _PaneData;
            public readonly AnimContentsArraySet _AnimData;
            public readonly SerializableFmt.PaneHierarchy _HierData;
            public readonly Hashtable _PaneWorldPosTbl = new Hashtable();


            public IntermediateData(
                SerializableFmt.Pane[] paneData,
                AnimContentsArraySet animData,
                SerializableFmt.PaneHierarchy hierData,
                Hashtable paneWorldPosTbl )
            {
                _PaneData = paneData;
                _AnimData = animData;
                _HierData = hierData;
                _PaneWorldPosTbl = paneWorldPosTbl;
            }
        }

        /// <summary>
        /// 内部で型変換に使用するクラス。
        /// (ファイル出力の際に使用するシリアライズクラスを流用しています。)
        /// </summary>
        ///
        static readonly RlytConverter _serializableRlytCvtr =
            new RlytConverter( string.Empty, LayoutEditorCore.MsgReporter );


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneSetDupulicator()
        {
            //
            // TODO: コンストラクタ ロジックをここに追加してください。
            //
        }


        #region ToIntermediate
        /// <summary>
        /// ペイン列をセーブデータに変換します。
        /// </summary>
        static SerializableFmt.Pane[]
            ConvertToSavePaneSet_( IPane[] srcPanes )
        {
            SerializableFmt.Pane[] dstPanes = new SerializableFmt.Pane[srcPanes.Length];
            for( int i = 0 ; i < srcPanes.Length ; i++ )
            {
                dstPanes[i] =
                    _serializableRlytCvtr.ConvertPaneToSerializable( srcPanes[i] ) as SerializableFmt.Pane;
            }
            return dstPanes;
        }



        /// <summary>
        /// ペイン列のもつアニメーションをセーブデータに変換します。
        /// </summary>
        static AnimContentsArraySet
            ConvertToSaveAnimContents_( IPane[] srcPanes )
        {
            RlanConverter anmCvtr = new RlanConverter( LayoutEditorCore.MsgReporter );

            AnimContentsArraySet animArraySet = new AnimContentsArraySet();
            for( int i = 0 ; i < srcPanes.Length ; i++ )
            {
                animArraySet.SetAnimation(srcPanes[i]);
            }

            return animArraySet;
        }

        /// <summary>
        /// ペイン列の持つ階層情報をセーブデータに変換します。
        /// </summary>
        static SerializableFmt.PaneHierarchy
            ConvertToSaveHierarchy_( IPane[] srcPanes )
        {
            SerializableFmt.PaneHierarchy paneHierarchy = null;
            if( srcPanes.Length > 0 )
            {
                // ルートペインを取得して、シーン全体の階層情報を出力します。
                IPane rootPane = srcPanes[0].OwnerSubScene.RootIPane;
                paneHierarchy = _serializableRlytCvtr.ConvertPaneHierarchyToSerializable( rootPane );
            }
            return paneHierarchy;
        }

        /// <summary>
        /// 親ペイン情報を変換します。
        /// </summary>
        /// <param name="srcPaneSet"></param>
        /// <returns></returns>
        static Hashtable ConvertToSaveParentPaneSet_( IPane[] srcPaneSet )
        {
            Hashtable parentPaneTable = new Hashtable();
            foreach( Pane pane in srcPaneSet )
            {
                IPane parentPane = pane.Parent as IPane;
                if( Array.IndexOf( srcPaneSet, parentPane ) == -1 )
                {
                    parentPaneTable.Add( pane.PaneName, new FVec2( pane.XInWorld, pane.YInWorld ) );
                }
            }

            return parentPaneTable;
        }

        #endregion ToIntermediate

        /// <summary>
        /// ペインセットを中間データ形式に複製します。
        /// </summary>
        static public IntermediateData ToIntermediate( IPane[] srcPaneSet, Option option )
        {
            Debug.Assert( srcPaneSet != null );


            // 0番要素には、SerializableFmt.PaneModify[]
            // 1番要素には、アニメーションデータ
            // 2番要素には、階層データ
            // を格納します。
            SerializableFmt.Pane[] paneData = null;
            AnimContentsArraySet animData = null;
            SerializableFmt.PaneHierarchy hierData = null;
            Hashtable paneWorldPosTbl = null;

            // -------------- ペイン情報のコピー
            paneData = ConvertToSavePaneSet_( srcPaneSet );

            // -------------- アニメーションのコピー
            if( ( option & Option.CopyAnimation ) != 0 )
            {
                animData = ConvertToSaveAnimContents_( srcPaneSet );
            }

            // -------------- 階層のコピー
            if( ( option & Option.CopyHierarchy ) != 0 )
            {
                hierData = ConvertToSaveHierarchy_( srcPaneSet );
            }

            // -------------- ワールド位置のコピー
            if( ( option & Option.ConvertTransToWorld ) != 0 )
            {
                paneWorldPosTbl = ConvertToSaveParentPaneSet_( srcPaneSet );
            }

            return new IntermediateData( paneData, animData, hierData, paneWorldPosTbl );
        }

        #region ToPaneSet
        /// <summary>
        /// 複製ペインセットにアニメーションをロードします。
        /// </summary>
        static void SetAnimationDataToPaneSet_(
            IPane[] dstPanes,
            AnimContentsArraySet contentsArraySet )
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            if (!subScene.IsAnimEditSeparateMode())
            {
                SetAnimationDataToPaneSetSingle_(dstPanes, contentsArraySet);
            }
            else
            {
                SetAnimationDataToPaneSetMulti_(dstPanes, contentsArraySet);
            }
        }

        static void SetAnimationDataToPaneSetSingle_(
            IPane[] dstPanes,
            AnimContentsArraySet contentsArraySet)
        {
            RlanConverter rlanCvtr = new RlanConverter(null);

            // 全ての種類のアニメーションの...
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                SerializableAnmFmt.AnimContent[] animContentSet = contentsArraySet.GetAnimContentArray(kind, AnimCurvesParam.__NoSelected__);

                // 全てのアニメーションコンテンツについて...
                foreach (RlanHelper.AnimLoadFunction loader in RlanHelper.GetAnimLoadFunctionSet(kind))
                {
                    RlanHelper.DoLoad(loader, dstPanes, animContentSet, AnimCurvesParam.__NoSelected__);
                }
            }
        }

        static void SetAnimationDataToPaneSetMulti_(
            IPane[] dstPanes,
            AnimContentsArraySet contentsArraySet)
        {
            RlanConverter rlanCvtr = new RlanConverter(null);

            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.BindTarget(subScene);

            // 現在選択中の区間名を保持
            var targetSection = (subScene.IAnimFrameSectionSet as AnimFrameSectionSet).TargetIAnimFrameSection;
            string targetSectionName = targetSection?.Name ?? String.Empty;

            // 区間タグごとにAnmCurveを生成
            AddAnimTag_(dstPanes);

            // カレントタグを切り替えながらすべてのカーブを走査する
            foreach (IAnimFrameSection animFrameSection in subScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
            {
                // 全ての種類のアニメーションの...
                foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                {
                    SerializableAnmFmt.AnimContent[] animContentSet = contentsArraySet.GetAnimContentArray(kind, animFrameSection.Name);

                    // 全てのアニメーションコンテンツについて...
                    foreach (RlanHelper.AnimLoadFunction loader in RlanHelper.GetAnimLoadFunctionSet(kind))
                    {
                        RlanHelper.DoLoad(loader, dstPanes, animContentSet, animFrameSection.Name);
                    }
                }
            }

            // テクスチャパターンを読み込み
            {
                SerializableAnmFmt.AnimContent[] animContentSet = contentsArraySet.GetAnimContentArray(ParamaterKind.Animation_TexturePattern, AnimCurvesParam.__NoSelected__);

                // 読み込み関数を実行
                foreach (RlanHelper.AnimLoadFunction loader in RlanHelper.GetAnimLoadFunctionSet(ParamaterKind.Animation_TexturePattern))
                {
                    RlanHelper.DoLoad(loader, dstPanes, animContentSet, AnimCurvesParam.__NoSelected__);
                }
            }

            // 参照設定を行ないます
            SubSceneHelper.MakeReferenceCurves(subScene);
        }

        /// <summary>
        /// 複製ペインセットに区間タグを追加します。
        /// </summary>
        private static void AddAnimTag_(IPane[] panes)
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;

            foreach (IPane pane in panes)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    foreach (IAnimFrameSection animFrameSection in subScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                    {
                        curveAttr.AddAnmTag(animFrameSection.Name);
                    }
                }
            }
        }

        #region 階層関連 ( SetHierarchyDataToPaneSet_ )
        /// <summary>
        /// 親をたどって、すでに階層が設定済みでないか確認します。
        /// </summary>
        static bool CheckHierarchySetFinished_( IPane pane, IPane[] dstPaneSet )
        {
            IHierarchyNode parentPane = pane.Parent;
            while( parentPane != null )
            {
                if( Array.IndexOf( dstPaneSet, parentPane ) != -1 )
                {
                    return true;
                }
                parentPane = parentPane.Parent;
            }
            return false;
        }

        /// <summary>
        /// ペイン名で階層情報を検索します。
        /// </summary>
        static SerializableFmt.PaneTree FindHierarchyDataSrc_(
            SerializableFmt.PaneTree srcPaneTree,
            string paneName )
        {
            if( srcPaneTree.name == paneName )
            {
                return srcPaneTree;
            }
            else
            {
                foreach( SerializableFmt.PaneTree subTree in srcPaneTree.paneTree )
                {
                    SerializableFmt.PaneTree result =
                        FindHierarchyDataSrc_( subTree, paneName );
                    if( result != null )
                    {
                        return result;
                    }
                }
            }
            return null;
        }

        // TOTO:消す
        /// <summary>
        /// ペイン名でペインを検索。
        /// </summary>
        static IPane FindPaneByName_( IPane[] dstPaneSet, string name )
        {
            foreach( IPane pane in dstPaneSet )
            {
                if( pane.PaneName == name )
                {
                    return pane;
                }
            }
            return null;
        }

        /// <summary>
        /// 階層を設定します。
        /// </summary>
        static void SetHierData_(
            IPane dstPane,
            IPane[] dstPaneSet,
            SerializableFmt.PaneTree srcHierData )
        {
            // 全ての子供ペインについて、
            foreach( SerializableFmt.PaneTree srcChildData in srcHierData.paneTree )
            {
                // 子供ペインを検索します。
                IPane dstChildPane = FindPaneByName_( dstPaneSet, srcChildData.name );

                // 発見できれば階層を設定し、再帰的に処理を行います。
                if( dstChildPane != null )
                {
                    dstPane.AddChildNodeByLoaclCoordinate( dstChildPane );
                    SetHierData_( dstChildPane, dstPaneSet, srcChildData );
                }
            }
        }

        /// <summary>
        /// 階層データをペインセットに設定します。
        /// </summary>
        static void SetHierarchyDataToPaneSet_(
            IPane[] dstPaneSet,
            SerializableFmt.PaneHierarchy srcPaneHierarchy )
        {
            foreach( IPane pane in dstPaneSet )
            {
                // すでに設定済みでなければ、
                if( !CheckHierarchySetFinished_( pane, dstPaneSet ) )
                {
                    // 対応するデータソースを 検索する。
                    SerializableFmt.PaneTree srcHierData =
                        FindHierarchyDataSrc_( srcPaneHierarchy.paneTree, pane.PaneName );

                    // 発見された データソースをペインに設定する。
                    SetHierData_( pane, dstPaneSet, srcHierData );
                }
            }
        }
        #endregion 階層関連

        /// <summary>
        /// コピーの結果親を失ったペインの位置をワールド系に調整します。
        /// </summary>
        static void AdjustPositionToWorldSpace_( IPane[] dstPanes, Hashtable paneWorldPosTbl )
        {
            foreach( Pane pane in dstPanes )
            {
                if( paneWorldPosTbl.ContainsKey( pane.PaneName ) )
                {
                    FVec2 worldPos = (FVec2)paneWorldPosTbl[pane.PaneName];
                    pane.X = worldPos.X;
                    pane.Y = worldPos.Y;
                }
            }
        }

        #endregion ToPaneSet

        /// <summary>
        /// 中間データ形式からペインセットを複製します。
        /// </summary>
        static public IPane[] ToPaneSet( IntermediateData intermediate )
        {
            SerializableFmt.Pane[] srcPanes = intermediate._PaneData;
            AnimContentsArraySet contentsArraySet = intermediate._AnimData;
            SerializableFmt.PaneHierarchy paneHierarchy = intermediate._HierData;
            Hashtable paneWorldPosTbl = intermediate._PaneWorldPosTbl;


            Debug.Assert( srcPanes != null );


            // ---------------------------------------------------------
            // 保存されている情報から、新しい、インスタンスの生成を行います。
            // ---------------------------------------------------------
            // ペインの情報を変換します。
            IPane[] dstPanes = new IPane[srcPanes.Length];
            for( int i = 0 ; i < srcPanes.Length ; i++ )
            {
                dstPanes[i] = _serializableRlytCvtr.ConvertPaneToInternal( srcPanes[i] ) as Pane;
            }

            // アニメーション情報をペイン列に反映します
            // Paseer の LoadAnimationToPaneByAnimKind_ を参考に
            if (contentsArraySet != null)
            {
                SetAnimationDataToPaneSet_(dstPanes, contentsArraySet);
            }

            // 階層情報をペイン列に反映します
            if ( paneHierarchy != null )
            {
                SetHierarchyDataToPaneSet_( dstPanes, paneHierarchy );
            }

            // 親を失ったペインの位置を調整します。
            if( paneWorldPosTbl != null )
            {
                AdjustPositionToWorldSpace_( dstPanes, paneWorldPosTbl );
            }

            return dstPanes;
        }

        /// <summary>
        /// シリアライズ可能なオブジェクトへ変換します。
        /// </summary>
        static IPane[] ClonePanes_(
            IPane[] srcPanes,
            Option option )
        {
            // 内部データに変換します。
            IntermediateData interData = ToIntermediate( srcPanes, option );

            // 内部データからアプリケーションデータを複製します。
            IPane[] dstPanes = ToPaneSet( interData );

            // コピーオプション
            PaneParamaterCopyOption copyOption = new PaneParamaterCopyOption();
            copyOption.CopyBaseValue = true;

            // baseValueを更新します
            foreach (IPane pane in srcPanes)
            {
                var targetPanes = dstPanes.Where(dstPane => dstPane.PaneName == pane.PaneName);

                // baseValueを更新します
                foreach (IPane targetPane in targetPanes)
                {
                    PaneParamaterPaster.PasteParamaters(copyOption, targetPane, pane);
                }
            }

            // 複製したペインを現在時間で再評価します
            foreach (IPane pane in dstPanes)
            {
                // dstPanesはOwnerSubSceneがnullなのでペイン単位で評価します
                PaneAnimationHelper.EvaluateAnimationForce(pane, GlobalTime.Inst.Time);
            }

            return dstPanes;
        }

        /// <summary>
        /// ペインを複製します。
        /// </summary>
        static public IPane[] ClonePaneSet( IPane[] srcPane )
        {
            return ClonePaneSet( srcPane, Option.CopyAll );
        }

        /// <summary>
        /// ペインを複製します。
        /// </summary>
        static public IPane[] ClonePaneSet(
            IPane[] srcPane,
            Option option )
        {
            return ClonePanes_( srcPane, option ) as IPane[];
        }
    }
}
