﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Diagnostics;
using System.Linq;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using static RlanHelper;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class RltpConverter
    {
        #region ------------- Rltp(読み込み)

        /// <summary>
        /// テクスチャサブセットを生成します。
        /// </summary>
        /// <returns></returns>
        static string[] BuildTextureNameSet_( FileFmt.RefRes[] refRes )
        {
            int numTex = refRes != null ? refRes.Length : 0;
            string[] result = new string[numTex];
            for( int i = 0 ; i < result.Length ; i++ )
            {
                result[i] = refRes[i].name;
            }

            return result;
        }



        /// <summary>
        /// シリアライズデータ => アプリケーションデータ
        /// </summary>
        static public void LoadRltpToMaterial( object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            // アニメーション対象を探します。
            // 失敗した時点で中断します。
            Pane dstPane = dstObject as Pane;
            if( dstPane == null )
            {
                return;
            }

            Material dstMaterial = dstPane.FindMaterialByName( srcAnimContent.name );
            if( dstMaterial == null )
            {
                return;
            }

            foreach( FileFmt.AnimTarget srcAnimTexMap in srcAnimContent.Items )
            {
                AppData.MaterialTexMap dstMatTex = dstMaterial.GetMaterialTexMapByIndex( srcAnimTexMap.id );
                if( dstMatTex == null )
                {
                    continue;
                }

                TextureChanger texChanger = dstMatTex.TextureChanger;
                Debug.Assert( texChanger != null );

                // テクスチャパターンアニメーションを初期化します
                // テクスチャリストの生成(妥当性の確認)
                // TODO:
                // 現在はテクスチャパターンアニメーションにのみ対応しています。
                // パレットパターンアニメーションにも対応する必要があります。


                // テクスチャサブセットへの登録
                string[] texNameSet = BuildTextureNameSet_( srcAnimTexMap.refRes );

                // シーンにテクスチャを読み込みます。
                // 読み込みは別所で完了しています。
                // Scene.Instance.ITextureMgr.RegisterITextureImageSetFromFile( texNameSet );


                // テクスチャパターンサブリストに設定します。
                if (!texChanger.TextureIndexAnmAttr.IsReadOnlyLocked())
                {
                    texChanger.TextureSubSet.AddTextures(texNameSet);
                }

                // バインドされている、テクスチャパターンアニメーションについて...
                // キーフレームを生成して、設定していきます
                // TODO:キャスト：APIを整理する、ローダーにはAnmCurveを見せてもよい。
                RlanHelper.SetHermiteKeyFrame_( texChanger.TextureIndexAnmAttr, srcAnimTexMap, tag);
            }
        }

        #endregion ------------- Rltp

        #region ---------- Rltp(保存)

        #region ----- private

        /// <summary>
        /// TextureChanger => AnimTexMap
        /// データが不正な場合は null を返します。
        /// </summary>
        public static IEnumerable<FileFmt.AnimTexPatternTarget>
            Convert_AppDataToAnimTexMap_( AppData.ITextureChanger texChanger, int slotIndex, string tag )
        {
            var items = RlanHelper.ExtractAnimTargetIfValid<AnimTexPatternTarget>(
                texChanger.TextureIndexIAnmAttr,
                ParamaterKind.Animation_TexturePattern,
                AnimTargetType.Image,
                Convert.ToByte(slotIndex),
                tag,
                tag == AnimCurvesParam.__NoSelected__).ToArray();

            if (tag == AnimCurvesParam.__NoSelected__)
            {
                Debug.Assert(items.Length == 1);
                FileFmt.AnimTexPatternTarget animTexMap = items[0];

                //
                // テクスチャサブセットの名前テーブルを作成します。
                //
                string[] texSubSet = texChanger.ITextureSubSet.TextureSubSetNameSet;
                animTexMap.refRes = new RefRes[texSubSet.Length];
                for (int i = 0; i < animTexMap.refRes.Length; i++)
                {
                    animTexMap.refRes[i] = new RefRes();
                    animTexMap.refRes[i].name = texSubSet[i];
                }

                if (animTexMap.refRes.Length == 0 && animTexMap.key.Length == 0)
                {
                    return Enumerable.Empty<AnimTexPatternTarget>();
                }
            }

            return items;
        }

        #endregion ----- private

        public static FileFmt.AnimContent[] SaveTexturePatternAnim(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<IMaterial, AnimTexPatternTarget>(
                tag,
                srcPane,
                x => x.UsedMaterials.Select(y => new AnimContentSource<IMaterial>(y.MaterialName, y)),
                (t, x) => x.IMaterialTexMapSet.SelectMany((y, i) => Convert_AppDataToAnimTexMap_(y.ITextureChanger, i, t))
                ).ToArray();
        }

        #endregion ---------- Rltp
    }
}
