﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace LECore.Save_Load.Rlan
{
    using Structures;
    using Structures.SerializableObject.Lan;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class BakedInfinityKeyBuilder
    {
        /// <summary>
        /// Infinity領域のキーフレーム焼付け処理を行います。
        ///
        /// キーフレームが時間順に格納されていることを前提としています。
        /// </summary>
        public static void BuildBakedKey(FileFmt.AnimContent[] animContentSet, int startFrame, int endFrame)
        {
            foreach (FileFmt.AnimContent animContent in animContentSet)
            {
                foreach (FileFmt.AnimTarget animTraget in animContent.Items)
                {
                    BakedAnimTargetBuilder bakedAnimTargetBuilder = new BakedAnimTargetBuilder(animTraget);

                    // 修正後のデータに置き換えます。
                    animTraget.key = bakedAnimTargetBuilder.BakeAnimTarget_(startFrame, endFrame);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        struct BakedAnimTargetBuilder
        {
            readonly FileFmt.AnimTarget  _SrcAnimTraget;
            readonly List<Hermite>        _DstKeySet;
            readonly Hermite[]            _SrcKeySet;
            readonly int                   _FirstKeyTime;
            readonly int                   _LastKeyTime;
            readonly int                   _KeyedAreaLength;

           #region 焼付け処理(暫定)


            /// <summary>
            ///
            /// </summary>
            void SlideKeyAtSameTime_( List<Hermite> newKeySet, Hermite newKey )
            {
                List<Hermite> sameKeySet = newKeySet.FindAll( delegate( Hermite key )
                {
                    if( key.frame == newKey.frame )
                    {
                        return true;
                    }
                    else
                    {
                        return false;
                    }
                } );

                if( sameKeySet != null && sameKeySet.Count != 0 )
                {
                    foreach( Hermite key in sameKeySet )
                    {
                        key.frame -= 0.001f;
                    }
                }
            }

            /// <summary>
            ///
            /// </summary>
            Hermite MakeNewKey_( int offsetFrame, Hermite src )
            {
                float time = (int)( src.frame + offsetFrame);
                FileFmt.AnimTargetType asdf = _SrcAnimTraget.target;

                Hermite newKey = CloneHelmiteKey_(src);
                newKey.frame = time;
                newKey.value = src.value;
                newKey.slope = src.slope;
                newKey.slopeType = src.slopeType;

                return newKey;
            }

            Hermite CloneHelmiteKey_(Hermite src)
            {
                if (src is StepU16)
                {
                    return new StepU16();
                }
                else if (src is StepBool)
                {
                    return new StepBool();
                }
                else if (src is HermiteU8)
                {
                    return new HermiteU8();
                }
                else if (src is Hermite)
                {
                    return new Hermite();
                }
                else
                {
                    Ensure.Operation.True(false, "Unknown key type.");
                    return new Hermite();
                }
            }

            /// <summary>
            /// キーフレーム情報をコピーします。
            /// </summary>
            void BakeOneKeySetLoop_( int offsetFrame )
            {
                Hermite newFirstKey = MakeNewKey_( offsetFrame, _SrcKeySet[0] );

                // 先頭と末尾は同一時間にキーが重複します。
                // 同一時間に２つ以上のキーは不正であるため、末尾のキーをずらします。
                SlideKeyAtSameTime_( _DstKeySet, newFirstKey );

                foreach( Hermite srcHermite in _SrcKeySet )
                {
                    Hermite newKey = MakeNewKey_( offsetFrame, srcHermite );
                    _DstKeySet.Add( newKey );
                }
            }

            /// <summary>
            /// 焼付けを行う必要があるか？
            /// </summary>
            bool ShouldInfinityAreaKeyBeBaked_( InfinityType type )
            {
                return ( type == InfinityType.Cycle && _KeyedAreaLength != 0 );
            }

            /// <summary>
            /// 焼き付けます
            /// </summary>
            void Bake_( int startFrame, int endFrame, int step )
            {
                for( int offset = startFrame ; offset < endFrame ; offset += step )
                {
                    BakeOneKeySetLoop_( offset );
                }
            }


            /// <summary>
            /// コンストラクタ
            /// </summary>
            public BakedAnimTargetBuilder(FileFmt.AnimTarget animTragetArg)
            {

                _DstKeySet = new List<Hermite>();
                _SrcAnimTraget = animTragetArg;
                _SrcKeySet = _SrcAnimTraget.key;

                if( _SrcKeySet.Length > 0 )
                {
                    _FirstKeyTime = (int)_SrcKeySet[0].frame;
                    _LastKeyTime = (int)_SrcKeySet[_SrcKeySet.Length - 1].frame;
                    _KeyedAreaLength = _LastKeyTime - _FirstKeyTime;
                }
                else
                {
                    _FirstKeyTime = 0;
                    _LastKeyTime = 0;
                    _KeyedAreaLength = 0;
                }
            }

            /// <summary>
            /// アニメーションターゲットを焼き付けます
            /// </summary>
            public Hermite[] BakeAnimTarget_( int startFrame, int endFrame )
            {
                // キーが無ければ処理しません。
                if( _SrcKeySet.Length > 0 )
                {
                    //------------------------------------------------
                    // preInfinity 領域について
                    if( ShouldInfinityAreaKeyBeBaked_( _SrcAnimTraget.preInfinityType ) )
                    {
                        // キーフレーム区間長単位で、
                        // startFrame以前のもっとも大きいフレームを計算します。
                        int preInfinityAreaLength = _FirstKeyTime - startFrame;
                        int copyStartFrame = - ( ( preInfinityAreaLength / _KeyedAreaLength ) + 1 ) * _KeyedAreaLength;

                        Bake_( copyStartFrame, 0, _KeyedAreaLength );
                    }

                    //------------------------------------------------
                    // オリジナルキーをコピー
                    BakeOneKeySetLoop_( 0 );

                    //------------------------------------------------
                    // PostInfinity 領域のキーフレームを書き出します。
                    if( ShouldInfinityAreaKeyBeBaked_( _SrcAnimTraget.postInfinityType ) )
                    {
                        Bake_( _KeyedAreaLength, endFrame - _FirstKeyTime, _KeyedAreaLength );
                    }
                }
                return _DstKeySet.ToArray();
            }


        }


        #endregion 焼付け処理(暫定)


    }
}
