﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;
using System.Xml.Serialization;
using System.Text;
using System.Collections.Generic;
using System.Linq;


namespace LECore.Save_Load
{
    using Structures;
    using Structures.SerializableObject.Lan;
    using AppData = Structures;

    /// <summary>
    /// Rlmcをインポートするクラスです。
    /// </summary>
    internal abstract class BaseSingleImporter : IImporter
    {
        LEMsgReporter _msgReporter = null;
        IImporterOption _options = IImporterOption.None;
        /// <summary>
        /// 情報出力クラスを設定します。
        /// </summary>
        public LEMsgReporter MsgReporter
        {
            set { _msgReporter = value; }
        }

        public IImporterOption ImporterOption
        {
            get { return _options; }
        }

        /// <summary>
        /// 出力ファイルの拡張子を取得します(短縮)。
        /// </summary>
        public abstract string[] GetFileExtensionShort();


        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public abstract string[] GetFileExtensionLong();

        /// <summary>
        /// FileIO イベントを抑止する
        /// </summary>
        public virtual bool StopFileIOEvent { get { return false; } }

        /// <summary>
        /// ロード前のシーンのクリーンアップ
        /// </summary>
        public virtual void BeforeLoadingCleanup( ISubScene targetScene )
        {
            // 何もしません
        }

        /// <summary>
        /// オプションを設定します。
        /// </summary
        public virtual void SetOptionFlag( IImporterOption option )
        {
            _options = option;
        }

        /// <summary>
        /// ファイルパスを調整する。
        /// </summary>
        public string TrimFilePath(string originalPath)
        {
            return Path.ChangeExtension(originalPath, GetFileExtensionShort().First());
        }

        /// <summary>
        ///
        /// </summary>
        protected void ReportMessage_( string msg )
        {
            if( _msgReporter != null )
            {
                if( (_options & IImporterOption.DoNotReportFailure) == 0 )
                {
                    _msgReporter.OutLog( "Infomation : ", msg );
                }
            }
        }

        #region BaseSingleImporter abstract メンバ
        /// <summary>
        /// シリアル化可能オブジェクトの型
        /// </summary>
        /// <returns></returns>
        public abstract Type   GetDeselializableObjType();

        /// <summary>
        /// シリアル化可能オブジェクトの型(追加の型情報です。)
        /// </summary>
        /// <returns></returns>
        public virtual Type[] GetExtraDeselializableObjTypes()
        {
            return new Type[0];
        }

        /// <summary>
        /// シリアル化可能オブジェクトに変換します。
        /// </summary>
        public abstract void ImportFromDeserializable_( ISubScene targetScene, string inPath, object srcDeserializable, LEMsgReporter msgReporter );
        #endregion BaseSingleImporter abstract メンバ


        /// <summary>
        /// ファイルを読み込みます。
        /// </summary>
        public bool DoLoad( string inPath, ISubScene targetScene )
        {
            bool bLoadSuccess = false;
            // XML シリアライズをおこない、ファイル書き出しをします。
            FileStream reader		        = null;
            XmlSerializer		 xmlSerializer	    = null;
            try
            {
                // ApplicationSettings 型の
                // XmlSerializer を作成します。
                xmlSerializer = XmlSerializerCache.GetXmlSerializer(
                    GetDeselializableObjType(),
                    GetExtraDeselializableObjTypes());

                if( File.Exists( inPath ) )
                {
                    reader = new FileStream(inPath, FileMode.Open, FileAccess.Read);

                    // ApplicationSettings クラスのこのインスタンスをデシリアル化します。
                    object deserializedAnmData = XmlSerializerCache.Deserialize(xmlSerializer, reader);

                    // XML デシリアライズしたクラスから、ツール内部クラスに変換します。
                    ImportFromDeserializable_( targetScene, inPath, deserializedAnmData, _msgReporter );

                    // 読み込み成功
                    bLoadSuccess = true;
                    _msgReporter.OutLog(
                        LECoreStringResMgr.Get( "LECORE_RLXXLOAD_LOAD_END", Path.GetFileName( inPath ) ), string.Empty );
                }
                else
                {
                    // ファイルが読み込めなかった...
                    ReportMessage_( string.Format( "Can't Load file [{0}].", inPath ) );
                }
            }
            catch(Exception ex)
            {
                // 各IImporterで例外を処理してもよいが、
                // しなくても ImporterExecutor が最低限の処理をおこなってくれます。
                // ここでは、特になにも例外に対する処理をおこないません。
                throw ex;
            }
            finally
            {
                // FileStream が開いている場合は閉じます。
                if(reader != null)
                {
                    reader.Close();
                }
            }

            // ロード成功ならアクセス日時を更新します。
            if( bLoadSuccess )
            {
                FileAttributes fileattributes = File.GetAttributes( inPath );
                if( ( fileattributes & FileAttributes.ReadOnly ) == 0 )
                {
                    try
                    {
                        File.SetLastAccessTime(inPath, DateTime.Now);
                    }
                    catch (Exception e)
                    {
                        // 例えば、読み込み用の FileStream を Dispose し忘れると、例外になることがある。
                        Debug.Assert(false, "filterattributes: " + fileattributes.ToString() + "\n" + e.ToString());

                        // 何もしない
                    }
                }
            }

            return bLoadSuccess;
        }
    }


}

