﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using LECore.Save_Load;
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Win32;

namespace LECore
{
    public static class LayoutAggregationHelper
    {
        /// <summary>
        /// コピーされたファイル
        /// </summary>
        public class CopiedFiles
        {
            public List<string> Texture { get; set; }
            public List<string> Font { get; set; }
            public List<string> Parts { get; set; }
            public List<string> Controls { get; set; }
            public List<string> UserDataPresets { get; set; }
            public string Project { get; set; }
            public List<string> Combiner { get; set; }

            // 重複する出力を避けるため、出力したファイルを記録しておく。
            public List<string> DstTexture { get; set; }
            public List<string> DstFont { get; set; }
            public List<string> DstParts { get; set; }
            public List<string> DstControls { get; set; }
            public List<string> DstUserDataPresets { get; set; }
            public string DstProject { get; set; }
            public List<string> DstCombiner { get; set; }

            public CopiedFiles()
            {
                Texture = new List<string>();
                Font = new List<string>();
                Parts = new List<string>();
                Controls = new List<string>();
                UserDataPresets = new List<string>();
                Combiner = new List<string>();

                DstTexture = new List<string>();
                DstFont = new List<string>();
                DstParts = new List<string>();
                DstControls = new List<string>();
                DstUserDataPresets = new List<string>();
                DstCombiner = new List<string>();
            }
        }

        public static void CreateDirIfNotExist(string exportPath, Action<string> onErrorReportAction)
        {
            CreateDirIfNotExsit_(exportPath, onErrorReportAction);
        }

        /// <summary>
        /// 読みとり専用でなければ上書きコピーを行います。
        /// </summary>
        static void CopyFileIfValid_(string srcPath, string dstPath)
        {
            if(File.Exists(dstPath))
            {
                FileAttributes attr = File.GetAttributes(dstPath);
                bool isReadOnly = attr.HasFlag(System.IO.FileAttributes.ReadOnly);
                if (!isReadOnly)
                {
                    File.Copy(srcPath, dstPath, true);
                }
            }
            else
            {
                File.Copy(srcPath, dstPath);
            }
        }

        /// <summary>
        /// コンバイナファイルを出力します。
        /// </summary>
        static void AggregateForCombinerUserShader(ISubScene taregetSubScene, string exportCombinerPath, Action<string> onErrorReportAction, ref CopiedFiles copiedFiles)
        {
            foreach (IRevHWMaterial[] revMat in taregetSubScene.IPaneArray.Select(t => PaneHelper.GetRevHWMatFromPane(t)))
            {
                for( int i = 0 ; i < revMat.Length ; i++ )
                {
                    string fileFullPath = revMat[i].ICombinerUserShader.FileName;
                    if (!string.IsNullOrWhiteSpace(fileFullPath))
                    {
                        string fileName = Path.GetFileName(fileFullPath);
                        string exportCombinerFile = Path.Combine(exportCombinerPath, fileName);

                        if (copiedFiles.DstCombiner.Contains(exportCombinerFile))
                        {
                            // 同一ファイル名で、フォルダが違う場合を検出し警告
                            if (!copiedFiles.Combiner.Contains(fileFullPath))
                            {
                                // 重複するファイル名 {0} が複数のフォルダで利用されてい為、正しく出力が行われませんでした。
                                // コンバイナファイルの設定を確認してください。
                                onErrorReportAction(LECoreStringResMgr.Get("ERROR_COMBINERFILE_DUPLICATE", fileName, fileFullPath, copiedFiles.Combiner.FirstOrDefault()));
                            }
                            continue;
                        }

                        // コンバイナファイルが一つでも存在する場合のみフォルダを生成する
                        CreateDirIfNotExsit_(exportCombinerPath, onErrorReportAction);

                        CopyFileIfValid_(fileFullPath, exportCombinerFile);

                        copiedFiles.Combiner.Add(fileFullPath);
                        copiedFiles.DstCombiner.Add(exportCombinerFile);
                    }
                }
            }
        }

        public static CopiedFiles Aggregate(ISubScene taregetSubScene, string exportPath, string exportLytPath, bool makeTexSubDir, Action<string> onErrorReportAction)
        {
            // 部品ルートフォルダが存在しない場合は即終了
            if (!Directory.Exists(Scene.Instance.PartsRootPath))
            {
                // 部品ルートフォルダが見つかりませんでした。
                // 部品パレットウインドウの設定を確認してください。
                onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_PARTSROOT_NOT_FOUND") + "\n" + Scene.Instance.PartsRootPath);
                return new CopiedFiles();
            }

            // サブフォルダを生成する。
            string exportPathTexRoot = Path.Combine(exportPath, "Texture");
            string exportPathTex = makeTexSubDir ? Path.Combine(exportPathTexRoot, Path.GetFileNameWithoutExtension(exportLytPath)) : exportPathTexRoot;
            string exportPathFnt = Path.Combine(exportPath, "Font");
            string exportPathParts = Path.Combine(exportPath, "Parts");
            string exportCombinerPath = Path.Combine(exportPath, "Combiner");

            CreateDirIfNotExsit_(exportPathTex, onErrorReportAction);
            CreateDirIfNotExsit_(exportPathFnt, onErrorReportAction);
            CreateDirIfNotExsit_(exportPathParts, onErrorReportAction);

            CopiedFiles copiedFiles = new CopiedFiles();

            // レイアウトファイルの出力。
            CopyLyt_(taregetSubScene, exportLytPath, exportPathTex, exportPathFnt, exportCombinerPath, copiedFiles, onErrorReportAction);

            // 依存部品レイアウトの出力
            foreach (var partsPane in taregetSubScene.GetPaneSet((pane) => pane.PaneKind == PaneKind.Parts))
            {
                // 参照レイアウト
                foreach (var flytFileName in GetPartsFilePaths_(partsPane))
                {
                    IPartsSubScene partsSubScene = LECore.LayoutEditorCore.Scene.FindPartsSubSceneByFileName(flytFileName);

                    string exportPartsFile = Path.Combine(exportPath, "Parts\\" + flytFileName);
                    string exportPathPartsTex = makeTexSubDir ? Path.Combine(exportPathTexRoot, "Common\\") : exportPathTexRoot;
                    if (copiedFiles.DstParts.Contains(exportPartsFile))
                    {
                        continue;
                    }

                    if (partsSubScene == null || !partsSubScene.IsLoaded)
                    {
                        onErrorReportAction(LECoreStringResMgr.Get("ERROR_PARTS_NOT_LOADED", flytFileName));
                        continue;
                    }

                    AggregateForCombinerUserShader(partsSubScene.SubScene, exportCombinerPath, onErrorReportAction, ref copiedFiles);

                    CreateDirIfNotExsit_(exportPathPartsTex, onErrorReportAction);

                    CopyLyt_(partsSubScene.SubScene, exportPartsFile, exportPathPartsTex, exportPathFnt, exportCombinerPath, copiedFiles, onErrorReportAction);

                    copiedFiles.Parts.Add(flytFileName);
                    copiedFiles.DstParts.Add(exportPartsFile);
                }
            }

            // コントロール定義ファイル
            foreach (string controlFilePath in Directory.GetFiles(Scene.Instance.PartsRootPath, AppConstants.ContorlFileExtPattern, SearchOption.AllDirectories))
            {
                string ctrlFieName = Path.GetFileName(controlFilePath);
                string newCtrlFilePath = Path.Combine(exportPathParts, ctrlFieName);
                if (copiedFiles.DstControls.Contains(newCtrlFilePath))
                {
                    continue;
                }

                try
                {
                    CopyFileIfValid_(controlFilePath, newCtrlFilePath);

                    copiedFiles.Controls.Add(controlFilePath);
                    copiedFiles.DstControls.Add(newCtrlFilePath);
                }
                catch (System.Exception ex)
                {
                    // ファイル [{0}] の処理中にエラーが発生しました。
                    onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", newCtrlFilePath) + "\n" + ex.ToString());
                }
            }

            // プロジェクトファイル
            foreach (string projectFilePath in Directory.GetFiles(Scene.Instance.PartsRootPath, AppConstants.ProjectFileExtPattern, SearchOption.TopDirectoryOnly))
            {
                string projectFileName = Path.GetFileName(projectFilePath);
                string newProjectFilePath = Path.Combine(exportPathParts, projectFileName);

                if (copiedFiles.Project == projectFileName)
                {
                    continue;
                }

                try
                {

                    CopyFileIfValid_(projectFilePath, newProjectFilePath);

                    copiedFiles.Project = projectFilePath;
                    copiedFiles.DstProject = newProjectFilePath;

                    // 読み取り専用でなければ...
                    if(!File.GetAttributes(newProjectFilePath).HasFlag(System.IO.FileAttributes.ReadOnly))
                    {
                        // プロジェクトファイル内の、スケーラブルフォント参照パスを書き換えます。
                        string content = File.ReadAllText(newProjectFilePath);

                        // <FilePath>..\..\..\Resources\font\nintendo_NTLG-DB_002.bfttf のような文字を置換する
                        content = Regex.Replace(content, "<FilePath>(.+?\\.(bfttf|bfotf|ttf|otf))", ReplaceFontPath_);

                        File.WriteAllText(newProjectFilePath, content);
                    }
                }
                catch (System.Exception ex)
                {
                    // ファイル [{0}] の処理中にエラーが発生しました。
                    onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", newProjectFilePath) + "\n" + ex.ToString());
                }
            }

            // 拡張ユーザー情報プリセット定義ファイル
            foreach (string flupFilePath in Directory.GetFiles(Scene.Instance.PartsRootPath, AppConstants.UserDataPresetFileExtPattern, SearchOption.AllDirectories))
            {
                string flupFieName = Path.GetFileName(flupFilePath);
                string newFlupFilePath = Path.Combine(exportPathParts, flupFieName);
                if (copiedFiles.DstUserDataPresets.Contains(newFlupFilePath))
                {
                    continue;
                }

                try
                {
                    CopyFileIfValid_(flupFilePath, newFlupFilePath);

                    copiedFiles.UserDataPresets.Add(flupFilePath);
                    copiedFiles.DstUserDataPresets.Add(newFlupFilePath);
                }
                catch (System.Exception ex)
                {
                    // ファイル [{0}] の処理中にエラーが発生しました。
                    onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", newFlupFilePath) + "\n" + ex.ToString());
                }
            }

            // コンバイナファイルの出力
            AggregateForCombinerUserShader(taregetSubScene, exportCombinerPath, onErrorReportAction, ref copiedFiles);

            return copiedFiles;
        }

        /// <summary>
        /// <FilePath>..\Font\nintendo_NTLG-DB_002.bfttf というパスに置換します。
        /// </summary>
        static string ReplaceFontPath_(Match m)
        {
            return "<FilePath>" + Path.Combine("..\\Font\\", Path.GetFileName(m.Groups[1].Value));
        }

        /// <summary>
        /// レイアウトファイルをコピーします。
        /// </summary>
        static void CopyLyt_(ISubScene subScene, string exportPartsFile, string partsTexDir, string exportPathFnt, string exportCombinerPath, CopiedFiles copiedFiles, Action<string> onErrorReportAction)
        {
            CopyRelatedResources_(subScene, partsTexDir, exportPathFnt, copiedFiles, onErrorReportAction);
            ExportOption exportOption = new ExportOption() {
                UseBaseValue = true,
                Frame = GlobalTime.Inst.Time
            };
            LayoutEditorCore.ExportToFileAll(subScene, exportPartsFile, exportOption);
            ModifyFilePath_(exportPartsFile, partsTexDir, exportPathFnt, exportCombinerPath);
        }

        /// <summary>
        /// 関係部品ファイルを取得する。
        /// </summary>
        static List<string> GetPartsFilePaths_(IPane partsPane)
        {
            List<string> relatedFiles = new List<string>(partsPane.IPartsLayout.EnumlateReferencingLayoutFiles());
            IPartsSubScene partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsPane.IPartsLayout.PartsLayoutName);
            if (partsSubScene != null && partsSubScene.IsDerivativeParts())
            {
                relatedFiles.Add(partsSubScene.BasePartsName);
            }

            return relatedFiles;
        }

        /// <summary>
        /// 関連するテクスチャ、フォントをコピー
        /// </summary>
        static void CopyRelatedResources_(
            ISubScene targetSubScene,
            string texturePath,
            string fontPath,
            CopiedFiles copiedFiles,
            Action<string> onErrorReportAction)
        {
            ISubScene subScene = targetSubScene;
            //--------------------------------------------------
            // テクスチャ
            foreach (ITextureImage texImg in subScene.ITextureMgr.ITextureImageSet)
            {
                // コピーするのはファイルテクスチャのみです。
                if (texImg.SourceType != Structures.LECoreInterface.TextureSourceType.File)
                {
                    return;
                }

                string newFileName = Path.Combine(texturePath, Path.GetFileName(texImg.FilePath));
                if (copiedFiles.DstTexture.Contains(newFileName))
                {
                    continue;
                }

                try
                {
                    // コピーします。(上書きします)
                    if (texImg.FilePath == newFileName)
                    {
                        continue;
                    }
                    CopyFileIfValid_(texImg.FilePath, newFileName);

                    copiedFiles.Texture.Add(texImg.FilePath);
                    copiedFiles.DstTexture.Add(newFileName);
                }
                catch
                {
                    // ファイル [{0}] の処理中にエラーが発生しました。
                    onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", newFileName));
                }
            }

            //--------------------------------------------------
            // フォント
            // ffnt や fcpx の依存するファイルのコピーをし、参照パスを書き換えています。
            foreach (ILEFont font in subScene.ILEFontManager.ILEFontSet)
            {
                if (!string.IsNullOrEmpty(font.FontPath))
                {
                    string newFileName = Path.Combine(fontPath, Path.GetFileName(font.FontPath));
                    if (copiedFiles.DstFont.Contains(newFileName))
                    {
                        continue;
                    }

                    try
                    {
                        if (font.FontPath == newFileName)
                        {
                            continue;
                        }
                        CopyFileIfValid_(font.FontPath, newFileName);

                        // 参照素材のコピーとパス調整
                        if (LEFontHelper.IsFontConvertSettingsFile(newFileName))
                        {
                            // ffnt
                            // <imageFilePath>..\..\..\Applications\FontSimple\Resources\ColorFont.bmp
                            // <orderFilePath>..\..\..\Applications\FontSimple\Resources\ColorFont.xlor</orderFilePath>
                            CopyAndAdjustSubResourceOfFontIfNeeded_(font.FontPath, newFileName, "(imageFilePath|orderFilePath)", "(bmp|tga|tiff|tif|xlor)", copiedFiles);
                        }
                        else if (LEFontHelper.IsComplexFontFile(newFileName))
                        {
                            // fcpx
                            // <path>../../../../../Resources/font/nintendo_NTLG-DB_002.bfttf</path>
                            CopyAndAdjustSubResourceOfFontIfNeeded_(font.FontPath, newFileName, "(path)", "(bfttf|bfotf|otf|ttf)", copiedFiles);
                        }

                        copiedFiles.Font.Add(font.FontPath);
                        copiedFiles.DstFont.Add(newFileName);
                    }
                    catch
                    {
                        // ファイル [{0}] の処理中にエラーが発生しました。
                        onErrorReportAction(LECoreStringResMgr.Get("SYSTEM_ERROR_IN_EXEC_FILE", newFileName));
                    }
                }
            }
        }

        /// <summary>
        /// 依存リソースのコピーとパスの調整を行います。
        /// </summary>
        /// <param name="srcFontPath">コピー元フォント</param>
        /// <param name="dstFontPath">コピー先フォント</param>
        /// <param name="elementPattern">要素のパターン文字列</param>
        /// <param name="extPattern">拡張子のパターン文字列</param>
        /// <param name="copiedFiles">コピー結果記録</param>
        static void CopyAndAdjustSubResourceOfFontIfNeeded_(string srcFontPath, string dstFontPath, string elementPattern, string extPattern, CopiedFiles copiedFiles)
        {
            string content = File.ReadAllText(dstFontPath);
            string srcBaseDir = Path.GetDirectoryName(srcFontPath);
            string dstBaseDir = Path.GetDirectoryName(dstFontPath);

            // 依存リソースのパスを検出する正規表現
            string filePathPattern = string.Format("<{0}>(.+?\\.{1})", elementPattern, extPattern);
            content = Regex.Replace(content, filePathPattern, new MatchEvaluator((m) =>
            {
                // 参照しているリソースをコピーしつつ
                string srcFullPath = Path.Combine(srcBaseDir, m.Groups[2].Value);
                string dstFullPath = Path.Combine(dstBaseDir, Path.GetFileName(m.Groups[2].Value));

                if (!copiedFiles.DstFont.Contains(dstFullPath))
                {
                    CopyFileIfValid_(srcFullPath, dstFullPath);

                    copiedFiles.Font.Add(srcFullPath);
                    copiedFiles.DstFont.Add(dstFullPath);
                }

                // ファイルパスをコピー先の位置に変更する。
                return "<" + m.Groups[1].Value + ">" + Path.Combine(".\\", Path.GetFileName(m.Groups[2].Value));
            }));

            // 書き換えた内容を書き戻す
            File.WriteAllText(dstFontPath, content);
        }

        /// <summary>
        ///
        /// </summary>
        static void CreateAndClearDirIfNotExsit_(string dir, Action<string> onErrorReportAction)
        {
            try
            {
                if (Directory.Exists(dir))
                {
                    Directory.Delete(dir, true);
                }

                Directory.CreateDirectory(dir);
            }
            catch
            {
                // フォルダの作成に失敗しました処理を中止します。
                onErrorReportAction(LECoreStringResMgr.Get("LAYOUTDLG_ERROR_MAKINGFOLDER") + dir);
                return;
            }
        }
        /// <summary>
        /// フォルダ生成
        /// </summary>
        static void CreateDirIfNotExsit_(string dir, Action<string> onErrorReportAction)
        {
            try
            {
                if (!Directory.Exists(dir))
                {
                    Directory.CreateDirectory(dir);
                }
            }
            catch
            {
                // フォルダの作成に失敗しました処理を中止します。
                onErrorReportAction(LECoreStringResMgr.Get("LAYOUTDLG_ERROR_MAKINGFOLDER") + dir);
                return;
            }
        }

        /// <summary>
        /// レイアウトファイルの参照パス情報を書き換えます。
        /// </summary>
        static void ModifyFilePath_(string lytPath, string texDirPath, string fontDirPath, string exportCombinerPath)
        {
            string content = File.ReadAllText(lytPath);

            // Textureパス
            {
                var matcheies = Regex.Matches(content, "textureFile imagePath=\"(.+?)\"", RegexOptions.None);
                foreach (Match match in matcheies)
                {
                    string filePath = match.Groups[1].Value;
                    string newPath = Path.Combine(texDirPath, Path.GetFileName(filePath));
                    newPath = Utility.GetRelativePath(lytPath, newPath);

                    content = content.Replace(filePath, newPath);
                }
            }

            // Fontパス
            {
                var matcheies = Regex.Matches(content, "fontFile path=\"(.+?)\"", RegexOptions.None);
                foreach (Match match in matcheies)
                {
                    string filePath = match.Groups[1].Value;
                    string newPath = Path.Combine(fontDirPath, Path.GetFileName(filePath));
                    newPath = Utility.GetRelativePath(lytPath, newPath);

                    content = content.Replace(filePath, newPath);
                }
            }

            // コンバイナファイル パス
            {
                const string combinerUserShaderFileNameKey = "CombinerUserShader fileName=";
                foreach (Match match in Regex.Matches(content, combinerUserShaderFileNameKey + "\"(.*?)\""))
                {
                    string relativeFilePath = match.Groups[1].Value;
                    string combinerFilePath = Path.Combine(exportCombinerPath, Path.GetFileName(relativeFilePath));
                    string newPath = Utility.GetRelativePath(lytPath, combinerFilePath);

                    // コンバイナファイルは複数設定される場合がり relativeFilePath だけだと更新箇所もさらに入れ替えが行われるので項目を含めて一致とします。
                    content = content.Replace(
                                    combinerUserShaderFileNameKey + "\""+ relativeFilePath + "\"",
                                    combinerUserShaderFileNameKey + "\"" + newPath + "\"");
                }
            }
            File.WriteAllText(lytPath, content);
        }

        /// <summary>
        /// 結果を報告する。
        /// </summary>
        static public string GetReportString(CopiedFiles copiedFiles)
        {
            // 更新結果をユーザに通知します。
            // 以下のファイルが処理されました。
            StringBuilder sb = new StringBuilder(LECoreStringResMgr.Get("SYSTEM_EXECUTE_BELOW") + Environment.NewLine);
            sb.AppendLine("-------------------------------------------------");

            sb.AppendLine(string.Format(" * Parts [num={0}]", copiedFiles.Parts.Count));
            foreach (string partsPath in copiedFiles.Parts)
            {
                sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(partsPath), partsPath));
            }
            sb.AppendLine();

            sb.AppendLine(string.Format(" * Textures [num={0}]", copiedFiles.Texture.Count));
            foreach (string texPath in copiedFiles.Texture)
            {
                sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(texPath), texPath));
            }
            sb.AppendLine();

            sb.AppendLine(string.Format(" * Fonts [num={0}]", copiedFiles.Font.Count));
            foreach (string font in copiedFiles.Font)
            {
                sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(font), font));
            }
            sb.AppendLine();

            sb.AppendLine(string.Format(" * Controls [num={0}]", copiedFiles.Controls.Count));
            foreach (string control in copiedFiles.Controls)
            {
                sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(control), control));
            }
            sb.AppendLine();

            sb.AppendLine(string.Format(" * UserDataPresets [num={0}]", copiedFiles.UserDataPresets.Count));
            foreach (string userDataPreset in copiedFiles.UserDataPresets)
            {
                sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(userDataPreset), userDataPreset));
            }
            sb.AppendLine();

            sb.AppendLine(string.Format(" * Project [num={0}]", string.IsNullOrEmpty(copiedFiles.Project) ? 0 : 1 ));
            sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(copiedFiles.Project), copiedFiles.Project));
            sb.AppendLine();

            if (copiedFiles.Combiner.Count > 0)
            {
                sb.AppendLine(string.Format(" * CombinerFile [num={0}]", copiedFiles.Combiner.Count));
                sb.AppendLine(string.Format("    {0}", LECoreStringResMgr.Get("WARNING_COMBINERFILE_IMAGEPATH")));
                foreach (string combinerPath in copiedFiles.Combiner)
                {
                    sb.AppendLine(string.Format("    {0} ({1})", Path.GetFileNameWithoutExtension(combinerPath), combinerPath));
                }
                sb.AppendLine();
            }

            return sb.ToString();
        }
    }
}
