﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

using namespace System;
using namespace System::Runtime::InteropServices;
using namespace System::Text;
using namespace System::Diagnostics;

#define NW4R_FROM_TOOL
#include "NW4RTextWriter.h"


#include "TextWriter.h"
#include "WideTextWriter.h"
#include "TagProcessor.h"
#include "ExtendedTagProcessor.h"
#include "NW4RFont.h"



namespace NW4R
{
namespace Font
{
        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::NW4RTextWriter( INW4RTextRenderer* )

           Description:  コンストラクタ

           Arguments:    pRenderer        実際に文字を描画する処理を委譲するクラス

           Returns:      なし。
        *---------------------------------------------------------------------------*/

        NW4RTextWriter::NW4RTextWriter( INW4RTextRenderer^ pRenderer )
        {
           m_pTextWriter = new UnManaged::WideTextWriter( );

           m_pTagProcessor = new UnManaged::WideTagProcessor;
           m_pTextWriter->SetTagProcessor( m_pTagProcessor );

           m_pRenderer = pRenderer;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::SetFont( NW4RResFont* )

           Description:  文字列描画クラスの使用しているフォントを設定します。

           Arguments:    pFont         フォント

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        void NW4RTextWriter::SetFont( INWFont^ pFont )
        {
           m_pTextWriter->SetFont( *pFont->GetUnManagedFont() );
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::GetFont()

           Description:  文字列描画クラスの使用しているフォントを取得します。

           Arguments:    なし。

           Returns:      フォント。
        *---------------------------------------------------------------------------*/
        NW4RResFont^ NW4RTextWriter::GetFont()
        {
           return m_pFont;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::SetScale( float, float )

           Description:  文字のスケールを設定します。

           Arguments:    hScale    水平方向スケール
                         vScale    垂直方向スケール

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        void NW4RTextWriter::SetScale( float hScale, float vScale )
        {
           m_pTextWriter->SetScale( hScale, vScale );
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::SetFontSize( float, float )

           Description:  文字の大きさを指定します。

           Arguments:    width    文字の幅
                         height   文字の高さ

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        void NW4RTextWriter::SetFontSize( float width, float height )
        {
           m_pTextWriter->SetFontSize( width, height );
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::CalcStringWidth( String^ , int )

           Description:  カーソル位置を設定します。

           Arguments:    str       描画する文字列
                         int       計算する文字数

           Returns:      描画した文字列の幅。
        *---------------------------------------------------------------------------*/
        void NW4RTextWriter::SetCursor( float x, float y )
        {
           m_pTextWriter->SetCursor( x, y );
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::ResetTagProcessor(bool)

           Description:  タグプロセッサをリセットします。

           Arguments:    isExtendedTagEnabled  拡張タグの解釈を有効にするかどうか

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        void NW4RTextWriter::ResetTagProcessor(bool isExtendedTagEnabled)
        {
            delete m_pTagProcessor;
            if (isExtendedTagEnabled)
            {
                m_pTagProcessor = new UnManaged::ExtendedTagProcessor;
            }
            else
            {
                m_pTagProcessor = new UnManaged::WideTagProcessor;
            }
           m_pTextWriter->SetTagProcessor(m_pTagProcessor);
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::CalcStringWidth( String^ , int )

           Description:  文字列の幅を計算します

           Arguments:    str       描画する文字列
                         int       計算する文字数

           Returns:      描画した文字列の幅。
        *---------------------------------------------------------------------------*/
        float NW4RTextWriter::CalcStringWidth
        (
           String^     str,
           int         length
        )
        {
           // 文字列のコピーを作成
           // 文字コードの変換も行われます。
           IntPtr ip = Marshal::StringToCoTaskMemUni( str );
           const wchar_t* pP = static_cast<const wchar_t*>( static_cast<void*>(ip) );

           float result = m_pTextWriter->CalcStringWidth( pP, length );

           // コピー領域を解放
           Marshal::FreeCoTaskMem( ip );

           return result;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::CalcStringHeight( String^ , int )

           Description:  文字列の高さを計算します。

           Arguments:    str       描画する文字列
                         int       計算する文字数

           Returns:      描画した文字列の高さ。
        *---------------------------------------------------------------------------*/
        float NW4RTextWriter::CalcStringHeight( String^ str, int length )
        {
            // 文字列のコピーを作成
            // 文字コードの変換も行われます。
            IntPtr ip = Marshal::StringToCoTaskMemUni( str );
            const wchar_t* pP = static_cast<const wchar_t*>( static_cast<void*>(ip) );

            float result = m_pTextWriter->CalcStringHeight( pP, length );

            // コピー領域を解放
            Marshal::FreeCoTaskMem( ip );

            return result;
        }



        /*---------------------------------------------------------------------------*
           Name:         NW4RTextWriter::Print( String^ )

           Description:  文字列を描画します。

           Arguments:    str       描画する文字列

           Returns:      描画した文字列の幅。
        *---------------------------------------------------------------------------*/
        // TODO:複数のAPI
        // TODO:可変長引数は、どのようにあつかうのか？？？
        float  NW4RTextWriter::Print( String^ str, array<float>^ lineOffset )
        {
           // 文字列のコピーを作成
           // 文字コードの変換も行われます。
           IntPtr ip = Marshal::StringToCoTaskMemUni( str );

           const wchar_t* pP = static_cast<const wchar_t*>( static_cast<void*>(ip) );


           // 文字列を描画します。
           // m_pTextWriter->Print() 実行の結果、
           // m_pTextWriter 内部にコマンドリストが生成される
           pin_ptr<float> pLineOffset = (lineOffset == nullptr || lineOffset->Length == 0) ? nullptr : &lineOffset[0];
           int count = lineOffset == nullptr ? 0 : lineOffset->Length;
           float bResult = m_pTextWriter->Print( pP, static_cast<int>(wcslen(pP)), count, pLineOffset );


           // 描画コマンドリストを解釈しながら、描画関数コールバックを呼び出します。
           // これは、コールバック処理が、マネージ、アンマネージ境界を
           // またぐ事を回避するための対策です。
           const UnManaged::CharWriter::RenderingTaskPtrArray& taskArray = m_pTextWriter->GetTaskArray();
           if (m_pTextWriter->GetDrawFromRightToLeftEnabled())
           {
               for (int i = (int)taskArray.size() - 1; i >= 0; i--)
               {
                   NW4RGlyph^ pG = gcnew NW4RGlyph(&taskArray[i]->glyph);

                   // i が int の値域をオーバーする可能性はほとんどないと想定して単純にキャストしています。
                   Debug::Assert(INT_MAX >= i);
                   m_pRenderer->DrawChar(pG, taskArray[i]->x, taskArray[i]->y, static_cast<int>(i));
               }
           }
           else
           {
               for (int i = 0; i < (int)taskArray.size(); i++)
               {
                   NW4RGlyph^ pG = gcnew NW4RGlyph(&taskArray[i]->glyph);

                   // i が int の値域をオーバーする可能性はほとんどないと想定して単純にキャストしています。
                   Debug::Assert(INT_MAX >= i);
                   m_pRenderer->DrawChar(pG, taskArray[i]->x, taskArray[i]->y, static_cast<int>(i));
               }
           }

           // コピー文字列領域を解放
           Marshal::FreeCoTaskMem( ip );

           return bResult;
        }
}
}
