﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UncompressHuffman.h"
#include "nw4r/misc.h"
#include "inlines.h"


namespace
{
    inline u8* GetNextNode( const u8* pTree, u32 select )
    {
        return (u8*)( ((std::size_t)pTree & ~(std::size_t)0x1) + ( ( (*pTree & 0x3F) + 1 ) * 2 ) + select );
    }
}

void CXInitUncompContextHuffman( CXUncompContextHuffman *context, void* dest )
{
    context->destp      = (u8*)dest;
    context->destCount  = 0;
    context->bitSize    = 0;
    context->treeSize   = -1;
    context->treep      = &context->tree[ 0 ];
    context->destTmp    = 0;
    context->destTmpCnt = 0;
    context->srcTmp     = 0;
    context->srcTmpCnt  = 0;
    context->headerSize = 4;
}


s32 CXReadUncompHuffman( CXUncompContextHuffman *context, const void* data, u32 len )
{
#define TREE_END_MASK   0x80U
    const u8* srcp = (const u8*)data;
    u32  select;
    u32  endFlag;

    while ( context->headerSize > 0 )
    {
        if ( --context->headerSize <= 2 )
        {
            context->destCount |= (*srcp << ((2 - context->headerSize) * 8));
        }
        else
        {
            context->bitSize = (u8)(*srcp & 0xF);
        }
        ++srcp;
        if ( --len == 0 )
        {
            return (context->headerSize == 0)? context->destCount : -1;
        }
    }

    // CXInitUncompContextHuffmanでtreeSizeが-1に設定される。
    // context->treeSizeが負ならばデータの一番先頭である。
    if ( context->treeSize < 0 )
    {
        context->treeSize = (s16)( ( *srcp + 1 ) * 2 - 1);
        *context->treep++ = *srcp++;
        len--;
    }

    // ハフマン符号テーブルを読み込む
    while ( context->treeSize > 0 )
    {
        if ( len == 0 )
        {
            return context->destCount;
        }
        *context->treep++ = *srcp++;
        context->treeSize--;
        len--;
        if ( context->treeSize == 0 )
        {
            context->treep = &context->tree[ 1 ];
        }
    }

    // 復号化処理
    while ( context->destCount > 0 )
    {
        // 4バイト単位でsrcデータを読み込む
        while ( context->srcTmpCnt < 32 )
        {
            if ( len == 0 )
            {
                return context->destCount;
            }
            context->srcTmp |= (*srcp++) << context->srcTmpCnt;
            len--;
            context->srcTmpCnt += 8;
        }

        // 読み込んだ32bitを復号化する。32bit分の処理が終われば次の4バイトを読む
        while ( context->srcTmpCnt > 0 )
        {
            select  = context->srcTmp >> 31;
            endFlag = (*context->treep << select) & TREE_END_MASK;
            context->treep = GetNextNode( context->treep, select );
            context->srcTmp <<= 1;
            context->srcTmpCnt--;

            if ( ! endFlag )
            {
                continue;
            }

            // ハフマンツリーの終端フラグが立っていた場合にはオフセットの先に
            // データが格納されている
            context->destTmp >>= context->bitSize;
            context->destTmp |= *context->treep << ( 32 - context->bitSize );
            context->treep = &context->tree[ 1 ];
            context->destTmpCnt += context->bitSize;
            // 4バイト単位で書き込む
            if ( context->destTmpCnt == 32 )
            {
                //*(u32*)context->destp = NW4R::Font::UnManaged::ReverseEndian( context->destTmp );
                *(u32*)context->destp = context->destTmp;
                context->destp     += 4;
                context->destCount -= 4;
                context->destTmpCnt = 0;
                if ( context->destCount <= 0 )
                {
                    return 0;
                }
            }
        }
    }

    return 0;
}


void CXUncompressHuffman( const void *srcp, void *destp )
{
#define TREE_END 0x80
    const u32 *pSrc          = (u32*)srcp;
    u32       *pDst          = (u32*)destp;
    u8        *treep         = (u8 *)pSrc + 4;
    u8        *treeStartp    = treep + 1;
    u32       dataBit        = *(u8*)pSrc & 0x0FU;
    u32       destTmp        = 0;
    u32       destTmpCount   = 0;
    u32       destTmpDataNum = 4 + ( dataBit & 0x7 );
    s32       destCount      = (s32)( NW4R::Font::UnManaged::ReverseEndian( *pSrc ) >> 8 );

    pSrc  = (u32*)( treep + ((*treep + 1) << 1) );
    treep = treeStartp;

    while ( destCount > 0 )
    {
        s32 srcCount = 32;
        u32 srcTmp   = NW4R::Font::UnManaged::ReverseEndian( *pSrc++ );      // エンディアン対策
        while ( --srcCount >= 0 )
        {
            u32 treeShift = (srcTmp >> 31) & 0x1;
            u32 treeCheck = *treep;
            treeCheck <<= treeShift;
            treep = (u8*)( (((std::size_t)treep >> 1) << 1) + (((*treep & 0x3f) + 1) << 1) + treeShift );
            if ( treeCheck & TREE_END )
            {
                destTmp >>= dataBit;
                destTmp |= *treep << (32 - dataBit);
                treep = treeStartp;
                if ( ++destTmpCount == destTmpDataNum )
                {
                    // 展開用バッファの末尾４バイトアライメントまではオーバーアクセスする
                    *pDst++ = NW4R::Font::UnManaged::ReverseEndian(destTmp); // エンディアン対策
                    destCount -= 4;
                    destTmpCount = 0;
                }
            }
            if ( destCount <= 0 )
            {
                break;
            }
            srcTmp <<= 1;
        }
    }
}
