﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "ArchiveFont.h"
#include "inlines.h"
#include "nw4r/misc.h"
#include "nw4r/math/arithmetic.h"
#include "ByteSwapUtil.h"
#include "FontEnvironment.h"

#include <cstring>


namespace NW4R
{
namespace Font
{
namespace UnManaged
{


/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::ArchiveFont()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ArchiveFont::ArchiveFont()
{
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::~ArchiveFont()

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ArchiveFont::~ArchiveFont()
{
}

#if 0
//
// FontGlyphGroupsAcs のバイトスワップ処理
//
typedef detail::ArchiveFontBase::FontGlyphGroupsAcs FontGlyphGroupsAcs;
static void SwapFontGlyphGroupsAcs_( FontGlyphGroupsAcs& gg )
{
    // ロードするシートのサイズを計算
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumSheet(); ++flagSetNo )
    {
        // フラグ列をスワップ
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            gg.SetUseSheetFlags( setNo, flagSetNo, Swap4Byte( gg.GetUseSheetFlags( setNo, flagSetNo ) ) );
        }

        // ロードするシートのサイズをスワップ
        u32 useSheets = 0x0;
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);
            useSheets |= gg.GetUseSheetFlags(setNo, flagSetNo);
        }
        for( int b = 0; b < 32; ++b )
        {
            if( ((useSheets << b) & (1u << 31)) != 0 )
            {
                int idx = flagSetNo * 32 + b;
                gg.SetSizeCompSheet( idx, Swap4Byte( gg.GetSizeCompSheet( idx ) ) );
            }
        }
    }

    // CWDH ブロック用バッファサイズをスワップ
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumCWDH(); ++flagSetNo )
    {
        // フラグ列をスワップ
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            gg.SetUseCWDHFlags( setNo, flagSetNo, Swap4Byte( gg.GetUseCWDHFlags( setNo, flagSetNo ) ) );
        }

        // CWDH ブロック用のサイズテーブルのスワップ
        u32 useCWDH   = 0x0;
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);
            useCWDH   |= gg.GetUseCWDHFlags(setNo, flagSetNo);
        }

        for( int b = 0; b < 32; ++b )
        {
            if( ((useCWDH << b) & (1u << 31)) != 0 )
            {
                int idx = flagSetNo * 32 + b;
                gg.SetSizeCWDH( idx, Swap4Byte( gg.GetSizeCWDH(idx) ) );
            }
        }
    }

    // CMAP ブロック用に必要なバッファサイズを計算
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumCMAP(); ++flagSetNo )
    {
        // フラグ列をスワップ
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            gg.SetUseCMAPFlags( setNo, flagSetNo, Swap4Byte( gg.GetUseCMAPFlags( setNo, flagSetNo ) ) );
        }

        // CMAP ブロック用のサイズテーブルのスワップ
        u32 useCMAP   = 0x0;
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);
            useCMAP   |= gg.GetUseCMAPFlags(setNo, flagSetNo);
        }

        for( int b = 0; b < 32; ++b )
        {
            if( ((useCMAP << b) & (1u << 31)) != 0 )
            {
                int idx = flagSetNo * 32 + b;
                gg.SetSizeCMAP( idx, Swap4Byte( gg.GetSizeCMAP(idx) ) );
            }
        }
    }
}
#endif

/* ------------------------------------------------------------------------
        構築/破棄
   ------------------------------------------------------------------------ */
/* static */ void
ArchiveFont::UnpackData( const void* brfnt )
{
    NW4R_POINTER_ASSERT( brfnt );
    void*       pBrfnt = const_cast<void*>( brfnt );
    BinaryBlockHeader* blockHeader;
    FontInformation* info = NULL;
    int nBlocks = 0;

    // バイトスワップします
    BinaryFileHeader* fileHeader = reinterpret_cast<BinaryFileHeader*>(pBrfnt);
    //SwapBinaryFileHeader_( fileHeader );


    blockHeader = reinterpret_cast<BinaryBlockHeader*>(
        reinterpret_cast<u8*>(fileHeader) + fileHeader->headerSize
    );

    while( nBlocks < fileHeader->dataBlocks )
    {
        NW4R_POINTER_ASSERT( blockHeader );


        // バイトスワップします
        //SwapBinaryBlockHeader_( blockHeader );


        switch( blockHeader->kind )
        {

        //--------------------------------------------------
        // GLGP ブロック
        case BINBLOCK_SIG_GLGR:
            {
                FontGlyphGroups* pGroup = reinterpret_cast<FontGlyphGroups*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader) );

                //SwapFontGlyphGroups_( pGroup );

                FontGlyphGroupsAcs gg( brfnt );
                //SwapFontGlyphGroupsAcs_( gg );
            }
            break;
        //--------------------------------------------------
        // INFO ブロック
        case BINBLOCK_SIG_FINF:
            // INFOブロックは1つでなければならない
            NW4R_ASSERT( info == NULL );
            {
                info = reinterpret_cast<FontInformation*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader));

                // バイトスワップします
                //SwapFontInformation_( info );
            }
            break;

        //--------------------------------------------------
        // TGLP ブロック
        case BINBLOCK_SIG_TGLP:
            // TGLP ブロックも1つでなければならないが複数存在しても致命的ではない
            {
                FontTextureGlyph* glyph = reinterpret_cast<FontTextureGlyph*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                //SwapFontTextureGlyph_( glyph );

                // シートデータのオフセットを解決してデータにアクセスします。
                // 別の部分の処理のために、元のデータはオフセットのまま残しておく必要があります。
                u8*  pSheetChunkHead = glyph->sheetImage;
                ResolveOffset(pSheetChunkHead, const_cast<void*>(brfnt));

                int numSheet = 0;
                while( numSheet < glyph->sheetNum )
                {
                    // シートデータの先頭４バイトには、シート、データサイズが格納されています。
                    // ここでは、ツールでデータを解釈するために、バイトスワップ処理を行います。
                    u32* pSheetSize = reinterpret_cast<u32*>(pSheetChunkHead);
                    //*pSheetSize = Swap4Byte( *pSheetSize );

                    // 次のシートデータへ。(sizeof(u32) = シートデータサイズ用u32 )
                    pSheetChunkHead += *pSheetSize + sizeof(u32);
                    numSheet++;
                }
            }
            break;

        //--------------------------------------------------
        // CWDHブロック
        case BINBLOCK_SIG_CWDH:
            {
                FontWidth* width = reinterpret_cast<FontWidth*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                //SwapFontWidth_( width );
            }
            break;

        //--------------------------------------------------
        // CMAPブロック
        case BINBLOCK_SIG_CMAP:
            {
                FontCodeMap* map = reinterpret_cast<FontCodeMap*>(
                    reinterpret_cast<u8*>(blockHeader) + sizeof(*blockHeader)
                );

                // バイトスワップします
                //SwapFontCodeMap_( map );
            }
            break;

        //--------------------------------------------------
        // unknown
        default:
             NW4R_ASSERTMSG(false, "The font has unknown block('%c%c%c%c').",
                (blockHeader->kind >> 24) & 0xFF,
                (blockHeader->kind >> 16) & 0xFF,
                (blockHeader->kind >>  8) & 0xFF,
                (blockHeader->kind >>  0) & 0xFF
                );
            break;
        }

        blockHeader = reinterpret_cast<BinaryBlockHeader*>(
            reinterpret_cast<u8*>(blockHeader) + blockHeader->size
        );
        nBlocks++;
    }
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::GetRequireBufferSize()

  Description:  Construct に必要なバッファのサイズを計算します。

  Arguments:    brfnt:          フォントリソースの先頭部分へのポインタ。
                glyphGroups:    ロードするグリフグループを指定する文字列。

  Returns:      Construct に必要なバッファのサイズを返します。
                brfnt が不正であるなどのエラーが発生した場合は 0 を返します。
 *---------------------------------------------------------------------------*/
/* static */ u32
ArchiveFont::GetRequireBufferSize(
    const void* brfnt,
    const char* glyphGroups /*= LOAD_GLYPH_ALL*/
)
{
    NW4R_POINTER_ASSERT( brfnt );
    NW4R_POINTER_ASSERT( glyphGroups );

    // 妥当性のチェック
    if( ! IsValidResource(brfnt, HEADER_SIZE) )
    {
        return 0;
    }

    FontGlyphGroupsAcs gg(brfnt);
    u32 numRequireSheets    = 0;
    u32 sizeSumCWDH         = 0;
    u32 sizeSumCMAP         = 0;

    // ロードするシートの枚数を計算
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumSheet(); ++flagSetNo )
    {
        u32 useSheets = 0x0;

        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);

            if( IsNullString(glyphGroups) || IncludeName(glyphGroups, setName) )
            {
                useSheets |= gg.GetUseSheetFlags(setNo, flagSetNo);
            }
        }

        numRequireSheets += nw4r::math::CntBit1(useSheets);
    }

    // CWDH ブロック用に必要なバッファサイズを計算
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumCWDH(); ++flagSetNo )
    {
        u32 useCWDH   = 0x0;

        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);

            if( IsNullString(glyphGroups) || IncludeName(glyphGroups, setName) )
            {
                useCWDH   |= gg.GetUseCWDHFlags(setNo, flagSetNo);
            }
        }

        for( int b = 0; b < 32; ++b )
        {
            if( ((useCWDH << b) & (1u << 31)) != 0 )
            {
                sizeSumCWDH += gg.GetSizeCWDH(flagSetNo * 32 + b) - sizeof(BinaryBlockHeader);
            }
        }
    }

    // CMAP ブロック用に必要なバッファサイズを計算
    for( int flagSetNo = 0; flagSetNo * 32 < gg.GetNumCMAP(); ++flagSetNo )
    {
        u32 useCMAP   = 0x0;

        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);

            if( IsNullString(glyphGroups) || IncludeName(glyphGroups, setName) )
            {
                useCMAP   |= gg.GetUseCMAPFlags(setNo, flagSetNo);
            }
        }

        for( int b = 0; b < 32; ++b )
        {
            if( ((useCMAP << b) & (1u << 31)) != 0 )
            {
                sizeSumCMAP += gg.GetSizeCMAP(flagSetNo * 32 + b) - sizeof(BinaryBlockHeader);
            }
        }
    }


    //---- 必要バッファサイズの計算
    const u32 sizeTable    = RoundUp(sizeof(u16) * gg.GetNumSheet(),       4);
    const u32 sizeFINF     = RoundUp(sizeof(FontInformation),              4);
    const u32 sizeTGLP     = RoundUp(sizeof(FontTextureGlyph),             4);
    const u32 sizeSumSheet = RoundUp(gg.GetSheetSize() * numRequireSheets, 4);

    // シート領域以降はシート展開時に圧縮データのストリーム展開用コンテキストとして使用するので
    // ある程度以上のサイズを確保する。
    // 通常はシート領域以降がこのサイズを下回ることは無いのでサイズ的な追加コストは発生しない。
    const u32 sizeForContext = static_cast<u32>(sizeof(CXUncompContextHuffman));
    const u32 sizeAfterSheet = Max(static_cast<u32>(sizeSumCWDH + sizeSumCMAP), sizeForContext);

    // RoundUp の中に入るのはバッファ上でシートより前に配置されるもの
    return RoundUp(sizeTable + sizeFINF + sizeTGLP,
                   NW::Font::NWFontEnvironment::TextureDataAlignSize) + sizeSumSheet + sizeAfterSheet;
}




/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::InitStreamingConstruct(
                                ConstructContext*, void*, u32, const char*)

  Description:  順次読み込みでフォントを構築するための初期化を行います。
                以降、この関数で初期化した pContext を引数として
                StreamingConstruct を呼び出します。

  Arguments:    pContext:       初期化するストリーム展開コンテキストへのポインタ。
                pBuffer:        フォントを構築するバッファへのポインタ。
                                展開が完了すると以降このバッファは
                                ArchiveFont の管理下となります。
                bufferSize:     pBuffer が指すバッファのサイズ。
                glyphGroups:    ロードするグリフグループを指定する文字列。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFont::InitStreamingConstruct(
    ArchiveFont::ConstructContext*  pContext,
    void*                           pBuffer,
    u32                             bufferSize,
    const char*                     glyphGroups /*= LOAD_GLYPH_ALL*/
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pContext );
    NW4R_POINTER_ASSERT( pBuffer );
    NW4R_BUFFERSIZE_ASSERT( bufferSize );
    NW4R_POINTER_ASSERT( glyphGroups );

    pContext->Init( pBuffer, bufferSize, glyphGroups );
    pContext->op = ConstructContext::ANALYZE_FILE_HEADER;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::StreamingConstruct(
                                        ConstructContext*, const void*, u32)

  Description:  グループセットを持つ圧縮済みフォントから順次読み込みで
                指定したグリフグループを展開し、フォントとして使用可能にします。
                最初に InitStreamingConstruct で pContext を初期化しておく必要があります。

  Arguments:    pContext:       使用するストリーム展開コンテキストへのポインタ。
                stream:         メモリ上にロードしたストリーム部分データへのポインタ。
                streamSize:     stream のサイズ。

  Returns:      処理結果が返ります。
                    CONSTRUCT_MORE_DATA:  後続のストリームデータが必要です。
                    CONSTRUCT_FINISH:     展開が完了しました。
                    CONSTRUCT_ERROR:      展開中にエラーが発生しました。
 *---------------------------------------------------------------------------*/
ArchiveFont::ConstructResult
ArchiveFont::StreamingConstruct(
    ArchiveFont::ConstructContext*  pContext,
    const void*                     stream,
    u32                             streamSize
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pContext );
    NW4R_POINTER_ASSERT( stream );
    NW4R_BUFFERSIZE_ASSERT( streamSize );

    //---- 既に構築済みなら失敗とする
    if( GetFINF() != NULL )
    {
        // ただし CONSTRUCT_FINISH を返した後であるなら
        // エラーではなく CONSTRUCT_FINISH とする。
        return pContext->HasMoreBlock() ? CONSTRUCT_ERROR: CONSTRUCT_FINISH;
    }

    ArchiveFont::ConstructResult ret = CONSTRUCT_CONTINUE;
    CachedStreamReader& s = pContext->GetStreamReader();
    s.Attach(stream, streamSize);

    while( ret == CONSTRUCT_CONTINUE )
    {
        switch( pContext->op )
        {
        case ConstructContext::DISPATCH:              ret = ConstructOpDispatch(pContext, &s);  break;
        case ConstructContext::ANALYZE_FILE_HEADER:   ret = ConstructOpAnalyzeFileHeader(pContext, &s);  break;
        case ConstructContext::ANALYZE_GLGR:          ret = ConstructOpAnalyzeGLGR(pContext, &s);  break;
        case ConstructContext::ANALYZE_FINF:          ret = ConstructOpAnalyzeFINF(pContext, &s);  break;
        case ConstructContext::ANALYZE_CMAP:          ret = ConstructOpAnalyzeCMAP(pContext, &s);  break;
        case ConstructContext::ANALYZE_CWDH:          ret = ConstructOpAnalyzeCWDH(pContext, &s);  break;
        case ConstructContext::ANALYZE_TGLP:          ret = ConstructOpAnalyzeTGLP(pContext, &s);  break;
        case ConstructContext::PREPAIR_COPY_SHEET:    ret = ConstructOpPrepairCopySheet(pContext, &s);  break;
        case ConstructContext::PREPAIR_EXPAND_SHEET:  ret = ConstructOpPrepairExpandSheet(pContext, &s);  break;
        case ConstructContext::COPY:                  ret = ConstructOpCopy(pContext, &s);  break;
        case ConstructContext::SKIP:                  ret = ConstructOpSkip(pContext, &s);  break;
        case ConstructContext::EXPAND:                ret = ConstructOpExpand(pContext, &s);  break;
        case ConstructContext::FATAL_ERROR:           ret = ConstructOpFatalError(pContext, &s); break;
        default:
            NW4R_ASSERTMSG(false, "invalid operation(%d)", pContext->op);
            pContext->op = ArchiveFont::ConstructContext::FATAL_ERROR;
            ret = CONSTRUCT_ERROR;
            break;
        }
    }

    //---- フォントリソースに対する処理終了時
    if( (ret == CONSTRUCT_FINISH) && (GetFINF() == NULL) )
    {
        void* pUserBuffer;
        FontInformation* pFontInfo;
        u16* pAdjustArray;

        pContext->Finish(&pUserBuffer, &pFontInfo, &pAdjustArray);
        SetResourceBuffer(pUserBuffer, pFontInfo, pAdjustArray);

        if( AdjustIndex(GetFINF()->alterCharIndex) == GLYPH_INDEX_NOT_FOUND )
        {
            GetFINF()->alterCharIndex = 0;
        }

        InitializeFontInfo();
    }

    return ret;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::Construct(void*, u32, const void*, const char*)

  Description:  グループセットを持つ圧縮済みフォントから指定した
                グリフグループを展開し、フォントとして使用可能にします。

  Arguments:    pBuffer:        フォントを構築するバッファへのポインタ。
                                関数が成功すると以降このバッファは
                                ArchiveFont の管理下となります。
                bufferSize:     pBuffer が指すバッファのサイズ。
                brfnt:          メモリ上にロード済みのバイナリリソース。
                glyphGroups:    ロードするグリフグループを指定する文字列。

  Returns:      展開に成功した場合は true, 失敗した場合は false を返します。
 *---------------------------------------------------------------------------*/
bool
ArchiveFont::Construct(
    void*           pBuffer,
    u32             bufferSize,
    const void*     brfnt,
    const char*     glyphGroups /*= LOAD_GLYPH_ALL*/
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pBuffer );
    NW4R_BUFFERSIZE_ASSERT( bufferSize );
    NW4R_POINTER_ASSERT( brfnt );
    NW4R_POINTER_ASSERT( glyphGroups );

    ArchiveFont::ConstructResult ret;
    ArchiveFont::ConstructContext context;
    const u32 fileSize = reinterpret_cast<const BinaryFileHeader*>(brfnt)->fileSize;
    NW4R_BUFFERSIZE_ASSERT( fileSize );

    InitStreamingConstruct(&context, pBuffer, bufferSize, glyphGroups);
    ret = StreamingConstruct(&context, brfnt, fileSize);

    return ret == CONSTRUCT_FINISH;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::Destroy()

  Description:  フォントを破棄します。

  Arguments:    なし。

  Returns:      構築時に与えたバッファへのポインタを返します。
 *---------------------------------------------------------------------------*/
void*
ArchiveFont::Destroy()
{
    return RemoveResourceBuffer();
}



/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::GetGlyph( Glyph*, CharCode )

  Description:  文字のグリフデータを取得します。

  Arguments:    glyph:  グリフデータを受け取るバッファへのポインタ。
                c:      グリフデータを取得する文字の文字コード。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFont::GetGlyph( Glyph* glyph, CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    GetGlyphFromIndex(glyph, index);
}



/* ------------------------------------------------------------------------
        グリフインデックス
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::GetGlyphFromIndex( Glyph*, GlyphIndex )

  Description:  グリフインデックスに対応するグリフデータを取得します。

  Arguments:    glyph:  グリフデータを格納するバッファへのポインタ。
                index:  取得するグリフデータのグリフインデックス。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFont::GetGlyphFromIndex(
    Glyph*      glyph,
    GlyphIndex  index
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( GetFINF() );
    const FontTextureGlyph& tg = *(GetFINF()->pGlyph);
    u16 adjustedIndex = AdjustIndex(index);

    if( adjustedIndex == GLYPH_INDEX_NOT_FOUND )
    {
        index = GetFINF()->alterCharIndex;
        adjustedIndex = AdjustIndex(index);
    }
    NW4R_ASSERT( adjustedIndex != GLYPH_INDEX_NOT_FOUND );

#if 0
    const u32 cellsInASheet = static_cast<u32>(tg.sheetRow * tg.sheetLine);
#else
    //const u32 sheetSeqLine = ( tg.sheetFormat ==  (u16)GX_TF_I4 ? tg.sheetLine/2 : tg.sheetLine );
    const u32 sheetSeqLine = tg.sheetLine;
    const u32 cellsInASheet = static_cast<u32>(tg.sheetRow * sheetSeqLine);
#endif
    const u32 sheetNo       = adjustedIndex / cellsInASheet;
    const u32 cellNo        = adjustedIndex % cellsInASheet;
    const u32 cellUnitX     = cellNo        % tg.sheetRow;
#if 0
    const u32 cellUnitY     = cellNo        / tg.sheetRow;
#else
    const u32 cellUnitY     = ( cellNo / tg.sheetRow ) % sheetSeqLine;
#endif
    const u32 cellPixelX    = cellUnitX     * (tg.cellWidth  + 1);
    const u32 cellPixelY    = cellUnitY     * (tg.cellHeight + 1);
    const u32 offsetBytes   = sheetNo * tg.sheetSize;
    const void* pSheet      = tg.sheetImage + offsetBytes;

    glyph->pTexture     = pSheet;
    glyph->sheetNo      = sheetNo;
    glyph->widths       = GetCharWidthsFromIndex(index);
    glyph->height       = tg.cellHeight;
#if 0
    glyph->texFormat    = static_cast<GXTexFmt>(tg.sheetFormat);
#else
    glyph->texFormat    = GX_TF_RGBA8;
#endif
    glyph->texWidth     = tg.sheetWidth;
    glyph->texHeight    = tg.sheetHeight;
    glyph->cellX        = static_cast<u16>(cellPixelX + 1);
    glyph->cellY        = static_cast<u16>(cellPixelY + 1);
}


    } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */
