﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_MATH_ARITHMETIC_H_
#define NW4R_MATH_ARITHMETIC_H_

#include "nw4r/math/config.h"
#include "nw4r/math/constant.h"

#include <cmath>

namespace nw4r { namespace math {



/* =======================================================================
        浮動少数に関する数学関数
   ======================================================================== */

/*-------------------------------------------------------------------------*
  Name:         F32AsU32(f32)

  Description:  f32 型の値をビット列を変更することなく u32 型にします。

  Arguments:    x:  u32型にする値。

  Returns:      x と等しいビット表現を持つ u32 型の値。
 *-------------------------------------------------------------------------*/
// u32 F32AsU32(f32 x)
//
// 説明： f32 型の値をビット列を変更することなく u32 型にします
// 引数： x： u32型 にする値。
// 返値： x と等しいビット表現を持つ u32 型の値。
NW4R_MATH_INLINE u32
F32AsU32(f32 x)
{
    return *reinterpret_cast<u32*>(&x);
}



/*-------------------------------------------------------------------------*
  Name:         U32AsF32(f32)

  Description:  u32 型の値をビット列を変更することなく f32 型にします。

  Arguments:    x:  f32型にする値。

  Returns:      x と等しいビット表現を持つ f32 型の値。
 *-------------------------------------------------------------------------*/
// f32 U32AsF32(u32 x)
//
// 説明： u32 型の値をビット列を変更することなく f32 型にします
// 引数： x: f32型にする値。
// 返値： x と等しいビット表現を持つ f32 型の値。
NW4R_MATH_INLINE f32
U32AsF32(u32 x)
{
    return *reinterpret_cast<f32*>(&x);
}



/*-------------------------------------------------------------------------*
  Name:         FGetExpPart(f32)

  Description:  浮動小数点数の指数部を取得します。

  Arguments:    f:  指数部を取得する数。

  Returns:      f の指数部を返します。
                ただし負の値はオフセット表現ではなく、2の補数表現です。
 *-------------------------------------------------------------------------*/
// s32 FGetExpPart(f32 f)
//
// 説明： 浮動小数点数の指数部を取得します。
// 引数： f:  指数部を取得する数。
// 返値： f の指数部を返します。ただし負の値はオフセット表現ではなく 2 の補数表現です。
NW4R_MATH_INLINE s32
FGetExpPart(f32 f)
{
    s32 s = static_cast<s32>((F32AsU32(f) >> 23) & 0xFF);
    return s - 127;
}



/*-------------------------------------------------------------------------*
  Name:         FGetMantPart(f32)

  Description:  浮動小数点数の仮数部を求めます。

  Arguments:    f:  仮数部を取得する数。

  Returns:      f の仮数部の値を返します。
                返り値は f と同じ符号を持っています。
 *-------------------------------------------------------------------------*/
// f32 FGetMantPart(f32 f)
//
// 説明： 浮動小数点数の仮数部を求めます。
// 引数： f:  仮数部を取得する数。
// 返値： f の仮数部を返します。返り値は f と同じ符号を持っています。
NW4R_MATH_INLINE f32
FGetMantPart(f32 f)
{
    // 指数部を 127 にする
    u32 u = (F32AsU32(f) & 0x807FFFFF) | 0x3F800000;
    return U32AsF32(u);
}



/*-------------------------------------------------------------------------*
  Name:         FSelect(f32, f32, f32)

  Description:  数の正負に応じて値を選択します。

  Arguments:    cond:   返り値を選択する条件。
                ifPos:  cond が0以上の場合の返り値。
                ifNeg:  cond が負の場合の返り値。

  Returns:      cond が 0 以上であれば ifPos を、そうでなければ
                ifNeg を返します。
 *-------------------------------------------------------------------------*/
// f32 FSelect(f32 cond, f32 ifPos, f32 ifNeg)
//
// 説明： 数の正負に応じて値を選択します。
// 引数： cond:   返り値を選択する条件。
//          ifPos:  cond が0以上の場合の返り値。
//          ifNeg:  cond が負の場合の返り値。
// 返値： cond が 0 以上であれば ifPos を、そうでなければ ifNeg を返します。
NW4R_MATH_INLINE f32
FSelect(register f32 cond, register f32 ifPos, register f32 ifNeg)
{
    register f32 ret;

#if defined(NW4R_MATH_BROADWAY)
    asm{ fsel   ret, cond, ifPos, ifNeg }
#else
    ret = (cond >= 0) ? ifPos: ifNeg;
#endif

    return ret;
}



/*-------------------------------------------------------------------------*
  Name:         FAbs(f32)

  Description:  絶対値を求めます。

  Arguments:    x:  絶対値を求める値。

  Returns:      x の絶対値を返します。
 *-------------------------------------------------------------------------*/
// f32 FAbs(f32 x)
//
// 説明： 絶対値を求めます
// 引数： x:  絶対値を求める値
// 返値： x の絶対値を返します
NW4R_MATH_INLINE f32
FAbs(register f32 x)
{
    register f32 ret;

#if defined(NW4R_MATH_BROADWAY)
    asm{ fabs   ret, x }
#else
    ret = std::fabsf(x);
#endif

    return ret;
}



/*-------------------------------------------------------------------------*
  Name:         FNAbs(f32)

  Description:  絶対値が等しく、符号が負である数を求めます。

  Arguments:    x:  元の数。

  Returns:      x と絶対値が等しく、符号が負である数を返します。
 *-------------------------------------------------------------------------*/
// f32 FNAbs(f32 x)
//
// 説明： 絶対値が等しく、符号が負である数を求めます
// 引数： x:  元の数。
// 返値： x と絶対値が等しく、符号が負である数を返します
NW4R_MATH_INLINE f32
FNAbs(register f32 x)
{
    register f32 ret;

#if defined(NW4R_MATH_BROADWAY)
    asm{ fnabs  ret, x }
#else
    ret = - FAbs(x);
#endif

    return ret;
}



/*-------------------------------------------------------------------------*
  Name:         FCopySign(f32, f32)

  Description:  一方の数の符号を他方の数にコピーします。

  Arguments:    abs:    絶対値の元となる数。
                sign:   符号を採用する数。

  Returns:      abs の絶対値と sign の符号を持った数を返します。
 *-------------------------------------------------------------------------*/
// FCopySign(f32 abs, f32 sign)
//
// 説明： 一方の数の符号を他方の数にコピーします
// 引数： abs:    絶対値の元となる数。
//          sign:   符号を採用する数。
// 返値： abs の絶対値と sign の符号を持った数を返します。
NW4R_MATH_INLINE f32
FCopySign(register f32 abs, register f32 sign)
{
    f32 pos = FAbs(abs);
    f32 neg = FNAbs(abs);

    return FSelect(sign, pos, neg);
}

namespace detail
{
    f32 FExp(f32 x);
    f32 FLog(f32 x);
}



/*-------------------------------------------------------------------------*
  Name:         FExp(f32)

  Description:  e^x を求めます。

  Arguments:    x:  指数の値。

  Returns:      e^x を返します。
 *-------------------------------------------------------------------------*/
// f32 FExp(f32 x)
//
// 説明： e^x を求めます
// 引数： x:  指数の値
// 返値： e^x を返します
NW4R_MATH_INLINE f32
FExp(f32 x)
{
#if defined(NW4R_MATH_BROADWAY)
    return detail::FExp(x);
#else
    return std::expf(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FLog(f32)

  Description:  自然対数を求めます。

  Arguments:    x:  自然対数を求める値。

  Returns:      x の自然対数を返します。
 *-------------------------------------------------------------------------*/
// f32 FLog(f32 x)
//
// 説明： 自然対数を求めます
// 引数： x:  自然対数を求める値
// 返値： x の自然対数を返します
NW4R_MATH_INLINE f32
FLog(f32 x)
{
    NW4R_MATH_WARNING(x > 0, "FLog: Input is out of the domain.");

#if defined(NW4R_MATH_BROADWAY)
    return (x > 0) ? detail::FLog(x): F_NAN;
#else
    return std::logf(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FLog10(f32)

  Description:  常用対数を求めます。

  Arguments:    x:  常用対数を求める値。

  Returns:      x の常用対数を返します。
 *-------------------------------------------------------------------------*/
// f32 FLog10(f32 x)
//
// 説明： 常用対数を求めます
// 引数： x:  常用対数を求める値
// 返値： x の常用対数を返します
NW4R_MATH_INLINE f32
FLog10(f32 x)
{
    NW4R_MATH_WARNING(x > 0, "FLog10: Input is out of the domain.");

//#if defined(NW4R_MATH_BROADWAY)
#if 0
#else
    return std::log10f(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FMod(f32, f32)

  Description:  剰余を計算します。

  Arguments:    x:  割られる数。
                y:  割る数。

  Returns:      x / y の剰余を返します。
 *-------------------------------------------------------------------------*/
// f32 FMod(f32 x, f32 y)
//
// 説明： 剰余を計算します
// 引数： x:  割られる数
//          y:  割る数
// 返値： x / y の剰余を返します
NW4R_MATH_INLINE f32
FMod(f32 x, f32 y)
{
//#if defined(NW4R_MATH_BROADWAY)
#if 0
#else
    return std::fmodf(x, y);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FModf(f32, f32*)

  Description:  浮動小数点数を整数部と小数部に分けます。
                整数部、小数部共に x と同じ符号を持ちます。

  Arguments:    x:  元の浮動小数点数。
                y:  整数部を格納するバッファへのポインタ。

  Returns:      小数部を返します。
 *-------------------------------------------------------------------------*/
// f32 FModf(f32 x, f32* y)
//
// 説明： 浮動小数点数を整数部と小数部に分けます。整数部、小数部共に x と同じ符号を持ちます。
// 引数： x:  元の浮動小数点数。
//          y:  整数部を格納するバッファへのポインタ。
// 返値： 小数部を返します
NW4R_MATH_INLINE f32
FModf(f32 x, f32* y)
{
//#if defined(NW4R_MATH_BROADWAY)
#if 0
#else
    return std::modff(x, y);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FCeil(f32)

  Description:  x を下回らない最小の整数値を求めます。

  Arguments:    x:  対象の値。

  Returns:      x を下回らない最小の整数値を返します。
 *-------------------------------------------------------------------------*/
// f32 FCeil(f32 x)
//
// 説明： x を下回らない最小の整数値を求めます
// 引数： x:  対象の値
// 返値： x を下回らない最小の整数値を返します
NW4R_MATH_INLINE f32
FCeil(f32 x)
{
//#if defined(NW4R_MATH_BROADWAY)
#if 0
#else
    return std::ceilf(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FFloor(f32)

  Description:  x を超えない最大の整数値を求めます。

  Arguments:    x:  対象の値

  Returns:      x を超えない最大の整数値を返します。
 *-------------------------------------------------------------------------*/
// f32 FFloor(f32 x)
//
// 説明： x を超えない最大の整数値を求めます
// 引数： x:  対象の値
// 返値： x を超えない最大の整数値を返します
NW4R_MATH_INLINE f32
FFloor(f32 x)
{
//#if defined(NW4R_MATH_BROADWAY)
#if 0
#else
    return std::floorf(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         F32ToS16(f32)

  Description:  f32型からs16型に変換します。

  Arguments:    x:  s16型に変換する値

  Returns:      x と同じ値を持ったs16型の値を返します。
 *-------------------------------------------------------------------------*/
// s16 F32ToS16(f32 x)
//
// 説明： f32型からs16型に変換します
// 引数： x:  s16型に変換する値
// 返値： x と同じ値を持ったs16型の値を返します
NW4R_MATH_INLINE s16
F32ToS16(f32 x)
{
#if defined(NW4R_MATH_BROADWAY)
    s16 rval;
    OSf32tos16(&x, &rval);
    return rval;
#else
    return s16(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         F32ToU16(f32)

  Description:  f32型からu16型に変換します。

  Arguments:    x:  u16型に変換する値

  Returns:      x と同じ値を持ったu16型の値を返します。
 *-------------------------------------------------------------------------*/
// u16 F32ToU16(f32 x)
//
// 説明： f32型からu16型に変換します
// 引数： x:  u16型に変換する値
// 返値： x と同じ値を持ったu16型の値を返します
NW4R_MATH_INLINE u16
F32ToU16(f32 x)
{
#if defined(NW4R_MATH_BROADWAY)
    u16 rval;
    OSf32tou16(&x, &rval);
    return rval;
#else
    return u16(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         U16ToF32(f32)

  Description:  u16型からf32型に変換します。

  Arguments:    x:  f32型に変換する値

  Returns:      x と同じ値を持ったf32型の値を返します。
 *-------------------------------------------------------------------------*/
// f32 U16ToF32(u16 x)
//
// 説明： u16型からf32型に変換します
// 引数： x:  f32型に変換する値
// 返値： x と同じ値を持ったf32型の値を返します
NW4R_MATH_INLINE f32
U16ToF32(u16 x)
{
#if defined(NW4R_MATH_BROADWAY)
    f32 rval;
    OSu16tof32(&x, &rval);
    return rval;
#else
    return f32(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         S16ToF32(f32)

  Description:  s16型からf32型に変換します。

  Arguments:    x:  f32型に変換する値

  Returns:      x と同じ値を持ったf32型の値を返します。
 *-------------------------------------------------------------------------*/
// f32 S16ToF32(s16 x)
//
// 説明： s16型からf32型に変換します
// 引数： x:  f32型に変換する値
// 返値： x と同じ値を持ったf32型の値を返します
NW4R_MATH_INLINE f32
S16ToF32(s16 x)
{
#if defined(NW4R_MATH_BROADWAY)
    f32 rval;
    OSs16tof32(&x, &rval);
    return rval;
#else
    return f32(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FInv(f32)

  Description:  逆数を高速低精度に求めます。

  Arguments:    x:  逆数を求める値。

  Returns:      x の逆数を返します。
 *-------------------------------------------------------------------------*/
// f32 FInv(f32 x)
//
// 説明： 逆数を高速低精度に求めます
// 引数： x:  逆数を求める値
// 返値： x の逆数を返します
NW4R_MATH_INLINE f32
FInv(f32 x)
{
    register f32 ix;

#if defined(NW4R_MATH_BROADWAY)
    register const f32 rx = x;
    asm{ fres ix, rx }
#else
    ix = 1.0f / x;
#endif

    return ix;
}




/*-------------------------------------------------------------------------*
  Name:         FrSqrt(f32)

  Description:  平方根の逆数を求めます。

  Arguments:    x:  平方根の逆数を求める値。

  Returns:      x の平方根の逆数を返します。
 *-------------------------------------------------------------------------*/
f32 FrSqrt(f32 x);



/*-------------------------------------------------------------------------*
  Name:         FSqrt(f32)

  Description:  平方根を求めます。

  Arguments:    x:  平方根を求める値。

  Returns:      x の平方根を返します。
 *-------------------------------------------------------------------------*/
// f32 FSqrt(f32 x)
//
// 説明： 平方根を求めます
// 引数： x:  平方根を求める値
// 返値： x の平方根を返します
NW4R_MATH_INLINE f32
FSqrt(f32 x)
{
    NW4R_MATH_WARNING(x >= 0, "FSqrt: Input is out of the domain.");

#if defined(NW4R_MATH_BROADWAY)
    return (x <= 0) ? 0.0f : x * FrSqrt(x);
#else
    return std::sqrtf(x);
#endif
}



/*-------------------------------------------------------------------------*
  Name:         FCbrt(f32)

  Description:  立方根を求めます。

  Arguments:    x:  立方根を求める値。

  Returns:      x の立方根を返します。
 *-------------------------------------------------------------------------*/
// FCbrt(f32 x)
//
// 説明： 立方根を求めます
// 引数： x:  立方根を求める値
// 返値： x の立方根を返します
NW4R_MATH_INLINE f32
FCbrt(f32 x)
{
#if defined(NW4R_CW3)
    return std::cbrtf(x);
#else
    return std::pow(x, 1/3.f);
#endif
}



f32 Hermite(f32 p1, f32 t1, f32 p2, f32 t2, f32 s);
f32 Bezier(f32 p1, f32 p2, f32 p3, f32 p4, f32 s);
f32 CatmullRom(f32 p0, f32 p1, f32 p2, f32 p3, f32 s);


/* =======================================================================
        整数関連
   ======================================================================== */

// 非インライン関数

u32 CntBit1(u32 x);
u32 CntBit1(const u32* first, const u32* last);
u32 DistBit(const u32* first1, const u32* last1, const u32* first2);
u32 RevBit(u32 x);
int IExp(int x, u32 n);
u32 ILog10(u32 x);

namespace detail
{
u32 CntLz_(u32 x);
}


// u32 CntLz(u32 x)
//
// 説明： MSBからの連続する 0 のビットを数えます。
// 引数： x: 対象のビット列
// 返値： MSBからの連続する 0 のビット数を返します。
NW4R_MATH_INLINE u32
CntLz(register u32 x)
{
#if defined(NW4R_MATH_BROADWAY)
    register u32 result;
    asm
    {
        cntlzw result, x;
    }
    return result;
#else
    return detail::CntLz_(x);
#endif
}


// bool IsPwr2(int x)
//
// 説明： 整数値が2の整数乗かどうかを判定します。
// 引数： x: 判定対象の整数値
// 返値： x が 2 の累乗であれば true を、そうでなければ false を返します。
//          ただし x が 0 の場合は true を返します。
NW4R_MATH_INLINE bool IsPwr2(int x) { return 0 == (x & (x - 1)); }

// int Rightmost1(int x)
//
// 説明： 1 になっているビットで一番右(LSB側)のビットを取得します。
// 引数： x: 対象とするビット列
// 返値： x で一番右の 1 となっているビット位置のみが 1 となっているビット列を返します。
NW4R_MATH_INLINE int Rightmost1(int x) { return x & (-x); }

// int Rightmost0(int x)
//
// 説明： 0 になっているビットで一番右(LSB側)のビットを取得します。
// 引数： x: 対象とするビット列
// 返値： x で一番右の 0 となっているビット位置のみが 1 となっているビット列を返します。
NW4R_MATH_INLINE int Rightmost0(int x) { return ~x & (x + 1); }

// u32 DistBit(u32 x, u32 y)
//
// 説明： ビット列同士の距離を計算します。
// 引数： x: ビット列 1
//          y: ビット列 2
// 返値： ビット列 1 とビット列 2 の距離を返します。
NW4R_MATH_INLINE u32 DistBit(u32 x, u32 y) { return CntBit1(x ^ y); }

// u32 CntTz(u32 x)
//
// 説明： LSBからの連続する 0 のビットを数えます。
// 引数： x: 対象のビット列
// 返値： LSBからの連続する 0 のビット数を返します。
NW4R_MATH_INLINE u32 CntTz(u32 x) { return 32 - CntLz(~x & (x - 1)); }

// u32 ILog2(u32 x)
//
// 説明： 整数の 2 を底とする対数を計算し、結果を整数で返します。
// 引数： x: 真数
// 返値： 整数の 2 を底とする対数を整数で返します。
NW4R_MATH_INLINE u32 ILog2(u32 x) { return 31 - CntLz(x); }

// u32 FloorPwr2(u32 x)
//
// 説明： 与えられた整数以下の最大の 2 の累乗を計算します。
// 引数： x: 対象の整数
// 返値： x 以下の最大の2の累乗を返します。
NW4R_MATH_INLINE u32 FloorPwr2(u32 x) { return 0x80000000 >> CntLz(x); }

// u32 CeilPwr2(u32 x)
//
// 説明： 与えられた整数以上の最小の 2 の累乗を計算します。
// 引数： x: 対象の整数
// 返値： x 以上の最小の 2 の累乗を返します。
NW4R_MATH_INLINE u32 CeilPwr2(u32 x) { return 0x80000000 >> (CntLz(x - 1) - 1); }





}}  // nw4r::math

/* NW4R_MATH_ARITHMETIC_H_ */
#endif
