﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 *  小物なユーティリティインライン関数を定義します。
 */



#ifndef NW4R_UT_INLINES_H_
#define NW4R_UT_INLINES_H_


#include <dolphin/types.h>
#include <nw4r/misc.h>

namespace NW4R
{
namespace Font
{
namespace UnManaged
{



namespace
{
    /*-------------------------------------------------------------------------*
      Name:         Abs

      Description:  引数の絶対値を返します。

      Arguments:    a

      Returns:      a の絶対値
     *-------------------------------------------------------------------------*/
    template <typename T>
    inline T
    Abs(T a)
    {
        return (a < 0) ? static_cast<T>(-a) : a;
    }

    template<>
    inline f32
    Abs<f32>(register f32 a)
    {
#ifdef GEKKO
        register f32 ret;
        asm{ fabs   ret, a }
        return ret;
#else
        return (a < 0) ? -a : a;
#endif
    }


    /*-------------------------------------------------------------------------*
      Name:         Min

      Description:  2つの数値を比較して小さい方の値を返します。

      Arguments:    a, b:   比較する数値

      Returns:      a, b のうち小さい方の値
     *-------------------------------------------------------------------------*/
    template <typename T>
    inline T
    Min(T a, T b)
    {
        return (a > b) ? b: a;
    }



    /*-------------------------------------------------------------------------*
      Name:         Max

      Description:  2つの数値を比較して大きい方の値を返します。

      Arguments:    a, b:   比較する数値

      Returns:      a, b のうち大きい方の値
     *-------------------------------------------------------------------------*/
    template <typename T>
    inline T
    Max(T a, T b)
    {
        return (a < b) ? b: a;
    }


    /*-------------------------------------------------------------------------*
      Name:         Clamp

      Description:  lowからhightの範囲に収まる値を返します。

      Arguments:    x:      比較する数値
                    low:    最小値
                    high:   最大値

      Returns:      x < low ならば low, x > high ならば high さもなくば x
     *-------------------------------------------------------------------------*/
    template <typename T>
    inline T
    Clamp(T x, T low, T high)
    {
        return (x > high) ? high : ( (x < low) ? low : x );
    }



    /*-------------------------------------------------------------------------*
      Name:         RoundUp

      Description:  x を base の倍数に切り上げます。

      Arguments:    x:      切り上げる数。
                    base:   切り上げの基準となる値。
                            2の累乗でなければなりません。

      Returns:      x を base の倍数に切り上げた値を返します。
     *-------------------------------------------------------------------------*/
    template <typename ValueT> ValueT RoundUp(ValueT x, unsigned int base);

    template <typename ValueT>
    inline void*
    RoundUp(ValueT* x, unsigned int base)
    {
        nw4r::IntPtr value = reinterpret_cast<nw4r::IntPtr>(x);
        nw4r::IntPtr rounded = (value + (base - 1)) & ~(base - 1);

        return reinterpret_cast<void*>(rounded);
    }

    template <typename ValueT>
    inline const void*
    RoundUp(const ValueT* x, unsigned int base)
    {
        nw4r::IntPtr value = reinterpret_cast<nw4r::IntPtr>(x);
        nw4r::IntPtr rounded = (value + (base - 1)) & ~(base - 1);

        return reinterpret_cast<const void*>(rounded);
    }

    template <typename ValueT>
    inline ValueT
    RoundUp(ValueT x, unsigned int base)
    {
        return static_cast<ValueT>( (x + (base - 1)) & ~(base - 1) );
    }


    /*-------------------------------------------------------------------------*
      Name:         RoundDown

      Description:  x を base の倍数に切り下げます。

      Arguments:    x:      切り下げる数。
                    base:   切り下げの基準となる値。2の累乗でなければなりません。

      Returns:      x を base の倍数に切り下げた値を返します。
     *-------------------------------------------------------------------------*/
    template <typename ValueT> ValueT RoundDown(ValueT x, unsigned int base);

    template <typename ValueT>
    inline void*
    RoundDown(ValueT* x, unsigned int base)
    {
        nw4r::IntPtr value = reinterpret_cast<nw4r::IntPtr>(x);
        nw4r::IntPtr rounded = value  & ~(base - 1);

        return reinterpret_cast<void*>(rounded);
    }

    template <typename ValueT>
    inline const void*
    RoundDown(const ValueT* x, unsigned int base)
    {
        nw4r::IntPtr value = reinterpret_cast<nw4r::IntPtr>(x);
        nw4r::IntPtr rounded = value  & ~(base - 1);

        return reinterpret_cast<const void*>(rounded);
    }

    template <typename ValueT>
    inline ValueT
    RoundDown(ValueT x, unsigned int base)
    {
        return static_cast<ValueT>( x & ~(base - 1) );
    }


    /*-------------------------------------------------------------------------*
      Name:         DivUp

      Description:  x を y で割って端数を切り上げます。

      Arguments:    x:  割られる数
                    y:  割る数

      Returns:      x を y で割って端数を切り上げた値を返します。
     *-------------------------------------------------------------------------*/
     inline int
     DivUp(int x, int y)
     {
        return (x + (y-1)) / y;
     }

    /*-------------------------------------------------------------------------*
      Name:         BitExtract

      Description:  ビット列から部分ビット列を抜き出します。

      Arguments:    bits:   抜き出し元のビット列
                    pos:    抜き出すビット列の最下位ビットのbits中での位置
                    len:    抜き出すビット長

      Returns:      抽出したビット列。
     *-------------------------------------------------------------------------*/
    template <typename OutputT>
    inline OutputT
    BitExtract(OutputT bits, int pos, int len=1)
    {
        const OutputT mask = static_cast<OutputT>( ( 1UL << len ) - 1 );
        return static_cast<OutputT>( (bits >> pos) & mask );
    }


    /*-------------------------------------------------------------------------*
      Name:         ReverseEndian

      Description:  エンディアンを反転します。

      Arguments:    x:      反転する数。

      Returns:      x のエンディアンを反転して返します。
     *-------------------------------------------------------------------------*/
    inline u32
    ReverseEndian( u32 x )
    {
        return static_cast<u32>(
            BitExtract( x,  0, 8 ) << 24 |
            BitExtract( x,  8, 8 ) << 16 |
            BitExtract( x, 16, 8 ) <<  8 |
            BitExtract( x, 24, 8 ) <<  0
        );
    }

    inline s32
    ReverseEndian( s32 x )
    {
        return static_cast<s32>(
            BitExtract( x,  0, 8 ) << 24 |
            BitExtract( x,  8, 8 ) << 16 |
            BitExtract( x, 16, 8 ) <<  8 |
            BitExtract( x, 24, 8 ) <<  0
        );
    }

    inline u16
    ReverseEndian( u16 x )
    {
        return static_cast<u16>(
            BitExtract( x,  0, 8 ) << 8 |
            BitExtract( x,  8, 8 ) << 0
        );
    }

    inline s16
    ReverseEndian( s16 x )
    {
        return static_cast<s16>(
            BitExtract( x,  0, 8 ) << 8 |
            BitExtract( x,  8, 8 ) << 0
        );
    }

    /*-------------------------------------------------------------------------*
      Name:         GetIntPtr

      Description:  ポインタからnw4r::IntPtrへのキャストをおこないます。

      Arguments:    ptr ポインタ

      Returns:      ptrをnw4r::IntPtrにキャストした値を返します。
     *-------------------------------------------------------------------------*/
    inline nw4r::IntPtr
    GetIntPtr( const void* ptr )
    {
        return reinterpret_cast<nw4r::IntPtr>( ptr );
    }


    /*-------------------------------------------------------------------------*
      Name:         GetOffsetFromPtr

      Description:  ２つのポインタアドレスのオフセット値をlongで取得します。

      Arguments:    start   開始アドレス
                    end     終了アドレス

      Returns:      ２つのポインタのオフセット値
     *-------------------------------------------------------------------------*/
    inline nw4r::PtrDiff
    GetOffsetFromPtr( const void* start, const void* end )
    {
        return static_cast<nw4r::PtrDiff>(GetIntPtr(end) - GetIntPtr(start));
    }


    /*-------------------------------------------------------------------------*
      Name:         AddOffsetToPtr

      Description:  ポインタにオフセット値を加えます。

      Arguments:    ptr     ポインタ
                    offset  オフセット値

      Returns:      ptrにoffsetを加えたアドレスを引数と同型のポインタとして返します。
     *-------------------------------------------------------------------------*/
    template <typename T>
    inline void*
    AddOffsetToPtr( void* ptr, T offset )
    {
        return reinterpret_cast<void*>( GetIntPtr(ptr) + offset );
    }
    template <typename T>
    inline const void*
    AddOffsetToPtr( const void* ptr, T offset )
    {
        return reinterpret_cast<const void*>( GetIntPtr(ptr) + offset );
    }

    inline void*
    AddU32ToPtr( void* ptr, u32 offset )
    {
        return AddOffsetToPtr( ptr, offset );
    }
    inline const void*
    AddU32ToPtr( const void* ptr, u32 offset )
    {
        return AddOffsetToPtr( ptr, offset );
    }

    /*-------------------------------------------------------------------------*
      Name:         ComparePtr

      Description:  ２つのポインタアドレスを比較します。

      Arguments:    a   比較元のポインタ
                    b   比較対象となるポインタ

      Returns:      > 0 aの指すアドレスがbの指すアドレスよりも大きい場合
                    = 0 aの指すアドレスとbの指すアドレスが等しい場合
                    < 0 aの指すアドレスがbの指すアドレスよりも小さい場合
     *-------------------------------------------------------------------------*/
    inline int
    ComparePtr( const void* a, const void* b )
    {
        return static_cast<int>( GetIntPtr( a ) - GetIntPtr( b ) );
    }

    /*-------------------------------------------------------------------------*
      Name:         NonCopyable

      Description:  コピー不可を表現するクラス
     *-------------------------------------------------------------------------*/
    class NonCopyable
    {
      protected:
        NonCopyable() {}
        ~NonCopyable() {}
      private:
        NonCopyable( const NonCopyable& );
        const NonCopyable& operator=( const NonCopyable& );
    };
}
        } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */



#endif //  NW4R_UT_INLINES_H_
