﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_FONTRESOURCEFORMAT_H_
#define NW4R_UT_FONTRESOURCEFORMAT_H_

#include <dolphin/types.h>
#include "binaryFileFormat.h"


#ifdef _MSC_VER // for VC
    #pragma warning( disable: 4200 )
    #pragma warning( disable: 4201 )
#endif  //_MSC_VER

namespace NW4R
{
namespace Font
{
namespace UnManaged
{

/* ------------------------------------------------------------------------
        シグネチャ定義
   ------------------------------------------------------------------------ */

// 通常フォントリソースファイルシグネチャ
const SigWord BINFILE_SIG_FONT              = NW4R_UT_MAKE_SIGWORD('F','F','N','T');

// オフセット解決済みバイナリファイルシグネチャ
const SigWord BINFILE_SIG_FONT_RESOLEVED    = NW4R_UT_MAKE_SIGWORD('F','F','N','U');

// アーカイブフォントリソースファイルシグネチャ
const SigWord BINFILE_SIG_FONTA             = NW4R_UT_MAKE_SIGWORD('F','F','N','A');

// バイナリブロックシグネチャ
const SigWord BINBLOCK_SIG_FINF             = NW4R_UT_MAKE_SIGWORD('F','I','N','F');
const SigWord BINBLOCK_SIG_CGLP             = NW4R_UT_MAKE_SIGWORD('C','G','L','P');
const SigWord BINBLOCK_SIG_TGLP             = NW4R_UT_MAKE_SIGWORD('T','G','L','P');
const SigWord BINBLOCK_SIG_CWDH             = NW4R_UT_MAKE_SIGWORD('C','W','D','H');
const SigWord BINBLOCK_SIG_CMAP             = NW4R_UT_MAKE_SIGWORD('C','M','A','P');
const SigWord BINBLOCK_SIG_KRNG             = NW4R_UT_MAKE_SIGWORD('K','R','N','G');

const SigWord BINBLOCK_SIG_GLGR             = NW4R_UT_MAKE_SIGWORD('G','L','G','R');
const SigWord BINBLOCK_SIG_HTGL             = NW4R_UT_MAKE_SIGWORD('H','T','G','L');

typedef u32    PtrOffset;
typedef u32    DataHead32;
typedef u16    DataHead16;

/* ------------------------------------------------------------------------
        リリースバージョン定義
   ------------------------------------------------------------------------ */

//
// バージョン情報
// Ver         変更点
// -------------------------------------
// 0.0         2005/08/22-
// 1.1         セルサイズに関する変更
// 1.2         height, width, ascent の追加
// 1.4         GLGP ブロック, 圧縮形式の追加
//

// リソースバージョンはプラットフォーム毎のDLLから取得するように変更しました。
#ifdef NW_PLATFORM_CTR
//const u32 FONT_FILE_VERSION         = NW_UT_MAKE_VERSION(4,0,0,0);
#elif NW_PLATFORM_CAFE
//const u32 FONT_FILE_VERSION         = NW_UT_MAKE_VERSION(4,0,0,0);
#else

#endif




/* ------------------------------------------------------------------------
        enum定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FontType

  Description:  フォントタイプ
 *---------------------------------------------------------------------------*/
enum FontType
{
    FONT_TYPE_GLYPH,            // BMP
    FONT_TYPE_NNGCTEXTURE,      // NNGC Texture
    NUM_OF_FONT_TYPE
};


/*---------------------------------------------------------------------------*
  Name:         FontEncoding

  Description:  文字列エンコーディングタイプ
 *---------------------------------------------------------------------------*/
enum FontEncoding
{
    FONT_ENCODING_UTF8,         // UTF-8
    FONT_ENCODING_UTF16,        // UTF-16
    FONT_ENCODING_SJIS,         // ShiftJIS
    FONT_ENCODING_CP1252,       // CP1252
    NUM_OF_FONT_ENCODING
};


/*---------------------------------------------------------------------------*
  Name:         FontMapMethod

  Description:  文字コードマッピングタイプ
 *---------------------------------------------------------------------------*/
enum FontMapMethod
{
    FONT_MAPMETHOD_DIRECT,      // インデックス = 文字コード - オフセット
    FONT_MAPMETHOD_TABLE,       // インデックス = mapInfo[文字コード - オフセット]
    FONT_MAPMETHOD_SCAN,        // インデックス = search(mapInfo, 文字コード)
    NUM_OF_FONT_MAPMETHOD
};


/*---------------------------------------------------------------------------*
  Name:         FontSheetFormat

  Description:  シートの形式 (テクスチャフォーマット)
 *---------------------------------------------------------------------------*/
enum FontSheetFormat
{
    FONT_SHEETFORMAT_TEXFMT_MASK     = 0x3FFF,  // GXTexFmt
    FONT_SHEETFORMAT_LINEAR_FLAG     = 0x4000,  // 1 ならリニア色空間用フォーマット
    FONT_SHEETFORMAT_COMPRESSED_FLAG = 0x8000   // 1 なら圧縮されている
};

/*---------------------------------------------------------------------------*
  Name:         FontTargetPlatform

  Description:  対象プラットフォーム
 *---------------------------------------------------------------------------*/
enum FontTargetPlatform
{
    FONT_FONTTARGETPLATFORM_CTR = 0,
    FONT_FONTTARGETPLATFORM_CAFE,
    NUM_OF_FONT_FONTTARGETPLATFORM_CAFE
};


/* ------------------------------------------------------------------------
        構造体定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWidths

  Description:  文字の各種幅
 *---------------------------------------------------------------------------*/
struct CharWidths
{
    s8 left;                    // 文字の左スペースの幅
    u8 glyphWidth;              // 文字のグリフ幅
    u8 charWidth;               // 文字の幅 = 左スペース幅 + グリフ幅 + 右スペース幅
};

/*---------------------------------------------------------------------------*
  Name:         CMapScanEntry

  Description:  文字コードとグリフインデックスペア
 *---------------------------------------------------------------------------*/
struct CMapScanEntry
{
    u32 ccode;                  // 文字コード
    u16 index;                  // グリフインデックス
    u16 padding;
};

/*---------------------------------------------------------------------------*
  Name:         CMapInfoScan

  Description:  MAPMETHOD_SCAN の場合の FontCodeMap.mapInfo
 *---------------------------------------------------------------------------*/
struct CMapInfoScan
{
    u16 num;              // entries の要素数
    u16 padding;
    CMapScanEntry entries[1];    // 文字コードからグリフインデックスへのマッピングリスト CMapScanEntry []
};


//---------------------------------------------------------------------------
//! @brief      一つ目の文字から二つ目のテーブルへのオフセットを
//! 			引くためのテーブルの要素です。
//---------------------------------------------------------------------------
struct KerningFirstTableElem
{
    u32 firstWord;		// 一つ目の文字
    u32 offset;			// 二つ目のテーブルへのオフセット。値はFontKerningTableの先頭アドレスからのオフセットを2で割ったものとなります。
};

//---------------------------------------------------------------------------
//! @brief      二つ目のテーブルの要素を表します。
//---------------------------------------------------------------------------
struct KerningSecondTableElem
{
    u32 secondWord;		// 二つ目の文字
    s16 kerningValue;	// カーニングの値
    u16 padding;
};

//---------------------------------------------------------------------------
//! @brief      二つ目のテーブルを表します。
//---------------------------------------------------------------------------
struct KerningSecondTable
{
    u16 secondWordNum;			        // このテーブルに含まれる二つ目の文字の種類
    u16 padding;
    KerningSecondTableElem	elems[1];	// 二分探索するための要素の配列。実際にはsecondWordNum個あります。
};



/* ------------------------------------------------------------------------
        ブロック本体定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FontGlyphGroups

  Description:  フォントを部分ロードするための情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontGlyphGroups
{
    u32 sheetSize;
    u16 glyphsPerSheet;
    u16 numSet;
    u16 numSheet;
    u16 numCWDH;
    u16 numCMAP;

    DataHead16 nameOffsets;  // numSet // u16[]
/*
    以下は可変長データが続きます

    (4 byte align)

    u32 sizeSheets[numSheet];       // 圧縮済みシートのサイズ
    u32 sizeCWDH[numCWDH];          // CWDH ブロックのサイズ
    u32 sizeCMAP[numCMAP];          // CMAP ブロックのサイズ
    u32 useSheets[numSet][numSheet/32];
    u32 useCWDH[numSet][numCWDH/32];
    u32 useCMAP[numSet][numCMAP/32];

    char names[numSet][name length];
*/
};



/*---------------------------------------------------------------------------*
  Name:         FontTextureGlyph

  Description:  フォントのグリフテクスチャを格納します。
 *---------------------------------------------------------------------------*/
struct FontTextureGlyph
{
    u8  cellWidth;              // セル幅=最大グリフ幅
    u8  cellHeight;             // セル高さ
    u8  sheetNum;               // シート数
    u8  maxCharWidth;           // 最大文字幅
    u32 sheetSize;              // テクスチャシートデータサイズ(byte単位)
    s16 baselinePos;            // ベースライン位置
    u16 sheetFormat;            // FontSheetFormat
    u16 sheetRow;               // シート内の横方向セル数
    u16 sheetLine;              // シート内の縦方向セル数
    u16 sheetWidth;             // シート幅(ピクセル単位)
    u16 sheetHeight;            // シート高さ(ピクセル単位)
    PtrOffset sheetImage;             // テクスチャデータへのポインタ // u8*
};

/*---------------------------------------------------------------------------*
  Name:         FontWidth

  Description:  各文字の文字幅情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontWidth
{
    u16                 indexBegin;     // widthTable の最初のエントリが対応するグリフインデックス
    u16                 indexEnd;       // widthTable の最後のエントリが対応するグリフインデックス
    PtrOffset           pNext;          // 次の FontWidth へのポインタ // FontWidth*
    DataHead32          widthTable;   // 幅情報の配列 // CharWidths []
};



/*---------------------------------------------------------------------------*
  Name:         FontCodeMap

  Description:  文字コードから、文字コードに対応するグリフの
                グリフイメージ配列中のインデックスへのマッピングを規定します。
 *---------------------------------------------------------------------------*/
struct FontCodeMap
{
    u32                 ccodeBegin;     // このブロックが担当する文字コード範囲の最初の文字コード
    u32                 ccodeEnd;       // このブロックが担当する文字コード範囲の最後の文字コード
    u16                 mappingMethod;  // マッピング方法 (FontMappingMethod型)
    u16                 reserved;       // 予約
    PtrOffset           pNext;          // 次の FontCodeMap へのポインタ // FontCodeMap*
    DataHead16           mapInfo;      // 文字コードマッピング情報 具体的な内容は mappingMethod に依る // u16[]
};

/*---------------------------------------------------------------------------*
  Name:         FontKerningTable

  Description:  カーニングの値を格納します。
 *---------------------------------------------------------------------------*/
struct FontKerningTable
{
    u16                     firstWordNum;   // 最初の文字が何パターンあるか
    KerningFirstTableElem	firstTable[1];	// 二分探索するための要素の配列。実際にはfirstWordNum個あります。
/*
    以下はサイズ可変のデータが続きます。

    (2 byte align)

    KerningSecondTable		secondTables[firstWordNum];
*/
};

/*---------------------------------------------------------------------------*
  Name:         FontInformation

  Description:  フォント全体に渡る情報を格納します。
 *---------------------------------------------------------------------------*/
struct FontInformation
{
    u8                  fontType;       // グリフデータタイプ (FontType型)
    u8                  height;         // フォントの高さ
    u8                  width;          // フォントの幅
    u8                  ascent;         // アセント
    s16                 linefeed;       // (*)== leading
    u16                 alterCharIndex; // (*)フォントに含まれない文字用のグリフのグリフインデックス
    CharWidths          defaultWidth;   // (*)文字幅情報を持たないグリフ用の文字幅情報
    u8                  encoding;       // 対応する文字列エンコーディング (FontEncoding型)
    PtrOffset           pGlyph;         // 唯一の FontGlyph へのポインタ // FontTextureGlyph*
    PtrOffset           pWidth;         // 最初の FontWidth へのポインタ // FontWidth*
    PtrOffset           pMap;           // 最初の FontCodeMap へのポインタ // FontCodeMap*
};



/* ------------------------------------------------------------------------
        ブロック定義
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         BinaryBlock

  Description:  NintendoWare 標準バイナリブロック
 *---------------------------------------------------------------------------*/
template <typename BlockBodyType>
struct BinaryBlock
{
    BinaryBlockHeader header;       // ブロックヘッダ
    BlockBodyType     body;         // ブロック本体
};

typedef BinaryBlock<FontGlyphGroups>    FontGlyphGroupsBlock;
typedef BinaryBlock<FontInformation>    FontInformationBlock;
typedef BinaryBlock<FontTextureGlyph>   FontTextureGlyphBlock;
typedef BinaryBlock<FontWidth>          FontWidthBlock;
typedef BinaryBlock<FontCodeMap>        FontCodeMapBlock;


        } // UnManaged
    } // namespace ut
} // namespace nw4r


#ifdef _MSC_VER
    #pragma warning( default: 4200 )
    #pragma warning( default: 4201 )
#endif  //_MSC_VER


#endif //  NW4R_UT_FONTRESOURCEFORMAT_H_
