﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_RESFONTBASE_H_
#define NW4R_UT_RESFONTBASE_H_

#include <dolphin/types.h>
#include "Font.h"
#include "fontResourceFormat.h"

namespace NW4R
{
namespace Font
{
namespace UnManaged
{
namespace detail
{


class ResFontBase
    : public Font
{
public:
    static const int            INVALID_IMAGE_PTR = 0x1;

    //! @brief 1 つのビットマップフォントに設定できる使用文字コード範囲のセットの最大個数です。
    static const int CharCodeRangeCountMax = 16;

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ/デストラクタ
                                ResFontBase();
    virtual                     ~ResFontBase();

    // フォント情報関連の初期化
    void* GetRgbaSheetImageBase() { return mRgbaSheetImageBase;}
    void InitializeRGBAImage(void* pRGBAImage) { mRgbaSheetImageBase = pRGBAImage;}
    bool IsReadyToDraw() const { return mRgbaSheetImageBase != NULL; }
    bool IsInvalid() const { return mRgbaSheetImageBase == (void*)INVALID_IMAGE_PTR; }

    //---- フォント全体情報アクセサ
    char*                       GetFilePtrOffsetBase()  { return m_pPtrOffsetBase; }
    FontInformation*            GetFINF()       { return mFontInfo; }
    const FontInformation*      GetFINF() const { return mFontInfo; }
    virtual int                 GetWidth() const;
    virtual int                 GetHeight() const;
    virtual int                 GetAscent() const;
    virtual int                 GetDescent() const;
    virtual int                 GetBaselinePos() const;
    virtual int                 GetCellHeight() const;
    virtual int                 GetCellWidth() const;
    virtual int                 GetMaxCharWidth() const;
    virtual Type                GetType() const;
    virtual GXTexFmt            GetTextureFormat() const;
    virtual int                 GetLineFeed() const;
    virtual const CharWidths    GetDefaultCharWidths() const;

    virtual void                SetDefaultCharWidths(
                                    const CharWidths& widths        // 新しいデフォルト幅
                                );

    virtual bool                SetAlternateChar(
                                    CharCode c );                   // 新しい代替文字

    virtual void                SetLineFeed(
                                    int linefeed );                 // 新しい改行幅

    //---- 文字単体情報アクセサ
    virtual int                 GetCharWidth(
                                    CharCode c                      // 幅を求める文字
                                ) const;

    virtual const CharWidths    GetCharWidths(
                                    CharCode c                      // 幅を求める文字
                                ) const;

    virtual void                GetGlyph(
                                    Glyph*      glyph,              // グリフ情報を受け取るバッファ
                                    CharCode    c                   // グリフ情報を取得する文字
                                ) const;

    virtual bool                HasGlyph(CharCode c) const;     // グリフ情報があるか

    virtual int            GetKerning(CharCode c0, CharCode c1) const;
    virtual bool           HasKerning() const; // カーニングの値を持っているか？
    virtual FontType        GetFontType() const;

    //---- 文字ストリーム処理
    virtual FontEncoding        GetEncoding() const;

    //---- フォント全体情報アクセサ
    // 追加：
    const FontTextureGlyph* GetFontTextureGlyph() const;

    //! @brief 使用文字コード範囲のセットを設定します。
    //!
    //! @param[in] count セットの配列の数です。
    //! @param[in] pFirst 文字コード範囲の先頭位置の配列です。
    //! @param[in] pLast 文字コード範囲の末尾位置の配列です。
    //!
    void SetCharCodeRange(int count, u32* pFirst, u32* pLast);

protected:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef u16                 GlyphIndex;


    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const GlyphIndex     GLYPH_INDEX_NOT_FOUND = 0xFFFF;

    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    #ifdef NW4R_UT_SUPPORT_OLD_FONT_1_1
        u8                          mWidth;
        u8                          mHeight;
        u8                          mAscent;
    #endif

    // リソース中のFINFブロック本体へのポインタ
    FontInformation*            mFontInfo;

    // 追加：
    // .NETモジュールのための RGBA8 形式に変換された画像バッファ
    // 常に作成されます。
    // SetResource() で 生成され、デストラクタで解放されます。
    void*                       mRgbaSheetImageBase;

    // 追加：
    // PtrOffset のベースとなるアドレス PtrOffset に加算して本来のポインタを復元します。
    char*                       m_pPtrOffsetBase;

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- メンバアクセス

    bool                        IsManaging(const void* ptr) const { return mResource == ptr; }
    void                        SetResourceBuffer(
        void*               pUserBuffer,
        FontInformation*    pFontInfo,
        FontKerningTable*	pKerningTable);
    void*                       RemoveResourceBuffer();


    //---- グリフインデックス
    GlyphIndex                  GetGlyphIndex(
                                    CharCode c                      // インデックスを求める文字
                                ) const;

    GlyphIndex                  FindGlyphIndex(
                                    CharCode c                      // インデックスを求める文字
                                ) const;

    GlyphIndex                  FindGlyphIndex(
                                    const FontCodeMap*  pMap,       // 探索対象のコードテーブル
                                    CharCode           c            // インデックスを求める文字
                                ) const;

    const CharWidths&           GetCharWidthsFromIndex(
                                    GlyphIndex  index               // 対象のグリフインデックス
                                ) const;

    const CharWidths&           GetCharWidthsFromIndex(
                                    const FontWidth*    pWidth,     // 探索対象のコードテーブル
                                    GlyphIndex          index       // インデックスを求める文字
                                ) const;

    void                        GetGlyphFromIndex(
                                    Glyph*      glyph,              // グリフ情報を受け取るバッファ
                                    GlyphIndex  index               // 対象のグリフインデックス
                                ) const;

    // 追加：.NET モジュールでテクスチャの検索を行う際に利用します。
    virtual int           GetSheetNum() const;

private:
    // 指定した文字コードが CharCodeRange の中に含まれているかを判別します。
    bool CheckCharCodeRange(u32 code) const;

    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    // リソースへのポインタ
    void*                       mResource;
    const FontKerningTable*	m_pKerningTable; //!< カーニングの情報を保持するテーブル

    int m_CharCodeRangeCount;
    u32 m_CharCodeRangeFirst[CharCodeRangeCountMax];
    u32 m_CharCodeRangeLast[CharCodeRangeCountMax];
};




        } /* namespace detail */
    } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */
#endif //  NW4R_UT_RESFONTBASE_H_
