﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using NintendoWare.Font.Runtime;
    using NintendoWare.Font.Win32;
    using System.Xml.Serialization;

    public partial class Program
    {
        private const string OptionHelp = "h";
        private const string OptionPlatform = "p";
        private const string OptionTileMode = "tile-mode";
        private const string OptionNoBntx = "nobntx";
        private const string OptionLogSilent = "silent";
        private const string OptionConvertCtr2ToCafe = "ctr2cafe";
        private const string OptionSkipValidation = "skipvalidation";
        private const string OptionUseDtd = "usedtd";
        public const string OptionInput = "i";
        public const string OptionFilter = "f";
        public const string OptionOutput = "o";
        private const string OptionInputAnti = "ia";
        private const string OptionInputUseScfontKerning = "isck";
        private const string OptionInputFormat = "ic";
        public const string OptionInputFile = "if";
        private const string OptionInputFile2 = "ig";
        private const string OptionInputSystemFont = "isysfont";
        private const string OptionInputName = "in";
        private const string OptionInputFaceName = "iface";
        public const string OptionInputOrder = "io";
        private const string OptionInputAverageWidth = "isw";
        private const string OptionInputWeight = "iwg";
        private const string OptionInputSize = "is";
        private const string OptionInputWidthType = "it";
        private const string OptionInputWidth = "iw";
        private const string OptionOutputAlter = "oa";
        private const string OptionOutputBottom = "ob";
        private const string OptionOutputBackgroundColor = "ocb";
        private const string OptionOutputGridColor = "ocg";
        private const string OptionOutputMarginColor = "ocm";
        private const string OptionOutputWidthBarColor = "ocw";
        private const string OptionOutputNullBlockColor = "ocn";
        private const string OptionOutputEncoding = "oe";
        public const string OptionOutputFile = "of";
        private const string OptionOutputDrawGrid = "og";
        private const string OptionOutputKerning = "ok";
        private const string OptionOutputNoKerning = "onok";

        private const string OptionOutputHeight = "oh";
        private const string OptionOutputLinefeed = "oh";
        private const string OptionOutputFormat = "oi";
        private const string OptionOutputLeft = "ol";
        private const string OptionOutputMarginX = "om";
        private const string OptionOutputMarginY = "on";
        private const string OptionOutputOrder = "oo";
        private const string OptionOutputGroup = "op";
        private const string OptionOutputRight = "or";
        private const string OptionOutputSheetPixels = "os";
        private const string OptionOutputTop = "ot";
        private const string OptionOutputWidth = "ow";
        private const string OptionOutputSheetSize = "ox";
        public const string OptionNoCache = "nocache";
        private const string OptionIgnoreFilter = "ignore-filter";

        private const string InputBmp = "bmp";
        private const string InputImage = "image";
        public const string InputImageFontConvertSettings = "ffnt";
        private const string InputFont = LibFormat.ExtensionFont;
        private const string InputWin = "win";
        private const string InputLC = "lc";
        private const string OutputBMP = "bmp";
        public const string OutputImage = "image";
        public const string OutputFont = LibFormat.ExtensionFont;
        private const string OutputOrder = "xlor";
        private const string OutputFormat = "format";
        private const string GpuEncodingEnabledOptions = "gpu-encoding";

        private const int DefFontSize = -1;
        private const int DefFontWeight = 400;

        private static SymbolicValue[] EncodingSymbol;
        private static readonly SymbolicValue[] WidthTypeSymbol =
        {
            new SymbolicValue("glyph",  WidthType.GlyphOnly),
            new SymbolicValue("keepsp", WidthType.GlyphOnlyKeepSpace),
            new SymbolicValue("char",   WidthType.IncludeMargin),
            new SymbolicValue("fixed",  WidthType.Fixed),
        };

        private static string[] PlatformNameSymbol;
        private static SymbolicValue[] GlyphImageFormatSymbol;
        private static readonly SymbolicValue[] ImageFormatSymbol =
        {
            new SymbolicValue("bmp", ImageFileFormat.Bmp),
            new SymbolicValue("tga", ImageFileFormat.Tga),
            new SymbolicValue("tif", ImageFileFormat.Tif),
            new SymbolicValue("ext", ImageFileFormat.Ext),
        };

        public static readonly CmdLine.OptionDef[] Options =
        {
            new CmdLine.OptionDef(OptionHelp,                     CmdLine.Option.None),
            new CmdLine.OptionDef(OptionPlatform,                 CmdLine.Option.One),
            new CmdLine.OptionDef(OptionTileMode,                 CmdLine.Option.One),
            new CmdLine.OptionDef(OptionNoBntx,                 CmdLine.Option.None),
            new CmdLine.OptionDef(OptionLogSilent,                 CmdLine.Option.None),
            new CmdLine.OptionDef(OptionConvertCtr2ToCafe,            CmdLine.Option.None),
            new CmdLine.OptionDef(OptionSkipValidation,            CmdLine.Option.None),
            new CmdLine.OptionDef(OptionUseDtd,                       CmdLine.Option.None),
            new CmdLine.OptionDef(OptionInput,                        CmdLine.Option.One),
            new CmdLine.OptionDef(OptionFilter,                       CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutput,                       CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputFile,                   CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputSystemFont,              CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputOrder,              CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputFormat,             CmdLine.Option.One, CheckOptionInputFormatWarning_),
            new CmdLine.OptionDef(OptionInputName,                   CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputFaceName,           CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputSize,                   CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputAverageWidth,           CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputWidthType,          CmdLine.Option.LTOne),
            new CmdLine.OptionDef(OptionInputWidth,              CmdLine.Option.One),
            new CmdLine.OptionDef(OptionInputAnti,               CmdLine.Option.None),
            new CmdLine.OptionDef(OptionInputUseScfontKerning,   CmdLine.Option.None),
            new CmdLine.OptionDef(OptionOutputFile,              CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputOrder,             CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputDrawGrid,          CmdLine.Option.None),
            new CmdLine.OptionDef(OptionOutputKerning,           CmdLine.Option.None),
            new CmdLine.OptionDef(OptionOutputNoKerning,           CmdLine.Option.None),
            new CmdLine.OptionDef(OptionOutputAlter,             CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputLinefeed,          CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputTop,                   CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputBottom,                CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputLeft,              CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputRight,             CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputWidth,             CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputEncoding,          CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputHeight,                CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputMarginX,          CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputMarginY,          CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputFormat,                CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputGroup,               CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputSheetSize,           CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputSheetPixels,         CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputBackgroundColor,    CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputGridColor,          CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputMarginColor,        CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputWidthBarColor,      CmdLine.Option.One),
            new CmdLine.OptionDef(OptionOutputNullBlockColor,     CmdLine.Option.One),
            new CmdLine.OptionDef(OptionNoCache,                  CmdLine.Option.None),
            new CmdLine.OptionDef(OptionIgnoreFilter,             CmdLine.Option.None),
            new CmdLine.OptionDef(GpuEncodingEnabledOptions,     CmdLine.Option.One),
        };

        public static HashSet<string> skipCacheSourceOptions = new HashSet<string>{
            OptionInputFile,
            OptionInputOrder,
            OptionFilter,
            OptionOutputFile,
        };

        /// <summary>
        /// A4 A6 LA4 LA6 に対する警告チェック
        /// </summary>
        private static string CheckOptionInputFormatWarning_(CmdLine options, StringList values)
        {
            if (ConverterEnvironment.PlatformDetails.CanHWSupports4BitFormat)
            {
                return string.Empty;
            }

            if (values.Count != 1)
            {
                return string.Empty;
            }

            if (_tcscmp(options[OptionInput], InputWin) != 0)
            {
                return string.Empty;
            }

            if (values[0] == "A4" || values[0] == "LA4")
            {
                return string.Format("[{0}] --- A obsolete param was detected. This option is only used for backward compatibility.", values[0]);
            }

            return string.Empty;
        }

        /// <summary>
        /// デフォルト文字幅
        /// </summary>
        private const WidthType DefaultWidthType = WidthType.IncludeMargin;

        /// <summary>
        /// デフォルトフォントエンコーディング
        /// </summary>
        private const CharEncoding DefaultFontEncoding = CharEncoding.UTF16;

        /// <summary>
        /// デフォルトイメージフォーマット
        /// </summary>
        private const ImageFileFormat DefaultImageFileFormat = ImageFileFormat.Ext;

        private static IPlatformDetails _platformDetails = new NullPlatformDetails();
        private static bool IsTemporaryFontLoaded = false;

        static Program()
        {
            InitEncodingSymbol_();
            InitPlatformNameSymbol_();

            // プラットフォームが決定するまでは、仮の値で初期化しておく。
            GlyphImageFormatSymbol = new SymbolicValue[0];
        }

        /// <summary>
        /// EncodingSymbol を初期化します。
        /// </summary>
        private static void InitEncodingSymbol_()
        {
            var symList = new List<SymbolicValue>();

            symList.Add(new SymbolicValue("sjis", CharEncoding.SJIS));
            if (ConverterEnvironment.IsCtr)
            {
                symList.Add(new SymbolicValue("uincode", CharEncoding.UTF16));// タイポ対策
                symList.Add(new SymbolicValue("unicode", CharEncoding.UTF16));
            }
            else
            {
                symList.Add(new SymbolicValue("utf8", CharEncoding.UTF8));
                symList.Add(new SymbolicValue("utf16", CharEncoding.UTF16));
            }
            symList.Add(new SymbolicValue("cp1252", CharEncoding.CP1252));

            EncodingSymbol = symList.ToArray();
        }

        /// <summary>
        /// GlyphImageFormatSymbol を初期化します。
        /// </summary>
        private static void InitGlyphImageFormatSymbol_()
        {
            var symList = new List<SymbolicValue>();
            foreach (var formatDesc in ConverterEnvironment.PlatformDetails.GetImageInputFormatFormatList())
            {
                if (symList.Find((item) => item.Value == (int)formatDesc.GlyphImageFormat) != null) { continue; }
                symList.Add(new SymbolicValue(formatDesc.GlyphImageFormat.ToString(), formatDesc.Description, formatDesc.GlyphImageFormat));
            }

            GlyphImageFormatSymbol = symList.ToArray();
        }

        private static void InitPlatformNameSymbol_()
        {
            var symList = new List<string>();
            foreach (var platformName in ConverterEnvironment.EnumratePlatFormNames(ConverterEnvironment.PlatformDllPrefix))
            {
                if (platformName == ConverterEnvironment.GenericPlatformDllName)
                {
                    symList.Add("Win");
                    continue;
                }

                symList.Add(platformName);
            }

            PlatformNameSymbol = symList.ToArray();
        }

        /// <summary>
        /// cygwin の仕様変更によって、1.7.10 以降の bash からツールが実行された場合、tmp と TMP など大文字・小文字違いの同名
        /// 環境変数が定義され、XmlSerializer内部で例外が発生してしまう問題があります。
        /// 対策として、小文字の環境変数を削除します。
        /// </summary>
        static void RemoveMultipleEnvironmentVariableWorkaround_(string envName)
        {
            // 環境変数を取得する。
            string envVal = Environment.GetEnvironmentVariable(envName);
            if (string.IsNullOrEmpty(envVal))
            {
                return;
            }

            // 手始めに一つ削除して、その後再度取得してみる。
            Environment.SetEnvironmentVariable(envName, null);
            string envVal2 = Environment.GetEnvironmentVariable(envName);
            if (!string.IsNullOrEmpty(envVal2))
            {
                // 2つ目が取得できた場合は、多重定義されている状況。
                // POSIX スタイルの環境変数を削除するようにする。
                if (envVal2.StartsWith("/"))
                {
                    Environment.SetEnvironmentVariable(envName, envVal);
                }
                return;
            }
            else
            {
                // 多重定義がなかったので、削除した環境変数を元に元しておく。
                Environment.SetEnvironmentVariable(envName, envVal);
            }
        }

        public static int Main(string[] args)
        {
            //System.Diagnostics.Stopwatch sw = new System.Diagnostics.Stopwatch();
            //sw.Start();

            //--------------------------------------------
            // cygwin 挙動変更による環境変数多重定義の回避コード
            RemoveMultipleEnvironmentVariableWorkaround_("tmp");
            RemoveMultipleEnvironmentVariableWorkaround_("temp");

            GlCm.InitLanguage();

            CommandLineProgressControl.CreateInstance();
            FontData font = new FontData();
            CharFilter filter = new CharFilter();
            GlyphOrder outputOrder = new GlyphOrder();
            FontReader reader = null;
            FontWriter writer = null;

            Rpt.InitForConsole();

            ConvertImageFontCache cache = null;

            try
            {
                Rpt._RPT0("------- analyze command line \n");
                CmdLine cmd = new CmdLine();

                bool readFfnt = false;
                cmd.AnalizedAnOption += (o, e) =>
                {
                    switch (e.Option)
                    {
                        case OptionInput:
                        case OptionInputFile:
                            // イメージ変換設定の場合
                            if (!readFfnt &&
                                cmd.Exists(OptionInput) && _tcscmp(cmd[OptionInput], InputImageFontConvertSettings) == 0 &&
                                cmd.Exists(OptionInputFile))
                            {
                                readFfnt = true;

                                // オプションの追加
                                AddImageFontConvertOptions(cmd);
                            }
                            break;
                    }
                };

                cmd.Analyze(Environment.CommandLine, Options, Options.Length);

                // 引数が無い
                if (cmd.GetNum() == 0)
                {
                    ShowUsage();
                    return 1;
                }


                if (readFfnt)
                {
                    // キャッシュを有効にする
                    if (!cmd.Exists(OptionNoCache) && _tcscmp(cmd[OptionOutput], OutputFont) == 0)
                    {
                        cache = new ConvertImageFontCache();
                    }
                }

                // プラットフォームが指定された場合は、警告を表示して、タイルモードを適切に設定する
                string platformName = string.Empty;
                if (cmd.Exists(OptionPlatform))
                {
                    platformName = cmd[OptionPlatform];

                    Console.Error.Write(Strings.ErrorPlatformNameObsolete);
                    cmd.SetOptionValue(OptionTileMode, ConverterEnvironment.GetTileModeFromPlatformName(platformName));
                }
                else
                {
                    // タイルモードが指定された場合は、プラットフォームを適切に設定する
                    if (cmd.Exists(OptionTileMode))
                    {
                        if (string.Equals(cmd[OptionTileMode], "linear", StringComparison.CurrentCultureIgnoreCase))
                        {
                            platformName = "Win";
                        }
                        else if (string.Equals(cmd[OptionTileMode], "Cafe", StringComparison.CurrentCultureIgnoreCase))
                        {
                            platformName = "Cafe";
                        }
                        else if (string.Equals(cmd[OptionTileMode], "NX", StringComparison.CurrentCultureIgnoreCase))
                        {
                            platformName = "NX";
                        }
                        else
                        {
                            platformName = "Unknown";
                        }
                    }
                    else
                    {
                        // ヘルプを表示する。
                        if (cmd.Exists(OptionHelp))
                        {
                            ShowUsage();
                            return 1;
                        }

                        // タイルモードを指定してください。
                        Console.Error.Write(Strings.ErrorNoTileModeNameSpecified);
                        return 1;
                    }
                }

                // プラットフォーム固有情報の初期化
                {
                    System.Diagnostics.Debug.Assert(string.IsNullOrEmpty(platformName));
                    System.Diagnostics.Debug.Assert(cmd.Exists(OptionTileMode));

                    // Generic DLL で初期化を行います。
                    // ツール内部的でのプラットフォーム名は、指定されたプラットフォーム名を採用します。
                    // テクスチャコンバーターに渡されるプラットフォーム名も、このプラットフォーム名となります。
                    // ほとんどのプラットフォームは、Generic DLL によって処理されます。
                    // 現状、プラットフォーム固有DLL は存在していません。
                    bool platformInitResult = ConverterEnvironment.InitializeTargetPlatform(platformName, cmd[OptionTileMode], "Win", cache);
                    if (!platformInitResult)
                    {
                        Console.Error.Write(Strings.ErrorInvalidPlatformNameSpecified);
                        return 1;
                    }

                    if (cache != null)
                    {
                        cache.AddCacheKey((int)ConverterEnvironment.PlatformDetails.OutputFontVersion);
                    }

                    // 出力対象とするプラットフォームを決定する。
                    _platformDetails = ConverterEnvironment.PlatformDetails;
                    _platformDetails.CUISetup(args);

                    InitGlyphImageFormatSymbol_();
                }

                // BNTX を利用しないかどうか
                ConverterEnvironment.UseBntx = !cmd.Exists(OptionNoBntx);
                ConverterEnvironment.IsLogSilent = cmd.Exists(OptionLogSilent);

                // Gpuエンコーディングを利用するかどうか
                if(cmd.Exists(GpuEncodingEnabledOptions))
                {
                    ConverterEnvironment.IsGpuEncodingEnabled = cmd[GpuEncodingEnabledOptions];
                }

                // ヘルプを表示する。
                if (cmd.Exists(OptionHelp))
                {
                    ShowUsage();
                    return 1;
                }

                // オプションに対する事前処理
                foreach (NintendoWare.Font.CmdLine.OptionDef optionDef in Options)
                {
                    // キャッシュキーに追加
                    if (cache != null)
                    {
                        if (!skipCacheSourceOptions.Contains(optionDef.Name))
                        {

                            cache.AddCacheKey(optionDef.Name);
                        }
                    }

                    StringList optionValue = cmd.FindOptionValueOf(optionDef.Name);
                    if (optionValue != null)
                    {
                        // オプションに付随する、警告メッセージがあれば表示します。
                        string warningMesasge = optionDef.GetWarningMesasge(cmd, optionValue);
                        if (!string.IsNullOrEmpty(warningMesasge))
                        {
                            Console.Error.Write("WARNING : option {0} => {1}\n", optionDef.Name, warningMesasge);
                        }

                        // キャッシュキーに追加
                        if (cache != null)
                        {
                            if (!skipCacheSourceOptions.Contains(optionDef.Name))
                            {
                                cache.AddCacheKey(optionValue.Count);

                                foreach (var value in optionValue)
                                {
                                    cache.AddCacheKey(value);
                                }
                            }
                        }
                    }
                    else
                    {
                        // キャッシュキーに追加
                        if (cache != null)
                        {
                            if (!skipCacheSourceOptions.Contains(optionDef.Name))
                            {
                                // 後に続くキーの追加方法を自由にするために
                                // オプションが指定されていないことを示すキーを追加する
                                cache.AddCacheKey(-1);
                            }
                        }
                    }
                }

                // 環境変数に対する事前処理
                if (cache != null)
                {
                    // 動作に影響のある環境変数だけ扱う
                    var variable = "NINTENDO_TEXTURE_CONVERTER_NVTT_BC123";
                    cache.AddCacheKey(variable);
                    cache.AddCacheKey(Environment.GetEnvironmentVariable(variable) ?? string.Empty);
                }


                Rpt._RPT0("------- create font reader\n");
                reader = CreateReader(cmd, cache);


                // キャッシュキーに追加
                if (cache != null)
                {
                    if (cmd.Exists(OptionFilter))
                    {
                        cache.AddCacheKeyFromFile(cmd[OptionFilter]);
                    }
                    else
                    {
                        cache.AddCacheKey(0);
                    }
                }

                var cacheFilePath = cache != null ? cache.GetCacheFilePath(): null;
                var outputPath = cmd[OptionOutputFile];
                if (cacheFilePath != null)
                {
                    FontIO.ValidateOutputPath(outputPath);

                    // キャッシュからコピー
                    cache.WaitCacheFileMutex();
                    if (cache.CopyToOutput(outputPath))
                    {
                        cache.ClearCache();
                        Rpt._RPT0(string.Format("------- copied font from cache {0}\n", cacheFilePath));
                        return 0;
                    }

                    // オプションの差し替え
                    cmd.SetOptionValue(OptionOutputFile, Path.ChangeExtension(cacheFilePath, LibFormat.ExtensionFont));
                }

                Rpt._RPT0("------- craete font writer\n");
                writer = CreateWriter(cmd);

                // かなり重い処理をしているので、スキップするオプションをつけた。
                if (!cmd.Exists(OptionSkipValidation))
                {
                    Rpt._RPT0("------- validate reader\n");
                    reader.ValidateInput();

                    Rpt._RPT0("------- validate writer\n");
                    writer.ValidateInput();
                }

                Rpt._RPT0("------- read order file\n");
                writer.GetGlyphOrder(outputOrder);

                Rpt._RPT0("------- read filter file\n");
                if (!cmd.Exists(OptionIgnoreFilter) && cmd.Exists(OptionFilter))
                {
                    filter.Load(cmd[OptionFilter], cmd.Exists(OptionUseDtd));
                }

                filter.SetOrder(outputOrder);
                //// filter.DumpFilter();    // for debug

                // 元となるフォントデータを読み込む
                Rpt._RPT0("------- read font\n");
                reader.ReadFontData(font, filter);

                // フィルタでpassになっている文字がすべてあるか確認
                if (!cmd.Exists(OptionIgnoreFilter) && cmd.Exists(OptionFilter))
                {
                    filter.CheckEqual(font);
                }

                // フォントデータの調整
                Rpt._RPT0("------- restruction font\n");
                font.ReflectGlyph();

                // フォントデータ書き出し
                Rpt._RPT0("------- write font\n");
                writer.WriteFontData(font, outputOrder);

                if (cacheFilePath != null)
                {
                    // 拡張子をキャッシュ用ファイルのものに変更
                    File.Move(Path.ChangeExtension(cacheFilePath, LibFormat.ExtensionFont), cacheFilePath);

                    // ファイルはキャッシュ側に生成されるので、キャッシュからコピー
                    if (!cache.CopyToOutput(outputPath))
                    {
                        Rpt._RPT1("Error: failed to copy font from cache {0}\n", cacheFilePath);
                        return 0;
                    }
                    Rpt._RPT0(string.Format("------- copied font from cache {0}\n", cacheFilePath));

                    // キャッシュの整理
                    cache.ClearCache();
                }
            }
            catch (GeneralException ge)
            {
                Console.Error.Write(Strings.IDS_ERROR_MSG, ge.GetMsg());
                Rpt._RPT1("Error: {0} \n", ge.GetMsg());
                return 2;
            }
            finally
            {
                if (cache != null)
                {
                    cache.Dispose();
                }
            }

            //sw.Stop();
            //Console.WriteLine("total-time : " + sw.ElapsedMilliseconds);

            return 0;
        }

        private static int SearchSymbol(SymbolicValue[] table, string target, int defaultValue)
        {
            foreach (var sv in table)
            {
                if (string.Equals(target, sv.Symbol, StringComparison.OrdinalIgnoreCase))
                {
                    return sv.Value;
                }
            }

            return defaultValue;
        }

        private static void ShowUsage()
        {
            // 対応イメージフォーマットを正しく表示するため
            // 未初期化なら、プラットフォームを Generic 的なモードで初期化したあと、GlyphImageFormatSymbol を初期化する。
            if (ConverterEnvironment.PlatformDetails == null)
            {
                bool platformInitResult = ConverterEnvironment.InitializeTargetPlatform("Win", "linear", "Win");
                System.Diagnostics.Debug.Assert(platformInitResult);
                InitGlyphImageFormatSymbol_();

                _platformDetails = ConverterEnvironment.PlatformDetails;
            }

            // プロダクト情報を取得
            var productInfo = GlCm.GetProductInfo();

            Console.WriteLine(productInfo + " - Version {0}", GlCm.GetVersionString(_platformDetails != null ? _platformDetails.UIPlatformName: null));

            ShowUsageCore();
        }

        private static WidthType ToWidthType(string wt)
        {
            WidthType ret = (WidthType)SearchSymbol(WidthTypeSymbol, wt, (int)WidthType.NUM);

            if (ret == WidthType.NUM)
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_WIDTH_TYPE, wt);
            }

            return ret;
        }

        /// <summary>
        ///
        /// </summary>
        private static uint ToCharCode(string str, CharEncoding enc)
        {
            uint alter;

            if (str.Length == 1)
            {
                alter = str[0];
            }
            else
            {
                var local = (uint)GlCm.ParseHexNumber(str);

                alter = GlCm.EncodingToUnicode(enc, local);

                if (alter == Runtime.RtConsts.InvalidCharCode)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_TO_UNICODE, local);
                }
            }

            return alter;
        }

        /// <summary>
        /// エンコーディング指定文字列→enum値
        /// </summary>
        private static CharEncoding ToEncodingType(string str)
        {
            CharEncoding ret = (CharEncoding)SearchSymbol(EncodingSymbol, str, (int)CharEncoding.NUM);

            if (ret == CharEncoding.NUM)
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_ENCODING, str);
            }

            return ret;
        }

        /// <summary>
        ///
        /// </summary>
        private static GlyphImageFormat ToGlyphImageFormat(string str)
        {
            if (str == string.Empty)
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_NO_COLOR_FORMAT);
            }

            int ret = SearchSymbol(GlyphImageFormatSymbol, str, -1);

            if (ret == -1)
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNONW_COLOR_FORMAT, str);
            }

            return (GlyphImageFormat)ret;
        }

        /// <summary>
        ///
        /// </summary>
        private static ImageFileFormat ToImageFormat(string str)
        {
            var format = (ImageFileFormat)SearchSymbol(
                ImageFormatSymbol,
                str,
                (int)ImageFileFormat.NUM);

            if (format == ImageFileFormat.NUM)
            {
                // TODO
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_IMAGE_FORMAT_STR, str);
            }

            return format;
        }

        /// <summary>
        ///
        /// </summary>
        private static void ToSheetSize(out int width, out int height, string str)
        {
            var splits = str.Split('x');
            if (splits.Length == 2)
            {
                try
                {
                    width = int.Parse(splits[0]);
                    height = int.Parse(splits[1]);
                    return;
                }
                catch (FormatException)
                {
                }
                catch (OverflowException)
                {
                }
            }

            throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_INVALID_SHEET_SIZE_FORMAT);
        }

        /// <summary>
        ///
        /// </summary>
        private static void ReadBpp(string str, out int bpp, out bool hasAlpha)
        {
            var descList = ConverterEnvironment.PlatformDetails.GetWindowsFontInputFormatList();
            var fmtDesc = descList.FirstOrDefault((desc) => desc.Name == str);
            if (fmtDesc == null)
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_ILLEGAL_FORMAT_SYNTAX, str);
            }

            bpp = fmtDesc.Bpp;
            hasAlpha = fmtDesc.HasAlpha;
        }

        /// <summary>
        /// FontReaderを構築
        /// </summary>
        private static FontReader CreateReader(CmdLine cmd, ConvertImageFontCache cache)
        {
            FontReader reader = CreateReaderImpl_(cmd, cache);
            if (reader != null)
            {
                reader.UseDtdValidation = cmd.Exists(OptionUseDtd);
            }

            return reader;
        }

        /// <summary>
        /// FontReaderを構築
        /// </summary>
        private static FontReader CreateReaderImpl_(CmdLine cmd, ConvertImageFontCache cache)
        {
            if (_tcscmp(cmd[OptionInput], InputBmp) == 0 || _tcscmp(cmd[OptionInput], InputImage) == 0)
            {
                if (cache != null)
                {
                    // ハッシュのソースに登録
                    // TODO: もっとうまくやれば効率的になるはず
                    cache.AddCacheKeyFromFile(cmd[OptionInputFile]);
                    cache.AddCacheKeyFromFile(cmd[OptionInputOrder]);
                    cache.AddCacheKeyFromFile(Path.ChangeExtension(cmd[OptionInputFile], ".bffkn"));
                }

                return new ImageFontReader(
                        cmd[OptionInputFile],
                        cmd[OptionInputOrder],
                        ToGlyphImageFormat(cmd[OptionInputFormat]),
                        cmd.Exists(OptionInputAnti));
            }
            else if (_tcscmp(cmd[OptionInput], InputFont) == 0)
            {
                // -i bffnt -o format 以外の組み合わせはエラーとする
                if (_tcscmp(cmd[OptionOutput], OutputFormat) == 0)
                {
                    return _platformDetails.GetFontReader(cmd[OptionInputFile]);
                }

                throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_INPUT_FORMAT, cmd[OptionInput]);
            }
            else if (_tcscmp(cmd[OptionInput], InputWin) == 0)
            {
                string path = null; // フォントのファイルパス

                // bfttfファイルパス が入力として指定されているなら読み込む。
                if (ConverterEnvironment.PlatformDetails.BfttfEnabled)
                {
                    if (!string.IsNullOrEmpty(cmd[OptionInputSystemFont]))
                    {
                        string sysFontPath = cmd[OptionInputSystemFont];
                        if (!File.Exists(sysFontPath))
                        {
                            throw GlCm.ErrMsg(ErrorType.CmdLine, string.Format(Strings.SystemFontNotFound, sysFontPath));
                        }

                        bool bResult = GlCm.InitSystemFontFromBfttf(sysFontPath);
                        if (!bResult)
                        {
                            throw GlCm.ErrMsg(ErrorType.CmdLine, string.Format(Strings.SystemFontInvalid, sysFontPath));
                        }

                        // フォント名は強制的に上書きされる。
                        cmd.SetOptionValue(OptionInputName, GlCm.SystemFontFaceName);

                        path = sysFontPath;
                    }
                }

                // ttf/otf/bfttf/bfotf ファイルパスが直接指定されている場合を処理します。
                {
                    string filePath = cmd[OptionInputName];
                    string ext = Path.GetExtension(cmd[OptionInputName]);
                    if (!string.IsNullOrEmpty(ext))
                    {
                        if (!File.Exists(cmd[OptionInputName]))
                        {
                            throw GlCm.ErrMsg(ErrorType.CmdLine, string.Format(Strings.IDS_ERR_FILE_NOT_EXISTS, filePath));
                        }

                        if (string.Compare(ext, ".ttf", false) != 0 &&
                            string.Compare(ext, ".otf", false) != 0 &&
                            string.Compare(ext, ".bfttf", false) != 0 &&
                            string.Compare(ext, ".bfotf", false) != 0)
                        {
                            throw GlCm.ErrMsg(ErrorType.CmdLine, string.Format(Strings.ErrorInvalidScalableFontFormat, filePath));
                        }

                        HandleScalableFontFileInput_(cmd);

                        path = filePath;
                    }
                }

                WidthType wt = cmd.Exists(OptionInputWidthType) ? ToWidthType(cmd[OptionInputWidthType]) : DefaultWidthType;

                // Win -ic 指定時の処理。
                int bpp;
                bool hasAlpha;
                ReadBpp(cmd[OptionInputFormat], out bpp, out hasAlpha);

                if (!cmd.Exists(OptionInputSize))
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_FONT_SIZE_NOT_SPECIFIED);
                }

                if ((wt == WidthType.Fixed) && (!cmd.Exists(OptionInputWidth)))
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_FIXED_WIDTH_NOT_SPECIFIED);
                }


                // 平均幅指定
                int avgwidth = cmd.Exists(OptionInputAverageWidth) ? cmd.Number(OptionInputAverageWidth, 0) : 0;
                return new WinFontReader(
                        path,
                        cmd[OptionInputName],
                        cmd.Number(OptionInputSize, DefFontSize),
                        cmd.Number(OptionInputWeight, DefFontWeight),
                        avgwidth,
                        bpp,
                        hasAlpha,
                        wt,
                        cmd.Number(OptionInputWidth, -1),
                        cmd.Exists(OptionInputAnti),
                        cmd.Exists(OptionInputUseScfontKerning),
                        IsTemporaryFontLoaded,
                        cmd.Exists(OptionInputFaceName) ? cmd[OptionInputFaceName] : null);
            }
            else
            {
                if (cmd[OptionInput].Length == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_REQUIRE_INPUT_FORMAT);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_INPUT_FORMAT, cmd[OptionInput]);
                }
            }
        }

        /// <summary>
        /// ttf otf ファイルパスが指定された場合の処理です。
        /// </summary>
        private static void HandleScalableFontFileInput_(CmdLine cmd)
        {
            string filePath = cmd[OptionInputName];
            string ext = Path.GetExtension(filePath);
            string faceName = "";

            var beforeAddFontFonts = GlCm.GetInstalledFontNames();
            {
                // Gdi にフォントを追加する処理。
                // フォントフェース名も取得できたら取得する。
                if (string.Compare(ext, ".bfttf", false) == 0)
                {
                    byte[] ttfData = BfttfHelper.Convert(filePath);

                    uint font;
                    IntPtr res = Gdi.AddFontMemResourceEx(ttfData, ttfData.Length, IntPtr.Zero, out font);

                    faceName = AcquireFaceName(cmd, ttfData, filePath);

                    // MonoTypeHelper.GetFontFaceNameFromFilePath の処理に備えて、ttf 拡張子にしておく。
                    filePath = Path.ChangeExtension(filePath, "ttf");

                    IsTemporaryFontLoaded = true;
                }
                else if (string.Compare(ext, ".bfotf", false) == 0)
                {
                    byte[] otfData = BfttfHelper.Convert(filePath);

                    uint font;
                    IntPtr res = Gdi.AddFontMemResourceEx(otfData, otfData.Length, IntPtr.Zero, out font);

                    faceName = AcquireFaceName(cmd, otfData, filePath);

                    // MonoTypeHelper.GetFontFaceNameFromFilePath の処理に備えて、otf 拡張子にしておく。
                    filePath = Path.ChangeExtension(filePath, "otf");

                    IsTemporaryFontLoaded = true;
                }
                else
                {
                    int result = Gdi.AddFontResourceEx(filePath, Gdi.FR_PRIVATE, IntPtr.Zero);

                    faceName = AcquireFaceName(cmd, null, filePath);
                }
            }

            // フォントバイナリからフェース名が取得できなかった場合の処理
            if (string.IsNullOrEmpty(faceName))
            {
                var afterAddFontFonts = GlCm.GetInstalledFontNames();

                // 未インストールフォントなら、数が変わるはず。増えたフォントからフォントフェース名を取得する。
                if (beforeAddFontFonts.Keys.Count != afterAddFontFonts.Keys.Count)
                {
                    var addedFontFamilyKey = afterAddFontFonts.Keys.Where((key) => !beforeAddFontFonts.ContainsKey(key)).FirstOrDefault();
                    faceName = afterAddFontFonts[addedFontFamilyKey].FaceName;
                }
                else
                {
                    // インストール済みフォントなら、レジストリを読んで、faceName を取得する。
                    faceName = MonoTypeHelper.GetFontFaceNameFromFilePath(filePath);
                }

                if (string.IsNullOrEmpty(faceName))
                {
                    if (string.Compare(ext, ".bfttf", false) == 0 || string.Compare(ext, ".bfotf", false) == 0)
                    {
                        faceName = Path.GetFileNameWithoutExtension(filePath);
                    }
                }
            }

            // エラー指定されたフォントファイルの、faceName が取得できませんでした。
            if (string.IsNullOrEmpty(faceName))
            {
                throw GlCm.ErrMsg(ErrorType.CmdLine, string.Format(Strings.CantGetFaceNameForScalableFont + " {0}", cmd[OptionInputName]));
            }

            // filePath を faceName で 上書きします。
            cmd.SetOptionValue(OptionInputName, faceName);
        }

        // フォントフェース名を取得する
        private static string AcquireFaceName(CmdLine cmd, byte[] fontBinary, string filePath)
        {
            if (cmd.Exists(OptionInputFaceName))
            {
                // フォントフェース名がオプションで直接指定された場合はそれを使用する
                return cmd[OptionInputFaceName];
            }
            if (fontBinary == null)
            {
                fontBinary = File.ReadAllBytes(filePath);
            }
            // フォントバイナリからフェース名を取得する
            return AcquireFaceNameFromBinary(fontBinary, filePath);
        }

        // フォントバイナリからフェース名を取得する
        private static string AcquireFaceNameFromBinary(byte[] ttfData, string fontPath)
        {
            string faceName = "";
            using (System.Drawing.Text.PrivateFontCollection privateFont = new System.Drawing.Text.PrivateFontCollection())
            {
                IntPtr pointer = IntPtr.Zero;
                try
                {
                    pointer = System.Runtime.InteropServices.Marshal.AllocHGlobal(ttfData.Length);
                    System.Runtime.InteropServices.Marshal.Copy(ttfData, 0, pointer, ttfData.Length);
                    privateFont.AddMemoryFont(pointer, ttfData.Length);
                    if (privateFont.Families != null && privateFont.Families.Length > 0)
                    {
                        faceName = privateFont.Families[0].Name;
                    }
                }
                catch
                {
                    throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_FAILED_TO_READ_FONT_FACE_NAME, fontPath);
                }
                finally
                {
                    System.Runtime.InteropServices.Marshal.FreeHGlobal(pointer);
                }
            }
            return faceName;
        }

        /// <summary>
        /// FontWriterを構築
        /// </summary>
        private static FontWriter CreateWriter(CmdLine cmd)
        {
            FontWriter writer = CreateWriterImpl_(cmd);
            if (writer != null)
            {
                writer.UseDtdValidation = cmd.Exists(OptionUseDtd);
            }

            return writer;
        }

        /// <summary>
        /// FontWriterを構築
        /// </summary>
        private static FontWriter CreateWriterImpl_(CmdLine cmd)
        {
            bool isKerningEnabled = !cmd.Exists(OptionOutputNoKerning);

            if (_tcscmp(cmd[OptionOutput], OutputBMP) == 0 ||
                _tcscmp(cmd[OptionOutput], OutputImage) == 0)
            {
                string file = cmd[OptionOutputFile];
                string order = cmd[OptionOutputOrder];
                bool isDrawGrid = !cmd.Exists(OptionOutputDrawGrid);
                int cellWidth = -1;
                int cellHeight = -1;
                int marginLeft = 0;
                int marginRight = 0;
                int marginTop = 0;
                int marginBottom = 0;
                bool isSizeOffset = cmd.Exists(OptionOutputWidth) || cmd.Exists(OptionOutputHeight);

                if (isSizeOffset)
                {
                    if (cmd.Exists(OptionOutputRight))
                    {
                        ProgressControl.Warning(Strings.IDS_WARN_OW_OH_OR);
                    }

                    if (cmd.Exists(OptionOutputBottom))
                    {
                        ProgressControl.Warning(Strings.IDS_WARN_OW_OH_OB);
                    }

                    cellWidth = cmd.Number(OptionOutputWidth, -1);
                    cellHeight = cmd.Number(OptionOutputHeight, -1);
                    marginLeft = cmd.Number(OptionOutputLeft, -1);
                    marginTop = cmd.Number(OptionOutputTop, -1);
                }
                else
                {
                    marginLeft = cmd.Number(OptionOutputLeft, 0);
                    marginTop = cmd.Number(OptionOutputTop, 0);
                    marginRight = cmd.Number(OptionOutputRight, marginLeft);
                    marginBottom = cmd.Number(OptionOutputBottom, marginTop);
                }

                ImageFileFormat format = cmd.Exists(OptionOutputFormat) ?
                    ToImageFormat(cmd[OptionOutputFormat]) : ImageFileFormat.Ext;
                var colorBackground = cmd.Exists(OptionOutputBackgroundColor) ?
                    GlCm.ToColor(cmd[OptionOutputBackgroundColor]) : GlCm.ColorToUint(FB.BackgroundColor);
                var colorGrid = cmd.Exists(OptionOutputGridColor) ?
                    GlCm.ToColor(cmd[OptionOutputGridColor]) : GlCm.ColorToUint(FB.GridColor);
                var colorMargin = cmd.Exists(OptionOutputMarginColor) ?
                    GlCm.ToColor(cmd[OptionOutputMarginColor]) : GlCm.ColorToUint(FB.MarginColor);
                var colorWidthBar = cmd.Exists(OptionOutputWidthBarColor) ?
                    GlCm.ToColor(cmd[OptionOutputWidthBarColor]) : GlCm.ColorToUint(FB.WidthBarColor);
                var colorNullBlock = cmd.Exists(OptionOutputNullBlockColor) ?
                    GlCm.ToColor(cmd[OptionOutputNullBlockColor]) : GlCm.ColorToUint(FB.NullBlockColor);

                return
                    new ImageFontWriter(
                        file,
                        order,
                        format,
                        isDrawGrid,
                        isKerningEnabled,
                        cellWidth,
                        cellHeight,
                        marginLeft,
                        marginTop,
                        marginRight,
                        marginBottom,
                        colorGrid,
                        colorMargin,
                        colorWidthBar,
                        colorNullBlock,
                        colorBackground);
            }
            else if (_tcscmp(cmd[OptionOutput], OutputFont) == 0)
            {
                CharEncoding enc = cmd.Exists(OptionOutputEncoding) ?
                    ToEncodingType(cmd[OptionOutputEncoding]) : DefaultFontEncoding;

                int sheetPixels = 0;
                if (cmd.Exists(OptionOutputSheetPixels))
                {
                    sheetPixels = cmd.Number(OptionOutputSheetPixels, 0) * 1024;
                }
                else if (cmd.Exists(OptionOutputSheetSize))
                {
                    int sheetWidth = 0;
                    int sheetHeight = 0;
                    ToSheetSize(out sheetWidth, out sheetHeight, cmd[OptionOutputSheetSize]);

                    return _platformDetails.GetFontWriter(
                        cmd[OptionOutputFile],
                        cmd[OptionOutputGroup],
                        GlyphDataType.Texture,
                        (cmd.Exists(OptionOutputAlter) ? ToCharCode(cmd[OptionOutputAlter], enc) : NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputLinefeed, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputWidth, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputLeft, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputRight, NitroFontWriter.UseDefault),
                        enc,
                        sheetWidth,
                        sheetHeight,
                        0,
                        ConverterEnvironment.IsOutputTargetLittleEndian,
                        cmd.Exists(OptionOutputKerning));
                }

                var alterChar = cmd.Exists(OptionOutputAlter) ? ToCharCode(cmd[OptionOutputAlter], enc) : NitroFontWriter.UseDefault;

                return _platformDetails.GetFontWriter(
                        cmd[OptionOutputFile],
                        cmd[OptionOutputGroup],
                        GlyphDataType.Texture,
                        alterChar,
                        cmd.Number(OptionOutputLinefeed, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputWidth, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputLeft, NitroFontWriter.UseDefault),
                        cmd.Number(OptionOutputRight, NitroFontWriter.UseDefault),
                        enc,
                        0,
                        0,
                        sheetPixels,
                        ConverterEnvironment.IsOutputTargetLittleEndian,
                        isKerningEnabled);
            }
            else if (_tcscmp(cmd[OptionOutput], OutputOrder) == 0)
            {
                return new XlorFontWriter(cmd[OptionOutputFile]);
            }
            else if (_tcscmp(cmd[OptionOutput], OutputFormat) == 0)
            {
                return new FormatFontWriter(cmd[OptionOutputFile]);
            }
            else
            {
                if (cmd[OptionOutput].Length == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_REQUIRE_OUTPUT_FORMAT);
                }
                else
                {
                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNOWN_OUTPUT_FORMAT, cmd[OptionOutput]);
                }
            }
        }

        public static int _tcscmp(string strA, string strB)
        {
            return string.Compare(strA, strB, StringComparison.InvariantCulture);
        }

        private class SymbolicValue
        {
            public readonly string Symbol;
            public readonly string Description = string.Empty;
            public readonly int Value;

            public SymbolicValue(string symbol, CharEncoding value)
            {
                this.Symbol = symbol;
                this.Value = (int)value;
            }

            public SymbolicValue(string symbol, WidthType value)
            {
                this.Symbol = symbol;
                this.Value = (int)value;
            }

            public SymbolicValue(string symbol, GlyphImageFormat value)
            {
                this.Symbol = symbol;
                this.Value = (int)value;
            }

            public SymbolicValue(string symbol, string description, GlyphImageFormat value)
            {
                this.Symbol = symbol;
                this.Description = description;
                this.Value = (int)value;
            }

            public SymbolicValue(string symbol, ImageFileFormat value)
            {
                this.Symbol = symbol;
                this.Value = (int)value;
            }
        }

        public static void AddImageFontConvertOptions(CmdLine cmd)
        {
            try
            {
                var ffntPath = cmd[OptionInputFile];
                imageFontConvertDescription description;
                var ffntBytes = File.ReadAllBytes(ffntPath);
                using (var stream = new MemoryStream(ffntBytes))
                {
                    description =
                        (imageFontConvertDescription)new XmlSerializer(typeof(imageFontConvertDescription)).Deserialize(stream);
                }

                var directory = Path.GetDirectoryName(ffntPath);
                if (description.additionalArguments != null)
                {
                    cmd.AnalyzeAdditionalOption(description.additionalArguments, Options, Options.Length);
                }

                cmd.SetOptionValue(OptionInputFile, Path.Combine(directory,
                    Environment.ExpandEnvironmentVariables(description.imageFilePath)));
                cmd.SetOptionValue(OptionInput, InputImage);
                cmd.SetOptionValue(OptionInputOrder, Path.Combine(directory,
                    Environment.ExpandEnvironmentVariables(description.orderFilePath)));

                if (!string.IsNullOrEmpty(description.filterFilePath))
                {
                    cmd.SetOptionValue(OptionFilter, Path.Combine(directory,
                        Environment.ExpandEnvironmentVariables(description.filterFilePath)));
                }
            }
            catch (Exception e)
            {
                // TODO: もう少し丁寧なエラーハンドリング
                throw e;
            }
        }
    }

    /// <summary>
    /// Bfttf のヘルパ
    /// </summary>
    internal static class BfttfHelper
    {
        /// <summary>
        /// bfttf を ttf に変換してファイルに書き出します。
        /// </summary>
        internal static byte[] Convert(string bfttfPath)
        {
            byte[] allData = File.ReadAllBytes(bfttfPath);

            const uint SCRAMBLE_SIGNETURE = 0x7f9a0218;
            uint scrambleKey = SwapRead32_(allData, 0) ^ SCRAMBLE_SIGNETURE;
            uint ttfSize = SwapRead32_(allData, 4) ^ scrambleKey;
            for (int i = 8; i < 8 + ttfSize; i += 4)
            {
                Unsclamble_(allData, i, scrambleKey);
            }

            byte[] result = new byte[allData.Length - 8];
            Array.Copy(allData, 8, result, 0, allData.Length - 8);

            return result;
        }

        /// <summary>
        /// byte配列の 指定位置 を スクランブルを解除します。
        /// </summary>
        private static void Unsclamble_(byte[] data, int offset, uint scrambleKey)
        {
            // エンディアンスワップして取得。
            uint val = SwapRead32_(data, offset);

            // スクランブル解除
            val ^= scrambleKey;

            // エンディアンを戻しつつ書き戻します。
            data[offset + 0] = (byte)((val >> 24) & 0xFF);
            data[offset + 1] = (byte)((val >> 16) & 0xFF);
            data[offset + 2] = (byte)((val >> 8) & 0xFF);
            data[offset + 3] = (byte)((val >> 0) & 0xFF);
        }

        /// <summary>
        /// 32 bit データをエンディアンスワップしつつ読む。
        /// </summary>
        private static uint SwapRead32_(byte[] data, int offset)
        {
            return (uint)((data[offset] << 24) | (data[offset + 1] << 16) | (data[offset + 2] << 8) | data[offset + 3]);
        }
    }
}
