﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using NintendoWare.Font.Win32;

namespace NintendoWare.Font
{
    public class MonoTypeHelper
    {
        /// <summary>
        /// レジストリを調べて、フォント名からフォントファイル名を得る。
        /// </summary>
        static public string GetFontFileName_(string fontName)
        {
            try
            {
                Microsoft.Win32.RegistryKey fontsKey = Microsoft.Win32.Registry.LocalMachine.OpenSubKey(@"Software\Microsoft\Windows NT\CurrentVersion\Fonts");

                return fontsKey.GetValue(string.Format("{0} (TrueType)", fontName), string.Empty) as string;
            }
            catch
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// フォント名からフォントファイルパスを得る。
        /// </summary>
        private static string GetFontFilePath_(string fontName)
        {
            try
            {
                System.Drawing.Font font = new System.Drawing.Font(fontName, 12.0f);
                System.Globalization.CultureInfo ci = new System.Globalization.CultureInfo("en-US");
                string fontEnName = font.FontFamily.GetName(ci.LCID);

                string fontFileName = GetFontFileName_(fontName);
                if (string.IsNullOrEmpty(fontFileName))
                {
                    return string.Empty;
                }

                string fontDir = Environment.GetFolderPath(Environment.SpecialFolder.Fonts);
                return Path.Combine(fontDir, fontFileName);
            }
            catch
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// レジストリ値がフォントファイル名と一致しているか
        /// </summary>
        static private bool IsMatchRegValueToFontFileName_(string regValueText, string fontFileName)
        {
            // 完全に一致している
            if (regValueText == fontFileName)
            {
                return true;
            }

            // 絶対パス表記で終端が一致している
            if (regValueText.EndsWith(fontFileName))
            {
                return true;
            }

            // ファイル名_XX.拡張子 というパターンが発見できるか調査する。
            // 同じフォントを複数回インストールすると、 xxx_1.ttf xxx_2.ttf とカウントアップする挙動の様子なのでこれに対応するため。
            string fileName = Path.GetFileNameWithoutExtension(fontFileName);
            string fileExt = Path.GetExtension(fontFileName);
            string pattern = string.Format("{0}_.+?{1}", fileName, fileExt);
            if (System.Text.RegularExpressions.Regex.IsMatch(regValueText, pattern))
            {
                return true;
            }

            // ファイル名の末尾の _数字 を取り去って再度検索する。
            // Windows がレジストリに記録している値から推測するに、ファイル名の末尾の数字を取り去って独自の数字を付加しているように見えるので。
            string newFileName = System.Text.RegularExpressions.Regex.Replace(fileName, "_\\d+?$", "");
            if (regValueText.StartsWith(newFileName))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// フォントファイルパスから、フォントフェイス名を取得します。
        /// </summary>
        public static string GetFontFaceNameFromFilePath(string fontFilePath)
        {
            string fontFileName = Path.GetFileName(fontFilePath);
            Microsoft.Win32.RegistryKey fontsKey = Microsoft.Win32.Registry.LocalMachine.OpenSubKey(@"Software\Microsoft\Windows NT\CurrentVersion\Fonts");
            foreach (var name in fontsKey.GetValueNames())
            {
                string fileNameValue = fontsKey.GetValue(name, string.Empty) as string;
                if (IsMatchRegValueToFontFileName_(fileNameValue, fontFileName))
                {
                    string faceName = name;
                    faceName = faceName.Replace("(TrueType)", "");
                    faceName = faceName.Replace("(OpenType)", "");
                    faceName = faceName.TrimEnd();

                    return faceName;
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// フォント生イメージを読み込む。
        /// </summary>
        private static IntPtr LoadRawFont_(string fontFilePath)
        {
            try
            {
                IntPtr rawFile;
                using (var fs = File.OpenRead(fontFilePath))
                {
                    byte[] fileRaw = null;
                    fileRaw = new Byte[fs.Length];
                    fs.Read(fileRaw, 0, (int)fs.Length);

                    // bfttf および bfotf の場合はデコードする
                    if (fontFilePath.Length > 6 && (fontFilePath.Substring(fontFilePath.Length - 6).ToLower() == ".bfttf" || fontFilePath.Substring(fontFilePath.Length - 6).ToLower() == ".bfotf"))
                    {
                        fileRaw = GlCm.Descramble_(fileRaw);
                    }

                    rawFile = Marshal.AllocHGlobal(fileRaw.Length);
                    Marshal.Copy(fileRaw, 0, rawFile, fileRaw.Length);
                }

                return rawFile;
            }
            catch
            {
                return IntPtr.Zero;
            }
        }

        /// <summary>
        /// MonoTypeエンジンのカーニングで、カーニング値を上書きする。
        /// </summary>
        public static void FixKerningByMonoTypeKerning(string fontPath, string fontName, int fontEmHeight, KerningPair[] kerns)
        {
            IntPtr workMem = IntPtr.Zero;
            IntPtr nameBuffer = IntPtr.Zero;
            IntPtr rawFile = IntPtr.Zero;

            try
            {
                // フォントフェース名でロードしてみる
                string fontFilePath = GetFontFilePath_(fontName);
                if (string.IsNullOrEmpty(fontFilePath))
                {
                    rawFile = IntPtr.Zero;
                }
                else
                {
                    rawFile = LoadRawFont_(fontFilePath);
                }
                if (rawFile == IntPtr.Zero)
                {
                    if (string.IsNullOrEmpty(fontPath))
                    {
                        // カーニング取得：ファイルパスが取得できなかった
                        GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_KERN_CANTGET_FONTFILEPATH, fontPath);
                        return;
                    }

                    // フォントパスでロードしてみる
                    rawFile = LoadRawFont_(fontPath);
                    if (rawFile == IntPtr.Zero)
                    {
                        // ファイルが読めなかった。
                        GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_KERN_CANTLOAD_FONT, fontName);
                        return;
                    }
                }

                const int WorkMemSize = 1024 * 1024 * 2;

                workMem = Marshal.AllocHGlobal(WorkMemSize);
                nameBuffer = Marshal.AllocHGlobal(50);

                NW4F.FontConverter.KerningReader kerningReader = new NW4F.FontConverter.KerningReader();
                kerningReader.InitializeFontEngine(workMem, WorkMemSize);
                kerningReader.LoadFont(nameBuffer, rawFile, 0, 50);
                kerningReader.SetFont(nameBuffer);
                kerningReader.SetScale(fontEmHeight << 16, 0, 0, fontEmHeight << 16);
                // FS_set_bold_pct... 動作を確認したところ、カーニング値に影響はないので指定しません。

                Int32 dx = 0;
                Int32 dy = 0;
                for (int i = 0; i < kerns.Length; i++)
                {
                    kerningReader.GetKerning(ref dx, ref dy, kerns[i].First, kerns[i].Second);
                    dx = dx >> 16;
                    if (dx != kerns[i].KernAmount)
                    {
                        kerns[i].KernAmount = dx;
                    }
                }

                kerningReader.FinalizeFontEngine();
            }
            finally
            {
                Marshal.FreeHGlobal(rawFile);
                Marshal.FreeHGlobal(nameBuffer);
                Marshal.FreeHGlobal(workMem);
            }
        }
    }
}
