﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Xml;
    using System.Xml.Schema;

    public abstract class HandlerBase
    {
        private string error;

        protected string Error
        {
            get { return this.error; }
            set { this.error = value; }
        }

        public static bool IsWhitespace(char c)
        {
            switch (c)
            {
            case '\u0020':  // 空白
            case '\u0009':  // タブ
            case '\u000D':  // 復帰
            case '\u000A':  // 改行
                return true;
            }

            return false;
        }

        public XmlReader CreateReader(string filePath, bool useDtdValidation)
        {
            var settings = new XmlReaderSettings();
            if (useDtdValidation)
            {
                settings.DtdProcessing = DtdProcessing.Parse; // settings.ProhibitDtd = false;
                settings.ValidationType = ValidationType.DTD;
                settings.ValidationEventHandler += new ValidationEventHandler(this.ValidationEventHandler);
            }
            else
            {
                settings.DtdProcessing = DtdProcessing.Ignore;
            }

            return XmlReader.Create(filePath, settings);
        }

        public void Parse(XmlReader reader)
        {
            var attrList = new AttributeList(reader);

            while (reader.Read())
            {
                switch (reader.NodeType)
                {
                case XmlNodeType.Element:
                    this.StartElement(reader.Name, attrList);
                    if (reader.IsEmptyElement)
                    {
                        this.EndElement(reader.Name);
                    }

                    break;
                case XmlNodeType.EndElement:
                    this.EndElement(reader.Name);
                    break;
                case XmlNodeType.Text:
                    this.Characters(reader.Value, reader.Value.Length);
                    break;
                }
            }
        }

        protected abstract void StartElement(string name, AttributeList attributes);

        protected abstract void EndElement(string name);

        protected abstract void Characters(string chars, int length);

        private void ValidationEventHandler(object sender, ValidationEventArgs e)
        {
            var buf = string.Format(
                Strings.IDS_ERR_SAX_MSG,
                e.Severity,
                e.Exception.LineNumber,
                e.Exception.LinePosition,
                e.Message);

            Rpt._RPT1("{0}\n", buf);

            if (this.error == null)
            {
                this.error = buf;
            }
        }

        public class AttributeList
        {
            private XmlReader reader;

            public AttributeList(XmlReader reader)
            {
                this.reader = reader;
            }

            public string GetValue(string name)
            {
                return this.reader.GetAttribute(name);
            }
        }
    }

    public class HandlerBaseException : Exception
    {
    }
}
