﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// このdefine値を有効にすると、NW4Rのコードにあるバグを
// 修正するコードが有効になります。
// ただし、正しく修正されているかどうかは未確認。
// #define NW_FONT_NITROFONTWRITER_BUGFIX  1

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.InteropServices;
    using NintendoWare.Font.Runtime;
    using NW4F.LayoutBinaryConverter;
    using TexConv = NW4F.LayoutBinaryConverter.TexConv;

    public abstract class NitroFontWriter : FontWriter
    {
        // font_ResourceFormat.h
        // FontSheetFormat の XXX_FLAG に対応する値です。
        [Flags]
        public enum GlyphImageFormatFlag
        {
            None = 0x0,
            FormatMask = NintendoWare.Font.FontSheetFormat.TexFmtMask, // 16ビット値から、フラグ以外のフォーマット値を取り出すためのマスク
            LinearFormat = NintendoWare.Font.FontSheetFormat.LinearFormatFlag, // リニアなのでタイリング解除の必要がない
            Compressed = NintendoWare.Font.FontSheetFormat.CompressedFlag, // 圧縮形式（現在は利用されていません）
        }

        public const int UseDefault = 0xFFFF;

        // DIRECT条件を満たす文字がこれ以上連続していればDIRECTにする
        // TODO 最適値
        private const int CmapDirectBlockThreshold = 80;

        // TABLE条件を満たす文字がこれ以上連続していればTABLEにする
        // TODO 最適値
        private const int CmapTableBlockThreshold = 40;

        private const string NoGroupedGlyphGroup = "*";

        private readonly string File;
        private readonly string GroupsFile;
        private readonly uint AlterCodeUser;
        private readonly int Linefeed;
        private readonly int DefaultWidth;
        private readonly int DefalutLeft;
        private readonly int DefaultRight;
        private readonly CharEncoding Encoding;
        private readonly GlyphDataType Type;
        private readonly int SheetWidth;
        private readonly int SheetHeight;
        private readonly int SheetPixels;
        private readonly SubGroupMap SubGroups = new SubGroupMap();
        private readonly GroupMap Groups = new GroupMap();
        private readonly IntArray CompSheetSizes = new IntArray();
        protected readonly TGLPInfo TglpInfo = new TGLPInfo();
        private readonly bool IsOutputTargetLittleEndian = false;
        private readonly bool IsOutKerningData = false;


        private uint alterCode;
        private ushort alterIndex;
        private bool isCompress;

        public NitroFontWriter(
            string file,
            string groupsFile,
            GlyphDataType type,
            uint alter,
            int linefeed,
            int width,
            int left,
            int right,
            CharEncoding encoding,
            int sheetWidth,
            int sheetHeight,
            int sheetPixels,
            bool isOutputTargetLittleEndian,
            bool isOutKerningData)
        {
            this.File = file;
            this.GroupsFile = groupsFile;
            this.AlterCodeUser = alter;
            this.Linefeed = linefeed;
            this.DefalutLeft = left;
            this.DefaultWidth = width;
            this.DefaultRight = right;
            this.Encoding = encoding;
            this.Type = type;
            this.isCompress = false;

            this.SheetWidth = sheetWidth;
            this.SheetHeight = sheetHeight;
            this.SheetPixels = sheetPixels;

            this.IsOutputTargetLittleEndian = isOutputTargetLittleEndian;
            this.IsOutKerningData = isOutKerningData;
        }

        /// <summary>
        /// 実機テクスチャフォーマットを出力するかを取得します。
        /// <remarks>
        /// 別々に指定する必要がないためエンディアン指定と、画像フォーマット指定は連動します。
        /// </remarks>
        /// </summary>
        private bool UseGX2TexureFormat { get { return !this.IsOutputTargetLittleEndian; } }

        private delegate void PixelWriter(ImageBase.Pixel pixel);

        //----------------------------------------------------------
        // プラットフォーム毎に処理をオーバーライドする処理
        //----------------------------------------------------------

        /// <summary>
        /// テクスチャフォーマットを調整します。
        /// </summary>
        protected abstract GlyphImageFormat ConvertGlyphImageFormat_(GlyphImageFormat gif);

        /// <summary>
        /// 実機テクスチャ列に変換します。
        /// </summary>
        protected abstract List<byte> ConvertToNativeImage_(RgbImage sheet_plane, GlyphImageFormat gif);

        /// <summary>
        /// 実機テクスチャのフォーマットを取得します。
        /// </summary>
        protected abstract ushort GlyphImageFormatToTextureFormat(GlyphImageFormat gif);

        /// <summary>
        /// テクスチャ変換向け実機テクスチャのフォーマットを取得します。
        /// </summary>
        protected abstract TexConv::TexelFormat GlyphImageFormatToTexConvTexelFormat(GlyphImageFormat gif);

        //----------------------------------------------------------

        public override void WriteFontData(FontData fontData, GlyphOrder order)
        {
            G2dFont font = new G2dFont();
            FontData encFontData = new FontData(fontData);

            // プログレスバー設定
            ProgressControl.GetInstance().SetStatusString(string.Format(Strings.IDS_STATUS_WRITE_NITRO, LibFormat.ExtensionFont));
            ProgressControl.GetInstance().ResetProgressBarPos();

            this.ConstructEncodedFontData(encFontData, fontData);

            // Default Width
            if (this.DefaultWidth != UseDefault)
            {
                if (this.DefaultWidth < 0 || encFontData.CellWidth.Value < this.DefaultWidth)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OUTOFRANGE_DEFAULTWIDTH, this.DefaultWidth, 0, encFontData.CellWidth.Value);
                }
            }

            // Default Right Space
            if (this.DefaultRight != UseDefault)
            {
                int left = (this.DefalutLeft == UseDefault) ? encFontData.DefaultWidth.Value.Left : this.DefalutLeft;
                int gwidth = (this.DefaultWidth == UseDefault) ? encFontData.DefaultWidth.Value.GlyphWidth : this.DefaultWidth;
                int right_min = sbyte.MinValue - (left + gwidth);
                int right_max = sbyte.MaxValue - (left + gwidth);

                if (this.DefaultRight < right_min || right_max < this.DefaultRight)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OUTOFRANGE_DEFAULTRIGHTSPACE, this.DefaultRight, right_min, right_max);
                }
            }

            // NITROフォントへ変換
            this.Convert(font, encFontData);

            // NITROフォント書き出し
            this.Save(font);
        }

        public override void GetGlyphOrder(GlyphOrder glyphOrder)
        {
            // nothing to do
        }

        /// <summary>
        /// 入力値チェック
        /// </summary>
        public override void ValidateInput()
        {
            // NITRO Font
            {
                ValidateOutputPath(this.File);
            }

            // glyph groups
            if (this.GroupsFile != string.Empty)
            {
                if (!IsFileExists(this.GroupsFile))
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_FILE_NOT_EXISTS, this.GroupsFile);
                }

                // 試しに読んでみる
                {
                    GlyphGroups groups = new GlyphGroups();

                    groups.Load(this.GroupsFile, this.UseDtdValidation);
                }
            }

            // Alternate Char
            if (this.AlterCodeUser != UseDefault)
            {
                if (this.AlterCodeUser == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_INVALID_ALTER);
                }
            }

            // Line Height
            if (this.Linefeed != UseDefault)
            {
                if (this.Linefeed < 0x00 || short.MaxValue < this.Linefeed)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OUTOFRANGE_LINEHEIGHT, this.Linefeed, 0, short.MaxValue);
                }
            }

            // Default Left Space
            if (this.DefalutLeft != UseDefault)
            {
                if (this.DefalutLeft < sbyte.MinValue || sbyte.MaxValue < this.DefalutLeft)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OUTOFRANGE_DEFAULTLEFTSPACE, this.DefalutLeft, sbyte.MinValue, sbyte.MaxValue);
                }
            }

            // sheet size
            if (this.SheetWidth != 0 || this.SheetHeight != 0)
            {
                int widthExp = 0;
                int heightExp = 0;

                for (int x = 32; x <= 1024; x *= 2)
                {
                    if (this.SheetWidth == x)
                    {
                        widthExp = x;
                    }
                }

                for (int x = 32; x <= 1024; x *= 2)
                {
                    if (this.SheetHeight == x)
                    {
                        heightExp = x;
                    }
                }

                if (widthExp == 0 || heightExp == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_INVALID_SHEET_SIZE, this.SheetWidth, this.SheetHeight);
                }
            }

            //// ////Default Width
            //// if( m_defaultWidth != USE_DEFAULT )
            //// {
            ////     if( m_defaultWidth < 0 || byte.MaxValue < m_defaultWidth )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                         _T(" DefaultWidth(=%d) out of range[%d ～ %d]"),
            ////                         m_defaultWidth, 0, byte.MaxValue );
            ////     }
            //// }
            //
            //// ////Default Right Space
            //// if( m_defaultRight != USE_DEFAULT )
            //// {
            ////     const int right_min = ;
            ////     const int right_max = sbyte.MaxValue - (m_de);
            ////     if( m_defaultRight < sbyte.MinValue || sbyte.MaxValue < m_defaultRight )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                         _T(" DefaultRightSpace(=%d) out of range[%d ～ %d]"),
            ////                         m_defaultRight, sbyte.MinValue, sbyte.MaxValue );
            ////     }
            //// }
        }

        private static int DereferenceComparator(Group a, Group b)
        {
            return a.CompareTo(b);
        }

        private static uint ConvertPixelA4(ImageBase.Pixel pixel)
        {
            int level = GlCm.RgbToGrayScale(pixel.IntColor) >> 4;
            return GlCm.InverseNumber((uint)level, 4);
        }

        private static int GetDefaultOrValue(int var, int def)
        {
            return var == UseDefault ? def : var;
        }

        private static void RegisterSubGroup(
            NitroFontWriter.SubGroupMap subMap,
            IntIntMap group2Sub,
            CodeExistsMap existsMap,
            GlyphGroups.Group g,
            CharEncoding encoding)
        {
            // サブグループ作成
            NitroFontWriter.SubGroup sub = new NitroFontWriter.SubGroup();

            /* 文字を登録
               g.chars は unicode 文字群
               pExistsMap はローカルコード文字群
             */

            foreach (var c in g.Chars)
            {
                uint local_code = GlCm.UnicodeToEncoding(encoding, c);

                if (local_code != Runtime.RtConsts.InvalidCharCode)
                {
                    var found = existsMap.ContainsKey(local_code);
                    if (found)
                    {
                        NitroFontWriter.GlyphCode gc = new GlyphCode(local_code, 0);
                        sub.CodeList.Add(gc);

                        existsMap[local_code] = true;
                    }
                }
            }

            if (sub.CodeList.Count == 0)
            {
                // 出力する文字がないので登録しない
                Rpt._RPT1("subgroup: {0} -> no glyph\n", g.Name);
            }
            else
            {
                // 登録
                Rpt._RPT4("subgroup: {0} -> {1:X4}-{2:X4} / num {3}\n", g.Name, sub.CodeList[0].CharCode, sub.CodeList[sub.CodeList.Count - 1].CharCode, sub.CodeList.Count);
                subMap.Add(sub);
                group2Sub.Add(g.Index, sub.Index);
            }
        }

        private static void MakeSubGroups(
            NitroFontWriter.SubGroupMap subMap,
            IntIntMap group2Sub,
            CodeExistsMap existsMap,
            GlyphGroups.GroupList srcGroups,
            CharEncoding encoding)
        {
            // グループリストの全てのグループに対して
            foreach (var i in srcGroups)
            {
                // 子グループを持たないのならば
                GlyphGroups.Group g = i;
                if (g.Groups.Count == 0)
                {
                    RegisterSubGroup(subMap, group2Sub, existsMap, g, encoding);
                }
                else
                {
                    // そうでなければ再帰
                    MakeSubGroups(subMap, group2Sub, existsMap, g.Groups, encoding);
                }
            }
        }

        private static void MakeGroup(
            IntIntMap list,
            NitroFontWriter.GroupMap groupMap,
            IntIntMap group2Sub,
            GlyphGroups.Group g)
        {
            IntIntMap subs = new IntIntMap();

            if (g.Groups.Count == 0)
            {
                // リーフなら
                // リストに自分のサブグループを追加する
                int value;
                var found = group2Sub.TryGetValue(g.Index, out value);
                if (found)
                {
                    subs[value] = 1;
                }
            }
            else
            {
                // ノードなら
                foreach (var i in g.Groups)
                {
                    MakeGroup(subs, groupMap, group2Sub, i);
                }
            }

            if (subs.Count > 0)
            {
                list.Add(subs);

                // 既にグループ登録されているか？
                Group value;
                bool found = groupMap.TryGetValue(g.Name, out value);

                // グループ登録
                if (!found)
                {
                    NitroFontWriter.Group pg = new NitroFontWriter.Group(
                        g.Index,
                        g.Name);
                    GlCm.MakeListFromMapKey(pg.SubGroups, subs);

                    groupMap[pg.Name] = pg;
#if DEBUG
                    Rpt._RPT2("group: {0} -> {1}:", g.Name, pg.Index);

                    foreach (var i in pg.SubGroups)
                    {
                        Rpt._RPT1(" {0}", i);
                    }

                    Rpt._RPT0("\n");
                }
                else
                {
                    Rpt._RPT2("group: {0} -> {1}:", g.Name, value.Index);

                    foreach (var i in value.SubGroups)
                    {
                        Rpt._RPT1(" {0}", i);
                    }

                    Rpt._RPT0("\n");
#endif
                }
            }
            else
            {
                Rpt._RPT1("group: {0} -> no glyphs\n", g.Name);
            }
        }

        private static void MakeGroups(
            NitroFontWriter.GroupMap groupMap,      // グループの登録先
            IntIntMap group2Sub,                // サブグループの対応リスト
            GlyphGroups.GroupList srcGroups)   // 元となるグループのリスト
        {
            IntIntMap subs = new IntIntMap();
            foreach (var i in srcGroups)
            {
                MakeGroup(subs, groupMap, group2Sub, i);
            }

            // どのグループにも含まれない文字によるサブグループを追加する
            {
                int value;
                var found = group2Sub.TryGetValue(-1, out value);
                if (found)
                {
                    NitroFontWriter.Group pg = new NitroFontWriter.Group(
                        groupMap.Count,
                        NoGroupedGlyphGroup);

                    pg.SubGroups.Add(value);
                    groupMap[pg.Name] = pg;
                }
            }
        }

        private static void MakeDefaultGroups(
            NitroFontWriter.GroupMap groupMap,
            NitroFontWriter.SubGroupMap subMap,
            FontData fontData)
        {
            // 出力する全ての文字から構成されるサブグループを作成する。
            {
                NitroFontWriter.SubGroup sub = new NitroFontWriter.SubGroup();
                GlyphList glist = fontData.GetGlyphList();

                // glist はローカル文字群
                foreach (var i in glist.GetEnum())
                {
                    NitroFontWriter.GlyphCode gc = new GlyphCode(i.GetCode(), 0);
                    sub.CodeList.Add(gc);
                }

                subMap.Add(sub);
            }

            // 上記サブグループだけを含むグループを作る
            {
                NitroFontWriter.Group pg = new NitroFontWriter.Group(
                    0,
                    string.Empty);
                pg.SubGroups.Add(0);

                groupMap[pg.Name] = pg;
                Rpt._RPT2("group: {0} -> {1}: 0\n", pg.Name, pg.Index);
            }
        }

        private void MakeSubGroups(
            NitroFontWriter.SubGroupMap subMap,
            IntIntMap group2Sub,
            GlyphGroups.GroupList srcGroups,
            FontData fontData,
            CharEncoding encoding)
        {
            CodeExistsMap existsMap = new CodeExistsMap();

            // 出力する全ての文字をリストアップする
            {
                GlyphList glist = fontData.GetGlyphList();

                // glist はローカルコード文字群
                foreach (var i in glist.GetEnum())
                {
                    existsMap.Add(i.GetCode(), false);
                }
            }

            // グループツリーの葉をサブグループとして登録する
            MakeSubGroups(subMap, group2Sub, existsMap, srcGroups, encoding);

            // 出力する文字のうちどのサブグループにも
            // 含まれない文字から構成されるサブグループを作成する。
            {
                NitroFontWriter.SubGroup sub = new NitroFontWriter.SubGroup();

                // existsMap はローカルコード文字群
                foreach (var i in existsMap)
                {
                    if (!i.Value)
                    {
                        NitroFontWriter.GlyphCode gc = new GlyphCode(i.Key, 0);
                        sub.CodeList.Add(gc);
                    }
                }

                if (sub.CodeList.Count == 0)
                {
                }
                else
                {
                    subMap.Add(sub);
                    group2Sub.Add(-1, sub.Index);
                }
            }
        }

        private void InitBitArray(G2dFont.BitArray bits, int numBits)
        {
            int numBitSet = GlCm.DIV_UP(numBits, 32);
            if (bits.Count != numBitSet)
            {
                if (bits.Count < numBitSet)
                {
                    for (int i = numBitSet - bits.Count; i > 0; --i)
                    {
                        bits.Add(0);
                    }
                }
                else
                {
                    bits.RemoveRange(numBitSet, bits.Count);
                }
            }

            for (int i = 0; i < numBitSet; ++i)
            {
                bits[i] = 0;
            }
        }

        private void BitArray_Set(G2dFont.BitArray bits, int index)
        {
            int bitIdx = index % 32;
            int bitSetIdx = index / 32;
            bits[bitSetIdx] |= 1u << (31 - bitIdx);
        }

        private void BitArray_Merge(G2dFont.BitArray bits, IndexList list)
        {
            foreach (var i in list)
            {
                this.BitArray_Set(bits, i);
            }
        }

        /*
            WriteFontData
                CnstructEncodedFontData
                Convert
                    CalcTGLPInfo
                        CalcTextureSize
                    ConstructSheetAssingment
                    CreateCMapBlock
                        CreateCMapBlockDirect
                        CreateCMapBlockTable
                        CreateCMapBlockScan
                    CreateGlyphBlock
                        ConvertFormat
                        CreateSheetPlane
                    CreateGroupBlock
                    CreateInfoBlock
                    CreateWidthBlock
        */

        /// <summary>
        /// エンコーディングを変換したフォントを作成
        /// </summary>
        /// <param name="encFontData"></param>
        /// <param name="fontData"></param>
        private void ConstructEncodedFontData(FontData encFontData, FontData fontData)
        {
            var srcList = fontData.GetGlyphList();
            var dstList = encFontData.GetGlyphList();

            dstList.Reserve(srcList.GetNum());

            // srcList は unicode 文字群
            foreach (var g in srcList.GetEnum())
            {
                Glyph sg = g;
                uint unicode = sg.GetCode();

                uint new_code = GlCm.UnicodeToEncoding(this.Encoding, unicode);

                if (new_code == Runtime.RtConsts.InvalidCharCode)
                {
                    Rpt._RPT2("{0:X4} {1}\n", unicode, Marshal.GetLastWin32Error());

                    ProgressControl.Warning(
                        Strings.IDS_WARN_CANT_REPRESENT_IN_LOCAL,
                        unicode,
                        unicode);
                }
                else
                {
                    dstList.AddGlyph(new Glyph(sg), new_code);
                }
            }

            dstList.SortByCode();
        }

        /// <summary>
        /// 変換
        /// </summary>
        /// <param name="font"></param>
        /// <param name="fontData"></param>
        private void Convert(G2dFont font, FontData fontData)
        {
            ProgressControl.GetInstance().SetProgressBarMax((int)(fontData.GetGlyphList().GetNum() * 3));

            // サイズチェック
            if (fontData.CellWidth.Value > byte.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OVERMAX_GLYPHWIDTH, fontData.CellWidth.Value, byte.MaxValue);
            }

            if (fontData.CellHeight.Value > byte.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OVERMAX_GLYPHHEIGHT, fontData.CellHeight.Value, byte.MaxValue);
            }

            if (fontData.Ascent.Value + fontData.Descent.Value > byte.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OVERMAX_FONTHEIGHT, fontData.Ascent.Value + fontData.Descent.Value, byte.MaxValue);
            }

            if (fontData.MaxCharWidth.Value > byte.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OVERMAX_MAXCHARWIDTH, fontData.MaxCharWidth.Value, byte.MaxValue);
            }

            if (fontData.BaselinePos.Value > short.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_OVERMAX_BASELINEPOS, fontData.BaselinePos.Value, short.MaxValue);
            }

            if (fontData.BaselinePos.Value < short.MinValue)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_UNDERMIN_BASELINEPOS, fontData.BaselinePos.Value, short.MinValue);
            }

            this.CalcTGLPInfo(this.TglpInfo, fontData);
            this.ConstructSheetAssingment(fontData);

            // INFOブロック作成
            Rpt._RPT0("create FINF block\n");
            this.CreateInfoBlock(font, fontData);

            // グリフブロック作成
            Rpt._RPT0("create TGLP block\n");
            this.CreateGlyphBlock(font, fontData);

            // 幅情報ブロック作成
            Rpt._RPT0("create CWDH block\n");
            this.CreateWidthBlock(font, fontData);

            // CMAPブロック作成
            Rpt._RPT0("create CMAP block\n");
            this.CreateCMapBlock(font, fontData);

            // KRNGブロック作成
            if (IsOutKerningData)
            {
                // カーニング情報の書き出しが必要にもかかわらずカーニング情報がない場合は警告
                if (fontData.KerningPairs == null || fontData.KerningPairs.Length == 0)
                {
                    ProgressControl.Warning(Strings.IDS_WARN_NO_KERNING_INFO);
                }
                else
                {
                    Rpt._RPT0("create KRNG block\n");
                    this.CreateKrngBlock(font, fontData);
                }
            }

            // グループブロック作成
            // 最後に行わなければならない
            Rpt._RPT0("create GLGR block\n");
            this.CreateGroupBlock(font, fontData);

            // ブロック間リンク作成
            Rpt._RPT0("make inter block link\n");
            font.MakeInterBlcokLink();
        }

        private void ConstructSheetAssingment(FontData fontData)
        {
            // グループをサブグループで表現
            if (this.GroupsFile != string.Empty)
            {
                GlyphGroups xggp = new GlyphGroups();
                xggp.Load(this.GroupsFile, this.UseDtdValidation);
                var srcGroups = xggp.GetGroups();

                if (srcGroups.Count > 0)
                {
                    IntIntMap group2sub = new IntIntMap();
                    MakeSubGroups(this.SubGroups, group2sub, srcGroups, fontData, this.Encoding);
                    MakeGroups(this.Groups, group2sub, srcGroups);
                    this.isCompress = true;
                }
            }

            // グループ指定がないなら全ての文字を単一のサブグループに入れる
            if (this.SubGroups.Count == 0 || this.Groups.Count == 0)
            {
                MakeDefaultGroups(this.Groups, this.SubGroups, fontData);
            }

            // 代替文字コードをローカルコードに変換
            {
                uint alterCode = this.AlterCodeUser;
                Rpt._RPT1("alter user: {0:X4}\n", alterCode);

                if (alterCode == Runtime.RtConsts.InvalidCharCode)
                { // Use Default
                    // 変換元フォントの設定を読み込む
                    if (fontData.AlterChar.HasValue)
                    {
                        alterCode = fontData.AlterChar.Value;
                    }
                }

                if (alterCode != Runtime.RtConsts.InvalidCharCode)
                {
                    uint unicode = alterCode;

                    // ローカルコードを用いる場合
                    alterCode = GlCm.UnicodeToEncoding(this.Encoding, unicode);

                    if (alterCode == Runtime.RtConsts.InvalidCharCode)
                    {
                        Rpt._RPT2("{0:X4} {1}\n", unicode, Marshal.GetLastWin32Error());
                        throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_TO_LOCAL, unicode, unicode);
                    }

                    Rpt._RPT1("alter uni: {0:X4}\n", alterCode);
                }
                else
                {
                    // 変換元でも指定されない場合は最初のグリフを使用する
                    alterCode = fontData.GetGlyphList().GetFirstItem().GetCode();
                    Rpt._RPT1("alter auto: {0:X4}\n", alterCode);
                }

                this.alterCode = alterCode;
            }

            // 代替文字を全てのサブグループに追加
            {
                foreach (var s in this.SubGroups)
                {
                    if (s.Value != null)
                    {
                        GlyphCode gc = new GlyphCode(this.alterCode, 0);
                        CodeList clist = s.Value.CodeList;

                        if (!clist.Contains(gc))
                        {
                            clist.Add(gc);
                        }
                    }
                }
            }

            // 共有文字を抽出
            {
                var share = new List<int>[0x110000];
                for (int i = 0; i < share.Length; ++i)
                {
                    share[i] = new List<int>();
                }

                // 文字コードの共有をカウント
                foreach (var s in this.SubGroups)
                {
                    if (s.Value != null)
                    {
                        CodeList clist = s.Value.CodeList;
                        int sindex = s.Value.Index;
                        foreach (var c in clist)
                        {
                            share[c.CharCode].Add(sindex);
                        }
                    }
                }

                int maxBasicSubGroupIndex = this.SubGroups.Count - 1;

                var codeSubIndexDic = new Dictionary<uint, int>();

                // 共有されている文字からなるサブグループを作成
                // 共有されている文字から上記サブグループへのマップを作成
                {
                    SharedCodeMap shareMap = new SharedCodeMap();

                    // 共有されている文字を集める
                    for (uint c = 0; c < 0x110000; ++c)
                    {
                        if (share[c].Count > 1)
                        {
                            // 内容をスワップする
                            List<int> temp;
                            if (!shareMap.TryGetValue(c, out temp))
                            {
                                temp = new List<int>();
                            }

                            shareMap[c] = share[c];
                            share[c] = temp;
                        }
                    }

                    // 共有元のサブグループリストごとにサブグループに割り当て
                    {
                        var sh0 = shareMap.GetEnumerator();

                        // 共有されている全ての文字に対して
                        while (sh0.MoveNext())
                        {
                            var sheet0Crnt = sh0.Current;

                            // 既に割り当てられているものはスキップ
                            if (sheet0Crnt.Value.Count > 0)
                            {
                                // 新規サブグループを作成
                                SubGroup sub = new SubGroup();
                                this.SubGroups.Add(sub);

                                GlyphCode gc = new GlyphCode(sheet0Crnt.Key, 0);
                                sub.CodeList.Add(gc);
                                codeSubIndexDic[sheet0Crnt.Key] = sub.Index;

                                // 残りの共有されている文字に対して
                                var sh1 = sh0;
                                while (sh1.MoveNext())
                                {
                                    var sheet1Crnt = sh1.Current;

                                    // 共有元のサブグループリスト外位置するなら
                                    if (sheet1Crnt.Value.SequenceEqual(sheet0Crnt.Value))
                                    {
                                        // 新しく作成したサブグループに割り当て
                                        GlyphCode sgc = new GlyphCode(sheet1Crnt.Key, 0);
                                        sub.CodeList.Add(sgc);
                                        codeSubIndexDic[sheet1Crnt.Key] = sub.Index;
                                        sheet1Crnt.Value.Clear();
                                    }
                                }

                                sheet0Crnt.Value.Clear();
                            }
                        }
                    }
                }

                // 共有されていた文字をもっていたサブグループの親に
                // 共有されていた文字を持つサブグループを追加
                {
                    // 全てのグループに対し
                    foreach (var g in this.Groups)
                    {
                        IndexList slist = g.Value.SubGroups;
                        int subNum = slist.Count;

                        // 全てのサブグループに対し
                        for (int sno = 0; sno < subNum; ++sno)
                        {
                            // 対象のグループが対象のサブグループを持っているなら
                            SubGroup value;
                            var s = this.SubGroups.TryGetValue(slist[sno], out value);
                            if (s && value != null)
                            {
                                CodeList clist = value.CodeList;

                                // サブグループが持つ全ての文字コードに対し
                                foreach (var c in clist)
                                {
                                    // 再マップが登録されていて
                                    int ivalue;
                                    var cs = codeSubIndexDic.TryGetValue(c.CharCode, out ivalue);
                                    if (cs)
                                    {
                                        IndexList ilist = g.Value.SubGroups;

                                        // 未登録のグループならば登録する
                                        if (!ilist.Contains(ivalue))
                                        {
                                            ilist.Add(ivalue);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                {
                    var delKeyList = new List<int>();   // 削除対象のキーのリスト
                    // 共有されていた文字を持っていたサブグループから
                    // 共有されていた文字を削除
                    foreach (var s in this.SubGroups)
                    {
                        if (s.Value != null
                           && s.Value.Index <= maxBasicSubGroupIndex)
                        {
                            CodeList clist = s.Value.CodeList;
                            clist.RemoveAll(c => codeSubIndexDic.ContainsKey(c.CharCode));

                            if (clist.Count == 0)
                            {
                                Rpt._RPT1("delete subgroup: {0} no glyph\n", s.Value.Index);

                                delKeyList.Add(s.Key);
                            }
                        }
                    }

                    foreach (var delKey in delKeyList)
                    {
                        this.SubGroups[delKey] = null;
                    }
                }
            }

            // この時点で全てのサブグループはユニークな文字のみをもつ

            // サブグループをシートに割り当て
            {
                var surplusList = new SurplusList();
                int sheetIdx = 0;

                // SurplusList の作成
                foreach (var s in this.SubGroups)
                {
                    SubGroup sub = s.Value;
                    if (sub != null)
                    {
                        var numGlyph = sub.CodeList.Count;
                        var surplus = numGlyph % this.TglpInfo.CellNum;
                        SheetSurplus ss = new SheetSurplus(sub, numGlyph, surplus);
                        surplusList.Add(ss);
                    }
                }

                // surplus 昇順ソート
                /*
                    オリジナルでは、std::sort()を使っている。
                    それに合わせて、surplusList.Sort() とすると、
                    どちらの関数の場合も同じ値の順番が維持されない仕様のためか、
                    オリジナルと結果が一致しない。
                    Enumerable.OrderBy の場合は順番を維持する仕様であり、
                    こちらを使うと現在の実装の方は順番が維持され、
                    また、偶然にもオリジナルと結果が一致する。
                */
                {
                    var newSurplusList = new SurplusList();
                    foreach (var sp in surplusList.OrderBy(sp => sp.Surplus))
                    {
                        newSurplusList.Add(sp);
                    }
                    surplusList = newSurplusList;
                }

                // 統合する必要が無いもの
                // シートの整数倍にきっちり収まる物
                {
                    // surplus 昇順を前提としたコード
                    var eraceCount = 0;
                    foreach (var sp in surplusList)
                    {
                        if (sp.Surplus > 0)
                        {
                            break;
                        }

                        var numSheet = GlCm.DIV_UP(sp.NumGlyph, this.TglpInfo.CellNum);
                        sp.Sub.SheetIndexBegin = sheetIdx;
                        sp.Sub.SheetIndexBeginOffset = 0;
                        sp.Sub.SheetIndexEnd = sheetIdx + numSheet - 1;
                        sheetIdx += numSheet;
                        Rpt._RPT3("sheet assign p: sheet {0} -> {1} sub {2}\n", sp.Sub.SheetIndexBegin, sp.Sub.SheetIndexEnd, sp.Sub.Index);
                        eraceCount++;
                    }

                    surplusList.RemoveRange(0, eraceCount);
                }

                // 3つ以上統合可能な物
                // -> 端数の和が 1 枚のシートに収まり
                //    かつ 1 枚のシートに収まらない物が 2 つ以下
                for (var i = 0; i < surplusList.Count;)
                {
                    int sumSurplus = 0;
                    List<int> bigSp = new List<int>();
                    List<int> uniteList = new List<int>();
                    {
                        SheetSurplus ssi = surplusList[i];
                        if (ssi.NumGlyph > this.TglpInfo.CellNum)
                        {
                            bigSp.Add(i);
                        }

                        uniteList.Add(i);
                        sumSurplus += ssi.Surplus;
                    }

                    // 統合対象を探す
                    for (var j = i + 1; j < surplusList.Count; ++j)
                    {
                        SheetSurplus ssj = surplusList[j];
                        if (sumSurplus + ssj.Surplus > this.TglpInfo.CellNum)
                        {
                            continue;
                        }

                        if (ssj.NumGlyph > this.TglpInfo.CellNum)
                        {
                            if (bigSp.Count > 1)
                            {
                                continue;
                            }

                            bigSp.Add(j);
                        }

                        uniteList.Add(j);
                        sumSurplus += ssj.Surplus;
                    }

                    // 統合する
                    if (uniteList.Count > 2)
                    {
                        int offset = 0;

                        Rpt._RPT1("sheet assign >2: sheet b {0} ", sheetIdx);
                        if (bigSp.Count > 0)
                        {
                            SheetSurplus sp = surplusList[bigSp[0]];
                            var numSheet = GlCm.DIV_UP(sp.NumGlyph, this.TglpInfo.CellNum);
                            sp.Sub.SheetIndexBegin = sheetIdx;
                            sp.Sub.SheetIndexBeginOffset = offset;
                            sp.Sub.SheetIndexEnd = sheetIdx + numSheet - 1;
                            sheetIdx += numSheet - 1;
                            offset += sp.NumGlyph % this.TglpInfo.CellNum;
                        }

                        Rpt._RPT1("c {0} ", sheetIdx);
                        for (var k = 0; k < uniteList.Count; ++k)
                        {
                            var isBig = false;

                            for (var b = 0; b < bigSp.Count; ++b)
                            {
                                if (uniteList[k] == bigSp[b])
                                {
                                    isBig = true;
                                    break;
                                }
                            }

                            if (!isBig)
                            {
                                SheetSurplus sp = surplusList[uniteList[k]];
                                Debug.Assert(sp.NumGlyph == sp.Surplus);
                                sp.Sub.SheetIndexBegin = sheetIdx;
                                sp.Sub.SheetIndexBeginOffset = offset;
                                sp.Sub.SheetIndexEnd = sheetIdx;
                                offset += sp.NumGlyph;
                                Debug.Assert(offset < this.TglpInfo.CellNum);
                            }
                        }

                        if (bigSp.Count > 1)
                        {
                            SheetSurplus sp = surplusList[bigSp[1]];
                            var numSheet = GlCm.DIV_UP(sp.NumGlyph, this.TglpInfo.CellNum);
                            sp.Sub.SheetIndexBegin = sheetIdx;
                            sp.Sub.SheetIndexBeginOffset = offset;
                            sp.Sub.SheetIndexEnd = sheetIdx + numSheet - 1;
                            sheetIdx += numSheet - 1;
                            offset += sp.NumGlyph % this.TglpInfo.CellNum;
                            Debug.Assert(offset < this.TglpInfo.CellNum);
                        }

                        Rpt._RPT1("e {0} ", sheetIdx);

                        sheetIdx++;

                        Rpt._RPT0("sub ");
                        for (int k = uniteList.Count - 1; k >= 0; --k)
                        {
                            Rpt._RPT1("{0} ", surplusList[uniteList[k]].Sub.Index);
                            surplusList.RemoveAt(uniteList[k]);
                        }

                        Rpt._RPT0("\n");

                        continue;
                    }

                    ++i;
                }

                // 2つ統合可能なもの
                for (var i = 0; i < surplusList.Count;)
                {
                    int sumSurplus = surplusList[i].Surplus;
                    int j;

                    for (j = i + 1; j < surplusList.Count; ++j)
                    {
                        if (sumSurplus + surplusList[j].Surplus <= this.TglpInfo.CellNum)
                        {
                            break;
                        }
                    }

                    if (j < surplusList.Count)
                    {
                        int offset = 0;
                        this.SetSheetSurplus(surplusList[i], ref sheetIdx, ref offset);
                        this.SetSheetSurplus(surplusList[j], ref sheetIdx, ref offset);
                        Debug.Assert(offset < this.TglpInfo.CellNum);

                        sheetIdx++;

                        Rpt._RPT3("sheet assign 2: sheet {0} -> {1} c {2} ", surplusList[i].Sub.SheetIndexBegin, surplusList[i].Sub.SheetIndexEnd, surplusList[j].Sub.SheetIndexBegin);

                        Rpt._RPT2("sub {0} {1}\n", surplusList[i].Sub.Index, surplusList[j].Sub.Index);
                        surplusList.RemoveAt(j);
                        surplusList.RemoveAt(i);

                        continue;
                    }

                    ++i;
                }

                /* 統合できないもの */

                foreach (var sp in surplusList)
                {
                    var numSheet = GlCm.DIV_UP(sp.NumGlyph, this.TglpInfo.CellNum);
                    sp.Sub.SheetIndexBegin = sheetIdx;
                    sp.Sub.SheetIndexBeginOffset = 0;
                    sp.Sub.SheetIndexEnd = sheetIdx + numSheet - 1;
                    sheetIdx += numSheet;
                    Rpt._RPT3("sheet assign 1: sheet {0} -> {1} sub {2}\n", sp.Sub.SheetIndexBegin, sp.Sub.SheetIndexEnd, sp.Sub.Index);
                }

                surplusList.Clear();

                // TODO: シートサイズの決定方法とタイミング
                this.TglpInfo.SheetNum = sheetIdx;
                this.TglpInfo.AllSheetCellNum = this.TglpInfo.SheetNum * this.TglpInfo.CellNum;
            }

            // シートのマージ
            {
                // なくてもよい
                // TODO
            }

            // グリフインデックスの割り当て
            {
                GlyphList glist = fontData.GetGlyphList();

                foreach (var s in this.SubGroups)
                {
                    if (s.Value != null)
                    {
                        CodeList clist = s.Value.CodeList;

                        // 文字コード順にソート
                        clist.Sort();

                        ushort gindex = (ushort)(
                                          (s.Value.SheetIndexBegin * this.TglpInfo.CellNum)
                                        + s.Value.SheetIndexBeginOffset);

                        for (var c = 0; c < clist.Count; ++c)
                        {
                            clist[c] = new GlyphCode(clist[c].CharCode, gindex++);

                            Glyph pg = glist.GetByCode(clist[c].CharCode);
                            if (pg == null)
                            {
                                if (G2dFont.IsUnicodeEncoding(this.Encoding))
                                {
                                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_NOT_INCLUDED_UNICODE, this.AlterCodeUser, this.alterCode);
                                }
                                else
                                {
                                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_NOT_INCLUDED_LOCAL, this.AlterCodeUser, this.alterCode);
                                }
                            }

                            pg.SetIndex(clist[c].GryphIndex);
                            //// _RPT2(_CRT_WARN, "%04X %04X\n", clist[c].ccode, clist[c].gindex);
                        }
                    }
                }
            }

            // 代替文字インデックス
            {
                Glyph pg = fontData.GetGlyphList().GetByCode(this.alterCode);

                if (pg != null)
                {
                    this.alterIndex = pg.GetIndex();
                }
                else
                {
                    if (G2dFont.IsUnicodeEncoding(this.Encoding))
                    {
                        throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_NOT_INCLUDED_UNICODE, this.AlterCodeUser, this.alterCode);
                    }
                    else
                    {
                        throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_ALTER_NOT_INCLUDED_LOCAL, this.AlterCodeUser, this.alterCode);
                    }
                }
            }
        }

        private void SetSheetSurplus(SheetSurplus sp, ref int sheetIdx, ref int offset)
        {
            var numSheet = GlCm.DIV_UP(sp.NumGlyph, this.TglpInfo.CellNum);
            sp.Sub.SheetIndexBegin = sheetIdx;
            sp.Sub.SheetIndexBeginOffset = offset;
            sp.Sub.SheetIndexEnd = sheetIdx + numSheet - 1;
            sheetIdx += numSheet - 1;
            offset += sp.NumGlyph % this.TglpInfo.CellNum;
        }

        /*
            ブロック作成
        */

        /// <summary>
        /// グループブロック作成
        /// </summary>
        /// <param name="font"></param>
        /// <param name="fontData"></param>
        private void CreateGroupBlock(
            G2dFont font,
            FontData fontData)
        {
            // グループが1つしか存在しないならグループブロックは作成しない
            if (this.Groups.Count < 2)
            {
                return;
            }

            // !! 暫定 !!
            // サブグループのCMAPリストを作成
            // 本来はこの場所で行うべきではない
            // 全サブグループで全CMAPをロードする
            {
                int numCMAP = font.GetNumBlock(Runtime.RtConsts.BinBlockSigCMAP);
                foreach (var s in this.SubGroups)
                {
                    if (s.Value != null)
                    {
                        s.Value.MapBlocks.Capacity = numCMAP;
                        for (int i = 0; i < numCMAP; ++i)
                        {
                            s.Value.MapBlocks.Add(i);
                        }
                    }
                }
            }

            // ソート済みグループリストを作成する
            List<Group> glist = new List<Group>();
            GlCm.MakeListFromMapValue<string, Group>(glist, this.Groups);
            glist.Sort(DereferenceComparator);

            // シートセットリストを作成する
            G2dFont.SheetSetList setList = new G2dFont.SheetSetList();
            {
                var numCWDH = font.GetNumBlock(Runtime.RtConsts.BinBlockSigCWDH);
                var numCMAP = font.GetNumBlock(Runtime.RtConsts.BinBlockSigCMAP);

                foreach (var g in glist)
                {
                    G2dFont.SheetSet set = new G2dFont.SheetSet(g.Name);

                    this.InitBitArray(set.UseSheets, this.TglpInfo.SheetNum);
                    this.InitBitArray(set.UseCWDH, numCWDH);
                    this.InitBitArray(set.UseCMAP, numCMAP);

                    foreach (var s in g.SubGroups)
                    {
                        SubGroup value;
                        var found = this.SubGroups.TryGetValue(s, out value);

                        if (found && value != null)
                        {
                            SubGroup sg = value;
                            for (int no = sg.SheetIndexBegin; no <= sg.SheetIndexEnd; ++no)
                            {
                                this.BitArray_Set(set.UseSheets, no);
                            }

                            this.BitArray_Merge(set.UseCWDH, sg.WidthBlocks);
                            this.BitArray_Merge(set.UseCMAP, sg.MapBlocks);
                        }
                    }

                    setList.Add(set);
                }
            }

            // グループブロックを登録する
            font.SetGlyphGroupsBlock(
                (uint)this.TglpInfo.SheetBytes,
                (ushort)this.TglpInfo.CellNum,
                (ushort)this.TglpInfo.SheetNum,
                (ushort)glist.Count,
                setList,
                this.CompSheetSizes);
        }

        /// <summary>
        /// INFOブロック作成
        /// </summary>
        /// <param name="font"></param>
        /// <param name="fontData"></param>
        private void CreateInfoBlock(G2dFont font, FontData fontData)
        {
            {
                Runtime.CharWidths widths = fontData.DefaultWidth.Value;
                int width = fontData.Width.Value;
                int ascent = fontData.Ascent.Value;
                int descent = fontData.Descent.Value;
                int linefeed = GetDefaultOrValue(this.Linefeed, fontData.LineHeight.Value);

                if (linefeed > short.MaxValue)
                {
                    ProgressControl.Warning(Strings.IDS_WARN_OVERMAX_LINEFEED, linefeed, short.MaxValue);
                }

                GlyphImageFormat imgFmt = fontData.OutputFormat.Value;
                GlyphDataType glyphDataType = (
                    imgFmt == GlyphImageFormat.LA4Packed || imgFmt == GlyphImageFormat.LA4PackedPassThrough ||
                    imgFmt == GlyphImageFormat.LA4PackedNoCompress || imgFmt == GlyphImageFormat.LA4PackedNoCompressPassThrough) ? GlyphDataType.PackedTexture : this.Type;

                font.SetInformationBlock(
                    (byte)glyphDataType,
                    (short)linefeed,
                    (byte)width,
                    (byte)(ascent + descent),
                    (byte)ascent,
                    this.alterIndex,
                    (sbyte)GetDefaultOrValue(this.DefalutLeft, widths.Left),
                    (byte)GetDefaultOrValue(this.DefaultWidth, widths.GlyphWidth),
                    (sbyte)GetDefaultOrValue(this.DefaultRight, widths.CharWidth - (widths.Left + widths.GlyphWidth)),
                    (byte)this.Encoding);
            }
        }

        // create glyph block
        private void CreateGlyphBlock(G2dFont font, FontData fontData)
        {
            int srcImageSize = this.TglpInfo.SheetBytes * this.TglpInfo.SheetNum;
            byte[] srcImage = new byte[srcImageSize];
            ushort sheetFormat;

            fontData.OutputFormat = ConvertGlyphImageFormat_(fontData.OutputFormat.Value);

            // 実機向けにテクスチャフォーマットを変換します。
            // (圧縮フォーマット化(指定があれば) + TextureArray化 + Tiling変換)
            {
                RgbImage sheet_plane = new RgbImage();
                this.CreateSheetPlane(sheet_plane, this.TglpInfo, fontData);

                List<byte> convertResultTotal = ConvertToNativeImage_(sheet_plane, fontData.OutputFormat.Value);
                if(convertResultTotal.Count == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_STATUS_ERROR);
                }

                srcImage = convertResultTotal.ToArray();
                srcImageSize = srcImage.Length;
                this.TglpInfo.SheetBytes = srcImageSize / this.TglpInfo.SheetNum;
                sheetFormat = GlyphImageFormatToTextureFormat(fontData.OutputFormat.Value);
            }

            var tgtImage = srcImage;
            int tgtImageSize = srcImageSize;

            // シート枚数上限オーバーの確認
            if (this.TglpInfo.SheetNum > byte.MaxValue)
            {
                ProgressControl.Warning(Strings.IDS_ERR_SHEETNUM_OVER255, this.TglpInfo.SheetNum);
            }

            // 圧縮形式フォントはCafeで廃止されました。
            Trace.Assert(!this.isCompress, "Compressed format is obsolete.");
            font.SetTextureGlyphBlock(
                (byte)this.TglpInfo.CellWidth,
                (byte)this.TglpInfo.CellHeight,
                (short)fontData.BaselinePos.Value,
                (byte)fontData.MaxCharWidth.Value,
                (uint)this.TglpInfo.SheetBytes,
                (byte)this.TglpInfo.SheetNum,
                sheetFormat,
                (ushort)this.TglpInfo.CellHCount,
                (ushort)this.TglpInfo.CellVCount,
                (ushort)this.TglpInfo.SheetWidth,
                (ushort)this.TglpInfo.SheetHeight,
                tgtImage,
                tgtImageSize);

            Rpt._RPT4("[TGLP] cw={0} ch={1} sw={2} sh={3}", this.TglpInfo.CellWidth, this.TglpInfo.CellHeight, this.TglpInfo.SheetWidth, this.TglpInfo.SheetHeight);
            Rpt._RPT4(" ch={0} cv={1} ss={2} sf={3}\n", this.TglpInfo.CellHCount, this.TglpInfo.CellVCount, this.TglpInfo.SheetBytes, fontData.OutputFormat.Value);
        }

        /// <summary>
        /// 色情報を反転させる必要があるかどうか？
        /// </summary>
        protected bool NeedToInvertColortValue(GlyphImageFormat gif)
        {
            return
                gif == GlyphImageFormat.A4 ||
                gif == GlyphImageFormat.A8 ||
                gif == GlyphImageFormat.LA4 ||
                gif == GlyphImageFormat.LA8 ||
                gif == GlyphImageFormat.LA4Packed||
                gif == GlyphImageFormat.LA4PackedPassThrough||
                gif == GlyphImageFormat.LA4PackedNoCompress ||
                gif == GlyphImageFormat.LA4PackedNoCompressPassThrough;
        }

        // create character width block
        private void CreateWidthBlock(G2dFont font, FontData fontData)
        {
            var list = fontData.GetGlyphList();
            CharWidths[] widthTable = new CharWidths[this.TglpInfo.AllSheetCellNum];
            ushort maxGlyphIndex = 0;
            ushort minGlyphIndex = 0xFFFF;

            // グリフインデックス―GlyphCode→文字コード―CGlyph→文字幅

            /* TODO: 現在は全てのグリフの幅情報を出力 */

            foreach (var s in this.SubGroups)
            {
                if (s.Value != null)
                {
                    CodeList clist = s.Value.CodeList;
                    foreach (var c in clist)
                    {
                        uint code = c.CharCode;
                        ushort gindex = c.GryphIndex;
                        var glyph = list.GetByCode(code);

                        if (glyph != null && gindex < this.TglpInfo.AllSheetCellNum)
                        {
                            Runtime.CharWidths pos = new Runtime.CharWidths();
                            pos.Left = (sbyte)glyph.Left();
                            pos.GlyphWidth = (byte)glyph.Width();
                            pos.CharWidth = (byte)glyph.CharFeed();
                            widthTable[(int)gindex] = pos;

                            minGlyphIndex = Math.Min(minGlyphIndex, gindex);
                            maxGlyphIndex = Math.Max(maxGlyphIndex, gindex);

                            ProgressControl.GetInstance().StepProgressBar();
                        }
                    }
                }
            }

            // ランタイムでは全てのセットで 0 番ブロックのみ読み込むようにする
            foreach (var s in this.SubGroups)
            {
                if (s.Value != null)
                {
                    s.Value.WidthBlocks.Add(0);
                }
            }

            if (minGlyphIndex <= maxGlyphIndex)
            {
                font.AddWidthBlock(
                    minGlyphIndex,
                    maxGlyphIndex,
                    widthTable);
            }
            else
            {
                // todo warning
            }
        }

        // create ccode map block
        private void CreateCMapBlock(G2dFont font, FontData fontData)
        {
            var list = fontData.GetGlyphList();
            var glyphs = new GlyphLinkedList();

            foreach (var g in list.GetEnum())
            {
                glyphs.AddLast(g);
            }

            /*
                SubGroupMap::iterator s;
                for( s = m_subGroups.begin(); s != m_subGroups.end(); ++s )
                {
                    CreateCMapBlockDirect(pFont, s->second, &glyphs, pFontData);
                    CreateCMapBlockTable(pFont, s->second, &glyphs, pFontData);
                    CreateCMapBlockScan(pFont, s->second, &glyphs, pFontData);
                }
            */
            this.CreateCMapBlockDirect(font, null, glyphs, fontData);
            this.CreateCMapBlockTable(font, null, glyphs, fontData);
            this.CreateCMapBlockScan(font, null, glyphs, fontData);
        }

        // create kerning block
        private void CreateKrngBlock(G2dFont font, FontData fontData)
        {
            font.AddKerningBlock(fontData.KerningPairs);
        }

        private void CreateCMapBlockDirect(
            G2dFont font,
            SubGroup sub,
            GlyphLinkedList glyphs,
            FontData fontData)
        {
            var range = new Range();

            LinkedListNode<Glyph> pos = glyphs.First;

            while (pos != null && pos != glyphs.Last.Next)
            {
                // DIRECT 対象ブロックを探す
                while (true)
                {
                    // ブロックを探す
                    var isFound = this.GetNextBlock(ref range, glyphs.Last, pos);

                    // もうない
                    if (!isFound)
                    {
                        return;
                    }

                    pos = range.Last.Next;

                    // 一定数以上連続しているならDIRECTブロック作成
                    if (range.Last.Value.GetCode() - range.First.Value.GetCode() + 1 >= CmapDirectBlockThreshold)
                    {
                        break;
                    }
                }

                // DIRECT ブロック作成
                {
                    ushort offset = range.First.Value.GetIndex();

                    Rpt._RPT4(
                        "DIRECT: 0x{0:X4}-0x{1:X4}  num {2}  offset {3}\n",
                        range.First.Value.GetCode(),
                        range.Last.Value.GetCode(),
                        range.Last.Value.GetCode() - range.First.Value.GetCode() + 1,
                        offset);

                    font.AddCMapBlock(
                        range.First.Value.GetCode(),
                        range.Last.Value.GetCode(),
                        FontMapMethod.Direct,
                        new ushort[] { offset });
                }

                ProgressControl.GetInstance().StepProgressBar(
                    range.Last.Value.GetIndex() - range.First.Value.GetIndex());

                // リストから削除
                glyphs.Erase(range.First, range.Last);
            }
        }

        private void CreateCMapBlockTable(
            G2dFont font,
            SubGroup sub,
            GlyphLinkedList glyphs,
            FontData fontData)
        {
            Range r = new Range();

            LinkedListNode<Glyph> pos = glyphs.First;

            while (pos != null && pos != glyphs.Last.Next)
            {
                // ブロックを探す
                bool isFound = this.GetNextBlock(ref r, glyphs.Last, pos);

                // もうない
#if NW_FONT_NITROFONTWRITER_BUGFIX
                if (!isFound)
                {
                    return;
                }
#else
                if (r.First == glyphs.Last.Next)
                {
                    return;
                }
#endif

                //// _RPT2(_CRT_WARN, "TABLE step1: 0x%04X-0x%04X\n",
                ////         (*(r.first))->GetCode(),
                ////         (*(r.last))->GetCode() );
                {
                    int glyphsNum = (int)r.Last.Value.GetCode() - (int)r.First.Value.GetCode() + 1;
                    int emptyNum = 0;

                    while (true)
                    {
                        var nextRange = new Range();
                        var prevRange = new Range();
                        float nextValue = 0;
                        float prevValue = 0;

                        // 次のブロックを加えた場合を評価
#if NW_FONT_NITROFONTWRITER_BUGFIX
                        if (r.Last != glyphs.Last)
#else
                        if (r.Last != glyphs.Last.Next)
#endif
                        {
                            LinkedListNode<Glyph> next = r.Last.Next;

                            var isNextFound = this.GetNextBlock(ref nextRange, glyphs.Last, next);
                            if (isNextFound)
                            {
                                var nextGlyphs = nextRange.Last.Value.GetCode() - nextRange.First.Value.GetCode() + 1;
                                var nextEmpty = nextRange.First.Value.GetCode() - r.Last.Value.GetCode() - 1;
                                var sumGlyphs = glyphsNum + nextGlyphs;
                                var range = sumGlyphs + emptyNum + nextEmpty;
                                nextValue = (float)sumGlyphs / range;

                                ////                    _RPT2(_CRT_WARN, " TABLE step2: Next 0x%04X-0x%04X ",
                                ////                        (*(rNext.first))->GetCode(), (*(rNext.last))->GetCode() );
                                ////                    _RPT3(_CRT_WARN, "g %d  e %d (v %1.3f)\n",
                                ////                        nNextGlyphs, nNextEmpty, vNext );
                            }
                        }

                        // 前のブロックを加えた場合を評価
                        if (r.First != glyphs.First)
                        {
                            LinkedListNode<Glyph> prev = r.First.Previous;

                            var isPrevFound = this.GetPrevBlock(ref prevRange, glyphs.First, prev);

                            if (isPrevFound)
                            {
                                int prevGlyphs = (int)prevRange.Last.Value.GetCode() - (int)prevRange.First.Value.GetCode() + 1;
                                int prevEmpty = (int)r.First.Value.GetCode() - (int)prevRange.Last.Value.GetCode() - 1;
                                int sumGlyphs = glyphsNum + prevGlyphs;
                                var range = sumGlyphs + emptyNum + prevEmpty;
                                prevValue = (float)sumGlyphs / range;

                                ////                    _RPT2(_CRT_WARN, " TABLE step2: Prev 0x%04X-0x%04X ",
                                ////                        (*(rPrev.first))->GetCode(), (*(rPrev.last))->GetCode() );
                                ////                    _RPT3(_CRT_WARN, "g %d  e %d (v %1.3f)\n",
                                ////                        nPrevGlyphs, nPrevEmpty, vPrev );
                            }
                        }

                        // 評価が高い方をブロックに追加
                        if (nextValue > prevValue)
                        {
                            if (nextValue < 0.5)
                            {
                                break;
                            }

                            int nextGlyphsNum = (int)nextRange.Last.Value.GetCode() - (int)nextRange.First.Value.GetCode() + 1;
                            int nextEmptyNum = (int)nextRange.First.Value.GetCode() - (int)r.Last.Value.GetCode() - 1;
                            glyphsNum += nextGlyphsNum;
                            emptyNum += nextEmptyNum;
                            r.Last = nextRange.Last;

                            //// _RPT0(_CRT_WARN, " TABLE step3: select Next\n");
                        }
                        else
                        {
                            if (prevValue < 0.5)
                            {
                                break;
                            }

                            int prevGlyphsNum = (int)prevRange.Last.Value.GetCode() - (int)prevRange.First.Value.GetCode() + 1;
                            int prevEmptyNum = (int)r.First.Value.GetCode() - (int)prevRange.Last.Value.GetCode() - 1;
                            glyphsNum += prevGlyphsNum;
                            emptyNum += prevEmptyNum;
                            r.First = prevRange.First;

                            //// _RPT0(_CRT_WARN, " TABLE step3: select Prev\n");
                        }

                        Debug.Assert((float)glyphsNum / (glyphsNum + emptyNum) >= 0.5);
                    }

                    // 一定数以上ならTABLEブロック登録
                    if (glyphsNum < CmapTableBlockThreshold)
                    {
                        pos = r.Last.Next;

                        continue;
                    }

                    ////        _RPT2(_CRT_WARN, " TABLE step4: 0x%04X-0x%04X ",
                    ////            (*(r.first))->GetCode(), (*(r.last))->GetCode() );
                    ////        _RPT3(_CRT_WARN, "g %d  e %d (v %1.3f)\n",
                    ////            nGlyphs, nEmpty, (float)(nGlyphs) / (nGlyphs + nEmpty) );
                }

                // TABLE ブロック作成
                {
                    uint firstCode = r.First.Value.GetCode();
                    uint lastCode = r.Last.Value.GetCode();
                    int num = (int)lastCode - (int)firstCode + 1;
                    var mapTable = new ushort[num];

                    for (int i = 0; i < num; ++i)
                    {
                        Glyph pg = fontData.GetGlyphList().GetByCode((uint)(firstCode + i));

                        ushort map = pg != null ? pg.GetIndex() : Runtime.RtConsts.InvalidGlyphIndex;
                        mapTable[i] = map;
                    }

                    font.AddCMapBlock(
                        firstCode,
                        lastCode,
                        FontMapMethod.Table,
                        mapTable);
                }

                pos = r.Last.Next;

                ProgressControl.GetInstance().StepProgressBar(
                    r.Last.Value.GetIndex() - r.First.Value.GetIndex());

                // リストから削除
                glyphs.Erase(r.First, r.Last);
            }
        }

        private bool GetNextBlock(
            ref Range r,
            LinkedListNode<Glyph> last,
            LinkedListNode<Glyph> pos)
        {
            // 連続するブロックを探す
            if (pos == null || pos == last.Next)
            {
                return false;
            }

            // 領域の始点
            uint nextc = pos.Value.GetCode() + 1;
            ushort nextg = (ushort)(pos.Value.GetIndex() + 1);

            LinkedListNode<Glyph> prePos = pos;
            LinkedListNode<Glyph> range_end = prePos.Next;

            // 連続するコードをたどる
            while (range_end != last.Next)
            {
                if (range_end.Value.GetCode() != nextc
                  || range_end.Value.GetIndex() != nextg)
                {
                    // 途切れた
                    break;
                }

                prePos = range_end;
                range_end = prePos.Next;
                ++nextc;
                ++nextg;
            }

            r.First = pos;
            r.Last = prePos;

            return true;
        }

        private bool GetPrevBlock(
                ref Range r,
                LinkedListNode<Glyph> begin,
                LinkedListNode<Glyph> pos)
        {
#if NW_FONT_NITROFONTWRITER_BUGFIX
            begin = begin.Previous;
#endif

            if (pos == null || pos == begin)
            {
                return false;
            }

            // 連続するブロックを探す
            LinkedListNode<Glyph> range_end = pos;

#if !NW_FONT_NITROFONTWRITER_BUGFIX
            range_end = range_end.Previous;
            if (range_end == begin)
            {
                return false;
            }
#endif

            uint prev = range_end.Value.GetCode() - 1;

            LinkedListNode<Glyph> range_begin = range_end.Previous;

            // 連続するコードをたどる
            while (range_begin != begin)
            {
                if (range_begin.Value.GetCode() != prev)
                {
                    // 途切れた
                    break;
                }

                range_begin = range_begin.Previous;
                --prev;
            }

            r.First = range_begin.Next;
            r.Last = range_end;

            return true;
        }

        private void CreateCMapBlockScan(
            G2dFont font,
            SubGroup sub,
            GlyphLinkedList glyphs,
            FontData fontData)
        {
            int maxNum = glyphs.Count;
            ushort outNum = 0;

            var list = new List<Runtime.CMapScanEntry>(maxNum);
            foreach (var glyph in glyphs)
            {
                CMapScanEntry ent;
                ent.Code = glyph.GetCode();
                ent.Index = glyph.GetIndex();
                ent.Padding = 0;
                list.Add(ent);
                outNum++;

                ProgressControl.GetInstance().StepProgressBar();
                //// _RPT1(_CRT_WARN, "SCAN: %04X\n", pGlyph->GetCode());
            }

            if (outNum <= 0)
            {
                return;
            }

            font.AddCMapBlock(
                0x0000,
                0xFFFF,
                FontMapMethod.Scan,
                list.ToArray());
        }

        /*
            パラメータ計算
        */

        private void CalcTGLPInfo(
            TGLPInfo pti,
            FontData fontData)
        {
            pti.SheetBpp = GlyphImageInfo.GetGlyphImageInfo(fontData.OutputFormat.Value).Bpp;
            pti.AllSheetCellNum = fontData.GetGlyphList().GetNum();

            pti.CellWidth = fontData.CellWidth.Value;
            pti.CellHeight = fontData.CellHeight.Value;

            if (this.SheetWidth == 0 || this.SheetHeight == 0)
            {
                this.CalcTextureSize(pti, this.SheetPixels);
            }
            else
            {
                pti.SheetWidth = this.SheetWidth;
                pti.SheetHeight = this.SheetHeight;
            }

            //---- LIEARフィルタ時の境界滲み対策
            pti.CellHCount = pti.SheetWidth / (pti.CellWidth + 1);
            pti.CellVCount = pti.SheetHeight / (pti.CellHeight + 1);
            pti.CellNum = pti.CellHCount * pti.CellVCount;



            pti.SheetNum = GlCm.DIV_UP(pti.AllSheetCellNum, pti.CellNum);
            pti.SheetBytes = GlCm.DIV_UP(pti.SheetWidth * pti.SheetHeight * pti.SheetBpp, 8);
        }

        /*
            フォントのシート1枚のサイズを計算する。

            縦横を32～1024の2の累乗全ての組み合わせで動かし、必要なシート枚数と余りサイズを
            求める。その中で、シート枚数が一番少ないもの、シート枚数が同じ組み合わせがある場合は
            余りサイズが小さいものを選び、引数ptiのSheetWidthとSheetHeightにセットする。
        */
        private void CalcTextureSize(TGLPInfo pti, int pixels)
        {
            int cellBytesSum = (pti.CellWidth + 1) * (pti.CellHeight + 1) * pti.AllSheetCellNum;
            int choosedUselessBytes = 0;
            int minSheetNum = int.MaxValue;

            for (int texWidth = 32; texWidth <= 1024; texWidth *= 2)
            {
                int cellAreaWidth = texWidth - 2;
                if (cellAreaWidth < pti.CellWidth)
                {
                    continue;
                }

                for (int texHeight = 32; texHeight <= 1024; texHeight *= 2)
                {
                    if ((pixels != 0) && (texWidth * texHeight != pixels))
                    {
                        continue;
                    }

                    int cellAreaHeight = texHeight - 2;
                    if (cellAreaHeight < pti.CellHeight)
                    {
                        continue;
                    }

                    int cellHCount = (cellAreaWidth + 1) / (pti.CellWidth + 1);
                    int cellVCount = (cellAreaHeight + 1) / (pti.CellHeight + 1);

                    int cellNum = cellHCount * cellVCount;
                    int sheetNum = GlCm.DIV_UP(pti.AllSheetCellNum, cellNum);
                    int sheetBytes = cellAreaWidth * cellAreaHeight * sheetNum;

                    int uselessBytes = sheetBytes - cellBytesSum;

                    if (sheetNum < minSheetNum || (sheetNum == minSheetNum && uselessBytes < choosedUselessBytes))
                    {
                        choosedUselessBytes = uselessBytes;
                        minSheetNum = sheetNum;
                        pti.SheetWidth = texWidth;
                        pti.SheetHeight = texHeight;
                    }
                }
            }

            if (minSheetNum == int.MaxValue)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_INVALID_SHEETPIXELS, pixels);
            }
        }

        /// <summary>
        /// v
        /// </summary>
        private void CreateSheetPlane(
            RgbImage image,
            TGLPInfo ti,
            FontData fontData)
        {
            Debug.Assert(ti.AllSheetCellNum > 0);
            GlyphList glist = fontData.GetGlyphList();
            RgbImage gimage = new RgbImage();

            image.Create(ti.SheetWidth, ti.SheetHeight * ti.SheetNum, 32);
            image.Clear(fontData.NullColor.Value, ImageBase.AlphaTransparent);
            image.EnableAlpha();

            // pFontData->GetGlyphList() がもつ CGlyphs に
            // 正しい文字コードが設定されていなければならない
            // m_subGroups.codeList に適切な文字コード/グリフインデックスペアが
            // 設定されていなければならない

            // 全てのサブグループに対してサブグループが持つグリフを
            // グリフインデックスから位置を計算してシート上に配置する
            {
                var cellWidth = ti.CellWidth + 1;
                var cellHeight = ti.CellHeight + 1;

                foreach (var s in this.SubGroups)
                {
                    if (s.Value != null)
                    {
                        CodeList clist = s.Value.CodeList;
                        foreach (var c in clist)
                        {
                            uint code = c.CharCode;
                            ushort gindex = c.GryphIndex;
                            int line = gindex / ti.CellHCount;
                            var glyph = glist.GetByCode(code);

                            if (glyph != null && line < this.TglpInfo.AllSheetCellNum)
                            {
                                int px = ((gindex % ti.CellHCount) * cellWidth) + 1;
                                int py = ((line / ti.CellVCount) * ti.SheetHeight)
                                       + ((line % ti.CellVCount) * cellHeight)
                                       + 1;

                                if (glyph != null)
                                {
                                    glyph.ExtractGlyphImage(gimage, fontData.BaselinePos.Value);
                                    image.Paste(gimage, px, py);
                                }
                            }
                            else
                            {
                                Rpt._RPT4("{0:X4} {1:X4} {2:X8} {3}\n", code, gindex, glyph, this.TglpInfo.AllSheetCellNum);
                            }
                        }
                    }
                }
            }
        }

        private void Save(G2dFont font)
        {
            int no;
            var isArchive = font.GetBlock(RtConsts.BinBlockSigGLGR) != null;
            var sig = isArchive ? RtConsts.BinFileSigFONTA : RtConsts.BinFileSigFONT;

            using (var binaryFile = BinaryFile.Open(this.File, FileMode.Create, FileAccess.Write))
            {
                var nnsFile = new NnsWriter(binaryFile, this.IsOutputTargetLittleEndian);

                nnsFile.WriteNnsBegin(sig, ConverterEnvironment.PlatformDetails.OutputFontVersion);

                {
                    nnsFile.WriteNnsBlock(font, RtConsts.BinBlockSigGLGR);
                    nnsFile.WriteNnsBlock(font, RtConsts.BinBlockSigFINF);
                    {
                        var blkCGLP = font.GetBlock(RtConsts.BinBlockSigCGLP);
                        var blkTGLP = font.GetBlock(Runtime.RtConsts.BinBlockSigTGLP);

                        Debug.Assert((blkCGLP != null) != (blkTGLP != null));

                        if (blkTGLP != null)
                        {
                            nnsFile.WriteNnsBlock(font, Runtime.RtConsts.BinBlockSigTGLP);
                        }
                        else if (blkCGLP != null)
                        {
                            nnsFile.WriteNnsBlock(font, Runtime.RtConsts.BinBlockSigCGLP);
                        }
                    }

                    no = 0;
                    while (nnsFile.WriteNnsBlock(font, Runtime.RtConsts.BinBlockSigCWDH, no))
                    {
                        no++;
                    }

                    no = 0;
                    while (nnsFile.WriteNnsBlock(font, Runtime.RtConsts.BinBlockSigCMAP, no))
                    {
                        no++;
                    }

                    no = 0;
                    while (nnsFile.WriteNnsBlock(font, Runtime.RtConsts.BinBlockSigKRNG, no))
                    {
                        no++;
                    }
                }

                nnsFile.WriteNnsEnd();
            }
        }

        public struct GlyphCode : IEquatable<GlyphCode>, IComparable<GlyphCode>
        {
            private readonly ushort gryphIndex;

            private uint charCode;

            public GlyphCode(uint charCode, ushort gryphIndex)
            {
                this.charCode = charCode;
                this.gryphIndex = gryphIndex;
            }

            public uint CharCode
            {
                get { return this.charCode; }
            }

            public ushort GryphIndex
            {
                get { return this.gryphIndex; }
            }

            public static bool operator ==(GlyphCode lhs, GlyphCode rhs)
            {
                return lhs.Equals(rhs);
            }

            public static bool operator !=(GlyphCode lhs, GlyphCode rhs)
            {
                return !lhs.Equals(rhs);
            }

            public bool Equals(GlyphCode rhs)
            {
                return this.charCode == rhs.charCode
                    && this.gryphIndex == rhs.gryphIndex;
            }

            public override bool Equals(object obj)
            {
                return obj is GlyphCode && this.Equals((GlyphCode)obj);
            }

            public override int GetHashCode()
            {
                return ((int)this.charCode << 16) | this.gryphIndex;
            }

            public int CompareTo(GlyphCode other)
            {
                return (int)this.charCode - (int)other.charCode;
            }
        }

        private struct Range
        {
            public LinkedListNode<Glyph> First { get; set; }

            public LinkedListNode<Glyph> Last { get; set; }
        }

        public class IndexList : List<int>
        {
        }

        public class CodeList : List<GlyphCode>
        {
        }

        public class SubGroup
        {
            private CodeList codeList = new CodeList();

            private IndexList widthBlocks = new IndexList();

            private IndexList mapBlocks = new IndexList();

            public CodeList CodeList
            {
                get { return this.codeList; }
            }

            public IndexList WidthBlocks
            {
                get { return this.widthBlocks; }
            }

            public IndexList MapBlocks
            {
                get { return this.mapBlocks; }
            }

            public int Index { get; set; }

            public int SheetIndexBegin { get; set; }

            public int SheetIndexBeginOffset { get; set; }

            public int SheetIndexEnd { get; set; }
        }

        public sealed class Group : IComparable<Group>
        {
            public readonly int Index;
            public readonly string Name;
            public readonly IndexList SubGroups = new IndexList();

            public Group(int index, string name)
            {
                this.Index = index;
                this.Name = name;
            }

            public int CompareTo(Group other)
            {
                return this.Index - other.Index;
            }
        }

        public class SubGroupMap : Dictionary<int, SubGroup>
        {
            // サブグループのインデックスを設定して追加する
            public void Add(SubGroup sg)
            {
                sg.Index = Count;
                this.Add(sg.Index, sg);
            }

            // サブグループのインデックスとして使われる size() が
            // 減少しないようにポインタをNULLにするだけで削除とする。
            public new void Remove(int no)
            {
                if (ContainsKey(no))
                {
                    this[no] = null;
                }
            }
        }

        public class GroupMap : Dictionary<string, Group>
        {
        }

        public class SheetSurplus : IComparable<SheetSurplus>
        {
            public SheetSurplus(SubGroup sub, int numGlyph, int surplus)
            {
                this.Sub = sub;
                this.NumGlyph = numGlyph;
                this.Surplus = surplus;
            }

            public SubGroup Sub { get; private set; }

            public int NumGlyph { get; private set; }

            public int Surplus { get; private set; }

            public int CompareTo(SheetSurplus other)
            {
                return this.Surplus - other.Surplus;
            }
        }

        protected class GlyphLinkedList : LinkedList<Glyph>
        {
            public void Erase(LinkedListNode<Glyph> first, LinkedListNode<Glyph> last)
            {
                if (null == first)
                {
                    throw new ArgumentNullException("first");
                }

                if (null == last)
                {
                    throw new ArgumentNullException("last");
                }

                var endPos = last.Next;
                var cur = first;
                while (cur != endPos)
                {
                    var target = cur;
                    cur = target.Next;
                    Remove(target);
                }
            }
        }

        private class SharedCodeMap : Dictionary<uint, List<int>>
        {
        }

        protected class TGLPInfo
        {
            /// <summary>
            /// シートの bpp
            /// </summary>
            public int SheetBpp { get; set; }

            /// <summary>
            /// シートの幅
            /// </summary>
            public int SheetWidth { get; set; }

            /// <summary>
            /// シートの高さ
            /// </summary>
            public int SheetHeight { get; set; }

            /// <summary>
            /// シートのバイト数
            /// </summary>
            public int SheetBytes { get; set; }

            /// <summary>
            /// シートの枚数
            /// </summary>
            public int SheetNum { get; set; }

            /// <summary>
            /// セルの幅
            /// </summary>
            public int CellWidth { get; set; }

            /// <summary>
            /// セルの高さ
            /// </summary>
            public int CellHeight { get; set; }

            /// <summary>
            /// シート内の横方向セル数
            /// </summary>
            public int CellHCount { get; set; }

            /// <summary>
            /// シート内の縦方向セル数
            /// </summary>
            public int CellVCount { get; set; }

            /// <summary>
            /// シート内でのセルの数
            /// </summary>
            public int CellNum { get; set; }

            /// <summary>
            /// 全シートでのセルの数
            /// </summary>
            public int AllSheetCellNum { get; set; }
        }

        private class CodeExistsMap : Dictionary<uint, bool>
        {
        }

        private class A4Writer
        {
            private readonly MemoryStream ms;
            private byte state;
            private byte value;

            public A4Writer(byte[] image)
            {
                this.ms = new MemoryStream(image);
            }

            /// <summary>
            /// 上位4bit, 下位4bit の順で格納する
            /// </summary>
            /// <param name="pixel"></param>
            public void WritePixelA4_HL(ImageBase.Pixel pixel)
            {
                uint p = ConvertPixelA4(pixel);
                this.state ^= 1;
                if (this.state != 0)
                {
                    this.value = (byte)p;
                }
                else
                {
                    p = ((uint)this.value << 4) | p;
                    this.ms.WriteByte((byte)p);
                }
            }

            /// <summary>
            /// 下位4bit, 上位4bit の順で格納する
            /// </summary>
            /// <param name="pixel"></param>
            public void WritePixelA4_LH(ImageBase.Pixel pixel)
            {
                uint p = ConvertPixelA4(pixel);
                this.state ^= 1;
                if (this.state != 0)
                {
                    this.value = (byte)p;
                }
                else
                {
                    p = this.value | ((uint)p << 4);
                    this.ms.WriteByte((byte)p);
                }
            }
        }


        private class SurplusList : List<SheetSurplus>
        {
        }
    }
}
