﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using System.Threading.Tasks;
    using NintendoWare.Font.Runtime;
    using NintendoWare.Font.Win32;
    using Alpha = System.Byte;

    public class ImageFontReader : FontReader
    {
        protected static readonly COLORREF DnotTransparentColor = 0xFF000000;

        private readonly string File;
        private readonly string Order;
        private readonly GlyphImageFormat ImageFormat;
        private readonly bool IsAntiLinear;

        private int blkHNum;
        private int blkVNum;
        private int blkWidth;
        private int blkHeight;
        private int cellWidth;
        private int cellHeight;

        public ImageFontReader(
                string file,
                string order,
                GlyphImageFormat gif,
                bool isAntiLinear)
        {
            this.File = file;
            this.Order = order;
            this.ImageFormat = gif;
            this.IsAntiLinear = isAntiLinear;
        }

        public override void ReadFontData(FontData fontData, CharFilter filter)
        {
            // プログレスバー設定
            ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_READ_BMP);
            ProgressControl.GetInstance().ResetProgressBarPos();

            fontData.OutputFormat = this.ImageFormat;

            var format = this.DetectFormat();

            // 処理状態を更新
            switch (format)
            {
            case ImageFileFormat.Bmp:
                ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_READ_BMP);
                ProgressControl.GetInstance().ResetProgressBarPos();
                break;

            case ImageFileFormat.Tga:
                ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_READ_TGA);
                ProgressControl.GetInstance().ResetProgressBarPos();
                break;
            }

            var image = LoadImage(this.File, format);

            GlyphOrder order = new GlyphOrder();
            RgbImage normalizedImage = new RgbImage();
            IntColor nullColor;
            ArrangeInfo ai = new ArrangeInfo();

            // xlor をロード
            order.Load(this.Order, this.UseDtdValidation);

            // 背景色を取得しピクセルフォーマットを標準化
            nullColor = this.PixelFormatNormalization(normalizedImage, image);
            fontData.NullColor = nullColor;

            // パラメータを計算し、妥当性を検査する
            this.InitCommonParameter(normalizedImage, order);
            ValidateImageFontFile(normalizedImage, this.blkHNum, this.blkVNum);

            // 配置情報を取得する
            this.LoadArrangeInfo(ai, normalizedImage);
            if (ai.Width > 0)
            {
                fontData.SetWidth(ai.Width);
            }
            else
            {
                fontData.Width = this.cellWidth;
            }

            if (ai.Ascent >= 0 && ai.Descent >= 0)
            {
                fontData.SetHeight(ai.Ascent, ai.Descent);
            }
            else
            {
                fontData.Ascent = ai.BaselinePos;
                fontData.Descent = this.cellHeight - ai.BaselinePos;
            }

            // カーニング情報を取得、フィルタも同時に行う
            try
            {
                using (var binaryFile = BinaryFile.Open(Path.ChangeExtension(this.File, ".bffkn"), FileMode.Open, FileAccess.Read))
                {
                    var br = new ByteOrderBinaryReader(binaryFile, true);
                    long fileLength = binaryFile.Length;

                    // 特別なヘッダがある場合は文字コードを 32bit として読み込む
                    bool is32BitMode = false;
                    if (fileLength > sizeof(uint))
                    {
                        uint header;
                        br.Read(out header);
                        if (header == KerningPair.HeaderOf32BitModeKerning)
                        {
                            is32BitMode = true;
                            fileLength -= sizeof(uint);
                        }
                        else
                        {
                            // ヘッダでなかった場合は位置を戻す
                            br.Seek(0, SeekOrigin.Begin);
                        }
                    }

                    // カーニング情報 1 つあたりのサイズ
                    long blockLength = is32BitMode ?
                        (sizeof(uint) + sizeof(uint) + sizeof(int)) :
                        (sizeof(ushort) + sizeof(ushort) + sizeof(int));

                    // カーニング情報の数
                    int kerningPairNum = (int)(fileLength / blockLength);
                    KerningPair[] pairs = new KerningPair[kerningPairNum];
                    for (int i = 0; i < kerningPairNum; i++)
                    {
                        if (is32BitMode)
                        {
                            // 32bit の場合はそのまま読み込める
                            br.Read(out pairs[i]);
                        }
                        else
                        {
                            // 16bit の場合は 32bit に変換しつつ読み込む
                            ushort charCode;
                            br.Read(out charCode);
                            pairs[i].First = (uint)charCode;
                            br.Read(out charCode);
                            pairs[i].Second = (uint)charCode;
                            br.Read(out pairs[i].KernAmount);
                        }
                    }
                    fontData.KerningPairs = FilterKerningPairs(pairs, filter);
                }
            }
            catch (GeneralException /* err */)
            {
                // カーニング情報がない、出力が必要なければ不要なのでスルーする
            }

            // グリフを取得する
            this.LoadGlyph(
                fontData.GetGlyphList(),
                normalizedImage,
                filter,
                order,
                ai.BaselinePos,
                nullColor);
        }

        public override void ValidateInput()
        {
            // Image File
            if (!IsFileExists(this.File))
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_FILE_NOT_EXISTS, this.File);
            }

            var image = LoadImage(this.File, this.DetectFormat());

            // Glyph Order
            ValidateOrderFile(this.Order, this.UseDtdValidation);

            // image
            GlyphOrder order = new GlyphOrder();
            order.Load(this.Order, this.UseDtdValidation);

            int width = image.Width;
            int height = image.Height;
            int blkHNum = order.GetHNum();
            int blkVNum = order.GetVNum();
            int blkWidth = width / blkHNum;
            int blkHeight = height / blkVNum;
            int areaWidth = blkWidth - (FB.BorderWidth * 2);
            int areaHeight = blkHeight - (FB.BorderWidth * 2);

            ValidateImageFontFile(image, blkHNum, blkVNum);

            for (int y = 0; y < blkVNum; ++y)
            {
                for (int x = 0; x < blkHNum; ++x)
                {
                    uint c = order.GetCharCode(x, y);

                    // このコードはNW4R時代からあるが、
                    // このタイミングでこの関数を呼び出すこと自体
                    // バグっているような気がするため、無効化。
                    //// ProgressControl.GetInstance().StepProgressBar();

                    if (c != RtConsts.InvalidCharCode)
                    {
                        var isSuccess = this.CheckMarginColor(
                                        image,
                                        (x * blkWidth) + FB.BorderWidth,
                                        (y * blkHeight) + FB.BorderWidth,
                                        areaWidth,
                                        areaHeight);

                        if (!isSuccess)
                        {
                            int px = (blkWidth * x) + (blkWidth / 2);
                            int py = (blkHeight * y) + (blkHeight / 2);
                            ProgressControl.Warning(
                                    Strings.IDS_WARN_OVER_DRAW_MARGIN,
                                    px,
                                    py);
                        }
                    }
                }
            }
        }

        private static int BitFold(int value, int src_len, int dst_len)
        {
            if (src_len < dst_len)
            {
                int src_max = (1 << src_len) - 1;
                int dst_max = (1 << dst_len) - 1;
                return value * dst_max / src_max;
            }
            else
            {
                return value >> (src_len - dst_len);
            }
        }

        private static bool IsToIntencity(GlyphImageFormat gif)
        {
            return GlyphImageInfo.GetGlyphImageInfo(gif).IsToIntencity;
        }

        private static bool HasAlpha(GlyphImageFormat gif)
        {
            return GlyphImageInfo.GetGlyphImageInfo(gif).HasAlpha;
        }

        private static void ValidateImageFontFile(ImageBase image, int blkHNum, int blkVNum)
        {
            int blkWidth = image.Width / blkHNum;
            int blkHeight = image.Height / blkVNum;
            int cellWidth = blkWidth - FB.NoCellAreaH;
            int cellHeight = blkHeight - FB.NoCellAreaV;

            if (cellWidth < 1)
            {
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_UNDERMIN_IMAGEWIDTH, image.Width, (FB.NoCellAreaH + 1) * blkHNum);
            }

            if (cellHeight < 1)
            {
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_UNDERMIN_IMAGEHEIGHT, image.Height, (FB.NoCellAreaV + 1) * blkVNum);
            }

            if (blkWidth * blkHNum != image.Width)
            {
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_NON_MULTIPLY_WIDTH, image.Width, blkHNum);
            }

            if (blkHeight * blkVNum != image.Height)
            {
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_NON_MULTIPLY_HEIGHT, image.Height, blkVNum);
            }
        }

        private static ImageBase LoadImage(string file, ImageFileFormat format)
        {
            switch (format)
            {
            case ImageFileFormat.Bmp:
                {
                    BitmapFile bmpfile = new BitmapFile(file);
                    return bmpfile.Load();
                }

            case ImageFileFormat.Tga:
                {
                    TgaFile tgafile = new TgaFile(file);
                    return tgafile.Load();
                }
            case ImageFileFormat.Tif:
                {
                    // 一時ファイルにコピー
                    string tempTiffFilePath = Path.Combine(Environment.GetEnvironmentVariable("TEMP"), Path.GetRandomFileName());
                    tempTiffFilePath = tempTiffFilePath + ".tif";
                    System.IO.File.Copy(file, tempTiffFilePath, true);

                    //読み取り専用属性を削除する
                    System.IO.File.SetAttributes(tempTiffFilePath, System.IO.File.GetAttributes(tempTiffFilePath) & (~System.IO.FileAttributes.ReadOnly));

                    // まず、TIFFを同名のBMPに変換する
                    string tempBmpFilePath = Path.ChangeExtension(tempTiffFilePath, ".bmp");
                    try
                    {
                        Process proc = new Process();
                        proc.StartInfo.FileName = Path.Combine(
                            Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), "TiffBmpCvtr.exe");
                        proc.StartInfo.Arguments = tempTiffFilePath;
                        proc.StartInfo.CreateNoWindow = true;
                        proc.StartInfo.UseShellExecute = false;
                        proc.Start();
                        proc.WaitForExit();

                        // そのあと、BMPを解析する。
                        BitmapFile bmpfile = new BitmapFile(tempBmpFilePath);
                        return bmpfile.Load();
                    }
                    finally
                    {
                        if (System.IO.File.Exists(tempTiffFilePath)) { System.IO.File.Delete(tempTiffFilePath); }
                        if (System.IO.File.Exists(tempBmpFilePath)) { System.IO.File.Delete(tempBmpFilePath); }
                    }
                }
            default:
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_FAIL_INPUT_IMAGE_DETECTION);
            }
        }

        private void LoadGlyph(
            GlyphList list,
            ImageBase image,
            CharFilter filter,
            GlyphOrder order,
            int baseline,
            IntColor nullColor)
        {
            ProgressControl.GetInstance().SetProgressBarMax(this.blkHNum * this.blkVNum);

            list.Reserve(this.blkVNum * this.blkHNum);
            for (int y = 0; y < this.blkVNum; ++y)
            {
                for (int x = 0; x < this.blkHNum; ++x)
                {
                    uint c = order.GetCharCode(x, y);

                    ProgressControl.GetInstance().StepProgressBar();

                    if (c != GlyphOrder.NullGlyph)
                    {
                        if (!filter.IsFiltered(c))
                        {
                            this.LoadOneGlyph(list, image, x, y, c, baseline, nullColor);
                        }
                    }
                    else
                    {
                        this.CheckGlyphEmpty(image, x, y, nullColor);
                    }
                }
            }
        }

        private void InitCommonParameter(ImageBase image, GlyphOrder order)
        {
            this.blkHNum = order.GetHNum();
            this.blkVNum = order.GetVNum();
            this.blkWidth = image.Width / this.blkHNum;
            this.blkHeight = image.Height / this.blkVNum;
            this.cellWidth = this.blkWidth - FB.NoCellAreaH;
            this.cellHeight = this.blkHeight - FB.NoCellAreaV;
        }

        private void LoadArrangeInfo(
            ArrangeInfo info,
            ImageBase image)
        {
            // 幅情報読み取り
            {
                int x_st = FB.NoCellAreaLeft - 1;
                int x_ed = FB.NoCellAreaLeft + this.cellWidth + 1;
                uint colorA = 0;
                uint colorB = 0;
                List<int> xColorA = new List<int>();
                List<int> xColorB = new List<int>();

                for (int x = x_st; x < x_ed; ++x)
                {
                    uint c = image.GetRGB(x, 0);

                    if ((xColorA.Count > 0) && (c == colorA))
                    {
                        xColorA.Add(x);
                    }
                    else if ((xColorB.Count > 0) && (c == colorB))
                    {
                        xColorB.Add(x);
                    }
                    else
                    {
                        if (xColorA.Count == 0)
                        {
                            colorA = c;
                            xColorA.Add(x);
                        }
                        else if (xColorB.Count == 0)
                        {
                            colorB = c;
                            xColorB.Add(x);
                        }
                        else
                        {
                            // 3 色目が見つかった
                            throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_ILLEGAL_HORIZONTAL_INFO);
                        }
                    }
                }

                if (xColorB.Count == 2)
                {
                    info.Width = xColorB[1] - xColorB[0] - 1;
                }
                else if (xColorA.Count == 2)
                {
                    info.Width = xColorA[1] - xColorA[0] - 1;
                }
                else if (xColorA.Count == 0 || xColorB.Count == 0)
                {
                    info.Width = -1;
                }
                else
                {
                    // どちらか片方の色は 0 または 2 ドットでなければならない
                    // TODO: ErrMsg
                    throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_ILLEGAL_HORIZONTAL_INFO);
                }
            }

            // 高さ情報読み取り
            {
                int y_st = FB.NoCellAreaTop - 1;
                int y_ed = FB.NoCellAreaTop + this.cellHeight + 1;
                uint colorA = 0;
                uint colorB = 0;
                IntArray yColorA = new IntArray();
                IntArray yColorB = new IntArray();

                for (int y = y_st; y < y_ed; ++y)
                {
                    uint c = image.GetRGB(0, y);

                    if ((yColorA.Count > 0) && (c == colorA))
                    {
                        yColorA.Add(y);
                    }
                    else if ((yColorB.Count > 0) && (c == colorB))
                    {
                        yColorB.Add(y);
                    }
                    else
                    {
                        if (yColorA.Count == 0)
                        {
                            colorA = c;
                            yColorA.Add(y);
                        }
                        else if (yColorB.Count == 0)
                        {
                            colorB = c;
                            yColorB.Add(y);
                        }
                        else
                        {
                            // 3 色目が見つかった
                            // TODO: ErrMsg
                            throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_ILLEGAL_VERTICAL_INFO);
                        }
                    }
                }

                if (yColorA.Count == yColorB.Count)
                {
                    // この場合はどちらがグリッド色か判定できないのでエラーにする
                    throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_UNSUPPORTED_VERTICAL_INFO);
                }

                // 数が少ない方が配置情報点カラー
                var infoColor = yColorA.Count < yColorB.Count ? yColorA : yColorB;

                switch (infoColor.Count)
                {
                case 1:
                    // ベースラインのみ
                    info.BaselinePos = infoColor[0] - FB.NoCellAreaTop;
                    info.Ascent = -1;
                    info.Descent = -1;
                    break;

                case 2:
                    // ベースラインとアセンダライン、ディセントは0
                    info.BaselinePos = infoColor[1] - FB.NoCellAreaTop;
                    info.Ascent = infoColor[1] - infoColor[0] - 1;
                    info.Descent = 0;
                    break;

                case 3:
                    // ベースラインとアセンダラインとディセンダライン
                    info.BaselinePos = infoColor[1] - FB.NoCellAreaTop;
                    info.Ascent = infoColor[1] - infoColor[0] - 1;
                    info.Descent = infoColor[2] - infoColor[1];
                    break;

                default:
                    // 配置情報点は 1～3 ピクセル存在しなければならない
                    if (infoColor.Count == 0)
                    {
                        throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_NO_BASELINE);
                    }
                    else
                    {
                        throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_ILLEGAL_VERTICAL_INFO);
                    }
                }
            }
        }

        private void LoadOneGlyph(
            GlyphList list,
            ImageBase image,
            int x,
            int y,
            uint c,
            int baseline,
            IntColor nullColor)
        {
            int cellX = (x * this.blkWidth) + FB.NoCellAreaLeft;
            int cellY = (y * this.blkHeight) + FB.NoCellAreaTop;
            RgbImage gimage = new RgbImage();
            RgbImage barimage = new RgbImage();

            // グリフを取得
            if (this.IsAntiLinear && HasAlpha(this.ImageFormat))
            {
                // 線形補間対策を行う
                gimage.Create(this.cellWidth + 2, this.cellHeight + 2, image.Bpp);
                gimage.Clear(nullColor, ImageBase.AlphaTransparent);
                gimage.Paste(image, cellX, cellY, 1, 1, this.cellWidth, this.cellHeight);
                this.AntiLinear(gimage);

                baseline++;
            }
            else
            {
                // 線形補間対策を行わない
                image.Extract(gimage, cellX, cellY, this.cellWidth, this.cellHeight);
            }

            // 幅線領域を取得
            image.Extract(
                barimage,
                cellX,
                cellY + this.cellHeight + FB.PaddingWidth,
                this.cellWidth,
                FB.BarWidth);

            var glyph = new Glyph();

            try
            {
                glyph.SetGlyphImageWithWidthBar(gimage, barimage, baseline, nullColor);
            }
            catch (GlyphException err)
            {
                switch (err.ErrorCode)
                {
                case Glyph.WidthLineSplitError:
                    {
                        int px = (this.blkWidth * x) + (this.blkWidth / 2);
                        int py = (this.blkHeight * y) + (this.blkHeight / 2);
                        throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_MULTI_WIDTHLINE, px, py);
                    }
                }
            }

            if (glyph.Left() != 0 || glyph.Width() != 0 || glyph.Right() != 0)
            {
                list.AddGlyph(glyph, c);
            }
        }

        private void AntiLinear(ImageBase image)
        {
            int width = image.Width;
            int height = image.Height;
            int temp_w = width + 2;
            int temp_h = height + 2;
            RgbImage tempImage = new RgbImage();

            tempImage.Create(temp_w, temp_h, 24);
            tempImage.Clear(0, ImageBase.AlphaTransparent);
            tempImage.Paste(image, 0, 0, 1, 1, width, height);

            for (int y = 0; y < height; ++y)
            {
                for (int x = 0; x < width; ++x)
                {
                    ImageBase.Pixel p = tempImage.GetPixel(x + 1, y + 1);

                    if (p.A == ImageBase.AlphaTransparent)
                    {
                        var px = new ImageBase.Pixel[8];
                        int n = 0;

                        // 隣接する 8 ピクセルの中から完全透明でないものを収集
                        for (int dx = 0; dx < 3; ++dx)
                        {
                            for (int dy = 0; dy < 3; ++dy)
                            {
                                if ((dx != 1) || (dy != 1))
                                {
                                    px[n] = tempImage.GetPixel(x + dx, y + dy);
                                    if (px[n].A != ImageBase.AlphaTransparent)
                                    {
                                        n++;
                                    }
                                }
                            }
                        }

                        // 集めたピクセルの平均を取る
                        if (n > 0)
                        {
                            int r = 0;
                            int g = 0;
                            int b = 0;
                            for (int i = 0; i < n; ++i)
                            {
                                r += px[i].R;
                                g += px[i].G;
                                b += px[i].B;
                            }

                            p.R = (byte)((r + (n / 2)) / n);
                            p.G = (byte)((g + (n / 2)) / n);
                            p.B = (byte)((b + (n / 2)) / n);
                            //// p.a = static_cast<BYTE>(IImage::ALPHA_TRANSPARENT);
                        }
                    }

                    image.SetRGB(x, y, p.R, p.G, p.B);
                }
            }
        }

        private void CheckGlyphEmpty(ImageBase image, int x, int y, IntColor nullColor)
        {
            int cellX = (x * this.blkWidth) + FB.NoCellAreaLeft;
            int cellY = (y * this.blkHeight) + FB.NoCellAreaTop;
            RgbImage gimage = new RgbImage();
            RgbImage barimage = new RgbImage();

            // グリフを取得
            image.Extract(gimage, cellX, cellY, this.cellWidth, this.cellHeight);
            image.Extract(
                barimage,
                cellX,
                cellY + this.cellHeight + FB.PaddingWidth,
                this.cellWidth,
                FB.BarWidth);

            if (!this.IsEmptyGlyph(gimage, barimage, nullColor))
            {
                int px = (this.blkWidth * x) + (this.blkWidth / 2);
                int py = (this.blkHeight * y) + (this.blkHeight / 2);

                ProgressControl.Warning(
                    Strings.IDS_WARN_NON_WHITE_NULL,
                    px,
                    py,
                    this.Order);
            }
        }

        private bool IsEmptyGlyph(ImageBase glyph, ImageBase bar, IntColor nullColor)
        {
            // セル内が 1 色で塗りつぶされているなら empty
            if (glyph.Height > 0 && glyph.Width > 0)
            {
                uint nullBlockColor = glyph.GetPixel(0, 0).IntColor;

                for (int y = 0; y < glyph.Height; y++)
                {
                    for (int x = 0; x < glyph.Width; x++)
                    {
                        if (glyph.GetPixel(x, y).IntColor != nullBlockColor)
                        {
                            return false;
                        }
                    }
                }
            }

            // 幅線領域内が背景色で塗りつぶされているなら empty
            for (int x = 0; x < bar.Width; x++)
            {
                if (bar.GetPixel(x, 0).IntColor != nullColor)
                {
                    return false;
                }
            }

            // 上記 2 つを満たした場合のみ empty
            return true;
        }

        /// <summary>
        /// 4bit フォーマットかどうか？
        /// 4 bitフォーマットの場合は、バイト長調整の処理をおこないます。
        /// </summary>
        private bool IsHalfbitFormat_()
        {
            return this.ImageFormat == GlyphImageFormat.A4 || this.ImageFormat == GlyphImageFormat.LA4;
        }

        // TODO
        private IntColor PixelFormatNormalization(RgbImage outImage, ImageBase inputImage)
        {
            const int TargetAlphaBpp = 8;
            const int TargetColorBpp = 24;
            outImage.Create(inputImage.Width, inputImage.Height, TargetColorBpp);
            IntColor nullColor;

            if (IsToIntencity(this.ImageFormat))
            {
                /*
                    I 出力の場合
                      インデックス: インデックス→グレースケール
                      ダイレクト:   明度→グレースケール
                */

                int srcLen = ConverterEnvironment.PlatformDetails.CanHWSupports4BitFormat && IsHalfbitFormat_() ? 4 : 8;
                int dstLen = outImage.ColorBpp / 3;
                bool isSrcImageIndexFormat = inputImage is IndexImage;

                // 主要ケースはループ外で分岐して最適な処理を行う。
                // (ソースが非インデックスフォーマット かつ 対処ビット長が減少する。)
                if (!isSrcImageIndexFormat && (srcLen >= dstLen))
                {
                    int srcExpansionBit = (8 - srcLen);
                    int dstTruncationBit = srcLen - dstLen;
                    int imgHeight = inputImage.Height;
                    Parallel.For(0, imgHeight, y =>
                    {
                        for (int x = 0; x < inputImage.Width; x++)
                        {
                            ImageBase.Pixel pixel = inputImage.GetPixel(x, y);

                            int raw_index = ((pixel.R + pixel.G + pixel.B) / 3) >> srcExpansionBit;
                            raw_index = raw_index >> dstTruncationBit;

                            outImage.SetRGBGray(x, y, (byte)(raw_index));
                        }
                    });
                }
                else
                {
                    for (int y = 0; y < inputImage.Height; y++)
                    {
                        for (int x = 0; x < inputImage.Width; x++)
                        {
                            int index;
                            ImageBase.Pixel pixel = inputImage.GetPixel(x, y);
                            if (isSrcImageIndexFormat)
                            {
                                uint color = pixel.Color;
                                index = BitFold((int)color, srcLen, dstLen);
                            }
                            else
                            {
                                int raw_index = ((pixel.R + pixel.G + pixel.B) / 3) >> (8 - srcLen);
                                index = BitFold(raw_index, srcLen, dstLen);
                            }

                            outImage.SetRGBGray(x, y, (byte)index);
                        }
                    }
                }
            }
            else
            {
                /*
                    RGB 出力の場合
                      インデックス: カラーパレット展開→bit拡張
                      ダイレクト:   bit拡張のみ
                */

                const int SrcLen = 8;
                const uint ElementMask = (1 << SrcLen) - 1;

                for (int y = 0; y < inputImage.Height; y++)
                {
                    for (int x = 0; x < inputImage.Width; x++)
                    {
                        var pixel = inputImage.GetRGB(x, y);
                        uint r = (pixel >> (SrcLen * 2)) & ElementMask;
                        uint g = (pixel >> (SrcLen * 1)) & ElementMask;
                        uint b = (pixel >> (SrcLen * 0)) & ElementMask;

                        outImage.SetRGB(x, y, (byte)r, (byte)g, (byte)b);
                    }
                }
            }

            nullColor = outImage.GetPixel(0, 0).Color;

            //---- アルファチャンネルの展開

            // 入力にαがあって出力にもアルファが必要な場合
            if (HasAlpha(this.ImageFormat) && inputImage.IsEnableAlpha)
            {
                int width = inputImage.Width;
                int height = inputImage.Height;

                // αチャンネルが1つの値で塗りつぶされていないかチェック
                Alpha a = inputImage.GetAlpha(0, 0);
                var isOneAlpha = true;

                for (int y = 0; y < height; ++y)
                {
                    for (int x = 0; x < width; ++x)
                    {
                        if (inputImage.GetAlpha(x, y) != a)
                        {
                            isOneAlpha = false;
                            goto break_2loop;
                        }
                    }
                }

            break_2loop:

                // αチャンネルが1つの値で塗りつぶされている場合はαなしとみなす
                if (isOneAlpha)
                {
                    ProgressControl.Warning(Strings.IDS_WARN_ONE_ALPHA);

                    // αチャンネルは 1bpp 相当にしてしまう。
                    Alpha transparent = 0;
                    Alpha opacity = (1 << TargetAlphaBpp) - 1;

                    for (int y = 0; y < inputImage.Height; y++)
                    {
                        for (int x = 0; x < inputImage.Width; x++)
                        {
                            Alpha alpha = (outImage.GetColor(x, y) == nullColor) ?
                                                        transparent : opacity;

                            outImage.SetAlpha(x, y, alpha);
                        }
                    }

                    outImage.EnableAlpha();
                }
                else
                { // αあり
                    int src_len = inputImage.AlphaBpp;
                    int dst_len = outImage.AlphaBpp;
                    int eff_len = GlyphImageInfo.GetGlyphImageInfo(this.ImageFormat).AlphaBits;

                    // 主要ケースはループ外で分岐して最適な処理を行う。
                    // (アルファビットが増減しないケース or 現状するケース)
                    int srcToEffTruncationBit = src_len - eff_len;
                    int effToDstTruncationBit = eff_len - dst_len;
                    if (srcToEffTruncationBit == 0 && effToDstTruncationBit == 0)
                    {
                        Parallel.For(0, height, y =>
                        {
                            for (int x = 0; x < width; ++x)
                            {
                                outImage.SetAlpha(x, y, (byte)inputImage.GetAlpha(x, y));
                            }
                        });
                    }
                    else
                    {
                        for (int y = 0; y < height; ++y)
                        {
                            for (int x = 0; x < width; ++x)
                            {
                                // TODO:
                                Alpha src_alpha = inputImage.GetAlpha(x, y);
                                Alpha eff_alpha = (byte)BitFold(src_alpha, src_len, eff_len);
                                Alpha dst_alpha = (byte)BitFold(eff_alpha, eff_len, dst_len);
                                outImage.SetAlpha(x, y, dst_alpha);
                            }
                        }
                    }

                    outImage.EnableAlpha();
                }
            }
            else
            {   // if (nullColor != DnotTransparentColor)
                // 入力にアルファがないか出力にαが不要な場合

                // αチャンネルは 1bpp 相当にしてしまう。
                Alpha transparent = 0;
                Alpha opacity = (1 << TargetAlphaBpp) - 1;

                for (int y = 0; y < inputImage.Height; y++)
                {
                    for (int x = 0; x < inputImage.Width; x++)
                    {
                        Alpha alpha = (outImage.GetColor(x, y) == nullColor) ?
                                                    transparent : opacity;

                        outImage.SetAlpha(x, y, alpha);
                    }
                }

                outImage.EnableAlpha();
            }

            return outImage.GetPixel(0, 0).IntColor;
        }

        private ImageFileFormat DetectFormat()
        {
            ImageFileFormat format = ImageFileFormat.Ext;

            // 拡張子がtifだったら...
            if (string.Compare(Path.GetExtension(this.File), ".tif", true) == 0)
            {
                return ImageFileFormat.Tif;
            }

            BitmapFile bmpfile = new BitmapFile(this.File);
            TgaFile tgafile = new TgaFile(this.File);

            var isBmp = bmpfile.HasValidIdentifier();
            var isTga = tgafile.HasValidIdentifier();

            if (isBmp && isTga)
            {
                // none
            }
            else if (isBmp)
            {
                format = ImageFileFormat.Bmp;
            }
            else if (isTga)
            {
                format = ImageFileFormat.Tga;
            }
            else
            {
                var isNotBmp = bmpfile.IsInvalid();
                var isNotTga = tgafile.IsInvalid();

                if (isNotBmp && isNotTga)
                {
                    // none
                }
                else if (isNotBmp)
                {
                    format = ImageFileFormat.Tga;
                }
                else if (isNotTga)
                {
                    format = ImageFileFormat.Bmp;
                }
            }

            if ((int)format <= (int)ImageFileFormat.Ext && (int)ImageFileFormat.NUM <= (int)format)
            {
                throw GlCm.ErrMsg(ErrorType.Image, Strings.IDS_ERR_FAIL_INPUT_IMAGE_DETECTION);
            }

            return format;
        }

        private uint CheckMarginColorHorizontal(ImageBase bmp, int x, int y, int w)
        {
            uint color = bmp.GetRGB(x, y);

            for (int px = x; px < x + w; px++)
            {
                if (bmp.GetRGB(px, y) != color)
                {
                    return 0xFFFFFFFF;
                }
            }

            return color;
        }

        private uint CheckMarginColorVertical(ImageBase bmp, int x, int y, int h)
        {
            uint color = bmp.GetRGB(x, y);

            for (int py = y; py < y + h; py++)
            {
                if (bmp.GetRGB(x, py) != color)
                {
                    return 0xFFFFFFFF;
                }
            }

            return color;
        }

        private bool CheckMarginColor(ImageBase bmp, int x, int y, int w, int h)
        {
            uint margin_color;

            // 上横線
            margin_color = this.CheckMarginColorHorizontal(bmp, x, y, w);
            if (margin_color == 0xFFFFFFFF)
            {
                return false;
            }

            // 中横線
            if (this.CheckMarginColorHorizontal(bmp, x, y + h - 3, w) != margin_color)
            {
                return false;
            }

            // 下横線
            if (this.CheckMarginColorHorizontal(bmp, x, y + h - 1, w) != margin_color)
            {
                return false;
            }

            // 左縦線
            if (this.CheckMarginColorVertical(bmp, x, y, h) != margin_color)
            {
                return false;
            }

            // 右縦線
            if (this.CheckMarginColorVertical(bmp, x + w - 1, y, h) != margin_color)
            {
                return false;
            }

            return true;
        }
    }
}
