﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Text;

    public class G2dFont : NnsData
    {
        private bool isOldVer;
        private int widthsSize;

        public G2dFont()
            : base()
        {
            this.isOldVer = false;
            this.widthsSize = Marshal.SizeOf(typeof(Runtime.CharWidths));
        }

        public static bool IsUnicodeEncoding(CharEncoding enc)
        {
            return enc == CharEncoding.UTF8 || enc == CharEncoding.UTF16;
        }

        public Runtime.CharWidths GetGlyphWidthFromIndex(FontWidthEx fontWidthEx, int idx)
        {
            return fontWidthEx.WidthTable[idx - fontWidthEx.Body.IndexBegin];
        }

        public bool IsOldVer()
        {
            return this.isOldVer;
        }

        /*
            ブロック登録
        */
        public void SetGlyphGroupsBlock(
                uint sheetSize,
                ushort glyphsPerSheet,
                ushort numSheet,
                ushort numSet,
                G2dFont.SheetSetList sheetSetList,
                IntArray compSheetSizes)
        {
            ushort numCWDH = (ushort)GetNumBlock(Runtime.RtConsts.BinBlockSigCWDH);
            ushort numCMAP = (ushort)GetNumBlock(Runtime.RtConsts.BinBlockSigCMAP);
            Debug.Assert(GetNumBlock(Runtime.RtConsts.BinBlockSigCWDH) == numCWDH);
            Debug.Assert(GetNumBlock(Runtime.RtConsts.BinBlockSigCMAP) == numCMAP);

            var body = new Runtime.FontGlyphGroups();
            body.SheetSize = sheetSize;
            body.GlyphsPerSheet = glyphsPerSheet;
            body.NumSheet = numSheet;
            body.NumSet = numSet;
            body.NumCWDH = numCWDH;
            body.NumCMAP = numCMAP;

            var groupsEx = new FontGlyphGroupsEx();
            groupsEx.Body = body;
            groupsEx.SheetSetList = sheetSetList;
            groupsEx.CompSheetSizes = compSheetSizes;

            groupsEx.SizeCWDH = new int[numCWDH];
            for (var i = 0; i < numCWDH; ++i)
            {
                groupsEx.SizeCWDH[i] = GetBlock(Runtime.RtConsts.BinBlockSigCWDH, i).Get4AlignedBlockSize(0);
            }

            groupsEx.SizeCMAP = new int[numCMAP];
            for (var i = 0; i < numCMAP; ++i)
            {
                groupsEx.SizeCMAP[i] = GetBlock(Runtime.RtConsts.BinBlockSigCMAP, i).Get4AlignedBlockSize(0);
            }

            SetBlock(Runtime.RtConsts.BinBlockSigGLGR, groupsEx);
        }

        public void SetInformationBlock(
                byte fontType,
                short linefeed,
                byte width,
                byte height,
                byte ascent,
                ushort alterCharIndex,
                sbyte defaultLeft,
                byte defaultWidth,
                sbyte defaultRight,
                byte encoding)
        {
            var info = new Runtime.FontInformation();
            byte charWidth = (byte)(defaultLeft + defaultWidth + defaultRight);

            info.FontType = fontType;
            info.Linefeed = linefeed;
            info.Width = width;
            info.Height = height;
            info.Ascent = ascent;
            info.AlterCharIndex = alterCharIndex;
            info.DefaultWidth.Left = defaultLeft;
            info.DefaultWidth.GlyphWidth = defaultWidth;
            info.DefaultWidth.CharWidth = charWidth;
            info.Encoding = encoding;

            var infoEx = new FontInformationEx();
            infoEx.Body = info;

            SetBlock(Runtime.RtConsts.BinBlockSigFINF, infoEx);
        }

        public void SetTextureGlyphBlock(
                byte cellWidth,
                byte cellHeight,
                short baselinePos,
                byte maxCharWidth,
                uint sheetSize,
                byte sheetNum,
                ushort sheetFormat,
                ushort sheetRow,
                ushort sheetLine,
                ushort sheetWidth,
                ushort sheetHeight,
                byte[] image,
                int imageSize)
        {
            var glyph = new Runtime.FontTextureGlyph();
            glyph.CellWidth = cellWidth;
            glyph.CellHeight = cellHeight;
            glyph.BaselinePos = baselinePos;
            glyph.MaxCharWidth = maxCharWidth;
            glyph.SheetSize = sheetSize;
            glyph.SheetNum = sheetNum;
            glyph.SheetFormat = sheetFormat;
            glyph.SheetRow = sheetRow;
            glyph.SheetLine = sheetLine;
            glyph.SheetWidth = sheetWidth;
            glyph.SheetHeight = sheetHeight;

            var glyphEx = new FontTextureGlyphEx();
            glyphEx.Body = glyph;
            glyphEx.SheetImage = image;
            glyphEx.SheetImageSize = imageSize;

            SetBlock(Runtime.RtConsts.BinBlockSigTGLP, glyphEx);
        }

        public void AddWidthBlock(
                ushort ccodeBegin,
                ushort ccodeEnd,
                Runtime.CharWidths[] data)
        {
            var charWidthNum = ccodeEnd - ccodeBegin + 1;

            var width = new Runtime.FontWidth();
            width.IndexBegin = ccodeBegin;
            width.IndexEnd = ccodeEnd;

            var widthEx = new FontWidthEx();
            widthEx.Body = width;
            widthEx.WidthTable = new Runtime.CharWidths[charWidthNum];

            Array.Copy(data, ccodeBegin, widthEx.WidthTable, 0, widthEx.WidthTable.Length);

            SetBlock(Runtime.RtConsts.BinBlockSigCWDH, widthEx);
        }

        public void AddCMapBlock(
                uint ccodeBegin,
                uint ccodeEnd,
                FontMapMethod mappingMethod,
                Runtime.CMapScanEntry[] entries)
        {
            var map = new Runtime.FontCodeMap();
            map.CodeBegin = ccodeBegin;
            map.CodeEnd = ccodeEnd;
            map.MappingMethod = (ushort)mappingMethod;

            var mapEx = new FontCodeMapEx();
            mapEx.Body = map;
            var scanEx = new CMapInfoScanEx();
            scanEx.Body = new Runtime.CMapInfoScan { Num = (ushort)entries.Length };
            scanEx.Entries = entries;
            mapEx.Scan = scanEx;

            SetBlock(Runtime.RtConsts.BinBlockSigCMAP, mapEx);
        }

        public void AddKerningBlock(KerningPair[] pairs)
        {
            var fontKerningTable = new FontKerningTable(pairs);
            SetBlock(Runtime.RtConsts.BinBlockSigKRNG, fontKerningTable);
        }

        /*
            フォントリソース構築
        */
        public void MakeInterBlcokLink()
        {
            GeneralBinaryBlockInfo blkInfo;
            int offset;
            int glyphBlockOffset;
            var widthBlockOffset = new List<int>();
            var cmapBlockOffset = new List<int>();

            widthBlockOffset.Capacity = 10;
            cmapBlockOffset.Capacity = 10;

            offset = Runtime.BinaryFileHeader.Length;

            // get offset / GLGP block
            {
                blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigGLGR);

                if (blkInfo != null)
                {
                    offset += blkInfo.Get4AlignedBlockSize(offset);
                }
            }

            // get offset / FINF block
            {
                blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigFINF);
                Debug.Assert(blkInfo != null);

                offset += blkInfo.Get4AlignedBlockSize(offset);
            }

            // get offset / CGLP block
            {
                GeneralBinaryBlockInfo blkCGLP = GetBlock(Runtime.RtConsts.BinBlockSigCGLP);
                GeneralBinaryBlockInfo blkTGLP = GetBlock(Runtime.RtConsts.BinBlockSigTGLP);
                glyphBlockOffset = offset + Runtime.BinaryBlockHeader.Length;

                Debug.Assert((blkCGLP != null) != (blkTGLP != null));

                if (blkTGLP != null)
                {
                    offset += blkTGLP.Get4AlignedBlockSize(offset);
                }
                else
                {
                    offset += blkCGLP.Get4AlignedBlockSize(offset);
                }
            }

            // get offset / CWDH block
            {
                int n = 0;

                blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigCWDH);

                while (blkInfo != null)
                {
                    widthBlockOffset.Add(offset + Runtime.BinaryBlockHeader.Length);
                    offset += blkInfo.Get4AlignedBlockSize(offset);
                    n++;
                    blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigCWDH, n);
                }

                widthBlockOffset.Add(0);
            }

            // get offset / CMAP block
            {
                int n = 0;

                blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigCMAP);

                while (blkInfo != null)
                {
                    cmapBlockOffset.Add(offset + Runtime.BinaryBlockHeader.Length);
                    offset += blkInfo.Get4AlignedBlockSize(offset);
                    n++;
                    blkInfo = GetBlock(Runtime.RtConsts.BinBlockSigCMAP, n);
                }

                cmapBlockOffset.Add(0);
            }

            // set offset / FINF block
            {
                var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigFINF);
                var infoEx = (FontInformationEx)pbbi.Body;

                //// pBlock->body.pGlyph = reinterpret_cast<LF_FontGlyph*>(glyphBlockOffset);
                infoEx.Body.PtrGlyph = (uint)glyphBlockOffset;
                infoEx.Body.PtrWidth = (uint)widthBlockOffset[0];
                infoEx.Body.PtrMap = (uint)cmapBlockOffset[0];
            }

            // set offset / CWDH block
            {
                int last = widthBlockOffset.Count - 1;

                for (int n = 0; n < last; ++n)
                {
                    var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigCWDH, n);
                    Debug.Assert(pbbi != null);
                    var widthEx = (FontWidthEx)pbbi.Body;

                    widthEx.Body.PtrNext = (uint)widthBlockOffset[n + 1];
                }
            }

            // set offset / CMAP block
            {
                int last = cmapBlockOffset.Count - 1;

                for (int n = 0; n < last; ++n)
                {
                    var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigCMAP, n);
                    Debug.Assert(pbbi != null);
                    var mapEx = (FontCodeMapEx)pbbi.Body;

                    mapEx.Body.PtrNext = (uint)cmapBlockOffset[n + 1];
                }
            }
        }

        public IBinarizable GenerateInstance(Signature32 kind)
        {
            if (kind == Runtime.RtConsts.BinBlockSigFINF)
            {
                return new FontInformationEx();
            }
            else if (kind == Runtime.RtConsts.BinBlockSigTGLP)
            {
                return new FontTextureGlyphEx();
            }
            else if (kind == Runtime.RtConsts.BinBlockSigCWDH)
            {
                return new FontWidthEx();
            }
            else if (kind == Runtime.RtConsts.BinBlockSigCMAP)
            {
                return new FontCodeMapEx();
            }

            return null;
        }

        public void Rebuild()
        {
            foreach (var mi in this.Blocks)
            {
                BlockList bl = mi.Value;

                for (var no = 0; no < bl.Count; no++)
                {
                    GeneralBinaryBlockInfo bb = bl[no];

                    if (bb.Kind == Runtime.RtConsts.BinBlockSigFINF)
                    {
                        Debug.Assert(bb.Body != null);
                        var infoEx = (FontInformationEx)bb.Body;

                        if (infoEx.Body.PtrGlyph != 0)
                        {
                            var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigTGLP);
                            Debug.Assert(pbbi != null);
                            Debug.Assert(pbbi.Body != null);
                            infoEx.Glyph = (FontTextureGlyphEx)pbbi.Body;
                        }

                        if (infoEx.Body.PtrWidth != 0)
                        {
                            var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigCWDH);
                            Debug.Assert(pbbi != null);
                            Debug.Assert(pbbi.Body != null);
                            infoEx.Width = (FontWidthEx)pbbi.Body;
                        }

                        if (infoEx.Body.PtrMap != 0)
                        {
                            var pbbi = GetBlockItnl(Runtime.RtConsts.BinBlockSigCMAP);
                            Debug.Assert(pbbi != null);
                            Debug.Assert(pbbi.Body != null);
                            infoEx.Map = (FontCodeMapEx)pbbi.Body;
                        }
                    }
                    else if (bb.Kind == Runtime.RtConsts.BinBlockSigCWDH)
                    {
                        Debug.Assert(bb.Body != null);
                        var widthEx = (FontWidthEx)bb.Body;

                        if (widthEx.Body.PtrNext != 0)
                        {
                            widthEx.Next = (FontWidthEx)bl[no + 1].Body;
                        }
                    }
                    else if (bb.Kind == Runtime.RtConsts.BinBlockSigCMAP)
                    {
                        Debug.Assert(bb.Body != null);
                        var mapEx = (FontCodeMapEx)bb.Body;

                        if (mapEx.Body.PtrNext != 0)
                        {
                            mapEx.Next = (FontCodeMapEx)bl[no + 1].Body;
                        }
                    }
                }
            }
        }

        public ushort GetGlyphIndex(uint c)
        {
            var blockInfo = GetBlock(Runtime.RtConsts.BinBlockSigCMAP);
            int no = 0;

            while (blockInfo != null)
            {
                var mapEx = (FontCodeMapEx)blockInfo.Body;
                if ((mapEx.Body.CodeBegin <= c)
                    && (c <= mapEx.Body.CodeEnd))
                {
                    ushort index = this.GetGlyphIndex(mapEx, c);
                    if (index != Runtime.RtConsts.InvalidGlyphIndex)
                    {
                        return index;
                    }

                    break;
                }

                no++;
                blockInfo = GetBlock(Runtime.RtConsts.BinBlockSigCMAP, no);
            }

            return Runtime.RtConsts.InvalidGlyphIndex;
        }

        public Runtime.CharWidths GetGlyphWidthInfo(ushort idx)
        {
            var blockInfo = GetBlock(Runtime.RtConsts.BinBlockSigCWDH);
            int no = 0;

            while (blockInfo != null)
            {
                var widthEx = (FontWidthEx)blockInfo.Body;
                var body = widthEx.Body;
                if ((body.IndexBegin <= idx)
                    && (idx <= body.IndexEnd))
                {
                    return this.GetGlyphWidthFromIndex(widthEx, idx);
                }

                no++;
                blockInfo = GetBlock(Runtime.RtConsts.BinBlockSigCWDH, no);
            }

            blockInfo = GetBlock(Runtime.RtConsts.BinBlockSigFINF);
            Debug.Assert(blockInfo != null);

            return ((FontInformationEx)blockInfo.Body).Body.DefaultWidth;
        }

        /*
            妥当性検査
         */
        public void ValidateHeader()
        {
            // signature
            if (Header.Signature != Runtime.RtConsts.BinFileSigFONT
                && Header.Signature != Runtime.RtConsts.BinFileSigFONTA
                && Header.Signature != Runtime.RtConsts.BinFileSigFONTCTR
                && Header.Signature != Runtime.RtConsts.BinFileSigFONTACTR)
            {
                throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_INVALID_FONT_FILE, LibFormat.ExtensionFont);
            }

            // byte order
            if (Header.ByteOrder != 0xFEFF)
            {
                if (Header.ByteOrder != 0xFFFE)
                {
                    throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_INVALID_FONT_FILE, LibFormat.ExtensionFont);
                }
            }

            // version
            uint version = ConverterEnvironment.PlatformDetails.InputFontVersion;
            if (Header.Version != version)
            {
                throw GlCm.ErrMsg(
                        ErrorType.Font,
                        Strings.IDS_ERR_UNSUPPORTED_VERSION,
                        LibFormat.ExtensionFont,
                        GetVersionString(Header.Version),
                        GetVersionString(version));
            }

            // file size
            // nothing to do

            // header size
            if (Header.HeaderSize != Runtime.BinaryFileHeader.Length)
            {
                throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_UNEXPECTED_HEADER_SIZE, LibFormat.ExtensionFont, Header.HeaderSize, Runtime.BinaryFileHeader.Length);
            }

            // data blocks
            if (Header.DataBlocks < 2)
            {
                throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_TOO_FEW_BLOCKS, Header.DataBlocks);
            }
        }

        public void AddCMapBlock(
                uint ccodeBegin,
                uint ccodeEnd,
                FontMapMethod mappingMethod,
                ushort[] data)
        {
            var map = new Runtime.FontCodeMap();
            map.CodeBegin = ccodeBegin;
            map.CodeEnd = ccodeEnd;
            map.MappingMethod = (ushort)mappingMethod;

            var mapEx = new FontCodeMapEx();
            mapEx.Body = map;
            mapEx.MapInfo = data;

            SetBlock(Runtime.RtConsts.BinBlockSigCMAP, mapEx);
        }

        private static string GetVersionString(uint version)
        {
            return string.Format(
                "{0}.{1}.{2}.{3}",
                        (version >> 24) & 0xFF,
                        (version >> 16) & 0xFF,
                        (version >> 8) & 0xFF,
                        (version >> 0) & 0xFF);
        }

        /*
            フォントデータアクセサ
         */
        private ushort GetGlyphIndex(FontCodeMapEx mapEx, uint c)
        {
            ushort index = Runtime.RtConsts.InvalidGlyphIndex;
            var map = mapEx.Body;

            switch ((FontMapMethod)map.MappingMethod)
            {
            case FontMapMethod.Direct:
                {
                    ushort offset = mapEx.MapInfo[0];
                    index = (ushort)(c - map.CodeBegin + offset);
                }

                break;

            case FontMapMethod.Table:
                {
                    int table_index = (int)c - (int)map.CodeBegin;

                    index = mapEx.MapInfo[table_index];
                }

                break;

            case FontMapMethod.Scan:
                {
                    var entries = mapEx.Scan.Entries;
                    int st = 0;
                    int ed = mapEx.Scan.Body.Num - 1;

                    while (st <= ed)
                    {
                        int md = st + ((ed - st) / 2);

                        if (entries[md].Code < c)
                        {
                            st = md + 1;
                        }
                        else if (c < entries[md].Code)
                        {
                            ed = md - 1;
                        }
                        else
                        {
                            index = entries[md].Index;
                            break;
                        }
                    }
                }

                break;

            default:
                Debug.Assert(false);
                break;
            }

            return index;
        }

        public class BitArray : List<uint>
        {
        }

        public class SheetSetList : List<SheetSet>
        {
        }

        public class SheetSet
        {
            private readonly BitArray useSheets = new BitArray();
            private readonly BitArray useCWDH = new BitArray();
            private readonly BitArray useCMAP = new BitArray();

            public SheetSet(string name)
            {
                this.Name = name;
            }

            public string Name { get; private set; }

            public BitArray UseSheets
            {
                get { return this.useSheets; }
            }

            public BitArray UseCWDH
            {
                get { return this.useCWDH; }
            }

            public BitArray UseCMAP
            {
                get { return this.useCMAP; }
            }
        }
    }

    public class FontGlyphGroupsEx : IBinarizable
    {
        public Runtime.FontGlyphGroups Body { get; set; }

        public G2dFont.SheetSetList SheetSetList { get; set; }

        public IntArray CompSheetSizes { get; set; }

        public int[] SizeCWDH { get; set; }

        public int[] SizeCMAP { get; set; }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            // 読むパターンはないので、ここに到達することはありません。
            Debug.Assert(false);
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            Debug.Assert(this.SheetSetList.Count == this.Body.NumSet);

            var startPos = (int)bw.BaseStream.Position;
            bw.Write(this.Body);

            var offsetSizeSheet = this.CalcOffsetToSizeSheets();

            // nameOffsets に関しては直前にはバイナリファイルヘッダと
            // 自身のブロックヘッダのみとしてグローバルオフセットも計算してしまう。

            // nameOffsets
            var nameOffset =
                Runtime.BinaryFileHeader.Length
                + Runtime.BinaryBlockHeader.Length
                + offsetSizeSheet
                + this.CalcSizeTable();
            foreach (var set in this.SheetSetList)
            {
                bw.Write((ushort)nameOffset);
                nameOffset += set.Name.Length + 1;
            }

            // sizeSheets
            bw.Seek(startPos + offsetSizeSheet, SeekOrigin.Begin);

            Debug.Assert(this.Body.NumSheet == this.CompSheetSizes.Count);
            foreach (var val in this.CompSheetSizes)
            {
                uint temp = (uint)val;
                bw.Write(GlCm.ROUND_UP(temp, 4));
            }

            // sizeCWDH
            Debug.Assert(this.Body.NumCWDH == this.SizeCWDH.Length);
            foreach (var val in this.SizeCWDH)
            {
                uint temp = (uint)val;
                bw.Write(temp);
            }

            // sizeCMAP
            Debug.Assert(this.Body.NumCMAP == this.SizeCMAP.Length);
            foreach (var val in this.SizeCMAP)
            {
                uint temp = (uint)val;
                bw.Write(temp);
            }

            // useSheets
            var numSheetFlags = GlCm.DIV_UP(this.Body.NumSheet, 32);
            foreach (var set in this.SheetSetList)
            {
                Debug.Assert(numSheetFlags == set.UseSheets.Count);
                WriteArray(bw, set.UseSheets);
            }

            // useCWDH
            var numCWDHFlags = GlCm.DIV_UP(this.Body.NumCWDH, 32);
            foreach (var set in this.SheetSetList)
            {
                Debug.Assert(numCWDHFlags == set.UseCWDH.Count);
                WriteArray(bw, set.UseCWDH);
            }

            // useCMAP
            var numCMAPFlags = GlCm.DIV_UP(this.Body.NumCMAP, 32);
            foreach (var set in this.SheetSetList)
            {
                Debug.Assert(numCMAPFlags == set.UseCMAP.Count);
                WriteArray(bw, set.UseCMAP);
            }

            // names
            foreach (var set in this.SheetSetList)
            {
                WriteName(bw, set.Name);
            }
        }

        public int GetBlockContentsSize(int offset)
        {
            var sizeNames = this.SheetSetList.Sum(set => set.Name.Length + 1);

            return this.CalcOffsetToSizeSheets()
                + this.CalcSizeTable()
                + sizeNames;
        }

        private static void WriteArray(ByteOrderBinaryWriter bw, G2dFont.BitArray bitArray)
        {
            foreach (var value in bitArray)
            {
                bw.Write(value);
            }
        }

        private static void WriteName(ByteOrderBinaryWriter bw, string str)
        {
            var bytes = Encoding.ASCII.GetBytes(str + '\0');
            bw.Write(bytes, 0, bytes.Length);
        }

        private int CalcOffsetToSizeSheets()
        {
            var size = Runtime.FontGlyphGroups.Length + (sizeof(ushort) * this.Body.NumSet);
            return GlCm.ROUND_UP(size, 4);
        }

        private int CalcSizeTable()
        {
            var numSheetFlags = GlCm.DIV_UP(this.Body.NumSheet, 32);
            var numCWDHFlags = GlCm.DIV_UP(this.Body.NumCWDH, 32);
            var numCMAPFlags = GlCm.DIV_UP(this.Body.NumCMAP, 32);

            var result = (this.Body.NumSheet + this.Body.NumCWDH + this.Body.NumCMAP) * sizeof(uint);
            result += (numSheetFlags + numCWDHFlags + numCMAPFlags) * this.Body.NumSet * sizeof(uint);
            return result;
        }
    }

    public class FontTextureGlyphEx : IBinarizable
    {
        public static int GlyphDataAlignSize = ConverterEnvironment.TextureDataAlignSize;

        private Runtime.FontTextureGlyph body;

        public Runtime.FontTextureGlyph Body
        {
            get { return this.body; }
            set { this.body = value; }
        }

        public byte[] SheetImage { get; set; }

        public int SheetImageSize { get; set; }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            var contentsEnd = br.BaseStream.Position + size;
            br.Read(out this.body);

            br.Seek((int)this.body.SheetImage, SeekOrigin.Begin);
            this.SheetImage = new byte[contentsEnd - (int)this.body.SheetImage];
            br.Read(this.SheetImage, 0, this.SheetImage.Length);
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            var bodyEnd =
                (int)bw.BaseStream.Position
                + Runtime.FontTextureGlyph.Length;
            var dataBegin = GlCm.ROUND_UP(bodyEnd, GlyphDataAlignSize);

            this.body.SheetImage = (uint)dataBegin;
            bw.Write(this.body);

            bw.Seek(dataBegin, SeekOrigin.Begin);

            bw.Write(this.SheetImage, 0, this.SheetImageSize);
        }

        public int GetBlockContentsSize(int offset)
        {
            var dataOffs = GlCm.ROUND_UP(offset + Runtime.FontTextureGlyph.Length, GlyphDataAlignSize);
            return dataOffs + this.SheetImageSize - offset;
        }
    }

    public class FontWidthEx : IBinarizable
    {
        private Runtime.FontWidth body;

        public Runtime.FontWidth Body
        {
            get { return this.body; }
            set { this.body = value; }
        }

        public FontWidthEx Next { get; set; }

        public Runtime.CharWidths[] WidthTable { get; set; }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            br.Read(out this.body);

            this.WidthTable = new Runtime.CharWidths[this.body.IndexEnd + 1 - this.body.IndexBegin];
            for (int i = 0; i < this.WidthTable.Length; ++i)
            {
                br.Read(out this.WidthTable[i]);
            }
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            bw.Write(this.body);

            foreach (var width in this.WidthTable)
            {
                bw.Write(width);
            }
        }

        public int GetBlockContentsSize(int offset)
        {
            return Runtime.FontWidth.Length + (this.WidthTable.Length * Runtime.CharWidths.Length);
        }
    }

    public class FontCodeMapEx : IBinarizable
    {
        private Runtime.FontCodeMap body;

        public Runtime.FontCodeMap Body
        {
            get { return this.body; }
            set { this.body = value; }
        }

        public FontCodeMapEx Next { get; set; }

        public ushort[] MapInfo { get; set; }

        public CMapInfoScanEx Scan { get; set; }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            br.Read(out this.body);

            switch ((FontMapMethod)this.body.MappingMethod)
            {
            case FontMapMethod.Direct:
                {
                    var val = br.ReadUInt16();
                    this.MapInfo = new ushort[] { val };
                }

                break;

            case FontMapMethod.Table:
                {
                    int numEntry = (int)this.body.CodeEnd - (int)this.body.CodeBegin + 1;

                    this.MapInfo = new ushort[numEntry];

                    for (int i = 0; i < numEntry; i++)
                    {
                        this.MapInfo[i] = br.ReadUInt16();
                    }
                }

                break;

            case FontMapMethod.Scan:
                {
                    Runtime.CMapInfoScan body;
                    br.Read(out body);

                    var ws = new CMapInfoScanEx();
                    ws.Body = body;
                    ws.Entries = new Runtime.CMapScanEntry[ws.Body.Num];

                    for (int i = 0; i < ws.Body.Num; i++)
                    {
                        br.Read(out ws.Entries[i]);
                    }

                    this.Scan = ws;
                }

                break;

            default:
                Debug.Assert(false);
                break;
            }
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            bw.Write(this.body);

            if (this.Scan != null)
            {
                bw.Write(this.Scan.Body);
                foreach (var ent in this.Scan.Entries)
                {
                    bw.Write(ent);
                }
            }
            else
            {
                foreach (var map in this.MapInfo)
                {
                    bw.Write(map);
                }
            }
        }

        public int GetBlockContentsSize(int offset)
        {
            return Runtime.FontCodeMap.Length
                + (this.Scan != null ?
                    Runtime.CMapInfoScan.Length + (Runtime.CMapScanEntry.Length * this.Scan.Entries.Length) :
                    this.MapInfo.Length * sizeof(ushort));
        }
    }

    public class FontKerningTable : IBinarizable
    {
        private KerningPair[] kerningPairs;
        private SortedList<uint, SortedList<uint, short>> tableContents;

        public FontKerningTable(KerningPair[] pairs)
        {
            this.kerningPairs = pairs;

            tableContents = new SortedList<uint, SortedList<uint, short>>();

            foreach (KerningPair pair in pairs)
            {
                SortedList<uint, short> list = null;
                if (tableContents.ContainsKey(pair.First))
                {
                    list = tableContents[pair.First];
                }
                else
                {
                    list = new SortedList<uint, short>();
                    tableContents.Add(pair.First, list);
                }
                list.Add(pair.Second, (short)pair.KernAmount);
            }
        }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            // 現状では必要ないので実装しない
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            // まず1つ目のテーブルに入れるオフセットを計算する
            var offsetTable = new System.Collections.Hashtable();
            int offset = sizeof(ushort) + sizeof(ushort) + tableContents.Count * 8;

            foreach (KeyValuePair<uint, SortedList<uint, short>> pair in tableContents)
            {
                offsetTable.Add(pair.Key, offset);
                offset += sizeof(ushort) + sizeof(ushort) + pair.Value.Count * 8;
            }

            //-------------- データの書き込み
            // 一文字目の数
            bw.Write((ushort)tableContents.Count);
            bw.Write((ushort)0); // Padding

            // 一つ目のテーブル
            foreach (KeyValuePair<uint, SortedList<uint, short>> pair in tableContents)
            {
                bw.Write(pair.Key);
                bw.Write((uint)((int)offsetTable[pair.Key]));
            }

            // 二つ目のテーブル
            foreach (KeyValuePair<uint, SortedList<uint, short>> pair in tableContents)
            {
                bw.Write((ushort)pair.Value.Count);
                bw.Write((ushort)0); // Padding
                foreach (KeyValuePair<uint, short> second_pair in pair.Value)
                {
                    bw.Write(second_pair.Key);
                    bw.Write(second_pair.Value);
                    bw.Write((ushort)0); // Padding
                }
            }
        }

        public int GetBlockContentsSize(int offset)
        {
            int size = sizeof(ushort) + sizeof(ushort) + tableContents.Count * 8;

            foreach (KeyValuePair<uint, SortedList<uint, short>> pair in tableContents)
            {
                size += sizeof(ushort) + sizeof(ushort) + pair.Value.Count * 8;
            }

            return size;
        }
    }

    public class FontInformationEx : IBinarizable
    {
        private Runtime.FontInformation body;

        public Runtime.FontInformation Body
        {
            get { return this.body; }
            set { this.body = value; }
        }

        public FontTextureGlyphEx Glyph { get; set; }

        public FontWidthEx Width { get; set; }

        public FontCodeMapEx Map { get; set; }

        public void Read(ByteOrderBinaryReader br, int size)
        {
            br.Read(out this.body);
        }

        public void Write(ByteOrderBinaryWriter bw)
        {
            bw.Write(this.body);
        }

        public int GetBlockContentsSize(int offset)
        {
            return Runtime.FontInformation.Length;
        }
    }

    public class CMapInfoScanEx
    {
        public Runtime.CMapInfoScan Body { get; set; }

        public Runtime.CMapScanEntry[] Entries { get; set; }
    }
}
